/*!
 * \file       dia_SecurityLevelALD.cpp
 *
 * \brief      Security level that interacts with ALD
 *
 * \details    Class dia_SecurityLevelALD provides a seed/key based method using
 *             the ALD (Authorization Level Daemon) protocol. A seed request is called
 *             at the ALD and is returned to the tester. The tester forwards the seed
 *             to the smart card server for encryption and retrieves the corresponding
 *             key. The key is send to the ECU. The ECU encrypts the stored seed and
 *             compares the result with the received key. If the values are matching
 *             access for the security level is granted to the tester.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSecurity
 *
 * \copyright  (c) 2015-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */


#ifndef __INCLUDED_DIA_SECURITY_LEVEL_ALD__
#include <common/framework/security/dia_SecurityLevelALD.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include <common/framework/sysadapters/dia_SystemAdapterFacade.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#define DIA_C_U8_ALD_LEVEL_UNKNOWN                    ((tU8) 0xFF)
#define DIA_C_U8_ALD_LEVEL_ALL_INTERFACES_LOCKED      ((tU8) 0x00)

//-----------------------------------------------------------------------------

dia_SecurityLevelALD::dia_SecurityLevelALD ( dia_SecurityLevelConfiguration& config )
   : dia_SecurityLevel(DIA_NAME_SECURITY_LEVEL_ALD,config),
     mActiveLevel(DIA_C_U8_ALD_LEVEL_UNKNOWN),
     mKeyID(0x0000),
     mIsPassedKeyValid(false)
{
   mNeedToRunSeedKeyPolicy = true;
   (void) setSysAdapterListener<dia_IAuthorizationLevelListener>(this);
}

//-----------------------------------------------------------------------------

dia_SecurityLevelALD::~dia_SecurityLevelALD ( void )
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IAuthorizationLevelListener>(this);
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SecurityLevelALD::~dia_SecurityLevelALD !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelALD::getInitializationVector ( std::vector<tU8>& initVector )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::getInitializationVector(std::vector<tU8>&)");

   initVector.clear();
   if ( (dia_getProperty(DIA_PROP_CM_ALD_SEED_INPUT_DATA,initVector) != DIA_SUCCESS) || (!initVector.size()) )
   {
      DIA_TR_ERR("##### FAILED TO CREATE INITIALIZATION VECTOR: DIA_PROP_CM_ALD_SEED_INPUT_DATA NOT AVAILABLE #####");
      return DIA_FAILED;
   }

   DIA_TR_INF("INITIALIZATION VECTOR: %s", dia::utils::bin2str(initVector,' ').c_str());

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelALD::getSeed ( void )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::getSeed()");

   bool errorDetected = true;

   std::vector<tU8> initVector;
   if ( getInitializationVector(initVector) == DIA_SUCCESS )
   {
      dia_IAuthorizationLevel* pAuthLevel = 0;
      if ( (querySysAdapterInterface<dia_IAuthorizationLevel>(&pAuthLevel) == DIA_SUCCESS) && pAuthLevel )
      {
         (void) setSysAdapterListener<dia_IAuthorizationLevelListener>(this);
         if ( pAuthLevel->requestSeed(initVector) == DIA_SUCCESS )
         {
            DIA_TR_ERR("dia_SecurityLevelALD::getSeed --- getSeed success.");
            errorDetected = false;
         }
      }
      else
      {
         DIA_TR_ERR("dia_SecurityLevelALD::getSeed --- querySysAdapterInterface FAILED!!!!");
      }
   }

   if ( errorDetected )
   {
      DIA_TR_ERR("dia_SecurityLevelALD::getSeed --- FAILED TO REQUEST SEED AT ALD !!");
      return DIA_FAILED;
   }

   return acceptEvent(dia_SecurityLevelFSM::evSeedRequested, 0 );
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelALD::acceptKey ( std::vector<tU8>& keyValue )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::acceptKey");

   bool errorDetected = true;

   DIA_TR_INF("RECEIVED KEY: %s", dia::utils::bin2str(keyValue,' ').c_str());

   // forwarding key
   dia_IAuthorizationLevel* pAuthLevel = 0;
   if ( (querySysAdapterInterface<dia_IAuthorizationLevel>(&pAuthLevel) == DIA_SUCCESS) && pAuthLevel )
   {
      (void) setSysAdapterListener<dia_IAuthorizationLevelListener>(this);
      if ( pAuthLevel->provideKey(keyValue) == DIA_SUCCESS )
      {
         DIA_TR_ERR("dia_SecurityLevelALD::acceptKey --- sendKey success.");
         errorDetected = false;
      }
   }
   else
   {
      DIA_TR_ERR("dia_SecurityLevelALD::acceptKey --- querySysAdapterInterface FAILED !");
   }

   // always set to false, as also in good case we have to wait for the ALD response
   mIsPassedKeyValid = false;

   // state machine handling
   if ( errorDetected )
   {
      DIA_TR_ERR("dia_SecurityLevelALD::acceptKey --- FAILED !!");
      mErrorCode = DIA_FAILED;
   }
   else
   {
      mErrorCode = acceptEvent(dia_SecurityLevelFSM::evKeyReceived,(void*) &keyValue,DIA_E_RESPONSE_PENDING);
   }

   return mErrorCode;
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelALD::vOnSeed ( const std::vector<tU8>& seedData )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::vOnSeed()");

   DIA_TR_INF("RECEIVED SEED: %s", dia::utils::bin2str(seedData,' ').c_str());

   // store seed data in seed attribute
   initializeSeed();
   mSeed = seedData;

   (void) acceptEvent(dia_SecurityLevelFSM::evOnSeedAvailable,0);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelALD::vOnKeyValidationResult ( tDiaResult resultCode )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::vOnKeyValidationResult()");


   mIsPassedKeyValid = false;
   if ( resultCode == DIA_SUCCESS )
   {
      mErrorCode = DIA_E_NO_ERROR;
      mIsPassedKeyValid = true;
      loadLockStatus();
   }
   else
   {
      mErrorCode = resultCode;
      DIA_TR_INF("dia_SecurityLevelALD::vOnKeyValidationResult Set mErrorCode=0x%08X.", mErrorCode);
   }

   DIA_TR_INF("dia_SecurityLevelALD::vOnKeyValidationResult = %s", ((mIsPassedKeyValid) ? "OK" : "NOK"));
   (void) acceptEvent(dia_SecurityLevelFSM::evOnKeyValidationDone,0,mErrorCode);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelALD::vOnLevel ( tU16 levelID )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::vOnLevel()");

   DIA_TR_INF("dia_SecurityLevelALD::vOnLevel: level = 0x%04x (mActiveLevel = 0x%04x)", levelID,mActiveLevel);

   if ( mActiveLevel == levelID) return;

   mActiveLevel = (tU8) levelID;
   // evaluate the ALD security level
   (void) acceptEvent(dia_SecurityLevelFSM::evOperationModeUpdate,0);
   loadLockStatus();
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelALD::vOnLevelRequested ( tU16 newLevelID )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::vOnLevelRequested()");

   DIA_TR_INF("dia_SecurityLevelALD::vOnLevelRequested: level = 0x%04x (mActiveLevel = 0x%04x)", newLevelID,mActiveLevel);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelALD::vOnLevelChanged ( tU16 newLevelID )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::vOnLevelChanged()");

   DIA_TR_INF("dia_SecurityLevelALD::vOnLevelChanged: level = 0x%04x (mActiveLevel = 0x%04x)", newLevelID,mActiveLevel);

   mActiveLevel = (tU8) newLevelID;
   // evaluate the ALD security level
   (void) acceptEvent(dia_SecurityLevelFSM::evOperationModeUpdate,0);
   loadLockStatus();
}

//------------------------------------------------------------------------------

bool
dia_SecurityLevelALD::isKeyInvalid ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::isKeyInvalid()");

   DIA_TR_INF("dia_SecurityLevelALD::isKeyInvalid => mIsPassedKeyValid=%d", mIsPassedKeyValid);

   return !mIsPassedKeyValid;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelALD::vFsmSendSeed ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::vFsmSendSeed");

   std::vector<tU8> seedValue;

   std::vector<tU8>::iterator iter = mSeed.begin();
   for ( ; iter != mSeed.end(); iter++ )
   {
      seedValue.push_back(*iter);
   }

   notifySeedResult(DIA_SUCCESS,seedValue);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelALD::vFsmHandleActivation ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::vFsmHandleActivation");

   bool isCMDiagnosisUnlocked = false;
   if ( dia_getProperty(DIA_PROP_FEATURE_ALD_CM_DIAGNOSIS_ENABLED,isCMDiagnosisUnlocked) != DIA_SUCCESS )
   {
      DIA_TR_ERR("##### ALD SECURITY LEVEL: CM DIAGNOSIS LOCKED (UNABLE TO READ CONFIG MANAGER PROPERTIES #####");
      return;
   }

   DIA_TR_INF("##### ALD SECURITY LEVEL: mActiveLevel = 0x%04x",mActiveLevel);
   DIA_TR_INF("##### ALD SECURITY LEVEL: CM DIAGNOSIS \"%s\" #####", (isCMDiagnosisUnlocked) ? "ENABLED" : "DISABLED");

   if ( !isCMDiagnosisUnlocked )
   {
      // ALD has disabled the CM diagnosis feature
      (void) acceptEvent(dia_SecurityLevelFSM::evDeactivate,0);
   }
   else
   {
      // ALD has enabled the CM diagnosis feature
      (void) acceptEvent(dia_SecurityLevelFSM::evActivate,0);
   }
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelALD::vFsmHandleFailedActivation ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::vFsmHandleFailedActivation");

   bool isCMDiagnosisUnlocked = false;
   if ( dia_getProperty(DIA_PROP_FEATURE_ALD_CM_DIAGNOSIS_ENABLED,isCMDiagnosisUnlocked) != DIA_SUCCESS )
   {
      DIA_TR_ERR("##### ALD SECURITY LEVEL: CM DIAGNOSIS LOCKED (UNABLE TO READ CONFIG MANAGER PROPERTIES #####");
      return;
   }

   DIA_TR_INF("##### ALD SECURITY LEVEL: mActiveLevel = 0x%04x",mActiveLevel);
   DIA_TR_INF("##### ALD SECURITY LEVEL: CM DIAGNOSIS \"%s\" #####", (isCMDiagnosisUnlocked) ? "ENABLED" : "DISABLED");

   if ( !isCMDiagnosisUnlocked )
   {
      // ALD has disabled the CM diagnosis feature
      (void) acceptEvent(dia_SecurityLevelFSM::evDeactivate,0);
   }
//   else
//   {
//      // ALD has enabled the CM diagnosis feature
//      (void) acceptEvent(dia_SecurityLevelFSM::evActivate,0);
//   }
//
//   (void) acceptEvent(dia_SecurityLevelFSM::evDeactivate,0);
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelALD::loadLockStatus ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::loadLockStatus()");

   //
   // ALD maintains file switches to indicate whether production diagnosis shall be enabled or disabled (permanently/temporary)
   //

   bool isCMDiagnosisUnlocked = false;
   if ( dia_getProperty(DIA_PROP_FEATURE_ALD_CM_DIAGNOSIS_ENABLED,isCMDiagnosisUnlocked) != DIA_SUCCESS )
   {
      DIA_TR_ERR("##### ALD SECURITY LEVEL: CM DIAGNOSIS LOCKED (UNABLE TO READ CONFIG MANAGER PROPERTIES #####");
      return;
   }

   DIA_TR_INF("##### ALD SECURITY LEVEL: mActiveLevel = 0x%04x",mActiveLevel);
   DIA_TR_INF("##### ALD SECURITY LEVEL: CM DIAGNOSIS \"%s\" #####", (isCMDiagnosisUnlocked) ? "ENABLED" : "DISABLED");

   if ( isCMDiagnosisUnlocked )
   {
      // ALD has enabled the CM diagnosis feature
      if ( mActiveLevel == DIA_C_U8_ALD_LEVEL_UNKNOWN )
      {
         DIA_TR_INF("##### CM DIAGNOSIS FEATURE ENABLED BY ALD BUT YET NO CONNECTION TO ALD ESTABLISHED");
         mIsDisabled = false;
      }
      else if ( mActiveLevel == DIA_C_U8_ALD_LEVEL_ALL_INTERFACES_LOCKED )
      {
         DIA_TR_INF("##### CM DIAGNOSIS FEATURE ENABLED BY ALD BUT ALD LEVEL IS 0");
         mIsDisabled = false;
      }
      else
      {
         (void) acceptEvent(dia_SecurityLevelFSM::evActivate,0);
      }
   }
   else
   {
      // ALD has disabled the CM diagnosis feature

      if ( mStatus == DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE )
      {
         (void) acceptEvent(dia_SecurityLevelFSM::evDeactivate,0);
      }
   }
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelALD::saveLockStatus ( dia_UID /*uid*/ ) const
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelALD::saveLockStatus()");

   // nothing to be done here as ALD has the mastership
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelALD::vFsmEvaluateOperationMode ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelALD::vFsmEvaluateOperationMode");

   loadLockStatus();

   if ( mActiveLevel != DIA_C_U8_ALD_LEVEL_UNKNOWN )
   {
      // we already have received the currently active security level
      if ( mpFSM ) mpFSM->acceptEvent(dia_SecurityLevelFSM::evOperationModeUpdate, 0 );
   }
   else
   {
     (void) setSysAdapterListener<dia_IAuthorizationLevelListener>(this);

#ifdef VARIANT_S_FTR_DISABLE_ALD_GET_ACTIVE_LEVEL
      DIA_TR_INF("dia_SecurityLevelALD::vFsmEvaluateOperationMode --- still waiting for the monitored ALD level property.");
#else
      // request level from ALD
      dia_IAuthorizationLevel* pAuthLevel = 0;
      if ( (querySysAdapterInterface<dia_IAuthorizationLevel>(&pAuthLevel) == DIA_SUCCESS) && pAuthLevel )
      {
         if ( pAuthLevel->getActiveLevel() == DIA_SUCCESS )
         {
            DIA_TR_ERR("dia_SecurityLevelALD::vFsmEvaluateOperationMode --- successfully requested active security level at ALD.");
         }
      }
      else
      {
         DIA_TR_ERR("dia_SecurityLevelALD::acceptKey --- querySysAdapterInterface FAILED !");
      }
#endif
   }
}

