/*!
 * \file       dia_SecurityLevel.cpp
 *
 * \brief      Abstract base class for all kinds of security levels
 *
 * \details    This file contains the definition of the abstract dia_SecurityLevel
 *             base class. It provides basic services common for all types of security
 *             levels using a seed/key mechanism
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSecurity
 *
 * \copyright  (c) 2014-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SECURITY_LEVEL__
#define __INCLUDED_DIA_SECURITY_LEVEL__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include "common/framework/application/dia_Lock.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY__
#include <common/framework/security/dia_security.h>
#endif

#ifndef __INCLUDED_DIA_SECURITY_LEVEL_CONFIGURATION__
#include <common/framework/security/dia_SecurityLevelConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_SECURITY_TIMER_CLIENT__
#include <common/framework/security/dia_SecurityTimerClient.h>
#endif

#include <common/framework/fsm/generated/dia_SecurityLevelFSM.h>

// forward declarations
class dia_SecurityModule;

/**
 * \class   dia_SecurityLevel
 *
 * \brief   This class is responsible for controlling the number of engines existing
 *          in the diagnosis application
 *
 * \ingroup diaCoreSecurity
 */

class dia_SecurityLevel
   : public dia_ObjectWithUID,
     public dia_SecurityTimerClient,
     protected dia_SecurityLevelFSM::FsmBehaviour
{
   friend class dia_SecurityManager;

   DECL_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(dia_SecurityLevel);

public:
   //! class constructor
   dia_SecurityLevel ( tCString name, dia_SecurityLevelConfiguration& config /*, tU16 udsID*/ );
   //! class destructor
   virtual ~dia_SecurityLevel ( void );

   //! initialize the seed
   virtual void initialize ( void );

   //! return status of the given security level
   virtual dia_eSecurityLevelStatus getStatus ( void ) const;

   //! method used to retrieve the seed value
   virtual tDiaResult getSeed ( void /*std::vector<tU8>& seedValue*/ ) = 0;
   //! method used to deliver a key
   virtual tDiaResult acceptKey ( std::vector<tU8>& keyValue ) = 0;

   //! install a signal handler
   virtual tDiaResult addSecurityModule ( dia_SecurityModule* pSecModule );
   //! install a signal handler
   virtual tDiaResult removeSecurityModule ( dia_UID moduleID );

   //! method called by the timer after a timer tick (resolution 1s)
   virtual tBool vOnSecurityTimerTick ( void );

   virtual tDiaResult lock ( void );
   virtual tDiaResult unlock ( void );

   //! return the number of invalid keys received by this security level
   virtual tU32 getInvalidKeyCounter ( void ) const { return mInvalidKeyCounter; }

   //! called by the session control object to notify about a session change
   virtual void onSessionChange ( tU8 newSession, tU8 oldSession );

   //! get configuration of security level
   const dia_SecurityLevelConfiguration& getConfiguration ( void ) const { return mConfig; }

   bool isActive ( void ) const { return ((mStatus == DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE) ? true : false); }

protected:
   //! load the configuration
   virtual tDiaResult loadConfiguration ( void ) { return DIA_SUCCESS; }
   //! load the configuration
   virtual tDiaResult loadConfiguration ( tU32 propID_InvalidKeyCounter, tU32 propID_RemainingLockTime );
   //! load the configuration
   virtual tDiaResult storeConfiguration ( void ) { return DIA_SUCCESS; }
   //! load the configuration
   virtual tDiaResult storeConfiguration ( tU32 propID_InvalidKeyCounter, tU32 propID_RemainingLockTime );

   virtual tDiaResult activate ( void );
   virtual tDiaResult deactivate ( dia_SecurityLockLevelID lockLevelID );

   // initialize the seed
   virtual void initializeSeed ( void );

   virtual void notifySeedResult ( tDiaResult resultCode, const std::vector<tU8>& seedValue );
   virtual void notifyKeyValidationResult ( tDiaResult resultCode );

   virtual tDiaResult acceptEvent ( dia_SecurityLevelFSM::FsmEvent event, void* pArg, tDiaResult initialErrorCode = DIA_E_NO_ERROR );  //lint !e1735: checked that default param is equal to sub class default param
   virtual tDiaResult acceptEventWithoutErrorInitialization ( dia_SecurityLevelFSM::FsmEvent event, void* pArg );

protected:
   //! default class constructor without implementation to prevent usage
   dia_SecurityLevel ( void );

   //! load the lock status of the security level
   virtual void loadLockStatus ( void ) {}
   virtual tDiaResult saveLockStatus ( dia_UID /*uid*/ ) const { return DIA_SUCCESS; }

   virtual bool isEnabled ( void ) { return (!mIsDisabled); }

   //
   // FSM methods
   //
   virtual void vFsmActivate ( void* /*pArg*/ );
   virtual void vFsmActivateAtStartup ( void* /*pArg*/ );
   virtual void vFsmCalculateSeed ( void* /*pArg*/ ) {}
   virtual void vFsmCopyKeyCtrlFromNvm ( void* /*pArg*/ ) {}
   virtual void vFsmDeactivate ( void* /*pArg*/ );
   virtual void vFsmEvaluateOperationMode ( void* /*pArg*/ ) {}
   virtual void vFsmHandleActivation ( void* /*pArg*/ );
   virtual void vFsmHandleFailedActivation ( void* pArg );
   virtual void vFsmInitializeSecurityLevel ( void* pArg );
   virtual void vFsmInitializeSeedAndKeyPolicy ( void* /*pArg*/ ) {}
   virtual void vFsmInvalidAttempt ( void* /*pArg*/ );
   virtual void vFsmLock ( void* /*pArg*/ );
   virtual void vFsmSaveKeyCtrlToNvm ( void* /*pArg*/ ) {}
   virtual void vFsmSendKeyAccepted ( void* /*pArg*/ );
   virtual void vFsmSendKeyNotAccepted ( void* /*pArg*/ );
   virtual void vFsmSendSeed ( void* /*pArg*/ ) { mWasSeedRequested = true; }
   virtual void vFsmSendSeedWhenActive ( void* /*pArg*/ ) {}
   virtual void vFsmSetErrCode_InvalidKey ( void* /*pArg*/ );
   virtual void vFsmSetErrCode_InvalidLevel ( void* /*pArg*/ );
   virtual void vFsmSetErrCode_RequestSequence ( void* /*pArg*/ );
   virtual void vFsmSetErrCode_RequiredTimeDelayNotExpired ( void* /*pArg*/ );
   virtual void vFsmSetRemainingLockTime ( void* /*pArg*/ );
   virtual void vFsmSimulateCalcSeed ( void* pArg );
   virtual void vFsmSimulateKeyAccepted ( void* pArg ) { vFsmSendKeyAccepted(pArg); }
   virtual void vFsmSimulateSeedResponse ( void* pArg ) { vFsmSendSeed(pArg); }
   virtual void vFsmStartKeyTimer ( void* /*pArg*/ ) {}
   virtual void vFsmStopKeyTimer ( void* /*pArg*/ ) {}
   virtual void vFsmUnlock ( void* /*pArg*/ );
   virtual void vFsmValidateKey ( void* /*pArg*/ ) {}

   // FSM guards
   virtual bool isKeyInvalid ( void* /*pArg*/ );
   virtual bool isLevelInvalid ( void* /*pArg*/ );
   virtual bool isLockRequired ( void* /*pArg*/ );
   virtual bool isSessionControlled ( void* /*pArg*/ ) { return mIsLevelUnderSessionControl; }
   virtual bool isSimulationEnabled ( void* /*pArg*/ ) { return mIsDisabled; }
   virtual bool needToRunSeedKeyPolicy ( void* /*pArg*/ ) { return mNeedToRunSeedKeyPolicy; }
   virtual bool wasSeedRequested ( void* /*pArg*/ ) { return mWasSeedRequested; }

protected:
   //! security level configuration
   dia_SecurityLevelConfiguration& mConfig; //lint !e1725: a reference is intended here. potential problem is resolved by blocking copy construction and assignment for this class
   //! security status of this level
   dia_eSecurityLevelStatus mStatus;
   //! this flag indicates whether the level is disabled by configuration
   bool mIsDisabled;
   //! this flag indicates whether the seed was already requested by the client
   bool mWasSeedRequested;
   //! this flag indicates if the data of mSeed is valid
   bool mIsSeedValid;
   //! this flag indicates whether the security level works in simulation or not
   bool mIsLevelUnderSessionControl;
   //! this flag indicates whether the security level works in simulation or not
   bool mIsSimulationEnabled;
   //! this flag indicates that the seed and key based mechanism shall be started when a seed request is received
   bool mNeedToRunSeedKeyPolicy;
   //! vector to store the generated seed
   std::vector<tU8> mSeed;

   //! the lock state indicates if the security level is locked or not
   bool mLockState;
   //! remaining attempts to access the security level
   tU32  mRemainingAttempts;
   //! remaining locking time in seconds if the security level is locked
   tU32  mRemainingLockTime;
   //! error information for an action is reported in the following variable
   tDiaResult mErrorCode;
   //! number of invalid keys received by this security level
   tU32  mInvalidKeyCounter;

   //! pointer to the FSM object
   dia_SecurityLevelFSM::Fsm* mpFSM;

   //! repository of signal controlled via remote control
   std::map<dia_UID,dia_SecurityModule*> mModuleRep;

   // synchronization lock object.
   dia_Lock mSyncObj;
};

#endif /* __INCLUDED_DIA_SECURITY_LEVEL__ */
