/*
 * dia_RoutineCtrlManager.cpp
 *
 *  Created on: 23.08.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_MANAGER__
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE__
#include "common/framework/protocols/uds/rtctrl/dia_Routine.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_TIMER__
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlTimer.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __DIA_UNIT_TESTING__

dia_RoutineCtrlManager*
getInstanceOfRoutineControlManager ( void )
{
   return dia_RoutineCtrlManager::getInstance();
}

void
releaseInstanceOfRoutineControlManager ( void )
{
   dia_RoutineCtrlManager::deleteInstance();
}

#endif

// implementation of the singleton methods
DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_RoutineCtrlManager)

//------------------------------------------------------------------------------

dia_RoutineCtrlManager::dia_RoutineCtrlManager ( void )
   : mRoutineCtrlStatus(DIA_EN_RTCTRL_STATUS_UNKNOWN),
     mpFSM(0),
     mIsSetup(false),
     mpTimer(getInstanceOfRoutineCtrlTimer()),
     mIsResultReady(FALSE)
{
    dia_tclFnctTrace trc("dia_RoutineCtrlManager::dia_RoutineCtrlManager");
    mActiveRoutineIter = mActiveRoutines.end();
}

//------------------------------------------------------------------------------

dia_RoutineCtrlManager::~dia_RoutineCtrlManager ( void )
{
   _BP_TRY_BEGIN
   {
      releaseInstanceOfRoutineCtrlTimer();
      mpTimer = 0;
      mActiveRoutines.clear();
      mActiveRoutineIter = mActiveRoutines.end();
      mRoutineRep.clear();
      if ( mpFSM )
      {
         OSAL_DELETE mpFSM;
         mpFSM = 0;
      }
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_RoutineCtrlManager::~dia_RoutineCtrlManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::setup ( void )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   tDiaResult retCode = DIA_FAILED;

   // create the state machine object
   if ( dia_RoutineCtrlManagerFSM::Fsm::createFSM (&mpFSM,this) == true )
   {
      DIA_TR_INF( "### dia_RoutineCtrlManager state machine instantiated and initialized ###");
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::tearDown ( void )
{
   if ( mpFSM )
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::addRoutine ( dia_Routine* pRoutine )
{
   if ( !pRoutine ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;
   dia_UID uid = pRoutine->getUID();

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter == mRoutineRep.end() )
   {
      mRoutineRep[uid] = pRoutine;
      DIA_TR_INF( "Added Routine to RtCtrlMgr: %s UID: %08x", pRoutine->getName(), uid);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::removeRoutine ( dia_UID uid )
{
   tDiaResult retCode = DIA_E_NOT_FOUND;

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      mRoutineRep.erase(uid);
      DIA_TR_INF( "Removed Routine from RtCtrlMgr: UID: %08x", uid);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_RoutineCtrlManager::numberOfRoutines ( void ) const
{
   return (tU16) mRoutineRep.size();
}

//------------------------------------------------------------------------------

dia_eRoutineStatus
dia_RoutineCtrlManager::getRoutineControlStatus ( void )
{
    dia::ScopeTrace trc(__PRETTY_FUNCTION__);

    // update the current status
    vUpdateRoutineCtrlStatus();

    // return the current status
    return mRoutineCtrlStatus;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::addRoutineControlListener ( dia_UID uid, dia_IRoutineControlListener* pListener )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   dia_Routine* pRoutine = 0;
   if ( (queryRoutine(uid,&pRoutine) != DIA_SUCCESS) || (!pRoutine) ) return DIA_E_NOT_FOUND;

   tDiaResult retCode = DIA_FAILED;
   std::map<dia_UID,dia_IRoutineControlListener*>::iterator iter = mListenerRep.find(uid);
   if ( iter == mListenerRep.end() )
   {
      DIA_TR_INF("ADDING ROUTINE CONTROL LISTENER FOR ROUTINE \"%s\"", pRoutine->getName());
      mListenerRep[uid] = pListener;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::removeRoutineControlListener ( dia_UID uid, dia_IRoutineControlListener* pListener )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   dia_Routine* pRoutine = 0;
   if ( (queryRoutine(uid,&pRoutine) != DIA_SUCCESS) || (!pRoutine) ) return DIA_E_NOT_FOUND;

   tDiaResult retCode = DIA_E_NOT_FOUND;
   std::map<dia_UID,dia_IRoutineControlListener*>::iterator iter = mListenerRep.find(uid);
   if ( iter != mListenerRep.end() )
   {
      mListenerRep.erase(uid);
      DIA_TR_INF("REMOVED ROUTINE CONTROL LISTENER FOR ROUTINE \"%s\"", pRoutine->getName());
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

//! update the RoutineCtrl status
void
dia_RoutineCtrlManager::vUpdateRoutineCtrlStatus ( void )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   tU8 u8RoutinesActive   = 0;
   tU8 u8RoutinesTimedOut = 0;
   tU8 u8RoutinesAborted  = 0;

   std::map<dia_UID,dia_Routine*>::iterator iter = mRoutineRep.begin();
   for ( ; iter != mRoutineRep.end(); ++iter )
   {
      dia_Routine* pRoutine = iter->second;

      if ( pRoutine != 0 )
      {
         switch ( pRoutine->getStatus() )
         {
         case DIA_EN_RTCTRL_STATUS_ABORTED:
            u8RoutinesAborted++;
            break;
         case DIA_EN_RTCTRL_STATUS_TIMED_OUT:
            u8RoutinesTimedOut++;
            break;
         case DIA_EN_RTCTRL_STATUS_IN_PROGRESS:
            u8RoutinesActive++;
            break;
         default:
            break;
         }  //lint !e788: not all items intentionally used within defaulted switch
      }
   }

   if ( u8RoutinesActive )
   {
      mRoutineCtrlStatus = DIA_EN_RTCTRL_STATUS_IN_PROGRESS;
   }
   else if ( u8RoutinesTimedOut )
   {
      mRoutineCtrlStatus = DIA_EN_RTCTRL_STATUS_TIMED_OUT;
   }
   else if ( u8RoutinesAborted )
   {
      mRoutineCtrlStatus = DIA_EN_RTCTRL_STATUS_ABORTED;
   }
   else
   {
      mRoutineCtrlStatus = DIA_EN_RTCTRL_STATUS_IDLE;
   }
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::startRoutine ( dia_UID uid, std::vector<tU8>& params, tU8 timerValue )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("Starting Routine %d", uid);

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      dia_Routine* pRoutine = mRoutineRepIter->second;

      retCode = pRoutine->start(params,timerValue);

      // now start the timer if required
      if ( pRoutine->getMonitoringMode() != DIA_EN_RTCTRL_MONITORING_MODE_NOT_SUPPORTED )
      {
         pRoutine->setMonitoringMode(DIA_EN_RTCTRL_MONITORING_MODE_ENABLED,timerValue);
         if ( mpTimer && (timerValue > 0) && (timerValue != DIA_C_U8_UDS_RTCTRL_TIMER_VALUE_INFINITE) )
         {
            mpTimer->addTimerClient(pRoutine);
            mpTimer->start(DIA_C_U32_UDS_RTCTRL_TIMER_DEFAULT_TIMEOUT,DIA_C_U32_UDS_RTCTRL_TIMER_DEFAULT_CYCLETIME);
            DIA_TR_INF( "Timer started ...");
         }
      }

      vUpdateRoutineCtrlStatus();

      if ( mRoutineRepIter->second->bIsResultReady() )
      {
         mRoutineRepIter->second->setStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
         if ( mpFSM ) mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evRoutineActivated,mRoutineRepIter->second);
      }

   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::stopRoutine( dia_UID uid, std::vector<tU8>& params )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("Stopping Routine %d", uid);

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      // if the timer is currently active we have to stop it
      mpTimer->removeTimerClient(mRoutineRepIter->second);

      retCode = mRoutineRepIter->second->stop(params);

      vUpdateRoutineCtrlStatus();

      if ( retCode == DIA_SUCCESS && mRoutineRepIter->second->bIsResultReady() )
      {
         mRoutineRepIter->second->setStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
         if ( mpFSM ) mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evRoutineDeactivated,mRoutineRepIter->second);
      }

   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::requestRoutineResult ( dia_UID uid, std::vector<tU8>& results )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("Requesting Results from Routine %d", uid);

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      retCode = mRoutineRep[uid]->requestResult(results);
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::onRoutineTimeout ( dia_UID uid )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   tDiaResult retCode = DIA_E_NOT_FOUND;

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      dia_Routine* pRoutine = mRoutineRep[uid];

      if ( pRoutine )
      {
         // if the timer is currently active we have to stop it
         mpTimer->removeTimerClient(pRoutine);

         pRoutine->vOnServiceTimeout();
         retCode = DIA_SUCCESS;
      }
   }

   if ( mpFSM ) mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evTimeout,0);

   return retCode;
}

//------------------------------------------------------------------------------

tBool
dia_RoutineCtrlManager::bIsResultReady ( dia_UID uid )
{
   tBool retCode = FALSE;

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      retCode = mRoutineRep[uid]->bIsResultReady();
   }

   return retCode;
}

//------------------------------------------------------------------------------

bool
dia_RoutineCtrlManager::isResultReady ( dia_UID uid )
{
   bool retCode = false;

   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      retCode = mRoutineRep[uid]->isResultReady();
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::queryRoutine (  dia_UID uid, dia_Routine** ppRoutine )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( !ppRoutine ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   *ppRoutine = 0;
   mRoutineRepIter = mRoutineRep.find(uid);
   if ( mRoutineRepIter != mRoutineRep.end() )
   {
      dia_Routine* pRoutine = mRoutineRep[uid];
      if ( pRoutine )
      {
         *ppRoutine = pRoutine;
         retCode = DIA_SUCCESS;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::queryRoutineByDID ( tU16 rid, dia_Routine** ppRoutine )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( !ppRoutine ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("Searching for Routine ID: %04x", rid);

   *ppRoutine = 0;
   std::map<dia_UID,dia_Routine*>::iterator iter = mRoutineRep.begin();
   for ( ; iter != mRoutineRep.end(); iter++ )
   {
      dia_Routine* pActRoutine = iter->second;

      if ( pActRoutine && (pActRoutine->getDID() == rid) )
      {
         DIA_TR_INF("Routine with UID %08x matches Routine ID: %04x", pActRoutine->getUID(), rid);
         *ppRoutine = pActRoutine;
         retCode = DIA_SUCCESS;
         break;
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::configureSessionHandling ( tU8 currentSession, tU8 nextSession, bool stopMode )
{
   tU16 key = (tU16) ((((tU16) currentSession) << 8) + ((tU16) nextSession));
   mSessionHandlingRep[key] = stopMode;
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlManager::configureSessionHandling ( tU8 currentSession, bool stopMode )
{
   return configureSessionHandling(currentSession,DIA_C_U8_UDS_SESSION_INVALID,stopMode);
}

//------------------------------------------------------------------------------

//void
//dia_RoutineCtrlManager::vOnSessionChanged ( tU8 /*newSession*/, tU8 /*oldSession*/ )
//{
//    dia::ScopeTrace trc(__PRETTY_FUNCTION__);
//
//   if ( oldSession == newSession ) {
//      return;
//   }
//
//   if ( oldSession == DIA_C_U8_UDS_SESSION_DEFAULT )
//   {
//      // this service is not allowed in the default session, so we have nothing to do
//      return;
//   }
//
//   switch ( newSession )
//   {
//   case DIA_C_U8_UDS_SESSION_EXTENDED:
//   case DIA_C_U8_UDS_SESSION_VW_EOL:
//   case DIA_C_U8_UDS_SESSION_DEVELOPMENT:
//      // we have nothing to do
//      break;
//
//   case DIA_C_U8_UDS_SESSION_DEFAULT:
//      // ioctrls and remote control is not allowed in the default session so they have to be stopped
//      {
//          msgBuffer[2] = DIA_C_U16_UDS_SUBID_REMOTE_CONTROL_STOP >> 8;
//          msgBuffer[3] = DIA_C_U16_UDS_SUBID_REMOTE_CONTROL_STOP  & 0xFF;
//
//          // Create an object containing the UDS msg. It is deleted by dia_tclDiagSessionUds::ReadyForReqRx::OnEnter
//          dia_tclDiagMsgBufferUds* pMsgBuff = OSAL_NEW dia_tclDiagMsgBufferUds (
//              &msgBuffer[0],
//              tU16(sizeof msgBuffer),
//              vRemoteControlConfTxOk,
//              dia_tclDiagMsgBuffer::holds_internal_request
//              );
//
//          NORMAL_M_ASSERT(pMsgBuff);
//          if (pMsgBuff)
//          {
//              // Create and send a ReqRx event to the uds session queue
//              dia_tclDiagSessionUds::getInstance()->vEmit(dia_tclDiagSessionUds::tclEventReqRx(pMsgBuff));
//          }
//          else
//          {
//            // tbd
//          }
//      }
//      break;
//
//   default:
//      break;
//   }
//}

void
dia_RoutineCtrlManager::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( oldSession == newSession ) return;

   tU8 defaultSession = 0;
   tDiaResult retCode = dia_getProperty(DIA_PROP_DEFAULT_SESSION,defaultSession);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("!!! dia_RoutineCtrlManager::vOnSessionChanged: FAILED TO RETRIEVE DEFAULT SESSION VALUE (errCode = 0x%08X", retCode);
      return;
   }

   // routine controls are not allowed in default session. if we are currently in the default session we are done
   if ( oldSession == defaultSession ) return;

   // per default we don't stop active iocontrols to retain backward compatibility
   bool needToStopActiveRoutines = false;

   tU16 key = (tU16) ((((tU16) oldSession) << 8) + ((tU16) newSession));
   std::map<tU16,bool>::iterator iter = mSessionHandlingRep.find(key);
   if ( iter != mSessionHandlingRep.end() )
   {
      needToStopActiveRoutines = iter->second;
   }
   else
   {
      // we don't have a specific rule for this particular session change, so we check for a generic rule
      key  = tU16((oldSession << 8) + DIA_C_U8_UDS_SESSION_INVALID);
      iter = mSessionHandlingRep.find(key);
      if ( iter != mSessionHandlingRep.end() )
      {
         needToStopActiveRoutines = iter->second;
      }
   }

   if ( needToStopActiveRoutines && mpFSM )
   {
      mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evTerminateAllRoutines,0);
   }
}

//-----------------------------------------------------------------------------

const std::map<dia_UID,dia_Routine*>&
dia_RoutineCtrlManager::getRoutines ( void ) const
{
   return mRoutineRep;
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vOnRoutineUpdate ( dia_Routine& routine )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   DIA_TR_INF("Update Routine UID: %08x Type: %d Name: %s", routine.getUID(), routine.getType(), routine.getName());

   dia_eRoutineStatus routineStatus = routine.getStatus();

   if ( routineStatus == DIA_EN_RTCTRL_STATUS_IN_PROGRESS )
   {
      if ( mpFSM ) mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evRoutineActivated, &routine);
   }

   else if ( routineStatus == DIA_EN_RTCTRL_STATUS_ABORTED )
   {
      if ( mpFSM ) mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evRoutineDeactivated, &routine);
   }
   else
   {
      DIA_TR_INF("routineStatus = %d", routineStatus);
   }

   dia_IRoutineControlListener* pListener = OSAL_NULL;
   querySysAdapterListener<dia_IRoutineControlListener>(&pListener);
   if ( pListener )
   {
      pListener->vOnRoutineUpdate(routine);
   }
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vFsmHandleTimeout ( void* /*pArg*/ )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vFsmCollectRoutines ( void* /*pArg*/ )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   // collect the uids of all active ioctrl signals
   mActiveRoutines.clear();
   mRoutineRepIter = mRoutineRep.begin();
   for ( ; mRoutineRepIter != mRoutineRep.end(); mRoutineRepIter++ )
   {
      if ( mRoutineRepIter->second )
      {
         switch ( mRoutineRepIter->second->eGetStatus() )
         {
         case DIA_EN_RTCTRL_STATUS_IN_PROGRESS:
         case DIA_EN_RTCTRL_STATUS_ABORTED:
         case DIA_EN_RTCTRL_STATUS_TIMED_OUT:
            mActiveRoutines[mRoutineRepIter->second->getUID()] = mRoutineRepIter->second;
            break;

         default:
            break;
         }  //lint !e788: not all items intentionally used within defaulted switch
      }
   }

   // initialize the iterator we are using to traverse the list of active signals
   mActiveRoutineIter = mActiveRoutines.begin();
}
// Enum that specifies who controls the signal. Either ECU itself or the tester

//enum dia_eRoutineStatus
//{
//   DIA_EN_RTCTRL_STATUS_UNKNOWN = 0,
//   DIA_EN_RTCTRL_STATUS_IDLE,
//   DIA_EN_RTCTRL_STATUS_IN_PROGRESS,
//   DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK,
//   DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK,
//   DIA_EN_RTCTRL_STATUS_ABORTED,
//   DIA_EN_RTCTRL_STATUS_TIMED_OUT,
//   DIA_EN_RTCTRL_STATUS_COUNT
//};

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vFsmAddRoutine ( void* pArg )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( !pArg ) return;

   dia_Routine* pRoutine = (dia_Routine*) pArg;
   mActiveRoutines[pRoutine->getUID()] = pRoutine;

//   vUpdateRoutineCtrlStatus();
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vFsmRemoveRoutine ( void* pArg )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( !pArg ) return;

   dia_Routine* pRoutine = (dia_Routine*) pArg;
   std::map<dia_UID,dia_Routine*>::iterator iter = mActiveRoutines.find(pRoutine->getUID());
   if ( iter != mActiveRoutines.end() )
   {
      mActiveRoutines.erase(pRoutine->getUID());
   }

   if ( mActiveRoutines.size() ) mActiveRoutineIter = mActiveRoutines.begin();

   vUpdateRoutineCtrlStatus();
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vFsmReset ( void* /*pArg*/ )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlManager::vFsmStopNextRoutine ( void* /*pArg*/ )
{
   dia::ScopeTrace trc(__PRETTY_FUNCTION__);

   if ( mActiveRoutineIter != mActiveRoutines.end() && mActiveRoutineIter->second )
   {
      dia_Routine* pRoutine = mActiveRoutineIter->second;
      // we want to stop the identified routine
      std::vector<tU8> params;
      pRoutine->stop(params);
      if ( pRoutine->bIsResultReady() )
      {
         // we don't have to wait. operation is complete. we can directly process the next signal
         pRoutine->setStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK /*DIA_EN_RTCTRL_STATUS_ABORTED*/);
         if ( mpFSM ) mpFSM->acceptEvent(dia_RoutineCtrlManagerFSM::evRoutineDeactivated,pRoutine);
      }
   }
}

//-----------------------------------------------------------------------------

bool
dia_RoutineCtrlManager::bFsmIsNoRoutineActive ( void* /*pArg*/ )
{
   return ( mActiveRoutines.size() > 0 ) ? false : true;
}


