/*
 * dia_IOCtrlManager.cpp
 *
 *  Created on: 09.07.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_IOCTRL_MANAGER__
#include <common/framework/protocols/uds/ioctrl/dia_IOCtrlManager.h>
#endif

#ifndef __INCLUDED_DIA_IOCTRL_SIGNAL__
#include <common/framework/protocols/uds/ioctrl/dia_IOCtrlSignal.h>
#endif

#ifndef __INCLUDED_DIA_IOCTRL_TIMER__
#include <common/framework/protocols/uds/ioctrl/dia_IOCtrlTimer.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_IOCONTROL_LISTENER__
#include <common/interfaces/dia_IIOControlListener.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

using namespace dia;

#define DIA_C_U8_INVALID_CONTROL_STATE       ((tU8) 0xFF)

#ifndef __DIA_UNIT_TESTING__

dia_IOCtrlManager*
getInstanceOfIOCtrlManager ( void )
{
   return dia_IOCtrlManager::getInstance();
}

void
releaseInstanceOfIOCtrlManager ( void )
{
   return dia_IOCtrlManager::deleteInstance();
}

#endif

// implementation of the singleton methods
DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_IOCtrlManager)

//------------------------------------------------------------------------------

dia_IOCtrlManager::dia_IOCtrlManager ( void )
   : mSyncObj("dia_IOCtrlManager_LK"),
     mOperationMode(DIA_EN_IOCTRL_REMOTECONTROL_STATE_OFF),
     mIOCtrlStatus(DIA_EN_IOCTRL_STATUS_INACTIVE),
     mActiveSignalUID(0),
     mActiveSignalControlState(DIA_C_U8_INVALID_CONTROL_STATE),
     mpFSM(0),
     mIsSetup(false),
     mpTimer(getInstanceOfIOCtrlTimer())
{
    ScopeTrace trc("dia_IOCtrlManager::dia_IOCtrlManager");
    mActiveSignalIter = mActiveSignals.end();
}

//------------------------------------------------------------------------------

dia_IOCtrlManager::~dia_IOCtrlManager ( void )
{
   _BP_TRY_BEGIN
   {
      mpTimer = 0;
      mActiveSignals.clear();
      mActiveSignalIter = mActiveSignals.end();
      mSignalRep.clear();
      if ( mpFSM )
      {
         DIA_DELETE mpFSM;
         mpFSM = 0;
      }
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_IOCtrlManager::~dia_IOCtrlManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::setup ( void )
{
   ScopeTrace oTrace("dia_IOCtrlManager::setup()");

   tDiaResult retCode = DIA_FAILED;

   // create the state machine object
   if ( dia_IOCtrlManagerFSM::Fsm::createFSM (&mpFSM,this) == true )
   {
      DIA_TR_INF( "### dia_IOCtrlManager state machine instantiated and initialized ###");
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::tearDown ( void )
{
   if ( mpFSM )
   {
      DIA_DELETE mpFSM;
      mpFSM = 0;
   }

   return DIA_SUCCESS;
}


//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::addSignal ( dia_IOCtrlSignal* pIOCtrlSignal )
{
   ScopeTrace oTrace("dia_IOCtrlManager::addSignal()");

   if ( !pIOCtrlSignal ) return DIA_E_INVALID_POINTER;

   LockScope oLock(mSyncObj);

   dia_UID uid = pIOCtrlSignal->getUID();
   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
   if ( iter == mSignalRep.end() )
   {
      mSignalRep[uid] = pIOCtrlSignal;
      return DIA_SUCCESS;
   }

   return DIA_FAILED;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::removeSignal ( dia_UID uid )
{
   ScopeTrace oTrace("dia_IOCtrlManager::removeSignal()");

   LockScope oLock(mSyncObj);

   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
   if ( iter != mSignalRep.end() )
   {
      mSignalRep.erase(uid);
      return DIA_SUCCESS;
   }

   return DIA_E_NOT_FOUND;
}

//------------------------------------------------------------------------------

tU16
dia_IOCtrlManager::numberOfSignals ( void ) const
{
   LockScope oLock(mSyncObj);
   return (tU16) mSignalRep.size();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::addIOControlListener ( dia_UID uid, dia_IIOControlListener* pListener )
{
   ScopeTrace oTrace("dia_IOCtrlManager::addIOControlListener()");

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   LockScope oLock(mSyncObj);

   dia_IOCtrlSignal* pSignal = 0;
   if ( (queryIOControlSignal(uid,&pSignal) != DIA_SUCCESS) || (!pSignal) ) return DIA_E_NOT_FOUND;

   std::map<dia_UID,dia_IIOControlListener*>::iterator iter = mListenerRep.find(uid);
   if ( iter == mListenerRep.end() )
   {
      DIA_TR_INF("ADDING IOCONTROL LISTENER FOR SIGNAL \"%s\"", pSignal->getName());
      mListenerRep[uid] = pListener;
      return DIA_SUCCESS;
   }

   return DIA_FAILED;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::removeIOControlListener ( dia_UID uid, dia_IIOControlListener* pListener )
{
   ScopeTrace oTrace("dia_IOCtrlManager::removeIOControlListener()");

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   LockScope oLock(mSyncObj);

   dia_IOCtrlSignal* pSignal = 0;
   if ( (queryIOControlSignal(uid,&pSignal) != DIA_SUCCESS) || (!pSignal) ) return DIA_E_NOT_FOUND;

   std::map<dia_UID,dia_IIOControlListener*>::iterator iter = mListenerRep.find(uid);
   if ( iter != mListenerRep.end() )
   {
      mListenerRep.erase(uid);
      DIA_TR_INF("REMOVED IOCONTROL LISTENER FOR SIGNAL \"%s\"", pSignal->getName());
      return DIA_SUCCESS;
   }

   return DIA_FAILED;
}

//------------------------------------------------------------------------------

dia_eIOCtrlStatus
dia_IOCtrlManager::getIOControlStatus ( void )
{
    ScopeTrace trc("dia_IOCtrlManager::getIOControlStatus");

    // update the current status
    vUpdateIOCtrlStatus();

    // return the current status
    return mIOCtrlStatus;
}

//------------------------------------------------------------------------------

void
dia_IOCtrlManager::vUpdateIOCtrlStatus ( void )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vUpdateIOCtrlStatus()");

   tU8 u8ActiveSignals   = 0;
   tU8 u8TimedOutSignals = 0;
   tU8 u8AbortedSignals  = 0;

   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.begin();
   for ( ; iter != mSignalRep.end(); ++iter )
   {
      dia_IOCtrlSignal* pSignal = iter->second;

      if ( pSignal != 0 )
      {
         switch ( pSignal->getStatus() )
         {
         case DIA_EN_IOCTRL_STATUS_INACTIVE_ABORTED:
            u8AbortedSignals++;
            break;
         case DIA_EN_IOCTRL_STATUS_INACTIVE_TIMEOUT:
            u8TimedOutSignals++;
            break;
         case DIA_EN_IOCTRL_STATUS_ACTIVE:
            u8ActiveSignals++;
            break;
         default:
            break;
         }  //lint !e788: not all items intentionally used within defaulted switch
      }
   }

   if ( u8ActiveSignals )
   {
      mIOCtrlStatus = DIA_EN_IOCTRL_STATUS_ACTIVE;
   }
   else if ( u8TimedOutSignals )
   {
      mIOCtrlStatus = DIA_EN_IOCTRL_STATUS_INACTIVE_TIMEOUT;
   }
   else if ( u8AbortedSignals )
   {
      mIOCtrlStatus = DIA_EN_IOCTRL_STATUS_INACTIVE_ABORTED;
   }
   else
   {
      mIOCtrlStatus = DIA_EN_IOCTRL_STATUS_INACTIVE;
   }
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::freeze ( dia_UID uid )
{
   ScopeTrace oTrace("dia_IOCtrlManager::freeze(dia_UID)");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   DIA_TR_INF("Freezing Signal %d", uid);

   dia_IOCtrlSignal* pSignal = 0;
   {
      LockScope oLock(mSyncObj);

      std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
      if ( iter != mSignalRep.end() )
      {
         mActiveSignalUID = uid;
         mActiveSignalControlState = DIA_C_U8_UDS_IOCTRL_FREEZE_CURRENT_STATE;
         pSignal = mSignalRep[mActiveSignalUID];
      }
   }

   if ( pSignal )
   {
      pSignal->initializeRequest();
      retCode = pSignal->handleFreeze();
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::freeze ( dia_IOCtrlSignal& signal )
{
   ScopeTrace oTrace("dia_IOCtrlManager::freeze(dia_IOCtrlSignal&)");
   return freeze(signal.getUID());
}

//------------------------------------------------------------------------------

bool
dia_IOCtrlManager::isFrozen ( dia_UID uid )
{
   ScopeTrace oTrace("dia_IOCtrlManager::isFrozen(dia_UID)");

   bool retCode = false;

   dia_IOCtrlSignal* pSignal = 0;
   {
      LockScope oLock(mSyncObj);

      std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
      if ( iter != mSignalRep.end() )
      {
         pSignal = iter->second;
      }
   }

   if ( pSignal )
   {
      retCode = pSignal->isFrozen();
      DIA_TR_INF("Frozen state of signal \"%s\": %s", pSignal->getName(), (retCode ? "FROZEN" : "NOT FROZEN") );
   }
   else
   {
      DIA_TR_INF("SIGNAL WITH UID 0x%08X NOT FOUND !!", uid);
   }

   return retCode;
}

//------------------------------------------------------------------------------

bool
dia_IOCtrlManager::isFrozen ( dia_IOCtrlSignal& signal )
{
   ScopeTrace oTrace("dia_IOCtrlManager::isFrozen(dia_IOCtrlSignal&)");
   return isFrozen(signal.getUID());
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::returnControlToECU ( dia_UID uid )
{
   ScopeTrace oTrace("dia_IOCtrlManager::returnControlToECU(uid)");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   dia_IOCtrlSignal* pSignal = 0;
   {
      LockScope oLock(mSyncObj);

      std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
      if ( iter != mSignalRep.end() )
      {
         mActiveSignalUID = uid;
         mActiveSignalControlState = DIA_C_U8_UDS_IOCTRL_RETURN_CTRL_TO_ECU;
         pSignal = mSignalRep[uid];
      }
   }

   if ( pSignal )
   {
      DIA_TR_INF("Return-Control-To-ECU for IOCtrl signal \"%s\"", pSignal->getName());

      pSignal->initializeRequest();
      // if the timer is currently active we have to stop it
      mpTimer->removeTimerClient(pSignal);
      retCode = pSignal->handleReturnControlToECU();

      vUpdateIOCtrlStatus();

      if ( retCode == DIA_SUCCESS && pSignal->isResultReady() )
      {
         mActiveSignalUID = 0;
         pSignal->eSetStatus(DIA_EN_IOCTRL_STATUS_INACTIVE);
         if ( mpFSM ) mpFSM->acceptEvent(dia_IOCtrlManagerFSM::evSignalDeactivated,pSignal);
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::returnControlToECU ( dia_IOCtrlSignal& signal )
{
   ScopeTrace oTrace("dia_IOCtrlManager::returnControlToECU(signal)");
   return returnControlToECU(signal.getUID());
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::shortTermAdjustment ( dia_UID uid, tU8 timerValue, std::vector<tU8>* ctrlValue )
{
   ScopeTrace oTrace("dia_IOCtrlManager::shortTermAdjustment(uid,timerValue,ctrlValue)");

   tDiaResult retCode = DIA_E_NOT_FOUND;

   dia_IOCtrlSignal* pSignal = 0;
   {
      LockScope oLock(mSyncObj);

      std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
      if ( iter != mSignalRep.end() )
      {
         pSignal = mSignalRep[uid];
      }
   }

   if ( pSignal )
   {
      // initialize members, e.g. set ready flag to FALSE
      pSignal->initializeRequest(timerValue,ctrlValue);
      // forward the request to the signal object
      retCode = pSignal->handleRequest(timerValue,ctrlValue);
      if ( retCode == DIA_SUCCESS )
      {
         mActiveSignalUID = uid;
         mActiveSignalControlState = DIA_C_U8_UDS_IOCTRL_SHORT_TERM_ADJUSTMENT;

         // now start the timer if required
         if ( pSignal->getMonitoringMode() != DIA_EN_IOCTRL_MONITORING_MODE_NOT_SUPPORTED )
         {
            pSignal->setMonitoringMode(DIA_EN_IOCTRL_MONITORING_MODE_SHORT_TERM_ADJUSTMENT,timerValue);
            if ( mpTimer && (timerValue > 0) && (timerValue != IOCTRL_U8_TIMER_TIMEOUT_INFINITE) )
            {
               // if the signal is already registered as timer client, we need to remove it
               (void) mpTimer->removeTimerClient(pSignal);
               (void) mpTimer->addTimerClient(pSignal);
               mpTimer->start(DIA_C_U32_UDS_IOCTRL_TIMER_DEFAULT_TIMEOUT,DIA_C_U32_UDS_IOCTRL_TIMER_DEFAULT_CYCLETIME);
               DIA_TR_INF( "Timer started ...");
            }
         }

         vUpdateIOCtrlStatus();

         if ( pSignal->isResultReady() )
         {
            pSignal->eSetStatus(DIA_EN_IOCTRL_STATUS_ACTIVE);
            if ( mpFSM ) mpFSM->acceptEvent(dia_IOCtrlManagerFSM::evSignalActivated,pSignal);
         }
      }
   }

   DIA_TR_INF("dia_IOCtrlManager::handleRequest returned %d ==> %s", retCode, (DIA_E_NOT_FOUND==retCode? "DIA_E_NOT_FOUND": (DIA_SUCCESS==retCode? "DIA_SUCCESS": "")));

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::shortTermAdjustment ( dia_IOCtrlSignal& signal, tU8 timerValue, std::vector<tU8>* ctrlValue )
{
   ScopeTrace oTrace("dia_IOCtrlManager::shortTermAdjustment(signal,timerValue,ctrlValue)");
   return shortTermAdjustment(signal.getUID(),timerValue,ctrlValue);
}

//------------------------------------------------------------------------------

void
dia_IOCtrlManager::handleTimeout ( void )
{
   ScopeTrace oTrace("dia_IOCtrlManager::handleTimeout()");

   dia_IOCtrlSignal* pSignal = 0;
   {
      LockScope oLock(mSyncObj);

      std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(mActiveSignalUID);
      if ( iter != mSignalRep.end() )
      {
         pSignal = mSignalRep[mActiveSignalUID];
      }
   }

   if ( pSignal )
   {
      // if the timer is currently active we have to stop it
      mpTimer->removeTimerClient(mSignalRep[mActiveSignalUID]);
      // forward the timeout to the active signal object
      mSignalRep[mActiveSignalUID]->handleTimeout();
      mActiveSignalUID = 0;
      mActiveSignalControlState = DIA_C_U8_INVALID_CONTROL_STATE;
   }

   acceptEvent(dia_IOCtrlManagerFSM::evTimeout,0);
}

//------------------------------------------------------------------------------

bool
dia_IOCtrlManager::isResultReady ( dia_UID uid )
{
   bool readyFlag = false;

   LockScope oLock(mSyncObj);
   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
   if ( iter != mSignalRep.end() )
   {
      readyFlag = mSignalRep[uid]->isResultReady();
   }

   return readyFlag;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::queryIOControlSignal ( dia_UID uid, dia_IOCtrlSignal** ppSignal )
{
   ScopeTrace trc("dia_IOCtrlManager::queryIOControlSignal(dia_UID, dia_IOCtrlSignal**)");

   if ( !ppSignal ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   LockScope oLock(mSyncObj);
   *ppSignal = 0;
   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.find(uid);
   if ( iter != mSignalRep.end() )
   {
      *ppSignal = mSignalRep[uid];
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::queryIOControlSignalByDID ( tU16 udsID, dia_IOCtrlSignal** ppSignal )
{
   ScopeTrace trc("dia_IOCtrlManager::queryIOControlSignalByDID(tU16,dia_IOCtrlSignal**)");

   if ( !ppSignal ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_E_NOT_FOUND;

   LockScope oLock(mSyncObj);
   *ppSignal = 0;
   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mSignalRep.begin();
   for ( ; iter != mSignalRep.end(); iter++ )
   {
      dia_IOCtrlSignal* pActSignal = iter->second;
      if ( pActSignal && pActSignal->getDID() == udsID )
      {
         *ppSignal = pActSignal;
         retCode = DIA_SUCCESS;
         break;
      }
   }

   return retCode;
}


//------------------------------------------------------------------------------

void
dia_IOCtrlManager::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   ScopeTrace trc("dia_IOCtrlManager::vOnSessionChanged");

   if ( oldSession == newSession ) return;

   tU8 defaultSession = 0;
   tDiaResult retCode = dia_getProperty(DIA_PROP_DEFAULT_SESSION,defaultSession);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("!!! dia_IOCtrlManager::vOnSessionChanged: FAILED TO RETRIEVE DEFAULT SESSION VALUE (errCode = 0x%08X", retCode);
      return;
   }

   // iocontrols are not allowed in default session. if we are currently in the default session we are done
   if ( oldSession == defaultSession ) return;

   // per default we don't stop active iocontrols to retain backward compatibility
   bool needToStopActiveSignals = false;

   tU16 key = (tU16) ((((tU16) oldSession) << 8) + ((tU16) newSession));

   {
      LockScope oLock(mSyncObj);
      std::map<tU16,bool>::iterator iter = mSessionHandlingRep.find(key);
      if ( iter != mSessionHandlingRep.end() )
      {
         needToStopActiveSignals = iter->second;
      }
      else
      {
         // we don't have a specific rule for this particular session change, so we check for a generic rule
         key  = tU16((oldSession << 8) + DIA_C_U8_UDS_SESSION_INVALID);
         iter = mSessionHandlingRep.find(key);
         if ( iter != mSessionHandlingRep.end() )
         {
            needToStopActiveSignals = iter->second;
         }
      }
   }

   if ( needToStopActiveSignals )
   {
      acceptEvent(dia_IOCtrlManagerFSM::evTerminateAllSignals,0);
   }
}

//-----------------------------------------------------------------------------

const std::map<dia_UID,dia_IOCtrlSignal*>&
dia_IOCtrlManager::getSignals ( void ) const
{
   return mSignalRep;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::configureSessionHandling ( tU8 currentSession, tU8 nextSession, bool stopMode )
{
   LockScope oLock(mSyncObj);
   tU16 key = (tU16) ((((tU16) currentSession) << 8) + ((tU16) nextSession));
   mSessionHandlingRep[key] = stopMode;
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlManager::configureSessionHandling ( tU8 currentSession, bool stopMode )
{
   return configureSessionHandling(currentSession,DIA_C_U8_UDS_SESSION_INVALID,stopMode);
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::acceptEvent ( dia_IOCtrlManagerFSM::FsmEvent event, void* pArg )
{
   LockScope oLock(mSyncObj);

   DIA_TR_INF("dia_IOCtrlManager Accept Event Id=%d ('%s')", event, dia_IOCtrlManagerFSM::getEventName(event));
   if ( mpFSM )
   {
       DIA_TR_INF("### State Before: %s", mpFSM->getStateName());
       mpFSM->acceptEvent(event,pArg);
       DIA_TR_INF("### State After : %s", mpFSM->getStateName());
   }
   else
   {
       DIA_TR_INF( "### Event Not Accepted By dia_IOCtrlManager. No FSM !!!");
   }
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vFsmHandleTimeout ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vFsmHandleTimeout()");
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vFsmCollectSignals ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vFsmCollectSignals()");

   LockScope oLock(mSyncObj);

   // collect the uids of all active ioctrl signals
   mActiveSignals.clear();
   mSignalRepIter = mSignalRep.begin();
   for ( ; mSignalRepIter != mSignalRep.end(); mSignalRepIter++ )
   {
      if ( mSignalRepIter->second )
      {
         switch ( mSignalRepIter->second->eGetStatus() )
         {
         case DIA_EN_IOCTRL_STATUS_INACTIVE_ABORTED:
         case DIA_EN_IOCTRL_STATUS_INACTIVE_TIMEOUT:
         case DIA_EN_IOCTRL_STATUS_ACTIVE:
            mActiveSignals[mSignalRepIter->second->getUID()] = mSignalRepIter->second;
            break;

         default:
            break;
         }  //lint !e788: not all items intentionally used within defaulted switch
      }
   }

   // initialize the iterator we are using to traverse the list of active signals
   mActiveSignalIter = mActiveSignals.begin();
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vFsmAddSignal ( void* pArg )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vFsmAddSignal()");

   if ( !pArg ) return;

   LockScope oLock(mSyncObj);
   dia_IOCtrlSignal* pSignal = (dia_IOCtrlSignal*) pArg;
   mActiveSignals[pSignal->getUID()] = pSignal;
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vFsmRemoveSignal ( void* pArg )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vFsmRemoveSignal()");

   if ( !pArg ) return;

   LockScope oLock(mSyncObj);

   dia_IOCtrlSignal* pSignal = (dia_IOCtrlSignal*) pArg;
   std::map<dia_UID,dia_IOCtrlSignal*>::iterator iter = mActiveSignals.find(pSignal->getUID());
   if ( iter != mActiveSignals.end() )
   {
      mActiveSignals.erase(pSignal->getUID());
   }

   if ( mActiveSignals.size() ) mActiveSignalIter = mActiveSignals.begin();
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vFsmReset ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vFsmReset");
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vFsmStopNextSignal ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vFsmStopNextSignal");

   dia_IOCtrlSignal* pSignal = 0;
   {
      LockScope oLock(mSyncObj);

      if ( mActiveSignalIter != mActiveSignals.end() && mActiveSignalIter->second )
      {
         pSignal = mActiveSignalIter->second;
      }
   }

   if ( pSignal )
   {
      // we want to stop the identified signal
      returnControlToECU(*pSignal);
      if ( pSignal->isResultReady() )
      {
         // we don't have to wait. operation is complete. we can directly process the next signal
         pSignal->setStatus(DIA_EN_IOCTRL_STATUS_INACTIVE);
         acceptEvent(dia_IOCtrlManagerFSM::evSignalDeactivated,pSignal);
      }
   }
}

//-----------------------------------------------------------------------------

bool
dia_IOCtrlManager::bFsmIsNoSignalActive ( void* /*pArg*/ )
{
   LockScope oLock(mSyncObj);
   return ( mActiveSignals.size() > 0 ) ? false : true;
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlManager::vOnSignalUpdate ( dia_IOCtrlSignal& signal )
{
   ScopeTrace oTrace("dia_IOCtrlManager::vOnSignalUpdate()");

   if ( !mpInstance ) return;
   if ( !mpInstance->mActiveSignalUID ) return;

   LockScope oLock(mpInstance->mSyncObj);

   // check if we are waiting for an update of this signal
   if ( mpInstance->mActiveSignalUID == signal.getUID() )
   {
      if ( signal.isResultReady() && (signal.getErrorInfo() == DIA_E_NOERROR) )
      {
         mpInstance->mActiveSignalUID = 0;

         switch ( mpInstance->mActiveSignalControlState )
         {
         case DIA_C_U8_UDS_IOCTRL_RETURN_CTRL_TO_ECU:
            {
               signal.eSetStatus(DIA_EN_IOCTRL_STATUS_INACTIVE);
               mpInstance->acceptEvent(dia_IOCtrlManagerFSM::evSignalDeactivated,&signal);
            }
            break;

         case DIA_C_U8_UDS_IOCTRL_SHORT_TERM_ADJUSTMENT:
            {
               signal.eSetStatus(DIA_EN_IOCTRL_STATUS_ACTIVE);
               mpInstance->acceptEvent(dia_IOCtrlManagerFSM::evSignalActivated,&signal);
            }
            break;

         default:
            break;
         }
	  }
   }

   std::map<dia_UID,dia_IIOControlListener*> copyOfListenerRep = mpInstance->mListenerRep;
   std::map<dia_UID,dia_IIOControlListener*>::iterator iter = copyOfListenerRep.begin();
   for ( ; iter != copyOfListenerRep.end(); ++iter )
   {
      if ( iter->second ) iter->second->vOnIOCtrlSignalUpdate(signal);
   }
}
