/*!
 * \file       dia_ServiceHandlerUDS.cpp
 *
 * \brief      Base class for diagnostic service handlers using the UDS standard
 *
 * \details    Base class for diagnostic service handlers using the UDS standard
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreProtocols
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_SERVICE_HANDLER_UDS__
#include <common/framework/protocols/uds/dia_ServiceHandlerUDS.h>
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include <common/framework/engine/dia_MessageBuffer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include <common/framework/application/dia_Application.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

using namespace std;

#define DIA_C_U8_INVALID_SID     ((tU8)  0xFF)
#define DIA_C_U16_INVALID_DID    ((tU16) 0xFFFF)

//-----------------------------------------------------------------------------

dia_ServiceHandlerUDS::dia_ServiceHandlerUDS ( tCString name, tU32 timeoutVal )
   : dia_ServiceHandler(name,timeoutVal),
     mSID(DIA_C_U8_INVALID_SID),
     mDID(DIA_C_U16_INVALID_DID),
     mDIDLen(0)
{}

//-----------------------------------------------------------------------------

dia_ServiceHandlerUDS::dia_ServiceHandlerUDS ( tCString name, tU8 sid, tU32 timeoutVal )
   : dia_ServiceHandler(name,timeoutVal),
     mSID(sid),
     mDID(DIA_C_U16_INVALID_DID),
     mDIDLen(0)
{}

//-----------------------------------------------------------------------------

dia_ServiceHandlerUDS::dia_ServiceHandlerUDS ( tCString name, tU8 sid, tU8 did, tU32 timeoutVal )
   : dia_ServiceHandler(name,timeoutVal),
     mSID(sid),
     mDID(did),
     mDIDLen(1)
{}

//-----------------------------------------------------------------------------

dia_ServiceHandlerUDS::dia_ServiceHandlerUDS ( tCString name, tU8 sid, tU16 did, tU32 timeoutVal )
   : dia_ServiceHandler(name,timeoutVal),
     mSID(sid),
     mDID(did),
     mDIDLen(2)
{}

//-----------------------------------------------------------------------------

dia_ServiceHandlerUDS::~dia_ServiceHandlerUDS ( void )
{}

//-----------------------------------------------------------------------------

tU8
dia_ServiceHandlerUDS::getSID ( void ) const
{
   return mSID;
}

//-----------------------------------------------------------------------------

tU16
dia_ServiceHandlerUDS::getDID ( void ) const
{
   return mDID;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_ServiceHandlerUDS::makeLookupKeys ( std::vector<dia_LookupKey*>& keys )
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace oTrace("dia_ServiceHandlerUDS::makeLookupKeys(...)");
#endif

   tDiaResult retCode = DIA_FAILED;

   if ( mLookupKeys.empty() )
   {
      dia_LookupKey* pKey = 0; //lint !e438 Warning: last value assigned to variable not used

      switch ( mDIDLen )
      {
      case 0:
         pKey = OSAL_NEW dia_LookupKey(mSID, UDS_U8_PARAMETER_NOT_USED, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED);
         break;

      case 1:
         pKey = OSAL_NEW dia_LookupKey(mSID,(tU8) mDID, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED);
         break;

      case 2:
         pKey = OSAL_NEW dia_LookupKey(mSID,(tU16) mDID,DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED);
         break;

      default:
         break;
      }

      if ( pKey )
      {
         mLookupKeys.push_back(pKey);
         keys.push_back(pKey);
         retCode = DIA_SUCCESS;
      }

      // make lint happy
      pKey = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pKey'. --> lifetime is controlled by dispatcher class within a diagnostic session
   } //lint !e438 Warning: last value assigned to variable 'pKey' not used

   return retCode; //lint !e438 Warning: last value assigned to variable 'pKey' not used
}

//-----------------------------------------------------------------------------

void
dia_ServiceHandlerUDS::vOnTimeout ( void )
{
#ifdef __DIA_UNIT_TESTING__
   DIA_TR_INF("dia_ServiceHandlerUDS::vOnTimeout");
#endif

   oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   vResReadyAndQuit();
}

//-----------------------------------------------------------------------------

void
dia_ServiceHandlerUDS::vSendPositiveResponse ( tU16 length, vector<tU8>* dataToAppend ) const
{
   dia_tclFnctTrace oTrace("dia_ServiceHandlerUDS::vSendPositiveResponse");

#ifdef __DIA_UNIT_TESTING__
   DIA_TR_INF("dia_ServiceHandlerUDS::vSendPositiveResponse length=%d", length);
   DIA_TR_INF("dia_ServiceHandlerUDS::vSendPositiveResponse dataToAppend->size()=%d", (dataToAppend!=NULL ? (tS32)dataToAppend->size() : -1) );
#endif

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(length);
   if ( dataToAppend && (length >= dataToAppend->size()) )
   {
      size_t offset = length - dataToAppend->size();
      for ( tU16 i=0; i < dataToAppend->size(); i++ )
      {
         //DIA_TR_INF("[%d] = 0x%02X", (offset+i), (*dataToAppend)[i]);
         oDiagMsgBuffer().setDataU8(tU16(offset+i), (*dataToAppend)[i]);
      }
   }
   else
   {
      if (NULL!=dataToAppend)
      {
         DIA_TR_ERR("dia_ServiceHandlerUDS::vSendPositiveResponse ERROR length=%u dataToAppend->size()=%zu", length, dataToAppend->size());
      }
      else
      {
         DIA_TR_ERR("dia_ServiceHandlerUDS::vSendPositiveResponse ERROR dataToAppend is NULL");
      }
   }
   vResReadyAndQuit();
}

//-----------------------------------------------------------------------------

void
dia_ServiceHandlerUDS::vSendNegativeResponse ( tU8 errCode ) const
{
#ifdef __DIA_UNIT_TESTING__
   DIA_TR_INF("dia_ServiceHandlerUDS::vSendNegativeResponse errCode=0x%02X", errCode);
#endif

    oDiagMsgBuffer().vSetNegResp(errCode);
    vResReadyAndQuit();
}

//-----------------------------------------------------------------------------

dia_MessageBuffer&
dia_ServiceHandlerUDS::oDiagMsgBuffer ( void ) const
{
#ifdef __DIA_UNIT_TESTING__
   DIA_ASSERT(NULL!=mpEngine);
#endif

   return *(mpEngine->getMessageBuffer());
}

//-----------------------------------------------------------------------------

void
dia_ServiceHandlerUDS::vResReadyAndQuit ( void ) const
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace("dia_ServiceHandlerUDS::vResReadyAndQuit");
#endif

//   oDiagSession().vEmit(dia_tclDiagSession::tclEventResReadyAndQuit());
//   getInstanceOfApplication()->postMessage( dia_Event(DIA_MSG_ID_REQUEST_PROCESSED,mpEngine) );
   getInstanceOfApplication()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventResReadyAndQuit());
}

