/*!
 * \file       dia_ResponsePendingHandler.cpp
 *
 * \brief      Handling of Response-Pending as Message-Handler of udd-comm-channel
 *
 * \details    
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreUDD
 *
 * \copyright  (c) 2019 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif


#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include "common/framework/engine/dia_EngineManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_DIAGSESSION_UDS__
#include "common/depricated/dia_tclDiagSessionUds.h"
#endif



#include "dia_ResponsePendingHandler.h"


#define DIA_C_U32_RESPONSE_PENDING_FIRST_TIMER_VAL_MS 50
#define DIA_C_U32_RESPONSE_PENDING_TIMER_VAL_MS 5000

namespace dia {

ResponsePendingHandler::ResponsePendingHandler (  ):
   mIsSetupDone(false),
   mP2FirstValMs(DIA_C_U32_RESPONSE_PENDING_FIRST_TIMER_VAL_MS),
   mP2ValMs(DIA_C_U32_RESPONSE_PENDING_TIMER_VAL_MS),
   mP2Timer(this, this->mRefCounter)
{
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::CTOR default");
   if ( ResponsePendingHandler::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF dia::ResponsePendingHandler FAILED ###");
   }

}

ResponsePendingHandler::ResponsePendingHandler ( dia_MessageHandler* nextHandler ):
   dia_MessageHandler(nextHandler),
   mIsSetupDone(false),
   mP2FirstValMs(DIA_C_U32_RESPONSE_PENDING_FIRST_TIMER_VAL_MS),
   mP2ValMs(DIA_C_U32_RESPONSE_PENDING_TIMER_VAL_MS),
   mP2Timer(this, this->mRefCounter)
{
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::CTOR");
   if ( ResponsePendingHandler::setup() != DIA_SUCCESS )
   {
      DIA_TR_INF("### SETUP OF dia::ResponsePendingHandler FAILED ###");
   }
}

ResponsePendingHandler::~ResponsePendingHandler ( void ) {
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::DTOR");
   tearDown();
}

tVoid ResponsePendingHandler::vOnSessionChanged ( tU8 /* newSession */, tU8 /* oldSession */ )
{
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::vOnSessionChange");
   mPendingResponse.reset();
}


bool ResponsePendingHandler::PendingResponse::hasSuppressPositiveResponse() {
   // uds: equestCorrectlyReceivedResponsePending
   // uds: (suppress-PosRspMsg-Indication-Bit) 
   bool res=false;
   switch (mSid) {
      case DIA_C_U8_UDS_SID_SESSION_CONTROL:
      case DIA_C_U8_UDS_SID_ECU_RESET:
      case DIA_C_U8_UDS_SID_READ_DTC_INFO:
      case DIA_C_U8_UDS_SID_SECURITY_ACCESS:
      case DIA_C_U8_UDS_SID_COMMUNICATION_CONTROL:
      case DIA_C_U8_UDS_SID_DYNMICALLY_DEFINE_DATA_IDENTIFIER:
      case DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER:
      case DIA_C_U8_UDS_SID_ROUTINE_CONTROL:
      case DIA_C_U8_UDS_SID_WRITE_MEMORY_BY_ADDRESS:
      case DIA_C_U8_UDS_SID_TESTER_PRESENT:
      case DIA_C_U8_UDS_SID_ACCESS_TIMING_PARAMETER:
      case DIA_C_U8_UDS_SID_CONTROL_DTC_SETTING:
      case DIA_C_U8_UDS_SID_RESPONSE_ON_EVENT:
      case DIA_C_U8_UDS_SID_LINK_CONTROL:
         if (mSubFn & 0x80) {
            res=true;
         }
         break;
      default:
         break;
         
   }
   return res;
}

tDiaResult ResponsePendingHandler::handleMessage ( dia_MessageBuffer& msg ) {
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::handleMessage");
      // check protocoll.
   if ( msg.getProtocol() != DIA_EN_PROTOCOL_UDS ) {
      return dia_MessageHandler::handleMessage(msg);
   }
   if (mPendingResponse.isActive()) {
      /* todo: ignore functional adressed tester-present*/
      msg.vSetNegResp(getInstanceOfFactory()->makeNRC(DIA_E_BUSY_REPEAT));
      msg.vSendResponse();
      return DIA_SUCCESS;
   }
   if (mPendingResponse.set(msg)) {
      mP2Timer.start(mP2FirstValMs);
   }

   // we don't consume any messages, just forward to next handler
   DIA_TR_INF("dia::ResponsePendingHandler::handleMessage: forward to next handler");

   return dia_MessageHandler::handleMessage(msg);
}

void ResponsePendingHandler::onTimeout ( MsgTimeoutP2 * /* msg */ ) {
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::vOnTimerElapsed");
   DIA_TR_INF("dia::ResponsePendingHandler::vOnTimerElapsed: isActive=%u", 
              mPendingResponse.isActive());

   // forceDefaultSession
   if (!mPendingResponse.isActive()) {
      return;
   }

   mPendingResponse.emit();
   if (mPendingResponse.isActive()) {
      mP2Timer.start(mP2ValMs);
   }
}

tVoid ResponsePendingHandler::onDiagnosisResponse (  tU8 const *au8MsgBuffer, tU16 u16Length ) {
   DIA_TR_INF("dia::ResponsePendingHandler::onDiagnosisResponse START isActive=%u u16Length=%u",
              mPendingResponse.isActive(), u16Length);
   if (!mPendingResponse.isActive()) {
      return;
   }
   if (!u16Length) {
      return;
   }
   // check for ack of SESSION_CONTROL
   uint8_t sid=au8MsgBuffer[0];
   if (u16Length >= 3 && sid==DIA_C_U8_UDS_SID_NEGATIVE_RESPONSE) {
      DIA_TR_INF("dia::ResponsePendingHandler::onDiagnosisResponse: got negative response");
      sid=au8MsgBuffer[1];
      uint8_t nackCode=au8MsgBuffer[2];
      if (nackCode==DIA_E_U8_UDS_RESPONSE_PENDING) {
         DIA_TR_INF("dia::ResponsePendingHandler::onDiagnosisResponse: ignore RESPONSE_PENDING");
         return;
      }
   }
   uint8_t sidWant=mPendingResponse.getSid();
   DIA_TR_INF("dia::ResponsePendingHandler::onDiagnosisResponse: check sid: got 0x%02x want 0x%02x", sid, sidWant);
   if (sid==mPendingResponse.getSid()) {
      mPendingResponse.reset();
      mP2Timer.stop();
   }
}


tDiaResult ResponsePendingHandler::setup ( void ) {
   tDiaResult retCode = DIA_SUCCESS;

   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::setup");
   if (mIsSetupDone) {
      return DIA_SUCCESS;
   }
   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      pEngine->getSessionController()->addListener(this);
   } else {
      retCode = DIA_FAILED;
   }
   mIsSetupDone=true;
   return retCode;
}

tDiaResult ResponsePendingHandler::tearDown ( void ) {
   dia_tclFnctTrace oTrace("dia::ResponsePendingHandler::tearDown");
   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      DIA_TR_ERR("##### UNABLE TO RETRIEVE POINTER TO DIAGNOSIS ENGINE #####");
      return DIA_FAILED;
   }

   pEngine->getSessionController()->removeListener(this);
   return DIA_SUCCESS;
}

void ResponsePendingHandler::PendingResponse::emit() {
   DIA_TR_INF("dia::ResponsePendingHandler::PendingResponse::emit: sid=0x%02x", mSid);
   mNumEmit++;
   if (!isActive()) {
      DIA_TR_INF("dia::ResponsePendingHandler::PendingResponse::emit: not Active!");
      return;
   }
   uint8_t responseBuffer[3];
   responseBuffer[0]=DIA_C_U8_UDS_SID_NEGATIVE_RESPONSE;
   responseBuffer[1]=mSid;
   if (mNumEmit <= mMaxNumEmit) {
      responseBuffer[2]=DIA_E_U8_UDS_RESPONSE_PENDING;
      if (!hasSuppressPositiveResponse()) {
         mFuncResponseSend(responseBuffer, (tU16)3, mCookie);
      }
   }
   else {
      responseBuffer[2]=DIA_E_U8_UDS_GENERAL_REJECT ;
      mFuncResponseSend(responseBuffer, (tU16)3, mCookie);
      reset();
   } 
}

bool ResponsePendingHandler::PendingResponse::set(dia_MessageBuffer& msg, tCookieType /* cookie */) {
   uint8_t offsetUdsStart=(msg.getFormat() == dia_MessageBuffer::format_length_and_data) ? 1 : 0;
   uint32_t numUdsBytes=msg.u16GetDataLength() - offsetUdsStart;
   if (!numUdsBytes) {
      return false;
   }
   mFuncResponseSend=msg.getFuncSendResponse();
   mSid=msg.getDataU8(offsetUdsStart);
   mSubFn=0;
   if (numUdsBytes >= 2) {
      mSubFn=msg.getDataU8((tU16)(offsetUdsStart + 1));
   }
   DIA_TR_INF("dia::ResponsePendingHandler::PendingResponse::set: sid=0x%02x", mSid);
   return true;
}

void ResponsePendingHandler::PendingResponse::reset() {
   DIA_TR_INF("dia::ResponsePendingHandler::PendingResponse::reset: sid=0x%02x", mSid);
   mFuncResponseSend=nullptr;
   mSid=0;
   mSubFn=0;
   mCookie=0;
   mNumEmit=0;
}


}
