/*!
 * \file       dia_AppStateChangeHandler.cpp
 *
 * \brief      Pre-service handler for all diag services (check not in state change,
 *             if not zero, send NRC 0x22 for specific service handler)
 * \details
 *
 * \component  Diagnosis
 *
 * \ingroup    diaServicesProject
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_APP_STATE_CHANGE_HANDLER__
#include <common/framework/protocols/uds/dia_AppStateChangeHandler.h>
#endif

#ifndef __INCLUDED_DIA_CMD_START_TEST__
#include "common/framework/test/dia_CmdStartTest.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_APPSTATE__
#include <common/framework/application/dia_PredicateAppState.h>
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include <common/framework/application/dia_CommandController.h>
#endif

namespace dia {

//-----------------------------------------------------------------------------

AppStateChangeHandler::AppStateChangeHandler ( void )
{
   ScopeTrace trc("dia::AppStateChangeHandler::AppStateChangeHandler");

}

//-----------------------------------------------------------------------------

AppStateChangeHandler::AppStateChangeHandler ( dia_MessageHandler* nextHandler )
   : dia_MessageHandler(nextHandler)
{
   ScopeTrace trc("dia::AppStateChangeHandler::AppStateChangeHandler(dia_MessageHandler*)");

}

//-----------------------------------------------------------------------------

AppStateChangeHandler::~AppStateChangeHandler ( void )
{
}

//-----------------------------------------------------------------------------

tDiaResult
AppStateChangeHandler::handleMessage ( dia_MessageBuffer& msg )
{
   ScopeTrace trc("dia::AppStateChangeHandler::handleMessage(dia_MessageBuffer&)");

   tDiaResult retCode = DIA_FAILED;

   if ( getInstanceOfAppController()->isApplicationStateChangeInProgress() )
   {
      // the expected run level is not yet reached
      Command* pCmd = new Command;
      if ( pCmd )
      {
         pCmd->addFunctor( new dia_FunctorOneArgNoReturnValue<AppStateChangeHandler,dia_MessageBuffer*>(this,&AppStateChangeHandler::postprocessMessage,&msg) );

         // add predicate
         (void) pCmd->addPredicate( new dia_PredicateAppState(false) );
         // add command to cmdController
         getInstanceOfCommandController()->addCommand(pCmd);

         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia::AppStateChangeHandler::handleMessage DIA_SUCCESS!");
      }
      else
      {
         DIA_TR_ERR("dia::AppStateChangeHandler::handleMessage TRIGGER FAILED!");
      }
      pCmd = 0;  //lint !e423 Warning: Creation of memory leak in assignment to 'pCmd'. --> lifetime is controlled by command controller
   }
   else
   {
      DIA_TR_INF("dia::AppStateChangeHandler::handleMessage no application state change in progress!");

      //Go on processing, do not wait for vehicle speed.
      retCode = dia_MessageHandler::handleMessage(msg);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

void
AppStateChangeHandler::postprocessMessage ( dia_MessageBuffer* msg )
{
   ScopeTrace trc("dia::AppStateChangeHandler::postprocessMessage(dia_MessageBuffer&)");

   if ( msg && (dia_MessageHandler::handleMessage(*msg) == DIA_E_NOT_AVAILABLE) )
   {
      getInstanceOfApplication()->postMessage(new dia_tclDiagSession::tclEventReqRx(msg));
      DIA_TR_INF("--- AppStateChangeHandler::postprocessMessage => Posted as UDS message");
   }
}


}
