/*!
 * \file       dia_ALDStatusHandler.cpp
 *
 * \brief      Pre-service handler for all diag production services
 *             (check if ALD status is unknown when the device is in production session)
 *
 * \details    If ALD level is unknown, wait until ALD sends the status.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaServicesProject
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_APP_STATE_CHANGE_HANDLER__
#include <common/framework/protocols/uds/dia_ALDStatusHandler.h>
#endif

#ifndef __INCLUDED_DIA_CMD_START_TEST__
#include "common/framework/test/dia_CmdStartTest.h"
#endif

#ifndef __INCLUDED_DIA_PREDICATE_APPSTATE__
#include <common/framework/application/dia_PredicateAppState.h>
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include <common/framework/application/dia_CommandController.h>
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include "common/framework/engine/dia_Session.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_AUTHORIZATIONLEVEL_LISTENER__
#include <common/interfaces/dia_IAuthorizationLevelListener.h>
#endif

#ifndef __INCLUDED_DIA_PREDICATE_ALDSTATUS__
#include <common/framework/application/dia_PredicateALDStatus.h>
#endif


namespace dia {

//-----------------------------------------------------------------------------

ALDStatusHandler::ALDStatusHandler ( void )
{
   ScopeTrace trc("dia::ALDStatusHandler::ALDStatusHandler");

}

//-----------------------------------------------------------------------------

ALDStatusHandler::ALDStatusHandler ( dia_MessageHandler* nextHandler )
   : dia_MessageHandler(nextHandler)
{
   ScopeTrace trc("dia::ALDStatusHandler::ALDStatusHandler(dia_MessageHandler*)");

}

//-----------------------------------------------------------------------------

ALDStatusHandler::~ALDStatusHandler ( void )
{
}

//-----------------------------------------------------------------------------

tDiaResult
ALDStatusHandler::handleMessage ( dia_MessageBuffer& msg )
{
   ScopeTrace trc("dia::ALDStatusHandler::handleMessage(dia_MessageBuffer&)");

   tDiaResult retCode = DIA_FAILED;

   if ( msg.getProtocol() == DIA_EN_PROTOCOL_UDS )
   {
      tU16 offset = (msg.getFormat() == dia_MessageBuffer::format_length_and_data) ? 2 : 1; /* one length byte plus one byte SID */
      tU8 sid = msg.getDataU8(tU16(offset-1));
      if ( DIA_C_U8_UDS_SID_SESSION_CONTROL != sid )
      {
         tU8 activeSession = 0x00;

         dia_EngineServer* pEngine = 0;
         if ( (getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS) && (NULL!=pEngine) )
         {
            activeSession = pEngine->getActiveSessionID();
            if (0x00!=activeSession)
            {
               DIA_TR_INF(" --- ACTIVE DIAGNOSTIC SESSION IS 0x%02x ", activeSession);
            }
            else
            {
               DIA_TR_ERR("dia::ALDStatusHandler::handleMessage Failed. Active session is not available.");
            }
         }
         else
         {
            DIA_TR_ERR("dia::ALDStatusHandler::handleMessage Failed. Engine server is not available.");
         }

         if ( DIA_C_U8_UDS_SESSION_RBCM_PRODUCTION == activeSession )
         {
            dia_IAuthorizationLevel* pAuthLevel = 0;
            if ( (querySysAdapterInterface<dia_IAuthorizationLevel>(&pAuthLevel) == DIA_SUCCESS) && pAuthLevel )
            {
               if ( pAuthLevel->getCurrentLevel() == DIA_C_U16_ALD_UNKNOWN_LEVEL )
               {
                  Command* pCmd = new Command;
                  if ( pCmd )
                  {
                     pCmd->addFunctor( new dia_FunctorOneArgNoReturnValue<ALDStatusHandler,dia_MessageBuffer*>(this,&ALDStatusHandler::postprocessMessage,&msg) );
                     (void) pCmd->addPredicate( new PredicateALDStatus );
                     getInstanceOfCommandController()->addCommand(pCmd);
                     retCode = DIA_SUCCESS;
                     DIA_TR_INF("dia::ALDStatusHandler::handleMessage DIA_SUCCESS! ALD status is unknown now.");
                  }
                  pCmd = 0;  //lint !e423 Warning: Creation of memory leak in assignment to 'pCmd'. --> lifetime is controlled by command controller
               }
               else
               {
                  DIA_TR_INF("dia::ALDStatusHandler::handleMessage ALD is known (0x%02X)", pAuthLevel->getCurrentLevel());
               }
            }
            else
            {
               DIA_TR_ERR("dia::ALDStatusHandler::handleMessage Error. Interface for authorization level not available.");
            }
         }
         else
         {
            DIA_TR_INF("dia::ALDStatusHandler::handleMessage activeSession 0x%02X is different than SESSION_RBCM_PRODUCTION.", activeSession);
         }
      }
      else
      {
         DIA_TR_INF("dia::ALDStatusHandler::handleMessage SID 0x%02X is different than UDS.", sid);
      }
   }
   else
   {
      DIA_TR_INF("dia::ALDStatusHandler::handleMessage Protocol 0x%02X is not SID_SESSION_CONTROL.", msg.getProtocol());
   }

   if ( DIA_SUCCESS != retCode )
   {
      retCode = dia_MessageHandler::handleMessage(msg);
      DIA_TR_INF("dia::ALDStatusHandler::handleMessage forward message to next handler! retCode=0x%08X.", retCode);
   }

   return retCode;
}

//-----------------------------------------------------------------------------

void
ALDStatusHandler::postprocessMessage ( dia_MessageBuffer* msg )
{
   ScopeTrace trc("dia::ALDStatusHandler::postprocessMessage(dia_MessageBuffer&)");

   if ( msg && (dia_MessageHandler::handleMessage(*msg) == DIA_E_NOT_AVAILABLE) )
   {
      getInstanceOfApplication()->postMessage(new dia_tclDiagSession::tclEventReqRx(msg));
      DIA_TR_INF("--- ALDStatusHandler::postprocessMessage => Posted as UDS message");
   }
}

}
