/*!
 * \file       dia_AudioManager.h
 *
 * \brief      Manager class that interacts with the audio component to switch audio sources
 *
 * \details    Manager class that interacts with the audio component to switch audio sources
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCorePlatform
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_AUDIO_MANAGER__
#define __INCLUDED_DIA_AUDIO_MANAGER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_IAUDIO_SOURCE_LISTENER__
#include <common/interfaces/dia_IAudioSourceListener.h>
#endif

#ifndef __INCLUDED_DIA_IAUDIO_MODE_CONTROL_LISTENER__
#include <common/interfaces/dia_IAudioModeControlListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_REMOTECONTROL_LISTENER__
#include <common/interfaces/dia_IRemoteControlListener.h>
#endif

#ifndef __INCLUDED_DIA_AUDIOSOURCESETFSM__
#include <common/framework/fsm/generated/dia_AudioSourceSetFSM.h>
#endif

namespace dia
{

class AudioManagerClient;

struct AudioSourceMatcher
{
   AudioSourceMatcher ( void );
   AudioSourceMatcher ( const char* namePattern, UID srcID );
   AudioSourceMatcher ( const char* namePattern, UID srcID, int startPos, int endPos );
   AudioSourceMatcher ( const char* namePattern, UID srcID, int startPos, int endPos, int startPosDevID );
   AudioSourceMatcher ( const AudioSourceMatcher& srcMatcher );
   ~AudioSourceMatcher ( void );
   AudioSourceMatcher& operator = ( const AudioSourceMatcher& obj );
   bool operator == ( const AudioSourceMatcher& matcher ) const;

   UID mSrcID;
   std::string mNamePattern;
   int mStartPos;
   int mEndPos;
   bool mIsDynamicSource;
   int mStartPosDeviceID;
};

class AudioManager
    : protected dia_AudioSourceSetFSM::FsmBehaviour,
      public dia_IAudioSourceListenerImpl,
      public dia::IAudioModeControlListenerImpl,
      public dia_IRemoteControlListenerImpl
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(AudioManager);
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(AudioManager);

public:

   dia_eAudioSource getRequestedAudioSource ( void ) const { return mRequestedAudioSource; }

   virtual tDiaResult setAudioSource ( dia_eAudioSource sourceID, AudioManagerClient& pClient );
   virtual tDiaResult getAudioSource ( AudioManagerClient& pClient );
   virtual void resetClient ( void );

   //----------------------------------------------------------------------------------------------
   //
   // Overloaded dia_IAudioSourceListenerImpl methods
   //
   //----------------------------------------------------------------------------------------------

   //! response to getAudioSourceList
   virtual void vOnAudioSourceList ( const std::vector<dia_tAudioSourceType>& audioSrcList );
   //! response to getAudioSourceList
   virtual void vOnAudioSinkList ( const std::vector<dia_tAudioSinkType>& audioSrcList );
   //! response to getConnections
   virtual void vOnAudioConnectionList ( const std::vector<dia_tAudioConnectionType>& audioConnectionList );
   //! response to getConnections
   virtual void vOnAudioConnectionStateChanged ( const dia_tAudioConnectionType& audioConnection );
   //! handle error while switching the source
   virtual void vOnAudioConnectionFailure ( tDiaResult errInfo );


   //----------------------------------------------------------------------------------------------
   //
   // Overloaded dia::IAudioModeControlListenerImpl methods
   //
   //----------------------------------------------------------------------------------------------

   //! response to setAudioMode
   virtual void vOnAudioModeControlUpdate ( const AudioMode& mode );

   //----------------------------------------------------------------------------------------------
   //
   // Overloaded dia_IRemoteControlListenerImpl methods
   //
   //----------------------------------------------------------------------------------------------

   //! remote control mode has been changed to the given mode
   virtual void vOnRemoteControlModeChanged ( dia_eRemoteCtrlMode newMode, dia_eRemoteCtrlMode prevMode );

   static tDiaResult addAudioSourceMatcher ( const AudioSourceMatcher& matcher );
   static tDiaResult clearAudioSourceMatchers ( void );
   static const std::list<AudioSourceMatcher>& getAudioSourceMatchers ( void ) { return mAudioSourceMatcherRep; }
   //static void addSourceMapping ( tU8 udsID, dia::UID srcID, const std::string& name );
   //! Map a UDS value to a source based on project requirement(usually on startup)
   static void addSourceMapping ( tU8 udsID, dia::UID srcID, const char* name );
   //! To return mSourceMapping by reference
   static std::map<tU8,AudioSourceMatcher>& getSourceMappings ( void ) { return  mSourceMapping; }
   //! To load mSourceMapping with default values
   static void vLoadSourceMappings();

protected:
   //! protected because we are using the singleton pattern
   AudioManager ( void );
   //! protected because we are using the singleton pattern
   virtual ~AudioManager ( void );

   //! setup (create objects,...)
   tDiaResult setup ( void );
   //! tear down (destroy objects,...)
   //lint -sem(dia::AudioManager::tearDown,cleanup)
   tDiaResult tearDown ( void );

   tDiaResult acceptEvent ( dia_AudioSourceSetFSM::FsmEvent event, void* pArg );

   // FSM actions
   virtual void vFsmCheckForAudioDiagMode ( void* pArg );
   virtual void vFsmFinalize ( void* pArg );
   virtual void vFsmInitialize ( void* pArg );
   virtual void vFsmNoActiveSourceFound ( void* pArg );
   virtual void vFsmRequestAudioDiagMode ( void* pArg );
   virtual void vFsmRequestAudioNormalMode ( void* pArg );
   virtual void vFsmRequestConnection ( void* pArg );
   virtual void vFsmRequestListOfConnections ( void* pArg );
   virtual void vFsmRequestListOfSinks ( void* pArg );
   virtual void vFsmRequestListOfSources ( void* pArg );
   virtual void vFsmReset ( void* pArg );
   virtual void vFsmSendResponse ( void* pArg );
   virtual void vFsmSetError_ConnectionFailed ( void* pArg );
   virtual void vFsmSetError_ConnectionNotSupported ( void* pArg );
   virtual void vFsmSetError_SinkNotSupported ( void* pArg );
   virtual void vFsmSetError_SourceNotSupported ( void* pArg );
   virtual void vFsmSendGetActiveSourceResponse ( void* /*pArg*/ );

   // FSM guards
   virtual bool isAudioDiagModeActive ( void* pArg );
   virtual bool isConnected ( void* pArg );
   virtual bool isConnectionActive ( void* pArg );
   virtual bool isConnectionFound ( void* pArg );
   virtual bool isSinkSupported ( void* pArg );
   virtual bool isSourceSupported ( void* pArg );
   virtual bool isActiveConnectionAvailable ( void* pArg );

 protected:
   //! pointer to the state machine
   dia_AudioSourceSetFSM::Fsm* mpFSM; //lint -sem(dia::AudioManager::tearDown,cleanup)

   tDiaResult mErrorInfo;

   dia_eAudioSource mActiveAudioSource;

   //! diagnosis internal audio source ID
   dia_eAudioSource mRequestedAudioSource;
   //! audio component specific source information
   dia_tAudioSourceType mAudioSource;
   //! flag to indicate that the requested audio source is supported
   bool mIsSourceSupported;

   //! diagnosis internal audio sink ID
   dia_eAudioSink mRequestedAudioSink;
   //! audio component specific sink information
   dia_tAudioSinkType mAudioSink;
   //! flag to indicate that the requested audio sink is supported
   bool mIsSinkSupported;

   //! flag to indicate if the diagnosis mode of audio component is active or not
   bool mIsAudioDiagModeActive;
   //! flag to indicate if the diagnosis mode of audio component has to be requested or not
   bool mNeedToRequestAudioDiagMode;
   //! flag to indicate if the requested connection is established
   bool mIsConnected;
   //! flag to indicate if the requested connection is established and active
   bool mIsConnectionActive;
   //! flag to indicate if the requested connection is existing
   bool mIsConnectionFound;
   //!
   bool mIsActiveConnectionAvailable;

   //! flag to indicate if the requested connection is established
   //std::list<AudioManagerClient*> mClientRep;
   AudioManagerClient* mpClient; //lint -sem(dia::AudioManager::tearDown,cleanup)

   static std::list<AudioSourceMatcher> mAudioSourceMatcherRep;
   static std::map<tU8,AudioSourceMatcher> mSourceMapping;
};

}

dia::AudioManager* getInstanceOfAudioManager ( void );
void releaseInstanceOfAudioManager ( void );

#endif /* __INCLUDED_DIA_AUDIO_MANAGER__ */
