/*
 * dia_AudioManager.cpp
 *
 *  Created on: 20.05.2016
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_AUDIO_MANAGER__
#include <common/framework/platform/cmd/dia_AudioManager.h>
#endif

#ifndef __INCLUDED_DIA_AUDIO_MANAGER_CLIENT__
#include <common/framework/platform/cmd/dia_AudioManagerClient.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_AUDIO_MODE_CONTROL__
#include "common/interfaces/dia_IAudioModeControl.h"
#endif

#ifndef __INCLUDED_DIA_REMOTECONTROL__
#include "common/framework/application/dia_RemoteControl.h"
#endif

#define DIA_C_U16_MAX_SOURCE_NAME_LENGTH           ((tU16) 100)
#define DIA_C_INT_SOURCE_NAME_STARTPOS_INVALID     ((int) -1)
#define DIA_C_INT_SOURCE_NAME_ENDPOS_INVALID       ((int) -1)
#define DIA_C_INT_DEVICE_ID_STARTPOS_INVALID       ((int) -1)


DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia::AudioManager)

dia::AudioManager*
getInstanceOfAudioManager ( void )
{
   return dia::AudioManager::getInstance();
}

void
releaseInstanceOfAudioManager ( void )
{
   dia::AudioManager::deleteInstance();
}

namespace dia
{

std::list<AudioSourceMatcher> AudioManager::mAudioSourceMatcherRep;
std::map<tU8,AudioSourceMatcher> AudioManager::mSourceMapping;
//------------------------------------------------------------------------------

AudioManager::AudioManager ( void )
   : mpFSM(0),
     mErrorInfo(DIA_SUCCESS),
     mActiveAudioSource(DIA_EN_AUDIO_SOURCE_UNKNOWN),
     mRequestedAudioSource(DIA_EN_AUDIO_SOURCE_UNKNOWN),
     mIsSourceSupported(false),
     mRequestedAudioSink(DIA_EN_AUDIO_SINK_AMPLIFIER_MAINFIELD),
     mIsSinkSupported(false),
     mIsAudioDiagModeActive(false),
     mNeedToRequestAudioDiagMode(false),
     mIsConnected(false),
     mIsConnectionActive(false),
     mIsConnectionFound(false),
     mIsActiveConnectionAvailable(false),
     mpClient(0)
{
   ScopeTrace trc("AudioManager::AudioManager");
}

//------------------------------------------------------------------------------

AudioManager::~AudioManager ( void )
{
   _BP_TRY_BEGIN
   {
      (void) AudioManager::tearDown();
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: AudioManager::~AudioManager !!!");
       DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
AudioManager::setup ( void )
{
   ScopeTrace trc("AudioManager::setup");

   (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);
   (void) unsetSysAdapterListener<dia::IAudioModeControlListener>(this);
   (void) setSysAdapterListener<dia_IRemoteControlListener>(this);

   if ( !mpFSM )
   {
      // create the state machine object
      if ( !(dia_AudioSourceSetFSM::Fsm::createFSM(&mpFSM,this)) )
      {
         DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
         return DIA_FAILED;
      }
   }

   (void) addAudioSourceMatcher(AudioSourceMatcher("AUX_1",DIA_EN_AUDIO_SOURCE_AUX_1));
   (void) addAudioSourceMatcher(AudioSourceMatcher("AUX_2",DIA_EN_AUDIO_SOURCE_AUX_2));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_AM",DIA_EN_AUDIO_SOURCE_TUNER_AM));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_DAB",DIA_EN_AUDIO_SOURCE_TUNER_DAB));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_FM",DIA_EN_AUDIO_SOURCE_TUNER_FM));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_TA_DAB",DIA_EN_AUDIO_SOURCE_TUNER_TA_DAB));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_TA_FM",DIA_EN_AUDIO_SOURCE_TUNER_TA_FM));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_XM",DIA_EN_AUDIO_SOURCE_TUNER_XM));
   (void) addAudioSourceMatcher(AudioSourceMatcher("MEDIA_CDDA#",DIA_EN_AUDIO_SOURCE_MEDIA_CDDA,0,11));
   (void) addAudioSourceMatcher(AudioSourceMatcher("MEDIA_PLAYER#",DIA_EN_AUDIO_SOURCE_MEDIA_PLAYER,0,13,13));
   (void) addAudioSourceMatcher(AudioSourceMatcher("NAVI_SPEECH",DIA_EN_AUDIO_SOURCE_NAVI_SPEECH));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE#",DIA_EN_AUDIO_SOURCE_PHONE,0,6,6));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE_BTAUDIO",DIA_EN_AUDIO_SOURCE_PHONE_BTAUDIO,0,13));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE_ECALL",DIA_EN_AUDIO_SOURCE_PHONE_ECALL));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE_EXTERNAL_1",DIA_EN_AUDIO_SOURCE_PHONE_EXTERNAL_1));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE_OUTBAND_RING",DIA_EN_AUDIO_SOURCE_PHONE_OUTBAND_RING));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE_INBAND_RING#",DIA_EN_AUDIO_SOURCE_PHONE_INBAND_RING,0,18,18));
   (void) addAudioSourceMatcher(AudioSourceMatcher("PHONE_SPEECHRECOGNITION",DIA_EN_AUDIO_SOURCE_PHONE_SPEECHRECOGNITION));
   (void) addAudioSourceMatcher(AudioSourceMatcher("SDS_SPEECHRECOGNITION",DIA_EN_AUDIO_SOURCE_SDS_SPEECHRECOGNITION));
   (void) addAudioSourceMatcher(AudioSourceMatcher("SMS_SPEECH",DIA_EN_AUDIO_SOURCE_SMS_SPEECH));
   (void) addAudioSourceMatcher(AudioSourceMatcher("SPI_ENTERTAIN",DIA_EN_AUDIO_SOURCE_SPI_ENTERTAIN));
   (void) addAudioSourceMatcher(AudioSourceMatcher("SPI_INFO",DIA_EN_AUDIO_SOURCE_SPI_INFO));
   (void) addAudioSourceMatcher(AudioSourceMatcher("SPI_PHONE",DIA_EN_AUDIO_SOURCE_SPI_PHONE));
   (void) addAudioSourceMatcher(AudioSourceMatcher("SPI_SPEECHRECOGNITION",DIA_EN_AUDIO_SOURCE_SPI_SPEECHRECOGNITION));
   (void) addAudioSourceMatcher(AudioSourceMatcher("MICROPHONE",DIA_EN_AUDIO_SOURCE_MICROPHONE));
   (void) addAudioSourceMatcher(AudioSourceMatcher("INTERNET_APPS",DIA_EN_AUDIO_SOURCE_INTERNET_APPS));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TUNER_DRM",DIA_EN_AUDIO_SOURCE_TUNER_DRM));
   (void) addAudioSourceMatcher(AudioSourceMatcher("TCU_SERVICE_CALL",DIA_EN_AUDIO_SOURCE_DCM));
   (void) addAudioSourceMatcher(AudioSourceMatcher("MICROPHONE2",DIA_EN_AUDIO_SOURCE_MICROPHONE2));

   return DIA_SUCCESS;
}

void AudioManager::vLoadSourceMappings()
{
   ScopeTrace trc("AudioManager::LoadSourceMappings()");
   mSourceMapping[0x00] = AudioSourceMatcher("TUNER_AM",DIA_EN_AUDIO_SOURCE_TUNER_AM);
   mSourceMapping[0x01] = AudioSourceMatcher("TUNER_FM",DIA_EN_AUDIO_SOURCE_TUNER_FM);
   mSourceMapping[0x02] = AudioSourceMatcher("UNKNOWN (RESERVED)",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x03] = AudioSourceMatcher("TUNER_DAB",DIA_EN_AUDIO_SOURCE_TUNER_DAB);
   mSourceMapping[0x04] = AudioSourceMatcher("TUNER_XM",DIA_EN_AUDIO_SOURCE_TUNER_XM);
   mSourceMapping[0x05] = AudioSourceMatcher("HD_RADIO (NOT_SUPPORTED)",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x06] = AudioSourceMatcher("UNKNOWN (RESERVED)",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x07] = AudioSourceMatcher("MICROPHONE",DIA_EN_AUDIO_SOURCE_MICROPHONE);
   mSourceMapping[0x08] = AudioSourceMatcher("UNKNOWN (RESERVED)",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x09] = AudioSourceMatcher("ZAUDIO (MEDIA_PLAYER)",DIA_EN_AUDIO_SOURCE_MEDIA_PLAYER);
   mSourceMapping[0x0A] = AudioSourceMatcher("CD",DIA_EN_AUDIO_SOURCE_MEDIA_CDDA);
   mSourceMapping[0x0B] = AudioSourceMatcher("DVD (NOT_SUPPORTED)",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x0C] = AudioSourceMatcher("TCU_SERVICE_CALL",DIA_EN_AUDIO_SOURCE_DCM);
   mSourceMapping[0x0D] = AudioSourceMatcher("RSE (NOT_SUPPORTED)",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x0E] = AudioSourceMatcher("AUX_1",DIA_EN_AUDIO_SOURCE_AUX_1);
   mSourceMapping[0x0F] = AudioSourceMatcher("AUX_2",DIA_EN_AUDIO_SOURCE_AUX_2);
   mSourceMapping[0x10] = AudioSourceMatcher("PHONE",DIA_EN_AUDIO_SOURCE_PHONE);
   mSourceMapping[0x11] = AudioSourceMatcher("PHONE_BTAUDIO",DIA_EN_AUDIO_SOURCE_PHONE_BTAUDIO);
   mSourceMapping[0x12] = AudioSourceMatcher("TA_DAB",DIA_EN_AUDIO_SOURCE_TUNER_TA_DAB);
   mSourceMapping[0x13] = AudioSourceMatcher("TA_FM",DIA_EN_AUDIO_SOURCE_TUNER_TA_FM);
   mSourceMapping[0x14] = AudioSourceMatcher("NAVI_SPEECH",DIA_EN_AUDIO_SOURCE_NAVI_SPEECH);
   mSourceMapping[0x15] = AudioSourceMatcher("TUNER_DRM",DIA_EN_AUDIO_SOURCE_TUNER_DRM);
   mSourceMapping[0x16] = AudioSourceMatcher("PHONE_ECALL",DIA_EN_AUDIO_SOURCE_UNKNOWN);
   mSourceMapping[0x17] = AudioSourceMatcher("MICROPHONE2",DIA_EN_AUDIO_SOURCE_MICROPHONE2);
   mSourceMapping[0xFF] = AudioSourceMatcher("UNKNOWN",DIA_EN_AUDIO_SOURCE_UNKNOWN);
}
//-----------------------------------------------------------------------------

tDiaResult
AudioManager::tearDown ( void )
{
   if ( mpFSM )
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }

   mpClient = 0;

   (void) unsetSysAdapterListener<dia_IRemoteControlListener>(this);

   (void) clearAudioSourceMatchers();

   return DIA_SUCCESS;
} //lint !e1578 Warning: prioX: Pointer member neither freed nor zeroed by cleanup function --> this is the cleanup method called from within class destructor

//------------------------------------------------------------------------------

void
AudioManager::resetClient ( void )
{
   ScopeTrace oTrace("AudioManager::resetClient()");

   if ( mpClient )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evReturnControl,0);
      mpClient = 0;
   }
}
void
AudioManager::addSourceMapping ( tU8 udsID, dia::UID srcID, const char* srcName )
{
   DIA_TR_INF("AudioManager::addSourceMapping source UID %d",srcID);
   mSourceMapping[udsID] = AudioSourceMatcher(srcName,srcID);
}

//------------------------------------------------------------------------------

tDiaResult
AudioManager::setAudioSource ( dia_eAudioSource sourceID, AudioManagerClient& client )
{
   ScopeTrace oTrace("AudioManager::setAudioSource()");

   DIA_TR_INF("AudioManager::setAudioSource(): sourceID=%d (0x%08x)", sourceID, sourceID);

   mErrorInfo = DIA_SUCCESS;
   mRequestedAudioSource = sourceID;
   mActiveAudioSource = DIA_EN_AUDIO_SOURCE_UNKNOWN;
   mpClient = &client;

   (void) acceptEvent( dia_AudioSourceSetFSM::evSetSource,0);

   DIA_TR_INF("AudioManager::setAudioSource(): mErrorInfo = 0x%08x", mErrorInfo);

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);
      (void) unsetSysAdapterListener<dia::IAudioModeControlListener>(this);
      DIA_TR_ERR("AudioManager::setAudioSource() --- ERROR DETECTED !!!");
   }

   return mErrorInfo;
}

//------------------------------------------------------------------------------

tDiaResult
AudioManager::getAudioSource ( AudioManagerClient& client )
{
   ScopeTrace oTrace("AudioManager::getAudioSource()");

   mErrorInfo = DIA_SUCCESS;
   mActiveAudioSource = DIA_EN_AUDIO_SOURCE_UNKNOWN;
   mpClient = &client;

   (void) acceptEvent( dia_AudioSourceSetFSM::evGetActiveSource,0);

   DIA_TR_INF("AudioManager::getAudioSource(): mErrorInfo = 0x%08x", mErrorInfo);

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);
      (void) unsetSysAdapterListener<dia::IAudioModeControlListener>(this);
      DIA_TR_ERR("AudioManager::getAudioSource() --- ERROR DETECTED !!!");
   }

   return mErrorInfo;
}

//-----------------------------------------------------------------------------

void
AudioManager::vOnAudioSourceList ( const std::vector<dia_tAudioSourceType>& audioSrcList )
{
   ScopeTrace oTrace("AudioManager::vOnAudioSourceList");

   DIA_TR_INF("AudioManager::vOnAudioSourceList search sourceEnum 0x%02x .", mRequestedAudioSource);

   for ( tU8 i=0; (i < audioSrcList.size()) && (!mIsSourceSupported); i++)
   {
      DIA_TR_INF("AudioManager::vOnAudioSourceList check sourceEnum 0x%02x with devID 0x%02x .", audioSrcList.at(i).mSourceEnum, audioSrcList.at(i).mSourceID);

      if ( audioSrcList.at(i).mSourceEnum == mRequestedAudioSource )
      {
         DIA_TR_INF("AudioManager::vOnAudioSourceList found sourceEnum 0x%02x with devID 0x%02x .", audioSrcList.at(i).mSourceEnum, audioSrcList.at(i).mSourceID);
         mAudioSource = audioSrcList.at(i);
         mIsSourceSupported = true;
      }
   }

   (void) acceptEvent( dia_AudioSourceSetFSM::evSourceUpdate,0);
}

//-----------------------------------------------------------------------------

void
AudioManager::vOnAudioSinkList ( const std::vector<dia_tAudioSinkType>& audioSinkList )
{
   ScopeTrace oTrace("AudioManager::vOnAudioSinkList");

   for ( tU8 i=0; (i < audioSinkList.size()) && (!mIsSinkSupported); i++)
   {
      DIA_TR_INF("AudioManager::vOnAudioSinkList check sink 0x%02x with ID 0x%02x .", audioSinkList.at(i).mSinkEnum, audioSinkList.at(i).mSinkID);

      if ( audioSinkList.at(i).mSinkEnum == mRequestedAudioSink )
      {
         DIA_TR_INF("AudioManager::vOnAudioSinkList found sink 0x%02x with ID 0x%02x .", audioSinkList.at(i).mSinkEnum, audioSinkList.at(i).mSinkID);
         mAudioSink = audioSinkList.at(i);
         mIsSinkSupported = true;
      }
   }

   (void) acceptEvent( dia_AudioSourceSetFSM::evSinkUpdate,0);
}

//-----------------------------------------------------------------------------

void
AudioManager::vOnAudioConnectionList ( const std::vector<dia_tAudioConnectionType>& audioConnectionList )
{
   ScopeTrace oTrace("AudioManager::vOnAudioConnectionList");

   mIsConnected        = false;
   mIsConnectionActive = false;
   mIsConnectionFound  = false;
   mIsActiveConnectionAvailable = false;

   std::vector<dia_tAudioConnectionType>::const_iterator iter = audioConnectionList.begin();
   for ( ; iter != audioConnectionList.end(); ++iter )
   {
      dia_tAudioConnectionType connection = (*iter);
      DIA_TR_INF("AudioManager::vOnAudioConnectionList check (source=0x%02X, sink=0x%02X, state=0x%02X).", connection.mSource.mSourceEnum, connection.mSink.mSinkEnum, connection.mState);

      if ( ( connection.mState == DIA_EN_AUDIO_CONNECTION_STATE_CONNECTED ) ) //|| ( connection.mState == DIA_EN_AUDIO_CONNECTION_STATE_CONNECTING ) )
      {
            mIsActiveConnectionAvailable = true;
			mActiveAudioSource = connection.mSource.mSourceEnum;
      }

      if ( (connection.mSource == mAudioSource) && (connection.mSink == mAudioSink ) )
      {
         mIsConnectionFound = true;
         if ( connection.mState == DIA_EN_AUDIO_CONNECTION_STATE_CONNECTED )
         {
            mIsConnected = true;
            mIsConnectionActive = true;
         }
         else if ( connection.mState == DIA_EN_AUDIO_CONNECTION_STATE_CONNECTING )
         {
            mIsConnected = true;
         }

         DIA_TR_INF("AudioManager::vOnAudioConnectionList found (connected=%s, active=%s).", (mIsConnected) ? "true" : "false", (mIsConnectionActive) ? "true" : "false");
      }
   }

   (void) acceptEvent( dia_AudioSourceSetFSM::evConnectionUpdate,0);
}

//-----------------------------------------------------------------------------

void
AudioManager::vOnAudioConnectionStateChanged ( const dia_tAudioConnectionType& audioConnection )
{
   ScopeTrace oTrace("AudioManager::vOnAudioConnectionStateChanged");

   DIA_TR_INF("AudioManager::vOnAudioConnectionStateChanged check (source=0x%02X, sink=0x%02X, state=0x%02X).", audioConnection.mSource.mSourceEnum, audioConnection.mSink.mSinkEnum, audioConnection.mState);

   if ( (mAudioSource == audioConnection.mSource) && (mAudioSink == audioConnection.mSink) )
   {
      mIsConnectionFound = true;
      if ( audioConnection.mState == DIA_EN_AUDIO_CONNECTION_STATE_CONNECTED )
      {
         mIsConnected = true;
         mIsConnectionActive = true;
         mIsActiveConnectionAvailable = true;
      }
      else if ( audioConnection.mState == DIA_EN_AUDIO_CONNECTION_STATE_CONNECTING )
      {
         mIsConnected = true;
      }

      DIA_TR_INF("AudioManager::vOnAudioConnectionStateChanged found connection (connected = %s, active =%s)", (mIsConnected) ? "true" : "false", (mIsConnectionActive) ? "true" : "false");
   }

   (void) acceptEvent( dia_AudioSourceSetFSM::evConnectionUpdate,0);
}

//-----------------------------------------------------------------------------

void
AudioManager::vOnAudioConnectionFailure ( tDiaResult errCode )
{
   ScopeTrace oTrace("AudioManager::vOnAudioConnectionFailure");

   mErrorInfo = errCode;
   (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);

}

//-----------------------------------------------------------------------------

void
AudioManager::vOnAudioModeControlUpdate ( const AudioMode& mode )
{
   ScopeTrace oTrace("AudioManager::vOnAudioModeControlUpdate");

   if ( mode.getMode() == dia::DIA_EN_AUDIO_MODE_DIAG_REMOTE_CONTROL )
   {
      mIsAudioDiagModeActive = mode.getState();
   }

   (void) acceptEvent( dia_AudioSourceSetFSM::evAudioDiagModeUpdate,0);
}

//-----------------------------------------------------------------------------

void
AudioManager::vOnRemoteControlModeChanged ( dia_eRemoteCtrlMode newMode, dia_eRemoteCtrlMode prevMode )
{
   ScopeTrace oTrace("AudioManager::vOnRemoteControlModeChanged");

   if ( (newMode == DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE) && (prevMode == DIA_EN_REMOTE_CONTROL_MODE_ACTIVE) )
   {
      (void) acceptEvent(dia_AudioSourceSetFSM::evReturnControl,0);
   }
}

//-----------------------------------------------------------------------------

tDiaResult
AudioManager::acceptEvent ( dia_AudioSourceSetFSM::FsmEvent event, void* pArg )
{
   ScopeTrace oTrace("AudioManager::acceptEvent(dia_AudioSourceSetFSM::FsmEvent,void*)");

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;

   DIA_TR_INF("AudioManager::acceptEvent - State (before): %s (Event = %s)", mpFSM->getStateName(), dia_AudioSourceSetFSM::getEventName(event));
   mpFSM->acceptEvent(event,pArg);
   DIA_TR_INF("AudioManager::acceptEvent - State (after) : %s", mpFSM->getStateName());

   return mErrorInfo;
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmCheckForAudioDiagMode ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmCheckForAudioDiagMode");

   mIsAudioDiagModeActive = (getInstanceOfRemoteControl()->getRemoteControlMode() == DIA_EN_REMOTE_CONTROL_MODE_ACTIVE) ? true : false;
   mNeedToRequestAudioDiagMode = mIsAudioDiagModeActive ? false : true;

   (void) acceptEvent( dia_AudioSourceSetFSM::evCheckedAudioDiagMode,0);
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmFinalize ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmFinalize");
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmInitialize ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmInitialize");

   mErrorInfo = DIA_E_NO_ERROR;
   (void) setSysAdapterListener<dia_IAudioSourceListener>(this);
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmRequestAudioDiagMode ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmRequestAudioDiagMode");

   if ( mIsAudioDiagModeActive )
   {
      DIA_TR_INF("AudioManager::vFsmRequestAudioDiagMode => AUDIO DIAGNOSIS MODE ALREADY ACTIVE !");
      return;
   }

   dia::IAudioModeControl* pInterface = 0;
   if ( (querySysAdapterInterface<dia::IAudioModeControl>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      (void) setSysAdapterListener<dia::IAudioModeControlListener>(this);
      AudioMode mode(DIA_EN_AUDIO_MODE_DIAG_REMOTE_CONTROL,true);
      if ( pInterface->setAudioMode(mode) != DIA_SUCCESS )
      {
         DIA_TR_ERR("AudioManager::vFsmRequestAudioDiagMode => ERROR: FAILED TO REQUEST AUDIO DIAGNOSIS MODE !");
         mErrorInfo = DIA_E_CONDITIONS_NOT_CORRECT;
      }
   }
   else
   {
      DIA_TR_ERR("AudioManager::vFsmRequestAudioDiagMode => ERROR: FAILED TO RETRIEVE AUDIO MODE CONTROL INTERFACE !");
      mErrorInfo = DIA_E_INTERFACE_NOT_AVAILABLE;
   }

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmRequestAudioNormalMode ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmRequestAudioNormalMode");

   if ( !mIsAudioDiagModeActive )
   {
      DIA_TR_INF("AudioManager::vFsmRequestAudioDiagMode => AUDIO NORMAL MODE ALREADY ACTIVE !");
      return;
   }

   if ( !mNeedToRequestAudioDiagMode )
   {
      DIA_TR_INF("AudioManager::vFsmRequestAudioDiagMode => AUDIO DIAG MODE WAS NOT REQUESTED !");
      return;
   }

   dia::IAudioModeControl* pInterface = 0;
   if ( (querySysAdapterInterface<dia::IAudioModeControl>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      AudioMode mode(DIA_EN_AUDIO_MODE_DIAG_REMOTE_CONTROL,false);
      if ( pInterface->setAudioMode(mode) != DIA_SUCCESS )
      {
         DIA_TR_ERR("AudioManager::vFsmRequestAudioNormalMode => ERROR: FAILED TO REQUEST AUDIO NORMAL MODE !");
         mErrorInfo = DIA_E_CONDITIONS_NOT_CORRECT;
      }
   }
   else
   {
      DIA_TR_ERR("AudioManager::vFsmRequestAudioNormalMode => ERROR: FAILED TO RETRIEVE AUDIO MODE CONTROL INTERFACE !");
      mErrorInfo = DIA_E_INTERFACE_NOT_AVAILABLE;
   }

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmRequestConnection ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmRequestConnection");

   dia_IAudioSource* pInterface = 0;
   if ( (querySysAdapterInterface<dia_IAudioSource>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      if ( pInterface->setAudioSource(mAudioSource.mSourceID,mAudioSink.mSinkID) != DIA_SUCCESS )
      {
         DIA_TR_ERR("AudioManager::vFsmRequestConnection => ERROR: FAILED TO REQUEST AUDIO CONNECTION !");
         mErrorInfo = DIA_E_CONDITIONS_NOT_CORRECT;
      }
   }
   else
   {
      DIA_TR_ERR("AudioManager::vFsmRequestConnection => ERROR: FAILED TO RETRIEVE AUDIO SOURCE INTERFACE !");
      mErrorInfo = DIA_E_INTERFACE_NOT_AVAILABLE;
   }

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmRequestListOfConnections ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmRequestListOfConnections");

   dia_IAudioSource* pInterface = 0;
   if ( (querySysAdapterInterface<dia_IAudioSource>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      if ( pInterface->getAudioConnections() != DIA_SUCCESS )
      {
         DIA_TR_ERR("AudioManager::vFsmRequestListOfConnections => ERROR: FAILED TO REQUEST AUDIO CONNECTION LIST !");
         mErrorInfo = DIA_E_CONDITIONS_NOT_CORRECT;
      }
   }
   else
   {
      DIA_TR_ERR("AudioManager::vFsmRequestListOfConnections => ERROR: FAILED TO RETRIEVE AUDIO SOURCE INTERFACE !");
      mErrorInfo = DIA_E_INTERFACE_NOT_AVAILABLE;
   }

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmRequestListOfSinks ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmRequestListOfSinks");

   dia_IAudioSource* pInterface = 0;
   if ( (querySysAdapterInterface<dia_IAudioSource>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      if ( pInterface->getAudioSinkList() != DIA_SUCCESS )
      {
         DIA_TR_ERR("AudioManager::vFsmRequestListOfSinks => ERROR: FAILED TO REQUEST AUDIO SINK LIST !");
         mErrorInfo = DIA_E_CONDITIONS_NOT_CORRECT;
      }
   }
   else
   {
      DIA_TR_ERR("AudioManager::vFsmRequestListOfSinks => ERROR: FAILED TO RETRIEVE AUDIO SOURCE INTERFACE !");
      mErrorInfo = DIA_E_INTERFACE_NOT_AVAILABLE;
   }

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmRequestListOfSources ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmRequestListOfSources");

   dia_IAudioSource* pInterface = 0;
   if ( (querySysAdapterInterface<dia_IAudioSource>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      if ( pInterface->getAudioSourceList() != DIA_SUCCESS )
      {
         DIA_TR_ERR("AudioManager::vFsmRequestListOfSources => ERROR: FAILED TO REQUEST AUDIO SOURCE LIST !");
         mErrorInfo = DIA_E_CONDITIONS_NOT_CORRECT;
      }
   }
   else
   {
      DIA_TR_ERR("AudioManager::vFsmRequestListOfSources => ERROR: FAILED TO RETRIEVE AUDIO SOURCE INTERFACE !");
      mErrorInfo = DIA_E_INTERFACE_NOT_AVAILABLE;
   }

   if ( mErrorInfo != DIA_SUCCESS )
   {
      (void) acceptEvent( dia_AudioSourceSetFSM::evError,0);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmReset ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmReset");

   mRequestedAudioSource         = DIA_EN_AUDIO_SOURCE_UNKNOWN;
   mActiveAudioSource            = DIA_EN_AUDIO_SOURCE_UNKNOWN;
   mAudioSource                  = dia_tAudioSourceType();
   mIsSourceSupported            = false;
   mRequestedAudioSink           = DIA_EN_AUDIO_SINK_AMPLIFIER_MAINFIELD;
   mAudioSink                    = dia_tAudioSinkType();
   mIsSinkSupported              = false;
   mIsConnected                  = false;
   mIsConnectionActive           = false;
   mIsConnectionFound            = false;
   mIsActiveConnectionAvailable  = false;

   (void) unsetSysAdapterListener<dia_IAudioSourceListener>(this);
   (void) unsetSysAdapterListener<dia::IAudioModeControlListener>(this);
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmSendResponse ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmSendResponse");

   DIA_TR_INF("AudioManager::vFsmSendResponse (%s) !!!", (mErrorInfo == DIA_SUCCESS) ? "SUCCESSFUL" : "DIA_FAILED");

   if ( mpClient)
   {
      mpClient->onAudioSourceSetResult(mRequestedAudioSource,mErrorInfo);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmSendGetActiveSourceResponse ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmSendGetActiveSourceResponse");

   if ( mpClient)
   {
      mpClient->onAudioSourceGetResult(mActiveAudioSource);
   }
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmSetError_ConnectionFailed ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmSetError_ConnectionFailed");
   mErrorInfo = DIA_E_AUDIO_CONNECTION_FAILED;
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmSetError_ConnectionNotSupported ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmSetError_ConnectionNotSupported");
   mErrorInfo = DIA_E_AUDIO_CONNECTION_NOT_SUPPORTED;
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmSetError_SinkNotSupported ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmSetError_SinkNotSupported");
   mErrorInfo = DIA_E_AUDIO_SINK_NOT_SUPPORTED;
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmSetError_SourceNotSupported ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmSetError_SourceNotSupported");
   mErrorInfo = DIA_E_AUDIO_SOURCE_NOT_SUPPORTED;
}

//-----------------------------------------------------------------------------

void
AudioManager::vFsmNoActiveSourceFound ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::vFsmNoActiveSourceFound");
   mActiveAudioSource = DIA_EN_AUDIO_SOURCE_UNKNOWN;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isAudioDiagModeActive ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isAudioDiagModeActive");
   return mIsAudioDiagModeActive;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isConnected ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isConnected");
   return mIsConnected;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isConnectionActive ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isConnectionActive");
   return mIsConnectionActive;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isConnectionFound ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isConnectionFound");
   return mIsConnectionFound;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isSinkSupported ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isSinkSupported");
   return mIsSinkSupported;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isSourceSupported ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isSourceSupported");
   return mIsSourceSupported;
}

//-----------------------------------------------------------------------------

bool
AudioManager::isActiveConnectionAvailable ( void* /*pArg*/ )
{
   ScopeTrace trc("AudioManager::isActiveConnectionAvailable");
   return mIsActiveConnectionAvailable;
}

//---------------------------------------------------------------------------------------------------------------------

struct AudioSourceMatcherComparator
{
   AudioSourceMatcherComparator ( const AudioSourceMatcher& matcher ) : mMatcher(matcher) {}
   bool operator() ( const AudioSourceMatcher& matcher ) const { return matcher == mMatcher;}

protected:
   AudioSourceMatcherComparator ( void );

private:
   const AudioSourceMatcher& mMatcher;
};

//------------------------------------------------------------------------------

tDiaResult
AudioManager::addAudioSourceMatcher ( const AudioSourceMatcher& matcher )
{
   ScopeTrace oTrace("dia::AudioManager::addAudioSourceMatcher*)");

   tDiaResult retCode = DIA_FAILED;

   std::list<AudioSourceMatcher>::iterator iter = std::find_if(mAudioSourceMatcherRep.begin(),mAudioSourceMatcherRep.end(),AudioSourceMatcherComparator(matcher));
   if ( iter == mAudioSourceMatcherRep.end() )
   {
      DIA_TR_INF("ADDING AUDIO SOURCE MATCHER \"%s\" (UID=0x%08x)", matcher.mNamePattern.c_str(),matcher.mSrcID);
      mAudioSourceMatcherRep.push_back(matcher);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
AudioManager::clearAudioSourceMatchers ( void )
{
   mAudioSourceMatcherRep.clear();
   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

AudioSourceMatcher::AudioSourceMatcher ( void )
   : mSrcID(DIA_EN_AUDIO_SOURCE_UNKNOWN),
     mNamePattern("UNKNOWN_AUDIO_SOURCE_MATCHER"),
     mStartPos(DIA_C_INT_SOURCE_NAME_STARTPOS_INVALID),
     mEndPos(DIA_C_INT_SOURCE_NAME_ENDPOS_INVALID),
     mIsDynamicSource(false),
     mStartPosDeviceID(DIA_C_INT_DEVICE_ID_STARTPOS_INVALID)
{}

//------------------------------------------------------------------------------

AudioSourceMatcher::AudioSourceMatcher ( const char* namePattern, UID srcID )
   : mSrcID(srcID),
     mNamePattern(namePattern),
     mStartPos(DIA_C_INT_SOURCE_NAME_STARTPOS_INVALID),
     mEndPos(DIA_C_INT_SOURCE_NAME_ENDPOS_INVALID),
     mIsDynamicSource(false),
     mStartPosDeviceID(DIA_C_INT_DEVICE_ID_STARTPOS_INVALID)
{}

//------------------------------------------------------------------------------

AudioSourceMatcher::AudioSourceMatcher ( const char* namePattern, UID srcID, int startPos, int endPos )
   : mSrcID(srcID),
     mNamePattern(namePattern),
     mStartPos(startPos),
     mEndPos(endPos),
     mIsDynamicSource(false),
     mStartPosDeviceID(DIA_C_INT_DEVICE_ID_STARTPOS_INVALID)
{}

//------------------------------------------------------------------------------

AudioSourceMatcher::AudioSourceMatcher ( const char* namePattern, UID srcID, int startPos, int endPos, int startPosDevID )
   : mSrcID(srcID),
     mNamePattern(namePattern),
     mStartPos(startPos),
     mEndPos(endPos),
     mIsDynamicSource(true),
     mStartPosDeviceID(startPosDevID)
{}

//------------------------------------------------------------------------------

AudioSourceMatcher::AudioSourceMatcher ( const AudioSourceMatcher& matcher )
   : mSrcID(matcher.mSrcID),
     mNamePattern(matcher.mNamePattern),
     mStartPos(matcher.mStartPos),
     mEndPos(matcher.mEndPos),
     mIsDynamicSource(matcher.mIsDynamicSource),
     mStartPosDeviceID(matcher.mStartPosDeviceID)
{}

//------------------------------------------------------------------------------

AudioSourceMatcher::~AudioSourceMatcher ( void )
{}

//------------------------------------------------------------------------------

AudioSourceMatcher&
AudioSourceMatcher::operator = ( const AudioSourceMatcher& obj )
{
   if ( this != &obj )
   {
      mSrcID            = obj.mSrcID;
      mNamePattern      = obj.mNamePattern;
      mStartPos         = obj.mStartPos;
      mEndPos           = obj.mEndPos;
      mIsDynamicSource  = obj.mIsDynamicSource;
      mStartPosDeviceID = obj.mStartPosDeviceID;
   }
   return *this;
}

//------------------------------------------------------------------------------

bool
AudioSourceMatcher::operator == ( const AudioSourceMatcher& matcher ) const
{
   if ( mSrcID != matcher.mSrcID ) return false;
   if ( mStartPos != matcher.mStartPos ) return false;
   if ( mEndPos != matcher.mEndPos ) return false;
   if ( mIsDynamicSource != matcher.mIsDynamicSource ) return false;
   if ( mStartPosDeviceID != matcher.mStartPosDeviceID ) return false;
   return (mNamePattern == matcher.mNamePattern) ? true : false;
}

}
