/*!
 * \file       dia_FactoryPluginPDXFlashing.cpp
 *
 * \brief      Factory used to instantiate/shutdown the flashing engine
 *
 * \details    Factory used to instantiate/shutdown the flashing engine
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreFactory
 *
 * \copyright  (c) 2011-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include "dia_FactoryPluginPDXFlashing.h"

#ifndef __INCLUDED_DIA_COMMON_FACTORY__
#include "common/framework/factory/dia_common_factory.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY_METHODS__
#include "common/framework/factory/dia_factoryMethods.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_FLASH_PDX__
#include <common/framework/engine/dia_EngineFlash.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_CONFIGURATION_FLASH_PDX__
#include <common/framework/engine/dia_EngineFlashConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include "common/framework/application/dia_File.h"
#endif

#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <string>

#ifndef __DIA_UNIT_TESTING__

dia::FactoryPluginPDXFlashing*
getInstanceOfFactoryPluginPDXFlashing ( void )
{
   return dia::FactoryPluginPDXFlashing::getInstance();
}

void
releaseInstanceOfFactoryPluginPDXFlashing ( void )
{
   dia::FactoryPluginPDXFlashing::deleteInstance();
}

#endif

static int filterPDXContainers ( const struct dirent* dir );

namespace dia {

// implementation of the singleton methods
DIA_IMPL_SINGLETON(FactoryPluginPDXFlashing)

std::list<std::string>
FactoryPluginPDXFlashing::mPDXFolderRep;

std::list<std::string>
FactoryPluginPDXFlashing::mPDXFileRep;

//-----------------------------------------------------------------------------

FactoryPluginPDXFlashing::FactoryPluginPDXFlashing ( void )
   : dia_ObjectWithUID("dia::FactoryPluginPDXFlashing"),
     mpEngine(0),
     mpConfig(0),
     mSyncObj(std::string(std::string("dia::FactoryPluginPDXFlashing") + "_LK").c_str()),
     mStartModeAutomatic(true),
     mIsSetupDone(false)
{}

//-----------------------------------------------------------------------------

FactoryPluginPDXFlashing::~FactoryPluginPDXFlashing ( void )
{
   mpEngine = 0;
   mpConfig = 0;
}

//-----------------------------------------------------------------------------

void
FactoryPluginPDXFlashing::addPDXFolder ( const std::string& pdxFolder )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::addPDXFolder(const std::string&)");

   if ( std::find(mPDXFolderRep.begin(),mPDXFolderRep.end(),pdxFolder) == mPDXFolderRep.end() )
   {
      DIA_TR_INF("##### ADDING PDX FOLDER = '%s'",pdxFolder.c_str());
      mPDXFolderRep.push_back(pdxFolder);
   }

   DIA_TR_INF("######################################################################");
   DIA_TR_INF("#");
   DIA_TR_INF("# NUMBER OF LOADED PDX CONTAINERS: %zu", mPDXFolderRep.size());
   DIA_TR_INF("#");
   DIA_TR_INF("######################################################################");

}

//-----------------------------------------------------------------------------

void
FactoryPluginPDXFlashing::setStartMode ( bool mode )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::setStartMode");
   mStartModeAutomatic = mode;
   DIA_TR_INF("##### STARTMODE = '%s'",mStartModeAutomatic ? "AUTOMATIC" : "MANUAL");
}

//-----------------------------------------------------------------------------

tDiaResult
FactoryPluginPDXFlashing::setup ( void )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::setup");

   if ( mIsSetupDone ) return DIA_E_ALREADY_STARTED;

   dia_LockScope mScopeLock(mSyncObj);

   // create the configuration object
   static std::string /*tCString*/ strConfigName = "dia::EngineFlashConfiguration";
   (void) makeEngineConfiguration(strConfigName);
   if ( mpConfig )
   {
      // create the engine instance and configure it
      DIA_TR_INF("###############################################");
      DIA_TR_INF("# MAKE FLASH ENGINE                           #");
      DIA_TR_INF("###############################################");

      (void) makeEngine(DIA_NAME_ENGINE_PDXFLASHING_UDS,*mpConfig); //DIA_NEW EngineFlash(DIA_NAME_ENGINE_PDXFLASHING_UDS,*mpConfig);

      DIA_TR_INF("###############################################");

      if ( mpEngine )
      {
         if ( mpEngine->setup() != DIA_SUCCESS )
         {
            DIA_TR_ERR("### SETUP OF PDX-Flashing-ENGINE FAILED !!! ###");
            DIA_DELETE mpEngine;
            mpEngine = 0;
         }
         else
         {
            DIA_TR_INF("###############################################");
            DIA_TR_INF("# START CONTROL (AUTOMATIC MODE = %s)",mStartModeAutomatic ? "TRUE" : "FALSE");
            DIA_TR_INF("###############################################");

            std::vector<tArgsType> vecArgs;
            if ( mpEngine->startControl(vecArgs,mStartModeAutomatic) != DIA_SUCCESS )
            {
               DIA_TR_ERR("### START OF PDX-FLASHING-ENGINE FAILED !!! ###");
            }

            mIsSetupDone = true;
         }
      }
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
FactoryPluginPDXFlashing::tearDown ( void )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::tearDownObjects");

   dia_LockScope mScopeLock(mSyncObj);

   if ( mpEngine && (mpEngine->shutdown() != DIA_SUCCESS) ) return DIA_FAILED;
   OSAL_DELETE mpEngine;
   mpEngine = 0;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
FactoryPluginPDXFlashing::startFlashing ( void )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::startFlashing");

   if ( mpEngine ) mpEngine->onStartFlashingEvent();
}

//-----------------------------------------------------------------------------

tDiaResult
FactoryPluginPDXFlashing::makeEngineConfiguration ( const std::string& name )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::makeEngineConfiguration");

   mpConfig = DIA_NEW EngineFlashConfiguration(name);

   return (mpConfig) ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
FactoryPluginPDXFlashing::makeEngine ( const std::string& name, EngineFlashConfiguration& config )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::makeEngine");

   mpEngine = DIA_NEW EngineFlash(name,config);

   return ( mpEngine ) ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

const std::list<std::string>&
FactoryPluginPDXFlashing::getPDXContainers ( void )
{
   mPDXFileRep.clear();

   std::list<std::string>::const_iterator pdxFolderIter = mPDXFolderRep.begin();
   for ( ; pdxFolderIter != mPDXFolderRep.end(); ++pdxFolderIter )
   {
      (void) scanPDXContainers(*pdxFolderIter);
   }

   return mPDXFileRep;
}

//-----------------------------------------------------------------------------

tDiaResult
FactoryPluginPDXFlashing::scanPDXContainers ( const std::string& folderName )
{
   ScopeTrace oTrace("dia::FactoryPluginPDXFlashing::scanPDXContainers(std::string&)");

   struct dirent** nameList = 0;

   int count = ::scandir(folderName.c_str(), &nameList, filterPDXContainers, alphasort) ;

   if ( count >= 0 )
   {
      for( int i = 0; i < count; i++ )
      {
         if ( !nameList[i] ) continue;
         DIA_TR_INF("scanPDXContainers: nameList[%02d]->d_name = %s",i,nameList[i]->d_name);

         std::string fullPdxFileName(folderName);
         fullPdxFileName.append("/");
         fullPdxFileName.append(nameList[i]->d_name);
         DIA_TR_INF("  + Found PDX container '%s'",fullPdxFileName.c_str());
         mPDXFileRep.push_back(fullPdxFileName);
         ::free(nameList[i]) ;
      }
      ::free(nameList);
   }
   else
   {
      //Something went wrong. I trace out errno information.
      DIA_TR_ERR("scanPDXContainers: scandir() returned with error! (err='%s')",strerror(errno));
      return DIA_FAILED;
   }

   return DIA_SUCCESS;


//   dia_FileDir dir(folderName);
//   if ( !dir.doesExist() ) return DIA_E_FILE_DOES_NOT_EXIST;
//
//   std::string cmdString = "ls " + folderName + "/*.pdx"; // command used to retrieve the list of shared libraries in the given folder
//   FILE* cmdPipe = ::popen(cmdString.c_str(), "r");
//   if ( !cmdPipe )
//   {
//      ::perror("popen");
//      return DIA_FAILED;
//   }
//
//   char buffer [MAX_BUFFER_SIZE];
//   char pdxFileName[MAX_BUFFER_SIZE];
//
//   DIA_TR_INF("Scanning folder '%s' for *.pdx files",folderName.c_str());
//
//   while ( ::fgets(buffer, MAX_BUFFER_SIZE, cmdPipe) )
//   {
//      // trim off the whitespace
//      char* whitespace = ::strpbrk(buffer," \t\n");
//      if ( whitespace ) *whitespace = '\0';
//      // append "$folderName/" to the front of the library name
//      ::sprintf(pdxFileName, "%s", buffer);
//
//      std::string fullPdxFileName(pdxFileName);
//      size_t found = fullPdxFileName.find_last_of("/\\");
//      std::string strippedPdxFileName(fullPdxFileName.substr(found+1));
//      DIA_TR_INF("  + Found PDX container '%s'",strippedPdxFileName.c_str());
//      mPDXFileRep.push_back(fullPdxFileName);
//   }
//
//   (void) ::pclose(cmdPipe);

   return DIA_SUCCESS;
}

//
//int main() {
//   string dir_path = "/home/tom/TEST";
//
//   struct dirent **name_list ;
//   int count ;
//
//   count = scandir(dir_path.c_str(), &name_list, pdx_filter, alphasort) ;
//
//   if( count >= 0 ){
//      for( int i = 0; i < count; i++ ){
//         cout << name_list[i]->d_name << "\n";
//          free(name_list[i]) ;
//      }
//      free(name_list);
//   }else{
//      //Something went wrong. I trace out errno information.
//      cout << "Ups, scandir() returned with error!\n";
//      cout << "errno: "<< strerror(errno) << endl;
//   }
//
//   return 0;
//}


}

int
filterPDXContainers ( const struct dirent* dir )
{
   size_t len = ::strlen( dir->d_name );

   if( len < 4 ) return 0 ;

   return ::strcmp(dir->d_name+len-4,".pdx") == 0 ;
}

