/*!
 * \file       dia_SessionADR.cpp
 *
 * \brief      Session for ADR3 related services. Service requests will be forwarded to ADR3 diagnosis core
 *
 * \details    Session for ADR3 related services. Service requests will be forwarded to ADR3 diagnosis core
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_SESSION_ADR__
#include "dia_SessionADR.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include "common/framework/engine/dia_LookupKey.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

//#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
//#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGNOSTICS_BASE
//#include "trcGenProj/Header/dia_SessionADR.cpp.trc.h"
//#endif

#include "../../ai_osal_linux/components/devices/dev_adr3ctrl/include/dev_adr3ctrl.h"

#define DIA_ADR_CONTROL_DRIVER_NAME             "/dev/adr3ctrl"

//-----------------------------------------------------------------------------

dia_SessionADR::dia_SessionADR ( tU8 sessionID, dia_EngineServer& engine )
   : dia_Session(sessionID,engine),
     dia_ServiceHandlerUDS("ADR3_BRIDGE"),
     mErrorCode(DIA_E_NOERROR),
     mpFSM(0)
{
   dia_tclFnctTrace oTrace("dia_SessionADR::dia_SessionADR ( tU8 sessionID, dia_EngineServer& engine )");
}

//-----------------------------------------------------------------------------

dia_SessionADR::~dia_SessionADR ( tVoid )
{
   _BP_TRY_BEGIN
   {
        OSAL_DELETE mpFSM;
        mpFSM = 0;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SessionADR::~dia_SessionADR !!!");
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::initialize ( tVoid )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::initialize");

   // we initialize the state machine exactly once after creation
   if ( mpFSM ) return;

   mpFSM = OSAL_NEW dia_ADRSessionFSM::Fsm(this);

   if ( mpFSM ) mpFSM->init();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SessionADR::onEnterSession ( tVoid )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::onEnterSession()");

   mErrorCode = DIA_E_NOERROR;

   if ( mpFSM ) mpFSM->acceptEvent(dia_ADRSessionFSM::evActivate,0);

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SessionADR::onLeaveSession ( tVoid )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::onLeaveSession()");

   mErrorCode = DIA_E_NOERROR;

   if ( mpFSM ) mpFSM->acceptEvent(dia_ADRSessionFSM::evDeactivate,0);

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SessionADR::onResetSession ( tVoid )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::onResetSession()");

   mErrorCode = DIA_E_NOERROR;

   if ( mpFSM ) mpFSM->acceptEvent(dia_ADRSessionFSM::evReset,0);

   return mErrorCode;
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vProcessRequest ( const std::vector<tArgsType>& vecArgs )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vProcessRequest()");

   mErrorCode = DIA_E_NOERROR;

   if (vecArgs.size())
   {
      // Arguments are not handled in vFsmForwardRequest()
      DIA_ASSERT_ALWAYS();
   }

   if ( mpFSM ) mpFSM->acceptEvent(dia_ADRSessionFSM::evRequestReceived, 0);
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vOnADRContainerMessageResponse ( std::vector<tU8>& adrResponse )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vOnADRContainerMessageResponse()");

   mErrorCode = DIA_E_NOERROR;

   if ( mpFSM ) mpFSM->acceptEvent(dia_ADRSessionFSM::evADRContainerReceived,&adrResponse);
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmReset ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmReset()");

   // reset the internal error code
   mErrorCode = DIA_E_NOERROR;

   // session is not active
   mIsActive = FALSE;

   // clear the container for the results
   mResults.clear();

   // unregister for ADR3 notifications
   (tVoid) unsetSysAdapterListener<dia_IADRListener>(this);
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmOnSessionActivated ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmOnSessionActivated()");

   dia_SessionController* pSessionMgr =  dia_Session::mpEngine->getSessionController();
   if ( pSessionMgr )
   {
      pSessionMgr->vOnSessionUpdate(*this);
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmOnSessionDeactivated ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmOnSessionDeactivated()");
   dia_SessionController* pSessionMgr =  dia_Session::mpEngine->getSessionController();
   if ( pSessionMgr )
   {
      pSessionMgr->vOnSessionUpdate(*this);
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmSendActivation ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmSendActivation()");

   dia_IADR* pADR = OSAL_NULL;
   if ( querySysAdapterInterface<dia_IADR>(&pADR) == DIA_SUCCESS )
   {
      (tVoid) setSysAdapterListener<dia_IADRListener>(this);

      std::vector<tU8> adrMessage;
      adrMessage.push_back(0x10);   // session control service
      adrMessage.push_back(0x61);   // ADR session identifier

      if ( pADR->sendADRContainerMessage(adrMessage) != DIA_SUCCESS )
      {
         mErrorCode = DIA_FAILED;
      }
   }

   if ( mErrorCode != DIA_E_NOERROR )
   {
      (tVoid) unsetSysAdapterListener<dia_IADRListener>(this);
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmSendDeactivation ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmSendDeactivation()");

   dia_IADR* pADR = OSAL_NULL;
   if ( querySysAdapterInterface<dia_IADR>(&pADR) == DIA_SUCCESS )
   {
      (tVoid) setSysAdapterListener<dia_IADRListener>(this);

      // TODO: might be reset by ECU reset request
      std::vector<tU8> adrMessage;
      adrMessage.push_back(0x11);   // ECU reset service
      adrMessage.push_back(0x01);   // hard reset

      if ( pADR->sendADRContainerMessage(adrMessage) != DIA_SUCCESS )
      {
         mErrorCode = DIA_FAILED;
      }
   }

   if ( mErrorCode != DIA_E_NOERROR )
   {
      (tVoid) unsetSysAdapterListener<dia_IADRListener>(this);
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmEvalADRContainer ( tVoid* pArg )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmEvalADRContainer()");

   if ( pArg )
   {
      std::vector<tU8>* pADRContainer = (std::vector<tU8>*) pArg;
      if ( pADRContainer->size() >= 2 )
      {
         if ( pADRContainer->at(0) == 0x50 )
         {
            if ( pADRContainer->at(1) == 0x61 )
            {
               DIA_TR_INF("RECEIVED POSITIVE RESPONSE FOR SESSION ACTIVATION FROM ADR3 !!");
               mIsActive = TRUE;
            }
            else
            {
               DIA_TR_INF("RECEIVED RESPONSE FOR UNKNOWN SESSION FROM ADR3 !!");
            }
         }
         else if ( pADRContainer->at(0) == 0x51 )
         {
            if ( pADRContainer->at(1) == 0x01 )
            {
               DIA_TR_INF("RECEIVED POSITIVE RESPONSE FOR SESSION DEACTIVATION FROM ADR3 !!");
               mIsActive = FALSE;

               vResetADR3();
            }
            else
            {
               DIA_TR_INF("RECEIVED RESPONSE FOR UNKNOWN SESSION FROM ADR3 !!");
            }
         }
         else
         {
            DIA_TR_INF("RECEIVED NO SESSION RESPONSE FROM ADR3 !!");
         }
      }
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmForwardRequest ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmForwardRequest()");

   dia_IADR* pADR = OSAL_NULL;
   if ( querySysAdapterInterface<dia_IADR>(&pADR) == DIA_SUCCESS )
   {
      // copy the message buffer except the length info in the first byte
      std::vector<tU8> adrMessage;
      for ( tU16 i=1; i<oDiagMsgBuffer().u16GetDataLength(); i++ )
      {
         adrMessage.push_back(oDiagMsgBuffer().u8GetData(i));
      }

      (tVoid) setSysAdapterListener<dia_IADRListener>(this);

      if ( pADR->sendADRContainerMessage(adrMessage) != DIA_SUCCESS )
      {
         mErrorCode = DIA_FAILED;
      }
   }

   if ( mErrorCode != DIA_E_NOERROR )
   {
      (tVoid) unsetSysAdapterListener<dia_IADRListener>(this);
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmForwardResponse ( tVoid* pArg )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmForwardResponse()");

   if ( !pArg ) return;

   std::vector<tU8>* pADRResponse = (std::vector<tU8>*) pArg;

   oDiagMsgBuffer().vSetDataLength((tU16) pADRResponse->size());
   for ( tU16 i=0; i < pADRResponse->size(); i++ )
   {
      oDiagMsgBuffer().setDataU8(i, (*pADRResponse)[i]);
   }
   vResReadyAndQuit();

   (tVoid) unsetSysAdapterListener<dia_IADRListener>(this);
}

//-----------------------------------------------------------------------------

tVoid
dia_SessionADR::vFsmDeactivationFailed ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::vFsmDeactivationFailed()");
}

//-----------------------------------------------------------------------------

bool
dia_SessionADR::bIsActivationComplete ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::bIsActivationComplete()");
   return ( mIsActive ) ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_SessionADR::bIsDeactivationComplete ( tVoid* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia_SessionADR::bIsDeactivationComplete()");
   return ( mIsActive ) ? false : true;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SessionADR::makeLookupKeys ( std::vector<dia_LookupKey*>& keys )
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SessionADR::makeLookupKeys");
#endif

   if ( mLookupKeys.empty() )
   {
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_ACTUAL_CALIBRATION_DATA,            DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_ActualCalibrationData
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_ANALOG_TUNERINFO_DEMODULATOR_1,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_AnalogTunerInformation_Demodulator_1
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_ANALOG_TUNERINFO_DEMODULATOR_2,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_AnalogTunerInformation_Demodulator_2
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_VERSION_SW,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_VERSION_HW,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_VERSION_HDSW,                       DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_VERSION_HDSPL,                      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_FRONTEND_CONFIGURATION_ID,          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_FrontendConfigurationID
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_BACKEND_AGC_LEVEL,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_BackendADCLevel
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_BOOT_MODE,                          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_BootMode
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_FLASH_CHIP_ID,                      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_FlashChipID
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_AMPLIFIER_CONFIGURATION,            DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - ADR3_AmplifierConfiguration

      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_MEMORY_BY_ADDRESS,   (tU8) DIA_C_U16_DID_ADR3_MEMORY_ADDRESS,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_MEMORY_BY_ADDRESS,  (tU8) DIA_C_U16_DID_ADR3_MEMORY_ADDRESS,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );

      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SET_PHYSICAL_TUNER_FREQUENCY_1,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetPhysicalTunerFrequency 1
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SET_PHYSICAL_TUNER_FREQUENCY_2,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetPhysicalTunerFrequency 2
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SET_PHYSICAL_TUNER_FREQUENCY_3,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetPhysicalTunerFrequency 3
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SET_PHYSICAL_TUNER_FREQUENCY_4,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetPhysicalTunerFrequency 4
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_AUDIO_ROUTER_1,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_AudioRouter_1
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_TUNE_DEMODULATOR_1,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_TuneDemodulator_1
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SET_VOLUME,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetVolume
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_ACTUAL_CALIBRATION_DATA,            DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_ActualAlignmentValues
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SET_SIGNAL_QUALITY,                   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetSignalQuality
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_TUNE_DEMODULATOR_2,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_TuneDemodulator_2
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_CONTROL_FRONTEND_CLOCK_OUTPUT,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_ControlFrontEndClockOutput
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_DAB_SET_SERVICE_ID,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_DABSetServiceID
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_DRM_SET_SERVICE_ID,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_DRMSetServiceID
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SNOOP_MODE,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SnoopMode
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_SWTICH_BACKGROUND_ANTENNA,          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SwitchBackgroundAntenna
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_DDS_CONFIGURATION,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_DDSConfiguration
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_AUDIO_TEST_TONE,                    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_AudioTesttone
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_AMPLIFIER_POWER_STATE,              DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_AudioTesttone
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_ADR3_AUDIO_SIGNAL_QUALITY,               DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_SetSignalQuality

      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_START_SEEK,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_StartSeek
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_STORE_ACTUAL_CALIBRATION_DATA,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_StoreActualCalibrationDataPersistently
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_START_CALIBRATION,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_Start_Calibration
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_CALC_CRC,                           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_CalcCRC
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_FORMAT_FFS,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_FormatFFS
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_CALC_MEASUREMENT_VAL_DEMODULATOR_1, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_DABCalcMeasurementValueDemodulator_1
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_DSB_CALC_NET_TO_BER,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_DABCalcNettoBER
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_CHECK_FRONTEND_REGISTER,            DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_CheckFrontendregister
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_READ_GPIO,                          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_CheckFrontendregister
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         (tU16) DIA_C_U16_DID_ADR3_WRITE_GPIO,                         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ADR3_CheckFrontendregister
}

   keys = mLookupKeys;

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tVoid
dia_SessionADR::vResetADR3 ( tVoid ) const
{
   dia_tclFnctTrace trc("dia_SessionADR::vResetADR3");

   OSAL_tIODescriptor hAdr3Device = OSAL_IOOpen(DIA_ADR_CONTROL_DRIVER_NAME, OSAL_EN_READWRITE);

   if ( OSAL_ERROR != hAdr3Device )
   {
      if ( OSAL_s32IOControl( hAdr3Device, OSAL_C_S32_IOCTRL_ADR3CTRL_RESET_ADR3, (tS32) NULL ) != OSAL_OK )
      {
         DIA_TR_ERR("##### FAILED TO RESET ADR3 AFTER ADR3 SESSION WAS LEFT ######");
      }
      OSAL_s32IOClose(hAdr3Device);
   }
}
