/**
 * \file    dia_ServiceHandlerWrapper.h
 *
 * \brief   This file contains the definition of the wrapper class for all kinds of
 *          service handlers. It forwards request directly to the associated handler.
 *          Objects of this class are used if we need objects with limited lifetime.
 *          The wrapper object can be destroyed, but the associated object is kept
 *          alive.
 *
 * \author  gib2hi
 * \date    2012/04/11
 */

#ifndef __INCLUDED_DIA_SERVICE_HANDLER_WRAPPER__
#define __INCLUDED_DIA_SERVICE_HANDLER_WRAPPER__

#ifndef __INCLUDED_DIA_SERVICE_HANDLER__
#include <common/framework/engine/dia_ServiceHandler.h>
#endif

class dia_LookupKey;

/**
 * \class   dia_ServiceHandlerWrapper
 *
 * \brief   wrapper class for all kinds of service handlers. It forwards request
 *          directly to the associated handler. Objects of this class are used if
 *          we need objects with limited lifetime. The wrapper object can be
 *          destroyed, but the associated object is kept alive.
 *
 * \ingroup diaCoreEngine
 */

class dia_ServiceHandlerWrapper
   : public dia_ServiceHandler
{
public:
   //! default class constructor
   dia_ServiceHandlerWrapper ( dia_ServiceHandler* pHandler ) : mpHandler(pHandler) {}
   //! class destructor
   virtual ~dia_ServiceHandlerWrapper ( void ) { mpHandler = 0; }

   //! method called to start processing of the given request
   virtual void vProcessRequest ( const std::vector<tArgsType>& vecArgs ) { if ( mpHandler ) mpHandler->vProcessRequest(vecArgs); }
   //! method case if the processing timer has been elapsed
   virtual void vOnTimeout ( void ) { if ( mpHandler ) mpHandler->vOnTimeout(); }

   //! retrieve the service handlers name
   virtual tCString getName ( void ) const { return ( mpHandler ) ? mpHandler->getName() : ""; }
   //! retrieve the service handlers name
   virtual tDiaResult setName ( tCString name ) { return ( mpHandler ) ? mpHandler->setName(name) : DIA_FAILED; }

   //! retrieve the value of the processing timeout
   virtual tU32  getTimeoutValue ( void ) const { return ( mpHandler ) ? mpHandler->getTimeoutValue() : DIA_C_U32_MAX_PROCESSING_TIME_SERVICE; }
   //! assign a new value to the timeout value
   virtual void setTimeoutValue ( tU32 value ) { if ( mpHandler ) mpHandler->setTimeoutValue(value); }

   //! setup a lookup key for the lookup table
   virtual tDiaResult makeLookupKeys ( std::vector<dia_LookupKey*>& keys ) { return ( mpHandler ) ? mpHandler->makeLookupKeys(keys) : DIA_FAILED; }

   //
   // deprecated methods (this class provides default implementations in order to ensure backward compatibility)
   //

   //! deprecated old method used to start processing of a request
   virtual void vOnReqRx ( void ) { if ( mpHandler ) mpHandler->vOnReqRx(); }
   //! deprecated old method used to deliver a response from a CCA component
   virtual void vOnCompRes ( amt_tclServiceData* msg ) { if ( mpHandler ) mpHandler->vOnCompRes(msg); }
   //! deprecated method
   virtual void vOnCallBack ( tU8 u8MsgBuffer[], const tU16 u16MsgLength ) { if ( mpHandler ) mpHandler->vOnCallBack(u8MsgBuffer,u16MsgLength); }
   //! deprecated method
   virtual tBool bOnNewReq ( tU8* pu8Buffer, tU16 u16MsgLen ) { return ( mpHandler ) ? mpHandler->bOnNewReq(pu8Buffer,u16MsgLen) : FALSE; }
   //! deprecated method used to begin post processing
   virtual void vOnBeginPpg ( void ) { if ( mpHandler ) mpHandler->vOnBeginPpg(); }
   //! deprecated method used to deliver a response from a CCA component for post processing
   virtual void vOnCompResPpg(amt_tclServiceData* msg) { if ( mpHandler ) mpHandler->vOnCompResPpg(msg); }
   //! deprecated method
   virtual void vOnTimeoutPpg ( void ) { if ( mpHandler ) mpHandler->vOnTimeoutPpg(); }
   //! deprecated method
   virtual tU32 u32MaxPgTime ( void ) const { return ( mpHandler ) ? mpHandler->u32MaxPgTime() : DIA_C_U32_MAX_PROCESSING_TIME_SERVICE; }
   //! deprecated method
   virtual tU32 u32MaxPpgTime ( void ) const { return ( mpHandler ) ? mpHandler->u32MaxPpgTime() : DIA_C_U32_MAX_PROCESSING_TIME_SERVICE; }
   //! deprecated method
   virtual tCString cszClassName() const { return ( mpHandler ) ? mpHandler->cszClassName() : ""; }

protected:
    //! deprecated default constructor without implementation
   dia_ServiceHandlerWrapper ( void );

protected:
   //! pointer to the concrete service handler
   dia_ServiceHandler* mpHandler;
};

#endif /* __INCLUDED_DIA_SERVICE_HANDLER_WRAPPER__ */
