/*!
 * \file       dia_LookupKey.cpp
 *
 * \brief      This file contains the definition of the dia_LookupKey class that is
 *             used to check if a service handler for incoming diagnosis request is
 *             available
 *
 * \details    This file contains the definition of the dia_LookupKey class that is
 *             used to check if a service handler for incoming diagnosis request is
 *             available
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#define DIA_U8_UDS_SID_POS      1
#define DIA_U8_UDS_SUBFUNC_POS  2

dia_LookupKey::tsKeyFormat
dia_LookupKey::mKeyFormatRep[DIA_C_U8_UDS_MAX_SID_VALUE];

//------------------------------------------------------------------------------

dia_LookupKey::tsKeyFormat::tsKeyFormat ( void )
   : mSID(0), mParaType(eParameterUnknown), mLengthType(eLengthUnknown)
{}

//------------------------------------------------------------------------------

dia_LookupKey::tsKeyFormat::tsKeyFormat ( tU8 sid, eParameterType paraType, eLengthType lengthType )
   : mSID(sid), mParaType(paraType), mLengthType(lengthType)
{}

//------------------------------------------------------------------------------

void
dia_LookupKey::tsKeyFormat::vClear ( void )
{
   mSID        = 0;
   mParaType   = eParameterUnknown;
   mLengthType = eLengthUnknown;
}

//------------------------------------------------------------------------------

void
dia_LookupKey::tsKeyFormat::vSet ( tU8 sid, eParameterType paraType, eLengthType lengthType )
{
   mSID        = sid;
   mParaType   = paraType;
   mLengthType = lengthType;
}

//------------------------------------------------------------------------------

dia_LookupKey::dia_LookupKey ( void )
    : mSID(0),
      mParaType(eParameterUnknown),
      mParameter(0),
      mLengthType(eLengthUnknown),
      mLength(0)
{}

//------------------------------------------------------------------------------

dia_LookupKey::dia_LookupKey ( tU8 sid, tU16 parameter, tU16 srvLen )
    : mSID(sid),
      mParaType(eParameterWord),
      mParameter(parameter),
      mLengthType(eLengthSpecified),
      mLength(srvLen)
{
   if ( ((tU16) parameter) == UDS_U16_PARAMETER_NOT_USED ) {
      mParaType  = eParameterUnknown;
      mParameter = 0;
   }

   if ( srvLen == UDS_U16_LENGTH_NOT_USED ) {
      mLengthType = eLengthUnknown;
      mLength = 0;
   }

   dia_LookupKey::mKeyFormatRep[sid].vSet(mSID,mParaType,mLengthType);
}

//------------------------------------------------------------------------------

dia_LookupKey::dia_LookupKey ( tU8 sid, tU8 parameter, tU16 srvLen )
    : mSID(sid),
      mParaType(eParameterByte),
      mParameter(parameter),
      mLengthType(eLengthSpecified),
      mLength(srvLen)
{
   if ( ((tU8) parameter) == UDS_U8_PARAMETER_NOT_USED ) {
      mParaType  = eParameterUnknown;
      mParameter = 0;
   }

   if ( srvLen == UDS_U16_LENGTH_NOT_USED ) {
      mLengthType = eLengthUnknown;
      mLength = 0;
   }

   dia_LookupKey::mKeyFormatRep[sid].vSet(mSID,mParaType,mLengthType);
}

//------------------------------------------------------------------------------


dia_LookupKey::dia_LookupKey ( const dia_MessageBuffer& msgBuffer )
   : mSID(0),
     mParaType(eParameterUnknown),
     mParameter(0),
     mLengthType(eLengthUnknown),
     mLength(0)
{
   // currently we only support the UDS protocol
   if ( msgBuffer.getProtocol() == DIA_EN_PROTOCOL_UDS )
   {
      mSID = msgBuffer.getDataU8(DIA_U8_UDS_SID_POS);
   }

   if ( mSID && dia_LookupKey::mKeyFormatRep[mSID].mSID != 0 )
   {
      // we have a valid entry in the repository

      mParaType = dia_LookupKey::mKeyFormatRep[mSID].mParaType;

      if ( mParaType == eParameterByte )
      {
         mParameter = msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS);
      }
      else if ( mParaType == eParameterWord )
      {
         if ( mSID == 0x31 )
         {
#ifdef __ENABLE_FEATURE_ROUTINE_CONTROL_FORMAT_NISSAN_LCN__
            if ( msgBuffer.u16GetDataLength() < 3) // we have at least 3 bytes (1 byte SID and 2 bytes DID)
            {
               mParameter = 0x0000;
            }
            else if ( msgBuffer.u16GetDataLength() == 3 )
            {
               // nissan specific routine control format (not UDS) --> supported local identifers
               tU8 nissanDID = msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS);

               if ( nissanDID == 0x00 )
               {
                  mParameter = (((tU16) 0xFF) << 8);
               }
               else
               {
                  mParameter = 0x0000;
               }
            }
            else if ( msgBuffer.u16GetDataLength() == 4 ) // nissan specific routine control format
            {
               // nissan specific routine control format (not UDS)
               mParameter = (((tU16) 0xFF) << 8) | ((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS));
            }
            else
            {
               mParameter = (((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS+1)) << 8) | ((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS+2));
            }
#else
            mParameter = (((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS+1)) << 8) | ((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS+2));
#endif
         }
         else
         {
            mParameter = (((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS)) << 8) | ((tU16) msgBuffer.getDataU8(DIA_U8_UDS_SUBFUNC_POS+1));
         }
      }
      else
      {
         // parameter type unknown or doubleword
         mParameter = 0x0000;
      }

      mLengthType = dia_LookupKey::mKeyFormatRep[mSID].mLengthType;

      if ( mLengthType == eLengthSpecified )
      {
         mLength = msgBuffer.u16GetDataLength();
      }
   }
}

//------------------------------------------------------------------------------

dia_LookupKey::dia_LookupKey ( const dia_LookupKey& key )
    : mSID(key.mSID),
      mParaType(key.mParaType),
      mParameter(key.mParameter),
      mLengthType(key.mLengthType),
      mLength(key.mLength)
{}

//-----------------------------------------------------------------------------

dia_LookupKey&
dia_LookupKey::operator = ( const dia_LookupKey& obj )
{
   if ( this != &obj )
   {
      mSID        = obj.mSID;
      mParaType   = obj.mParaType;
      mParameter  = obj.mParameter;
      mLengthType = obj.mLengthType;
      mLength     = obj.mLength;
   }
   return *this;
}

//------------------------------------------------------------------------------

dia_LookupKey::~dia_LookupKey ( void )
{}

//------------------------------------------------------------------------------

bool
dia_LookupKey::equal ( const dia_LookupKey& key ) const
{
   if ( (mSID        == key.mSID       ) && (mParaType   == key.mParaType  ) &&
        (mParameter  == key.mParameter ) && (mLengthType == key.mLengthType) &&
        (mLength     == key.mLength    ))
   {
      return true;
   }

   return false;
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::less ( const dia_LookupKey& key ) const
{
   if ( mSID < key.mSID ) {
      return true;
   }
   if ( mSID == key.mSID ) {
      if ( mParaType < key.mParaType ) {
         return true;
      }
      if ( mParaType == key.mParaType ) {
         if ( mParameter < key.mParameter) {
            return true;
         }
         if ( mParameter == key.mParameter) {
            if ( mLengthType < key.mLengthType ) {
               return true;
            }
            if ( mLengthType == key.mLengthType ) {
               if ( mLength < key.mLength ) {
                  return true;
               }
            }
         }
      }
   }

   return false;
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::operator == ( const dia_LookupKey& key ) const
{
    return equal(key);
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::operator != ( const dia_LookupKey& key ) const
{
    return !(equal(key));
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::operator <  ( const dia_LookupKey& key ) const
{
    return less(key);
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::operator >  ( const dia_LookupKey& key ) const
{
    return key.less(*this);
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::operator <= ( const dia_LookupKey& key ) const
{
    return !(less(key));
}

//------------------------------------------------------------------------------

bool
dia_LookupKey::operator >= ( const dia_LookupKey& key ) const
{
    return key.less(*this);
}


