/*!
 * \file       dia_EngineServerConfiguration.cpp
 *
 * \brief      Configuration for a server engine
 *
 * \details    Configuration for a server engine
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2014-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER_CONFIGURATION__
#include "common/framework/engine/dia_EngineServerConfiguration.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_ACCESS_INFO__
#include "common/framework/engine/dia_ServiceAccessInfo.h"
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include "common/framework/engine/dia_LookupKey.h"
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include "common/framework/engine/dia_Session.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SECURITY__
#include "common/framework/security/dia_common_security.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

const tU8 MINIMAL_LENGTH_OF_IOCONTROL_MESSAGE = 4U;         // according to UDS standard (ISO 14229-1:2013)
const tU8 MINIMAL_LENGTH_OF_ROUTINECONTROL_MESSAGE = 4U;    // according to UDS standard (ISO 14229-1:2013)
const tU8 MINIMAL_LENGTH_OF_ANY_MESSAGE = 2U;

using namespace std;

//! load information about the supported diagnostic services like SessionControl, ECUReset, ...
tDiaResult
dia_EngineServerConfiguration::updateServiceInfo ( const std::map<tU8,dia_ServiceAccessInfo*>&  srvInfo )
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::updateServiceInfo()");

   std::map<tU8,dia_ServiceAccessInfo*>::const_iterator iter = srvInfo.begin();
   for ( ; iter != srvInfo.end(); ++iter )
   {
      if ( mServiceRep.find(iter->second->getServiceID()) != mServiceRep.end() )
      {
         mServiceRep.erase(iter->second->getServiceID());
      }

      mServiceRep[iter->second->getServiceID()] = iter->second;
   }

   return DIA_SUCCESS;
}

tDiaResult
dia_EngineServerConfiguration::updateServiceSubfunctionInfo ( const std::map< tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >& srvSubfuncInfo )
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::updateServiceSubfunctionInfo()");

   std::map< tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >::const_iterator iter = srvSubfuncInfo.begin();
   for ( ; iter != srvSubfuncInfo.end(); ++iter )
   {
      const std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >& accessInfoRep = iter->second;

      if ( accessInfoRep.empty() ) continue;

      // iterate over all service access info item
      std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >::const_iterator iter2 = accessInfoRep.begin();
      for ( ; iter2 != accessInfoRep.end(); ++iter2 )
      {
         tU32 sessionID = iter2->first;
         ::boost::shared_ptr<dia_ServiceAccessInfo> pAccessInfo = iter2->second;

         tU32 key = (pAccessInfo->getServiceID() << 16) | pAccessInfo->getSubFunctionID();

         if ( mSubFuncRep.find(key) != mSubFuncRep.end() )
         {
            // we have found an entry for the given key generated from SID and DID
            if ( mSubFuncRep[key].find(sessionID) != mSubFuncRep[key].end() )
            {
               // an entry for the given session already exists and must be replaced
               (mSubFuncRep[key]).erase(sessionID);
            }
         }

         (mSubFuncRep[key])[sessionID] = pAccessInfo;
      }
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::loadSessions ( const std::vector<tU8>& supportedSessions )
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::loadSessions()");
   mSupportedSessions = supportedSessions;
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::updateSessions ( const std::vector<tU8>& supportedSessions )
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::updateSessions()");
   std::vector<tU8>::const_iterator iter1 = supportedSessions.begin();
   for ( ; iter1 != supportedSessions.end(); ++iter1 )
   {
      bool found = false;
      std::vector<tU8>::iterator iter2 = mSupportedSessions.begin();
      for ( ; iter2 != mSupportedSessions.end(); ++iter2 )
      {
         if ( (*iter1) == (*iter2) )
         {
            found = true;
            break;
         }
      }

      if ( !found ) mSupportedSessions.push_back(*iter1);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::loadServiceInfo ( const vector<ServiceInfo>& srvInfoRep )
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::loadServiceInfo()");

   // Add each row of allowed sessions to the AccessControl object
   for ( tU16 i=0; i < srvInfoRep.size(); i++ )
   {
      const ServiceInfo& item = srvInfoRep[i];

      if ( item.mAccessInfo != DIA_C_U8_UDS_SERVICE_NOT_SUPPORTED )
      {
         // Create a vector of the allowed sessions.
         vector<tU8> sessions;
         for ( tU16 j=0; j<DIA_C_U16_MAX_SESSION_PER_ENGINE; j++ ) {
            if ( item.mSupportedSessions[j] != 0x00 ) {
               sessions.push_back(item.mSupportedSessions[j]);
            }
         }

         // Create a vector of the allowed security levels
         vector<dia_UID> secLevels;
         for ( tU16 k=0; k<DIA_C_U16_MAX_SECURITY_LEVELS_PER_ENGINE; k++ ) {
            secLevels.push_back(item.mSupportedSecurityLevels[k]);
         }

         // create the data item
         dia_ServiceAccessInfo* pAccessInfoItem = OSAL_NEW dia_ServiceAccessInfo(item.mSID,(tU8) item.mDIDLen,sessions,secLevels);
         if ( pAccessInfoItem )
         {
            pAccessInfoItem->setSupportMode(item.mAccessInfo);
            // add the new info item to the lookup table
            mServiceRep[item.mSID] = pAccessInfoItem;
         }
         else
         {
            //tbd
         }
      }
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::loadServiceInfo ( const std::vector<ServiceDataItem>& srvDataRep )
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::loadServiceInfo()");

   std::vector<ServiceDataItem>::const_iterator srvDataIter = srvDataRep.begin();
   for ( ; srvDataIter != srvDataRep.end(); ++srvDataIter )
   {
      const ServiceDataItem& item = *srvDataIter;

      // Create a vector of the allowed sessions.
      std::vector<tU8> sessions;
      sessions.reserve(DIA_C_U16_MAX_SESSION_PER_ENGINE);
      for ( tU16 j=0; j<DIA_C_U16_MAX_SESSION_PER_ENGINE; j++ )
      {
         if ( item.mSupportedSessions[j] == 0x00 ) continue;
         sessions.push_back(item.mSupportedSessions[j]);
      }

      // Create a vector of the allowed security levels
      std::vector<dia_UID> secLevels;
      secLevels.reserve(DIA_C_U16_MAX_SECURITY_LEVELS_PER_ENGINE);
      for ( tU16 k=0; k<DIA_C_U16_MAX_SECURITY_LEVELS_PER_ENGINE; k++ )
      {
         if ( (item.mSupportedSecurityLevels[k] == DIA_EN_SECURITY_LEVEL_UNKNOWN) || (item.mSupportedSecurityLevels[k] == 0) ) continue;
         secLevels.push_back(item.mSupportedSecurityLevels[k]);
      }

      // create the data item
      ::boost::shared_ptr<dia_ServiceAccessInfo> pAccessDataItem(OSAL_NEW dia_ServiceAccessInfo(item.mSID,item.mDID,(tU8) item.mDIDLen,item.mRunLevel,sessions,secLevels));
      if ( pAccessDataItem )
      {
         tU32 key = (pAccessDataItem->getServiceID() << 16) | pAccessDataItem->getSubFunctionID();

         if ( mSubFuncRep.find(key) == mSubFuncRep.end() )
         {
            // iterate over all sessions configured for the given service access info item
            std::vector<tU8>::const_iterator sessionIter = sessions.begin();
            for ( ; sessionIter != sessions.end(); ++sessionIter )
            {
               (mSubFuncRep[key])[*sessionIter] = pAccessDataItem;
            }
         }
         else
         {
            // iterate over all sessions configured for the given service access info item
            std::vector<tU8>::const_iterator sessionIter = sessions.begin();
            for ( ; sessionIter != sessions.end(); ++sessionIter )
            {
               if ( mSubFuncRep[key].find(*sessionIter) != mSubFuncRep[key].end() )
               {
                  // an entry for the given session already exists and must be replaced
                  (mSubFuncRep[key]).erase(*sessionIter);
               }

               (mSubFuncRep[key])[*sessionIter] = pAccessDataItem;
            }
         }
      }
      else
      {
         //tbd
      }
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::isAccessAllowed ( const dia_LookupKey& key ) const
{
    dia_tclFnctTrace trc("dia_EngineServerConfiguration::isAccessAllowed()");

    tDiaResult retCode = DIA_SUCCESS;

    tU32 u32key = (((tU32) key.getSID()) << 16) | key.getSubFuncID();

    // try to find the key in the lookup table
    std::map<tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >::const_iterator itr = mSubFuncRep.find(u32key);

    if ( itr == mSubFuncRep.end() )
    {
       DIA_TR_INF( "dia_EngineServerConfiguration: SERVICE NOT CONFIGURED (SID=0x%02x, DID=0x%02x) !!",key.getSID(),key.getSubFuncID());
       retCode = DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED;
       switch ( key.getSID() )
       {
       case DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER:
       case DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER:
       case DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER:
       case DIA_C_U8_UDS_SID_ROUTINE_CONTROL:
       case DIA_C_U8_UDS_SID_READ_MEMORY_BY_ADDRESS:
       case DIA_C_U8_UDS_SID_WRITE_MEMORY_BY_ADDRESS:
          retCode = DIA_E_ACCESS_DID_NOT_SUPPORTED;
          break;
       default:
          break;
       }
       return retCode;
    }

    if ( !mpEngine )
    {
       DIA_TR_INF( "dia_EngineConfiguration: ACCESS DENIED (GENERAL_PROGRAMMING_FAILURE)");
       return DIA_E_ACCESS_GENERAL_PROGRAMMING_FAILURE;
    }

    dia_EngineServer* pEngine = static_cast<dia_EngineServer*>(mpEngine);

    tU8 session = pEngine->getActiveSessionID();

    // The key/service exists in the map
    if ( itr->second.find(session) == itr->second.end() )
    {
       // The subfunction or DID is not allowed in the active session.
       DIA_TR_INF( "dia_EngineConfiguration: ACCESS DENIED IN SESSION 0x%02x", session);
       retCode = DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_SESSION;
       switch ( key.getSID() )
       {
       case DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER:
       case DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER:
       case DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER:
       case DIA_C_U8_UDS_SID_ROUTINE_CONTROL:
       case DIA_C_U8_UDS_SID_READ_MEMORY_BY_ADDRESS:
       case DIA_C_U8_UDS_SID_WRITE_MEMORY_BY_ADDRESS:
          retCode = DIA_E_ACCESS_DID_NOT_SUPPORTED_BY_SESSION;
          break;
       default:
          break;
       }
    }
    else
    {
       DIA_TR_INF("dia_EngineConfiguration: Access allowed in session 0x%02x", session);
    }

    return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::isServiceAllowed ( tU8 sid, tU16 len, tU8 session ) const
{
    dia_tclFnctTrace trc("dia_EngineServerConfiguration::isServiceAllowed()");

    tDiaResult retCode = DIA_SUCCESS;

    // try to find the key in the lookup table
    map<tU8,dia_ServiceAccessInfo*>::const_iterator itr = mServiceRep.find(sid);

    if ( itr == mServiceRep.end() )
    {
       DIA_TR_ERR( "###  isServiceAllowed: SID=0x%02X NOT FOUND IN REP (len=%d, session=0x%02X) ###", sid, len, session);
       return DIA_E_ACCESS_SERVICE_NOT_SUPPORTED;
    }

    if ( !(itr->second->checkSession(session)) )
    {
       if ( itr->second->getSupportMode() == DIA_C_U8_UDS_SERVICE_SUPPORTED_SELECTIVELY )
       {
          DIA_TR_ERR( "###  getSupportMode returned SERVICE_SUPPORTED_SELECTIVELY (len=%d, session=0x%02X) ###", sid, session);
          retCode = DIA_E_ACCESS_SERVICE_NOT_SUPPORTED;
       }
       else
       {
          // The service is not allowed in this session.
          DIA_TR_INF( "isServiceAllowed: Service Not Supported in Active Session (0x%02X)", session);
          retCode = DIA_E_ACCESS_SERVICE_NOT_SUPPORTED_BY_SESSION;
       }
    }
    else
    {
       tU8 minLen = tU8(itr->second->getSubIDSize() + 1); // sid + did
       if ( len < minLen ) retCode = DIA_E_ACCESS_INVALID_FORMAT;
    }

    return retCode;
}

//------------------------------------------------------------------------------

#define DIA_U8_MSG_POSITION_SID     1

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::validateRequest ( const dia_MessageBuffer& request, const dia_Session* pSession, const std::list<dia_SecurityLevel*>& securityLevels ) const
{
    dia_tclFnctTrace trc("dia_EngineServerConfiguration::validateRequest(dia_MessageBuffer,pSession,securityLevels)");

    // at this time we only support the UDS protocol
    if ( validateProtocol(request) != DIA_SUCCESS )
    {
       DIA_TR_INF("### DIAGNOSIS PROTOCOL NOT SUPPORTED ###");
       return DIA_FAILED;
    }

    tU8  sid = request.getDataU8(DIA_U8_MSG_POSITION_SID);
    tU16 len = tU16(request.u16GetDataLength()-1); /* length byte removed */

    DIA_TR_INF("### VALIDATING UDS REQUEST SID=0x%x, LEN=%d ###", sid, len);

    // if the tester sends an invalid request with the SID we use for internal messages, we have to reject it properly
    if ( (sid == DIA_C_U8_UDS_SID_DIAG_COMMAND) && (!(request.bIsInternalRequest())) )
    {
       DIA_TR_INF("### SERVICE NOT SUPPORTED ###");
       return DIA_E_ACCESS_SERVICE_NOT_SUPPORTED;
    }

    if ( validateLengthOrFormat(sid,len) != DIA_SUCCESS )
    {
       DIA_TR_INF("### INVALID LENGHT OR FORMAT ###");
       return DIA_E_ACCESS_INVALID_FORMAT;
    }

    if ( mpEngine && pSession /*mpEngine->getActiveSession()*/ )
    {
       tDiaResult retCodeService = isServiceAllowed(sid,len,pSession->getID());
       if ( retCodeService != DIA_SUCCESS )
       {
          DIA_TR_INF("### SERVICE NOT ALLOWED (ERR=0x%08x) ###", retCodeService);
          return retCodeService;
       }

       // here we know that the service itself is supported, so we now need to check the subfunction
       dia_LookupKey key(request);

       tDiaResult retCodeSubFunc = isAccessAllowed(key);
       if ( retCodeSubFunc != DIA_SUCCESS )
       {
          DIA_TR_INF("### SUBFUNCTION NOT ALLOWED (ERR=0x%08x) ###", retCodeSubFunc);
          return retCodeSubFunc;
       }

       // here we know that the service and the subfunction are supported, so we now need to check the security access level

       if ( !(request.isRequestInternal()) )
       {
          tDiaResult retCodeSecurityAccess = isSecurityAccessAllowed(key,pSession->getID(),securityLevels);
          if ( retCodeSecurityAccess != DIA_SUCCESS )
          {
             DIA_TR_INF("### SECURITY ACCESS NOT ALLOWED (ERR=0x%08x) ###", retCodeSecurityAccess);
             return retCodeSecurityAccess;
          }
       }

       return DIA_SUCCESS;
    }

    return DIA_E_ACCESS_SERVICE_NOT_SUPPORTED;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::validateProtocol ( const dia_MessageBuffer& request) const
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::validateProtocol()");

   return ( request.getProtocol() == DIA_EN_PROTOCOL_UDS ) ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::validateLengthOrFormat ( tU8 sid, tU16 length ) const
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::validateLengthOrFormat()");

   tDiaResult retCode = DIA_SUCCESS;

   tU8 minimalLength = 0;
   tU8 minimalLengthDefault = MINIMAL_LENGTH_OF_ANY_MESSAGE;
   tU32 propID = 0;

   switch ( sid )
   {
   case DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER:
      propID = DIA_PROP_CM_MIN_LEN_IOCONTROL_MESSAGE;
      minimalLengthDefault = MINIMAL_LENGTH_OF_IOCONTROL_MESSAGE;
      break;

   case DIA_C_U8_UDS_SID_ROUTINE_CONTROL:
      propID = DIA_PROP_CM_MIN_LEN_ROUTINECTRL_MESSAGE;
      minimalLengthDefault = MINIMAL_LENGTH_OF_ROUTINECONTROL_MESSAGE;
      break;

   default:
      break;
   }

   if ( propID )
   {
      if ( dia_getProperty(propID,minimalLength) != DIA_SUCCESS )
      {
         minimalLength = minimalLengthDefault;
      }
   }

   if ( length < (tU16) minimalLength)
   {
      retCode = DIA_E_ACCESS_INVALID_FORMAT;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::querySessions ( const dia_LookupKey& key, std::vector<tU8>& supportedSessions ) const
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::querySessions(key,sessions)");
#endif

   tU32 subFuncKey = (((tU32) key.getSID()) << 16) | ((tU32) key.getSubFuncID());

#ifdef __DIA_UNIT_TESTING__
   DIA_TR_INF("### KEY TO QUERY IS 0x%08x ###", subFuncKey);
#endif

   std::map<tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >::const_iterator iter =  mSubFuncRep.find(subFuncKey);
   if ( iter == mSubFuncRep.end() )
   {
      DIA_TR_INF("### KEY 0x%04x WAS NOT FOUND IN SERVICE ACCESS REPOSITORY !!! ###", subFuncKey);
      return DIA_FAILED;
   }

   supportedSessions.clear();
   const std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >& accessInfoRep = iter->second;
   std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >::const_iterator itemIter = accessInfoRep.begin();
   for ( ; itemIter != accessInfoRep.end(); ++itemIter )
   {
      supportedSessions.push_back((tU8) itemIter->first);
   }

   return DIA_SUCCESS;
}

/**
 * returns a list of all available SubFuncIDs (DIDs) according the defined parameters (session & bitMask)
 * @param session          session for which the function shall read the data
 * @param service          id of service
 * @param supportedIdents  list where response is stored
 * @param bitMask          if given only DIDs which match it will be returned. Usecase: if you have a certain high-byte you can read all low-bytes of that range
 */

tDiaResult
dia_EngineServerConfiguration::querySubFuncIDs ( const dia_Session* session, tU8 service, std::vector<tU8>& supportedIdents, tU16 bitMask ) const
{
   dia_tclFnctTrace trc("dia_EngineServerConfiguration::querySubFuncIDs()");
   tDiaResult retCode = DIA_SUCCESS;
   std::map<tU8,std::vector<tU8> > mapServicesSupported; // key = high byte ; data is list of low bytes

   DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs reading all DIDs for the following limitations:");
   DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs session = 0x%02x", session->getID());
   DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs service = 0x%02x", service);
   if ( bitMask != 0xFFFF )
   {
      DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs bitMask = 0x%04x", bitMask);
   }

   /* ---------------------------------------- */
   /* 1.) Extract needed data from mSubFuncRep */
   /* ---------------------------------------- */
   // 1 dia_ServiceAccessInfo == 1 registered Service
   std::map<tU32, std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >::const_iterator it = mSubFuncRep.begin();
   for ( ; it != mSubFuncRep.end(); ++it )
   {
      tU16 did = (tU16) it->first;
      tU8  sid = (tU8) (it->first >> 16);
      DIA_TR_INF("did = 0x%04x",did);
      const std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >& itemRep = it->second;
      if ( (sid == service) && (itemRep.find(session->getID()) != itemRep.end()) )
      {
         DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs DID 0x%04x is written back for Service 0x%02x .", did, service);
         // extract high and low byte
         tU8 highByte = (tU8) (did >> 8);
         tU8 lowByte  = (tU8) did;
         // store low byte in MAP with high-byte as key
         mapServicesSupported[highByte].push_back(lowByte);
      }
   }
   // print overall result of first step
   DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs MAP[%d] initialized with values:", mapServicesSupported.size());
   std::map<tU8,std::vector<tU8> >::const_iterator it2 = mapServicesSupported.begin();
   for ( ; it2 != mapServicesSupported.end(); ++it2)
   {
      DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs MAP-Key[0x%02x] with %d elements.", it2->first, it2->second.size() );
   }

   /* ---------------------------------------------------- */
   /* 2.) Copy requested data into parameter-list variable */
   /* ---------------------------------------------------- */
   if ( (bitMask == 0xFFFF /* Default */) || (bitMask == 0xDFFF /* Generated Mask for High Bytes */) )
   {
      DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs Copy list of high Bytes to return parameter.");
      std::map<tU8,std::vector<tU8> >::const_iterator it3 = mapServicesSupported.begin();
      for ( ; it3 != mapServicesSupported.end(); ++it3 )
      {
         supportedIdents.push_back(it3->first);
      }
   }
   else
   {
      tU8 hb = tU8(bitMask >> 8);

      std::map<tU8,std::vector<tU8> >::const_iterator itr = mapServicesSupported.find(hb);
      if ( itr != mapServicesSupported.end() )
      {
         DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs Copy %d low Bytes from high Byte 0x%02x to return parameter.", mapServicesSupported.find(hb)->second.size(), hb);
         // HB is in map and further LB are available
         for ( std::vector<tU8>::const_iterator itr_lb = mapServicesSupported.find(hb)->second.begin(); itr_lb != mapServicesSupported.find(hb)->second.end(); ++itr_lb )
         {
            DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs Copy high Byte 0x%02x -> Low Byte 0x%02x to response.", hb, *itr_lb);
            supportedIdents.push_back(*itr_lb);
         }
      }
      else
      {
         DIA_TR_INF("dia_EngineServerConfiguration::querySubFuncIDs no DIDs with high Byte 0x%02x .", hb);
      }
   }
   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_EngineServerConfiguration::isSecurityAccessAllowed ( const dia_LookupKey& key, tU8 sessionID, const std::list<dia_SecurityLevel*>& securityLevels ) const
{
    dia_tclFnctTrace trc("dia_EngineServerConfiguration::isSecurityAccessAllowed(key,sessionID,securityLevels)");

    // try to find the key in the lookup table
    tU32 u32key = (((tU32) key.getSID()) << 16) | key.getSubFuncID();
    std::map<tU32,std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> > >::const_iterator itr = mSubFuncRep.find(u32key);
    if ( itr == mSubFuncRep.end() )
    {
       DIA_TR_INF("dia_EngineServerConfiguration: LOOKUP KEY NOT FOUND");
       return DIA_SUCCESS;
    }

    // The key/service exists in the map

    // note at this point: sessions are checked before security levels are checked, so we don't need to check the session again here
    const std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >& accessInfoRep = itr->second;
    std::map<tU32,::boost::shared_ptr<dia_ServiceAccessInfo> >::const_iterator itr2 = accessInfoRep.find(sessionID);
    if ( itr2 != accessInfoRep.end() )
    {
       if ( !(itr2->second->checkSecurityLevels(securityLevels)) )
       {
          // The service is not allowed in this session.
          DIA_TR_INF( "dia_EngineConfiguration: SECURITY ACCESS DENIED !!");
          return DIA_E_ACCESS_REJECTED_BY_SECURITY_LEVEL;
       }
       else
       {
          DIA_TR_INF("dia_EngineConfiguration: SECURITY ACCESS ALLOWED !!");
       }
    }

    return DIA_SUCCESS;
}
