/*!
 * \file       dia_EngineManager.cpp
 *
 * \brief      Manager class that manages different kinds of engines
 *
 * \details    Manager class that manages and maintains different kinds of diagnosis engines,
 *             e.g. the UDS engine, Run-In engine, etc.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2014-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_CLIENT__
#include <common/framework/engine/dia_EngineClient.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

using namespace dia;

DIA_IMPL_SINGLETON(dia_EngineManager)

#ifndef __DIA_UNIT_TESTING__

dia_EngineManager*
getInstanceOfEngineManager ( void )
{
   return dia_EngineManager::getInstance();
}

void
releaseInstanceOfEngineManager ( void )
{
   dia_EngineManager::deleteInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_EngineManager::dia_EngineManager ( void )
   : mSyncObj("dia_EngineManager_LK")
{}

//-----------------------------------------------------------------------------

dia_EngineManager::~dia_EngineManager ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::setup ( void )
{
   ScopeTrace trc("dia_EngineManager::setup");

   tDiaResult retCode = DIA_SUCCESS;

   LockScope oLock(mSyncObj);

   std::map<dia_UID,dia_Engine*>::iterator iter = mEngineRep.begin();
   for ( ; iter != mEngineRep.end(); ++iter )
   {
      dia_Engine* pEngine = iter->second;
      if ( pEngine )
      {
         DIA_TR_INF("dia_EngineManager::setup Engine(name='%s',UID=0x%08X,pEngine=0x%p)", pEngine->getName(), iter->first, pEngine);
         if ( pEngine->setup() != DIA_SUCCESS )
         {
            DIA_TR_ERR("FAILED TO SETUP ENGINE (ADDR=0x%p)", pEngine);
            retCode = DIA_FAILED;
            break;
         }
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::shutdown ( void )
{
   ScopeTrace trc("dia_EngineManager::shutdown");

   tDiaResult retCode = DIA_SUCCESS;

   LockScope oLock(mSyncObj);

   std::map<dia_UID,dia_Engine*>::iterator iter = mEngineRep.begin();
   for ( ; iter != mEngineRep.end(); ++iter )
   {
      dia_Engine* pEngine = iter->second;
      if ( (!pEngine) || (pEngine->shutdown() != DIA_SUCCESS) )
      {
         DIA_TR_ERR("FAILED TO SHUTDOWN ENGINE (NAME='%s',ADDR=0x%p)", pEngine?pEngine->getName():"-", pEngine);
         retCode = DIA_FAILED;
         break;
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::addEngine ( dia_Engine& engine )
{
   ScopeTrace trc("dia_EngineManager::addEngine");

   tDiaResult retCode = DIA_FAILED;

   // we use a hash code of the engine name as unique identifier
   dia_UID id = dia_getHashCodeFromString(engine.getName());

   LockScope oLock(mSyncObj);

   std::map<dia_UID,dia_Engine*>::iterator iter = mEngineRep.find(id);
   if ( iter == mEngineRep.end() )
   {
      DIA_TR_INF("### ADDING ENGINE TO REPOSITORY (NAME='%s', ID=%u, ADDR=0x%p) ###", engine.getName(), id, (&engine) );
      mEngineRep[id] = &engine;
      retCode = DIA_SUCCESS;
   }
   else
   {
      if ( ::strcmp(engine.getName(),iter->second->getName()) != 0 )
      {
         // same UID although the names are equal
         DIA_TR_ERR("### Detected UID collision (UID = 0x%08x, 1st engine = \"%s\", 2nd engine = \"%s\" ###", id, engine.getName(), iter->second->getName());
         retCode = DIA_E_UID_COLLISION;
      }
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::queryEngine ( dia_UID id, dia_Engine** ppEngine )
{
   ScopeTrace trc("dia_EngineManager::queryEngine(dia_UID,dia_Engine**)");

   tDiaResult retCode = DIA_FAILED;

   LockScope oLock(mSyncObj);

   if ( ppEngine )
   {
      std::map<dia_UID,dia_Engine*>::iterator iter = mEngineRep.find(id);
      if ( iter != mEngineRep.end() )
      {
         *ppEngine = iter->second;
         DIA_TR_INF("### ENGINE FOUND (ID=0x%08X, ADDR=0x%p) ###", id, (*ppEngine));
         retCode = DIA_SUCCESS;
      }
      else
      {
         DIA_TR_INF("### ENGINE NOT FOUND (ID WAS 0x%08X) ###", id);
      }
   }
   else
   {
      DIA_TR_ERR("### INVALID POINTER ###");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::queryEngine ( tCString name, dia_Engine** ppEngine )
{
   ScopeTrace trc("dia_EngineManager::queryEngine(tCString,dia_Engine**)");
   return this->queryEngine(dia_getHashCodeFromString(name),ppEngine);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::queryEngineServer ( dia_UID id, dia_EngineServer** ppEngineServer )
{
   ScopeTrace trc("dia_EngineManager::queryEngineServer(dia_UID,dia_EngineServer**)");

   dia_Engine* pEngine = 0;
   tDiaResult retVal = DIA_FAILED;
   tDiaResult retCode = this->queryEngine(id,&pEngine);

   if ( (retCode == DIA_SUCCESS) && pEngine && pEngine->isServer() )
   {
      *ppEngineServer = static_cast<dia_EngineServer*>(pEngine);
      retVal = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_EngineManager::queryEngineServer retCode=0x%08X", retCode);
      if (NULL!=pEngine)
      {
         DIA_TR_ERR("dia_EngineManager::queryEngineServer pEngine->isServer() returned %s", (pEngine->isServer() ? "TRUE": "FALSE"));
      }
      else
      {
         DIA_TR_ERR("dia_EngineManager::queryEngineServer pEngine is NULL");
      }
   }

   DIA_TR_INF("dia_EngineManager::queryEngineServer returned 0x%08X", retVal);

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_EngineManager::queryEngineClient( dia_UID id, dia_EngineClient** ppEngineController )
{
   ScopeTrace trc("dia_EngineManager::queryEngineClient(dia_UID,dia_EngineClient**)");

   dia_Engine* pEngine = 0;
   tDiaResult retCode = this->queryEngine(id,&pEngine);

   if ( (retCode == DIA_SUCCESS) && pEngine && pEngine->isClient() )
   {
      *ppEngineController = static_cast<dia_EngineClient*>(pEngine);
      return DIA_SUCCESS;
   }

   return DIA_FAILED;
}


