/*!
 * \file       dia_EngineFlash.cpp
 *
 * \brief      PDX flashing engine (client engine)
 *
 * \details    PDX flashing engine (client engine) is used to update system configuration via PDX containers
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreEngine
 *
 * \copyright  (c) 2014-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_ENGINE_FLASH_PDX__
#include <common/framework/engine/dia_EngineFlash.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include <common/framework/engine/dia_Session.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_CONFIGURATION_FLASH_PDX__
#include <common/framework/engine/dia_EngineFlashConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include <common/framework/protocols/uds/dia_MessageBufferUDS.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include "common/framework/application/dia_File.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_PDX_PARSER_STRATEGY_TINYXML__
#include <common/framework/odx/pdx/dia_PDXParserStrategyTinyXML.h>
#endif

#ifndef __INCLUDED_DIA_PDX_CATALOG__
#include <common/framework/odx/pdx/dia_PDXCatalog.h>
#endif

#ifndef __INCLUDED_DIA_PDX_ABLOCK__
#include <common/framework/odx/pdx/dia_PDXABlock.h>
#endif

#ifndef __INCLUDED_DIA_PDX_FILE__
#include <common/framework/odx/pdx/dia_PDXFile.h>
#endif

#ifndef __INCLUDED_DIA_ODX__
#include <common/framework/odx/dia_ODX.h>
#endif

#ifndef __INCLUDED_DIA_ODX_TYPES__
#include <common/framework/odx/dia_ODXTypes.h>
#endif

#ifndef __INCLUDED_DIA_ODX_FLASH__
#include <common/framework/odx/dia_ODXFlash.h>
#endif

#ifndef __INCLUDED_DIA_ODX_FLASH_DATA__
#include <common/framework/odx/dia_ODXFlashData.h>
#endif

#ifndef __INCLUDED_DIA_ODX_FLASH_DATA_REF__
#include <common/framework/odx/dia_ODXFlashDataRef.h>
#endif

#ifndef __INCLUDED_DIA_ODX_FLASH_DATA_INTERN__
#include <common/framework/odx/dia_ODXFlashDataIntern.h>
#endif

#ifndef __INCLUDED_DIA_ODX_FLASH_DATA_EXTERN__
#include <common/framework/odx/dia_ODXFlashDataExtern.h>
#endif

#ifndef __INCLUDED_DIA_ODX_ECU_MEM__
#include <common/framework/odx/dia_ODXEcuMem.h>
#endif

#ifndef __INCLUDED_DIA_ODX_MEM__
#include <common/framework/odx/dia_ODXMem.h>
#endif

#ifndef __INCLUDED_DIA_ODX_DATA_BLOCK__
#include <common/framework/odx/dia_ODXDataBlock.h>
#endif

#ifndef __INCLUDED_DIA_ODX_SESSION__
#include <common/framework/odx/dia_ODXSession.h>
#endif

#ifndef __INCLUDED_DIA_ODX_EXPECTED_IDENT__
#include <common/framework/odx/dia_ODXExpectedIdent.h>
#endif

#ifndef __INCLUDED_DIA_ODX_IDENT_VALUE__
#include <common/framework/odx/dia_ODXIdentValue.h>
#endif

#ifndef __INCLUDED_DIA_ODX_DATA_BLOCK_REF__
#include <common/framework/odx/dia_ODXDataBlockRef.h>
#endif

#ifndef __INCLUDED_DIA_ODX_SEGMENT__
#include <common/framework/odx/dia_ODXSegment.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY_PLUGIN_PDX_FLASHING__
#include <common/framework/factory/dia_FactoryPluginPDXFlashing.h>
#endif

#ifndef __INCLUDED_DIA_CRC_CALCULATOR__
#include "common/framework/utils/dia_CRCCalculator.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#ifndef __INCLUDED_DIA_TEST_CONTROLLER__
#include "common/framework/test/dia_TestController.h"
#endif


#include <fstream>     //lint !e537 : repeatedly included header file without standard include guard
#include <stdlib.h>    //lint !e537 !e451 : repeatedly included header file without standard include guard
#include <sys/types.h> //lint !e537 : repeatedly included header file without standard include guard
#include <sys/wait.h>  //lint !e537 : repeatedly included header file without standard include guard

namespace dia {

#define PDXFLASH_QUEUE_SIZE      100
#define PDXFLASH_STR_BUFF_SIZE   200

static tCString strPDXFlashThreadName  = "DIA_PDX";
static tCString strPDXFlashQueueName   = "DIA_PDXQ";

#define DIA_C_U32_FLASH_ENGINE_INVALID_TIMER_ID             ((tU32) 0xFFFFFFFF)
#define DIA_C_U16_INVALID_BLOCK_SIZE                        ((tU16) 0x0000)
#define DIA_C_U8_DEFAULT_SESSION_ID_SYSTEM                  ((tU8)  0x01)
#define DIA_C_U8_DEFAULT_SESSION_ID_PROGRAMMING             ((tU8)  0x60)

static IdentInfo
gIndentInfo[] = {
      { "VariantID", 0xdc5273e7, 0xC008 }
};

//-----------------------------------------------------------------------------

EngineFlash::EngineFlash ( const std::string& name, EngineFlashConfiguration& config )
   : dia_EngineClient(name.c_str(),config),
     dia_ActiveObject(strPDXFlashThreadName,DIA_PROP_ENGINE_PDX_FLASH_THREAD_PRIO,DIA_PROP_ENGINE_PDX_FLASH_THREAD_STACK_SIZE),
     mConfig(config),
     mpFSM(0),
     mpMsgQueue(0),
     mIsRunning(false),
     mHasCatalogFlashContainer(false),
     mIsDataTransferComplete(false),
     mIsPDXContainerAvailable(false),
     mNeedToRemovePDXContainer(false),
     mIsTimerRunning(false),
     needToLoadPDXContainers(true),
     mpResponseData(0),
     mResponseLength(0),
     mEngineTimerID(DIA_C_U32_FLASH_ENGINE_INVALID_TIMER_ID),
     mpParser(0),
     mpPDXCatalog(0),
     mPDXFlashContainerFile(0),
     mpODXF(0),
     mCurrentJob(0),
     mBlockSize(0),
     mBlockSizeInBytes(0),
     mBlockSequenceNumber(0),
     mCRC(0),
     mSessionToBeRestored(DIA_C_U8_DEFAULT_SESSION_ID_SYSTEM),
     mSession(DIA_C_U8_DEFAULT_SESSION_ID_PROGRAMMING),
     mErrorCode(DIA_E_NO_ERROR)
{
   ScopeTrace trc("EngineFlash::EngineFlash");

   oEngineTimer.s32Create();
   mEngineTimerID = oEngineTimer.getID();
}

//-----------------------------------------------------------------------------

EngineFlash::~EngineFlash ( void )
{
   // destructors should not throw exceptions (--> lint), but functions called in the
   // could possibly throw excepections. So we catch them here and raise an assert
   _BP_TRY_BEGIN
   {
      (void) EngineFlash::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: EngineFlash::~EngineFlash !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
EngineFlash::setup ( void )
{
   ScopeTrace trc("dia::EngineFlash::setup");

   // we setup the engine object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### PDX FLASHING ENGINE WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   // create the state machine object
   if ( !(dia_EngineFlashFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
      return DIA_FAILED;
   }

   //
   // create message queue
   //
   mpMsgQueue = OSAL_NEW dia_Queue<PDXFlashingEvent>(strPDXFlashQueueName,PDXFLASH_QUEUE_SIZE);
   if ( !mpMsgQueue || (mpMsgQueue->open() != DIA_SUCCESS) ) return DIA_FAILED;

   DIA_TR_INF("### READING PDX FLASH ENGINE CONFIGURATION (ENGINE ADDR=0x%p)... ###",this);
   DIA_TR_INF("  +--> PDX FLASH ENGINE CONFIGURATION NOT YET AVAILABLE !!");
   mIsSetupDone = TRUE;

   return mIsSetupDone ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
EngineFlash::tearDown ( void )
{
   ScopeTrace oTrace("dia::EngineFlash::tearDown()");

   if ( mpMsgQueue )
   {
      tU32 dummy = 0;

      mpMsgQueue->addElement((PDXFlashingEvent*) &dummy);

      tS32 s32WaitCount = 0;
      while ( s32WaitCount < 10 )
      {
         DIA_TR_INF("Sleep for 100ms");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
         OSAL_s32ThreadWait(100);
#else
         usleep(100*1000); // DiagOmitSleepWarning: Required for thread synchronization
#endif
         ++s32WaitCount;
      }

      mpMsgQueue->close();
   }

   OSAL_DELETE mpFSM;
   mpFSM = 0;

   oEngineTimer.s32Delete();
   oEngineTimer.removeTimerListener(this);
   mEngineTimerID = DIA_C_U32_FLASH_ENGINE_INVALID_TIMER_ID;

   OSAL_DELETE mpMsgQueue;
   mpMsgQueue = 0;

   mpResponseData = 0;

   return DIA_SUCCESS;
} //lint !e1578: Pointer member dia_EngineRunIn::mCmdIter neither freed nor zeroed by cleanup function. No heap allocated for iterator

//-----------------------------------------------------------------------------

const EngineFlashConfiguration&
EngineFlash::getFlashConfiguration ( void ) const
{
//   return dynamic_cast<const dia_EngineRunInConfiguration&>(mConfig);
   return mConfig;
}

//-----------------------------------------------------------------------------

tDiaResult
EngineFlash::startControl ( std::vector<tArgsType>& /*args*/, bool automaticStartMode )
{
   ScopeTrace oTrace("EngineFlash::startControl");

   // check if the Run-In was already started
   if ( mIsRunning ) return DIA_SUCCESS;

   mIsRunning = true;
   postEvent(dia_EngineFlashFSM::evStart);

   DIA_TR_INF("EngineFlash::startControl: automaticStartMode = '%s'",automaticStartMode ? "true" : "false");
   tDiaResult retCode = automaticStartMode ? startThread() : DIA_SUCCESS;

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
EngineFlash::execute ( void )
{
   ScopeTrace oTrace("EngineFlash::execute");
   return startThread();
}

//-----------------------------------------------------------------------------

void
EngineFlash::onStartFlashingEvent ( void )
{
   ScopeTrace oTrace("EngineFlash::onStartFlashingEvent");
//   if ( !mIsRunning ) return;
   postEvent(dia_EngineFlashFSM::evStart);
}

//-----------------------------------------------------------------------------

tDiaResult
EngineFlash::stopControl ( void )
{
   postEvent(dia_EngineFlashFSM::evTerminate);
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vThreadEntrypointObject ( void )
{
   dia::ScopeTrace oTrace("dia_EngineRunIn::vThreadEntrypointObject");

   PDXFlashingEvent* pEvent = 0;

   bool isDone = false;

   do
   {
      pEvent = 0;
      if ( mpMsgQueue && (mpMsgQueue->getElement(&pEvent) == DIA_SUCCESS) )
      {
         DIA_TR_INF("pEvent = %s", pEvent ? "not null" : "null");

         if ( pEvent )
         {
            dia_EngineFlashFSM::FsmEvent fsmEvent = *((dia_EngineFlashFSM::FsmEvent*) pEvent);
            acceptEvent(fsmEvent,0);

            DIA_TR_INF("Message Processed !!");
            DIA_DELETE pEvent;
            pEvent = 0;
         }
         else
         {
            isDone = true;
         }
      }
      else
      {
         isDone = true;
      }
   }
   while ( !isDone ) ;

   DIA_TR_ERR("############################################################");
   DIA_TR_ERR("#");
   DIA_TR_ERR("# ENGINE PDX-FLASHING THREAD IS GETTING TERMINATED");
   DIA_TR_ERR("#");
   DIA_TR_ERR("############################################################");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   // Ask OSAL what the last ERROR to occur was
   tU32 u32ErrorReason = OSAL_u32ErrorCode();
   if ( u32ErrorReason != OSAL_E_NOERROR ) DIA_TR_ERR("msg queue error: %d", u32ErrorReason);
#endif
}

//-----------------------------------------------------------------------------

void
EngineFlash::postEvent ( dia_EngineFlashFSM::FsmEvent event )
{
   PDXFlashingEvent* pEvent = OSAL_NEW PDXFlashingEvent;  //lint !e429: custodial pointer is freed by after engine has processed the message
   if ( pEvent )
   {
      DIA_TR_INF("+------------------------------------------------------------------------------------");
      DIA_TR_INF("| Posting Event Id=%d ('%s') to Flash Engine Queue", event, dia_EngineFlashFSM::getEventName(event));
      DIA_TR_INF("+------------------------------------------------------------------------------------");
      *pEvent = (PDXFlashingEvent) event;
      if (mpMsgQueue) mpMsgQueue->addElement(pEvent);
   }
   pEvent = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pEvent'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'pEvent' not used

//-----------------------------------------------------------------------------

void
EngineFlash::acceptEvent ( dia_EngineFlashFSM::FsmEvent event, void* /*pArg*/ )
{
   DIA_TR_INF("Accept Event Id=%d ('%s')", event, dia_EngineFlashFSM::getEventName(event));
   if ( mpFSM )
   {
       DIA_TR_INF("### State Before: %s", mpFSM->getStateName());
       mpFSM->acceptEvent(event,0);
       DIA_TR_INF("### State After : %s", mpFSM->getStateName());
   }
   else
   {
       DIA_TR_INF( "### Event Not Accepted By PDX-Flash-Engine. No FSM !!!");
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmInitialize ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmInitialize");
   mSession = mConfig.getRequestedSession();
   mNeedToRemovePDXContainer = false;
   mIsTimerRunning = false;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmAssembleData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmAssembleData");

   if ( mBlockSequenceNumber > mBlockData.size() ) return;

   if ( mConfig.getCRCCalculationMode() == true )
   {
      mBlockData[mBlockSequenceNumber-1].push_back((tU8) (mCRC >> 8));
      mBlockData[mBlockSequenceNumber-1].push_back((tU8) (mCRC >> 0));
   }

   DIA_TR_INF("      +--> Data to be transferred: '%s'", dia::utils::bin2str(mBlockData[mBlockSequenceNumber-1].data(),(int) mBlockData[mBlockSequenceNumber-1].size(),' ').c_str());
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmCalcCRC ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmCalcCRC");

   if ( mBlockSequenceNumber > mBlockData.size() ) return;

   mCRC = 0;

   if ( mConfig.getCRCCalculationMode() == true )
   {
      // calculate checksum
      dia_CRCCalculator* pCrcCalc = getInstanceOfCRCCalculator();
      if ( pCrcCalc )
      {
         DIA_TR_INF("      +--> Calculating CRC for data '%s'", dia::utils::bin2str(mBlockData[mBlockSequenceNumber-1].data(),(int) mBlockData[mBlockSequenceNumber-1].size(),' ').c_str());
         mCRC = pCrcCalc->calcCrCCITT(&(mBlockData[mBlockSequenceNumber-1])[0],(tU16) mBlockData[mBlockSequenceNumber-1].size(),DIA_EN_CRC_FORMAT_BIG_ENDIAN);
         // TODO: proper error handling
         DIA_TR_INF("      +--> Calculated CRC: 0x%04x", mCRC);
      }
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmCheckForPDXContainer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmCheckForPDXContainer");

//   static bool needToLoadPDXContainers = true;

   FactoryPluginPDXFlashing* pdxFlashingFactory = getInstanceOfFactoryPluginPDXFlashing();
   if ( !pdxFlashingFactory ) return;

   if ( needToLoadPDXContainers )
   {
      const std::list<std::string>& pdxFiles = pdxFlashingFactory->getPDXContainers();
      std::list<std::string>::const_iterator pdxIter = pdxFiles.begin();
      for ( ; pdxIter != pdxFiles.end(); ++pdxIter )
      {
         mPDXContainers.push_back(*pdxIter);
      }

      DIA_TR_INF("######################################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# NUMBER OF LOADED PDX CONTAINERS: %zu", mPDXContainers.size());
      DIA_TR_INF("#");
      DIA_TR_INF("######################################################################");
      needToLoadPDXContainers = false;
   }

   if ( mPDXContainers.size() == 0 )
   {
      DIA_TR_INF("##### NO PDX CONTAINER FOUND #####");
      return;
   }

   std::string pdxContainerFullName = mPDXContainers.front();
   mPDXContainers.pop_front();
   DIA_TR_INF("  +--> Selected PDX container '%s'", pdxContainerFullName.c_str());
   size_t found = pdxContainerFullName.find_last_of("/\\");
   std::string pdxContainerStrippedName(pdxContainerFullName.substr(found+1));
   mPDXFolder = pdxContainerFullName.substr(0,pdxContainerFullName.find_last_of("/") + 1);
   mPDXContainer = pdxContainerStrippedName;
   mIsPDXContainerAvailable  = true;
   mNeedToRemovePDXContainer = true;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmDiagnosisActive ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmDiagnosisActive");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmDiagnosisInactive ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmDiagnosisInactive");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmEvaluateJob ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmEvaluateJob");

   if ( !mCurrentJob ) return;

   switch ( mCurrentJob->mID )
   {
   case DIA_EN_FLASH_JOB_ID_EXPECTED_IDENT:
      {
         DIA_TR_INF("##### PROCESSING EXPECTED IDENT #####");
         postEvent(dia_EngineFlashFSM::evExpectedIdent);
      }
      break;
   case DIA_EN_FLASH_JOB_ID_WDBI:
      {
         DIA_TR_INF("##### PROCESSING WDBI #####");
         postEvent(dia_EngineFlashFSM::evWriteDataByID);
      }
      break;
   case DIA_EN_FLASH_JOB_ID_DATA_DOWNLOAD:
      {
         DIA_TR_INF("##### PROCESSING DATA DOWNLOAD #####");
         postEvent(dia_EngineFlashFSM::evDataDownload);
      }
      break;
   default:
      {
         DIA_TR_INF("##### PROCESSING UNKNOWN JOB #####");
         postEvent(dia_EngineFlashFSM::evUnknownJob);
      }
      break;
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmEvaluateResponseDownloadRequest ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmEvaluateResponseDownloadRequest");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmEvaluateResponseTransferData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmEvaluateResponseTransferData");

   if ( !mpResponseData ) return;

   if ( isPositiveResponseReceived() && (mResponseLength >= 2) )
   {
      if ( mBlockSequenceNumber == mBlockData.size() )
      {
         mIsDataTransferComplete = true;
      }
      postEvent(dia_EngineFlashFSM::evDataTransferred);
   }
   else
   {
      postEvent(dia_EngineFlashFSM::evDataTransferFailed);
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmEvaluateResponseTransferExit ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmEvaluateResponseTransferExit");

   if ( !mpResponseData ) return;

   postEvent(isPositiveResponseReceived() ? dia_EngineFlashFSM::evProcessed : dia_EngineFlashFSM::evProcessingFailed);
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmEvaluateResponseWDBI ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmEvaluateResponseWDBI");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmEvalueteResponseExpectedIdent ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmEvalueteResponseExpectedIdent");

   if ( (!mpResponseData) || (!mCurrentJob) ) return;

   if ( isPositiveResponseReceived() && (mResponseLength > 3) )
   {
      bool errDetected = true;
      odx::ExpectedIdent* pExpectedIdent = (odx::ExpectedIdent*) mCurrentJob->mCookie;
      const std::list<odx::IdentValue*>& listIdentValues = pExpectedIdent->getIdentValues();
      if ( listIdentValues.size() )
      {
         const odx::IdentValue* pIdentValue = listIdentValues.front();
         if ( pIdentValue->getType() == "A_ASCIISTRING" )
         {
            if ( mResponseLength > 3 ) // has data
            {
               tU16 lengthToCompare = tU16(mResponseLength - 3);
               DIA_TR_INF("EXPECTED DATA LENGTH: %zu",pIdentValue->getTextValue().length());
               DIA_TR_INF("RECEIVED DATA LENGTH: %u",lengthToCompare);
               if ( lengthToCompare == pIdentValue->getTextValue().length() )
               {
                  tU16 matchCount = 0;
                  for ( tU16 i=0,j=3; i<lengthToCompare; i++,j++ )
                  {
                     if ( mpResponseData[j] == pIdentValue->getTextValue().at(i) ) matchCount++;
                  }
                  if ( matchCount == lengthToCompare )
                  {
                     DIA_TR_INF("##### EXPECTED IDENT MATCHES #####");
                     errDetected = false;
                  }
                  else
                  {
                     DIA_TR_INF("##### EXPECTED IDENT DOES NOT MATCH #####");
                  }
               }
            }
         }
      }

      postEvent(errDetected ? dia_EngineFlashFSM::evProcessingFailed : dia_EngineFlashFSM::evProcessed);
   }
   else
   {
      postEvent(dia_EngineFlashFSM::evProcessingFailed);
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmFinalize ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmFinalize");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmFinalizeFlashing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmFinalizeFlashing");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmFlashingComplete ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmFlashingComplete");
}

#define DIA_C_U16_BLOCK_SEQUENCE_NUMBER_INVALID       ((tU16) 0xFFFF)

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmGetNextBlock ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmGetNextBlock");

   mBlockSequenceNumber++;

   if ( mBlockSequenceNumber > mBlockData.size() )
   {
      // data transmission done
      mBlockSequenceNumber = DIA_C_U16_BLOCK_SEQUENCE_NUMBER_INVALID;
      mIsDataTransferComplete = true;
   }

   postEvent(dia_EngineFlashFSM::evDataTransfer);
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmGetNextJob ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmGetNextJob");

   if ( mCurrentJob )
   {
      delete mCurrentJob;
      mCurrentJob = 0;
   }

   if ( mJobList.empty() )
   {
      postEvent(dia_EngineFlashFSM::evDone);
      return;
   }

   mCurrentJob = mJobList.front();
   mJobList.pop_front();
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleCatalogError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleCatalogError");
   mErrorCode = DIA_E_FLASH_ENGINE_CATALOG_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleDataError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleDataError");
   mErrorCode = DIA_E_FLASH_ENGINE_DATA_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleDataTransferError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleDataTransferError");
   mErrorCode = DIA_E_FLASH_ENGINE_TRANSFER_DATA_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleDisableFlushingError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleDisableFlushingError");
   mErrorCode = DIA_E_FLASH_ENGINE_DISABLE_FLUSH_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleEnableFlushingError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleEnableFlushingError");
   mErrorCode = DIA_E_FLASH_ENGINE_ENABLE_FLUSH_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleExpectedIdentError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleExpectedIdentError");
   mErrorCode = DIA_E_FLASH_ENGINE_EXPECTED_IDENT_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleFlashContainerError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleFlashContainerError");
   mErrorCode = DIA_E_FLASH_ENGINE_FLASH_CONTAINER_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleRequestDownloadError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleRequestDownloadError");
   mErrorCode = DIA_E_FLASH_ENGINE_REQUEST_DOWNLOAD_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandlerTransferExitError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandlerTransferExitError");
   mErrorCode = DIA_E_FLASH_ENGINE_TRANSFER_EXIT_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleSessionChangeError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleSessionChangeError");
   mErrorCode = DIA_E_FLASH_ENGINE_SESSION_CHANGE_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleTimeout ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleTimeout");
   mErrorCode = DIA_E_TIMEOUT;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleUnknownJob ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleUnknownJob");
   mErrorCode = DIA_E_FLASH_ENGINE_UNKNOWN_JOB;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleUnpackingError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleUnpackingError");
   mErrorCode = DIA_E_FLASH_ENGINE_UNPACKING_FAILED;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleWDBIError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleWDBIError");
   mErrorCode = DIA_E_FLASH_ENGINE_WDBI_ERROR;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmHandleError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmHandleError");

   // further error handling strategies might be added in future ...
   // Error Handling for KDS Write Failure
   dia_TestController* pTestController = getInstanceOfTestController();
   if ( pTestController )
      {
         DIA_TR_INF("--- dia_EventThread::vWaitThread => KDS Write Fail ...");
         (void) pTestController->runTests(DIA_EN_TESTCONDITION_KDS_WRITE_FAIL);
      }
   else
      {
         DIA_TR_ERR("!!! dia_EventThread::vWaitThread => ERROR: KDS pTestController == NULL");
      }

   postEvent(dia_EngineFlashFSM::evErrorHandled);
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmInitializeFlashing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmInitializeFlashing");

   DIA_TR_INF("  +--> LOADING DID MAPPING TABLE ...");
   tU16 repSize = static_cast<tU16>(sizeof(gIndentInfo) / sizeof(IdentInfo));
   DIA_TR_INF("    +--> %d Elements found",repSize);
   for ( tU16 i=0; i<repSize; i++ )
   {
      mDIDLookupTable[gIndentInfo[i].mUID] = &(gIndentInfo[i]);
      DIA_TR_INF("    +--> Added DID 0x%04x ('%s')", gIndentInfo[i].mDID, gIndentInfo[i].mName.c_str());
   }

   DIA_IMPL_LIST_REPOSITORY_TEAR_DOWN(FlashJob,mJobList);
   mCurrentJob = 0;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmLoadJobs ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmLoadJobs");

   vLoadExpectedIdents();
   vLoadDataBlocksWDBI();
   vLoadDataBlocksDownload();
}

//-----------------------------------------------------------------------------

void
EngineFlash::vLoadExpectedIdents ( void )
{
   ScopeTrace oTrace("EngineFlash::vLoadExpectedIdents()");

   if ( !mpODXF ) return;
   const odx::Flash* pFlash = mpODXF->getFlash();
   if ( !pFlash ) return;
   const std::list<odx::EcuMem*>& listEcuMems = pFlash->getEcuMems();
   if ( listEcuMems.empty() ) return;
   const odx::Mem* pMem = listEcuMems.front()->getMem();
   if ( !pMem ) return;
   const std::list<odx::Session*>& listSessions = pMem->getSessions();
   if ( listSessions.empty() ) return;
   const odx::Session* pSession = listSessions.front();

   const std::list<odx::ExpectedIdent*>& listExpectedIdents = pSession->getExpectedIdents();
   std::list<odx::ExpectedIdent*>::const_iterator cIter = listExpectedIdents.begin();
   for ( ; cIter != listExpectedIdents.end(); ++cIter )
   {
      DIA_TR_INF("  +--> Loading Expected Ident '%s'", (*cIter)->getShortName().c_str());
      dia_UID uid = dia_getHashCodeFromString((*cIter)->getShortName().c_str());
      std::map<dia::UID,IdentInfo*>::iterator mapIter = mDIDLookupTable.find(uid);
      if ( (mapIter != mDIDLookupTable.end()) && mapIter->second )
      {
         DIA_TR_INF("    +--> Expected Ident found. DID = 0x%04x", mapIter->second->mDID);
         std::vector<tU8> data;
         mJobList.push_back(DIA_NEW FlashJob(DIA_EN_FLASH_JOB_ID_EXPECTED_IDENT,0x22,mapIter->second->mDID,data,(void*) *cIter));
      }
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vLoadDataBlocksWDBI ( void )
{
   ScopeTrace oTrace("EngineFlash::vLoadDataBlocksWDBI()");

   if ( !mpODXF ) return;

   const odx::Flash* pFlash = mpODXF->getFlash();
   if ( !pFlash ) return;
   const std::list<odx::EcuMem*>& listEcuMems = pFlash->getEcuMems();
   if ( listEcuMems.empty() ) return;
   const odx::Mem* pMem = listEcuMems.front()->getMem();
   if ( !pMem ) return;
   const std::list<odx::Session*>& listSessions = pMem->getSessions();
   if ( listSessions.empty() ) return;
   const odx::Session* pSession = listSessions.front();

   const std::list<odx::DataBlockRef*>& listDataRefs = pSession->getDataBlockRefs();
   std::list<odx::DataBlockRef*>::const_iterator dbRefIter = listDataRefs.begin();
   for ( ; dbRefIter != listDataRefs.end(); ++dbRefIter )
   {
      const odx::DataBlockRef* dbRef = *dbRefIter;
      const odx::DataBlock* pDataBlock = dbRef->getDataBlock();
      if ( pDataBlock->getType() == odx::DIA_ODX_DATABLOCK_TYPE_WDBI )
      {
         DIA_TR_INF("  +--> Loading Jobs For Datablock '%s' (Type = '%s')", pDataBlock->getShortName().c_str(),pDataBlock->getType().c_str());
         tU16 sizeInBytes = (tU16) (pDataBlock->getSegments().front()->getUncompressedSize() - 2); // need to subtract the 2 bytes for the included DID
         DIA_TR_INF("    +--> Size in bytes: %d",sizeInBytes);
         const std::vector<tU8>& dataVec = ((const odx::FlashDataIntern*) (pDataBlock->getFlashDataRef()->getFlashData()))->getData(); //->>getSegments().front()->getUncompressedSize(); //>getFlashDataRef()
         tU16 did = (tU16) ((dataVec[0] << 8) + dataVec[1]);
         DIA_TR_INF("    +--> DID: 0x%04x",did);
         std::vector<tU8> data;
         for ( tU16 j=0,k=2; k<dataVec.size(); j++,k++ )
         {
            data.push_back(dataVec[k]);
         }
         mJobList.push_back(DIA_NEW FlashJob(DIA_EN_FLASH_JOB_ID_WDBI,0x2E,did,data,(void*) *dbRefIter));
      }
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vLoadDataBlocksDownload ( void )
{
   ScopeTrace oTrace("EngineFlash::vLoadDataBlocksDownload()");

   if ( !mpODXF ) return;

   const odx::Flash* pFlash = mpODXF->getFlash();
   if ( !pFlash ) return;
   const std::list<odx::EcuMem*>& listEcuMems = pFlash->getEcuMems();
   if ( listEcuMems.empty() ) return;
   const odx::Mem* pMem = listEcuMems.front()->getMem();
   if ( !pMem ) return;
   const std::list<odx::Session*>& listSessions = pMem->getSessions();
   if ( listSessions.empty() ) return;
   const odx::Session* pSession = listSessions.front();

   const std::list<odx::DataBlockRef*>& listDataRefs = pSession->getDataBlockRefs();
   std::list<odx::DataBlockRef*>::const_iterator dbRefIter = listDataRefs.begin();
   for ( ; dbRefIter != listDataRefs.end(); ++dbRefIter )
   {
      const odx::DataBlockRef* dbRef = *dbRefIter;
      const odx::DataBlock* pDataBlock = dbRef->getDataBlock();
      if ( pDataBlock->getType() == odx::DIA_ODX_DATABLOCK_TYPE_DOWNLOAD )
      {
         DIA_TR_INF("  +--> Loading Jobs For Datablock '%s' (Type = '%s')", pDataBlock->getShortName().c_str(),pDataBlock->getType().c_str());
         tU32 sizeInBytes = pDataBlock->getSegments().front()->getUncompressedSize();
         DIA_TR_INF("    +--> Size in bytes : %d",sizeInBytes);
         tU32 sourceAddr  = pDataBlock->getSegments().front()->getSourceStartAddress(); //((odx::FlashDataExtern*) (pDataBlock->getFlashDataRef()->getFlashData()))->g>getData(); //->>getSegments().front()->getUncompressedSize(); //>getFlashDataRef()
         DIA_TR_INF("    +--> Source address: 0x%08x",sourceAddr);
         std::vector<tU8> data;
         data.push_back(0x00); // FMT
         data.push_back(0x33); // ADDRFMT
         data.push_back((tU8) (sourceAddr  >> 16));
         data.push_back((tU8) (sourceAddr  >>  8));
         data.push_back((tU8) (sourceAddr  >>  0));
         data.push_back((tU8) (sizeInBytes >> 16));
         data.push_back((tU8) (sizeInBytes >>  8));
         data.push_back((tU8) (sizeInBytes >>  0));
         mJobList.push_back(DIA_NEW FlashJob(DIA_EN_FLASH_JOB_ID_DATA_DOWNLOAD,0x34,0x0000,data,(void*) *dbRefIter));
      }
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmLogError ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmLogError");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmReadData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmReadData");

   if ( !mCurrentJob ) return;

   const odx::DataBlockRef* dbRef = (const odx::DataBlockRef*) this->mCurrentJob->mCookie;
   const odx::DataBlock* pDataBlock = dbRef->getDataBlock();
   DIA_TR_INF("    +--> Readding data of Datablock '%s' (Type = '%s')", pDataBlock->getShortName().c_str(),pDataBlock->getType().c_str());
   const odx::FlashDataExtern* pFlashData = (const odx::FlashDataExtern*) pDataBlock->getFlashDataRef()->getFlashData();
   DIA_TR_INF("      +--> FlashDataRef = '%s'",pDataBlock->getFlashDataRef()->getIdRef().c_str()); //setDataFileName
   std::string dataFileNameWithPath = mPDXFolderUnpacked + "/" + pFlashData->getDataFileName();
   DIA_TR_INF("      +--> Reading data from '%s'",dataFileNameWithPath.c_str()); //setDataFileName

   ifstream fileFlashData(dataFileNameWithPath.c_str(),ios::binary);
   fileFlashData.seekg((long long) 0,ios::end);
   tU32 fileSizeInBytes = (tU32) fileFlashData.tellg();
   DIA_TR_INF("      +--> File size is %d bytes",fileSizeInBytes);
   mFlashData.clear();
   mFlashData.reserve(fileSizeInBytes);
   mFlashData.resize(fileSizeInBytes);
   fileFlashData.seekg((long long) 0, ios::beg);
   fileFlashData.read(&mFlashData[0], (int) fileSizeInBytes);
   DIA_TR_INF("      +--> %zu bytes read from file",mFlashData.size());
   DIA_TR_INF("      +--> Data: '%s'", dia::utils::bin2str((tU8*) (&mFlashData[0]),(int) mFlashData.size(),' ').c_str());

   mBlockData.clear();
   mIsDataTransferComplete = false;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmReadODXFlashContainer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmReadODXFlashContainer");

   if ( mPDXFolderUnpacked.empty() || (!mPDXFlashContainerFile) ) return;

   if ( !mpParser )
   {
      mpParser = new odx::PDXParserStrategyTinyXML(std::string("PDXParser"));
      if ( !mpParser )
      {
         DIA_TR_ERR("##### FAILED TO CREATE PDX PARSER #####");
         return;
      }
   }

   std::string odxFileNameWithPath = mPDXFolderUnpacked + "/" + mPDXFlashContainerFile->getFileName();
   DIA_TR_INF("  +--> Parsing ODX-F File '%s'", odxFileNameWithPath.c_str());

   if ( mpParser->parseOdxFlashContainer(odxFileNameWithPath) == DIA_SUCCESS )
   {
      DIA_TR_INF("  +--> Parsing SUCCEEDED");
      mpODXF = mpParser->getODX();
      postEvent(dia_EngineFlashFSM::evFlashContainerOk);
   }
   else
   {
      DIA_TR_INF("  +--> Parsing FAILED");
      postEvent(dia_EngineFlashFSM::evFlashContainerError);
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmReadPDXCatalog ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmReadPDXCatalog");

   if ( mPDXFolderUnpacked.empty() ) return;

   if ( !mpParser )
   {
      mpParser = new odx::PDXParserStrategyTinyXML(std::string("PDXParser"));
      if ( !mpParser )
      {
         DIA_TR_ERR("##### FAILED TO CREATE PDX PARSER #####");
         return;
      }
   }

   std::string pdxFileNameWithPath = mPDXFolderUnpacked + "/index.xml";
   DIA_TR_INF("  +--> Parsing XML File '%s'", pdxFileNameWithPath.c_str());

   if ( mpParser->parseIndexXML(pdxFileNameWithPath) == DIA_SUCCESS )
   {
      DIA_TR_INF("  +--> Parsing SUCCEEDED");
      mpPDXCatalog = mpParser->getPdxCatalog();
      postEvent(dia_EngineFlashFSM::evCatalogAvailable);
   }
   else
   {
      DIA_TR_INF("  +--> Parsing FAILED");
      postEvent(dia_EngineFlashFSM::evCatalogError);
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmRemovePDXContainer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmRemovePDXContainer");

   removePDXContainer();
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmRestoreActiveSession ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmRestoreActiveSession");
   mSession = mSessionToBeRestored;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestDisableFlushing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestDisableFlushing");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestDownloadRequest ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestDownloadRequest");

   if ( !mCurrentJob ) return;

   DIA_TR_INF("--- Sending Data Download Request To Service Engine ...");

   std::vector<tU8> data;
   data.push_back(0x00); // place holder, will be set later
   data.push_back(0x34);
   data.insert(data.end(), mCurrentJob->mData.begin(), mCurrentJob->mData.end());
   tU8 dataLength = (data.size() > 1) ? ((tU8) (data.size()-1)) : 0;
   data[0] = dataLength;

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = DIA_NEW dia_MessageBufferUDS(data.data(), (tU16) data.size(), onDownloadResponse, dia_MessageBuffer::holds_request_internal); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via FlashEngine: Length: %zu Data: %s", data.size(), dia::utils::bin2str(data.data(),(int) data.size(),' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used

//-----------------------------------------------------------------------------

void
EngineFlash::onDownloadResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("EngineFlash::onDownloadResponse");

   DIA_TR_SM("UDS msg TX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   EngineFlash* pEngine = getInstanceOfFactoryPluginPDXFlashing()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = DIA_NEW tU8[pEngine->mResponseLength];

      //copy message data to the buffer
      ::memcpy(pEngine->mpResponseData, data, length);

      pEngine->postEvent((pEngine->isPositiveResponseReceived()) ? dia_EngineFlashFSM::evProcessed : dia_EngineFlashFSM::evProcessingFailed);
   }

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestEnableFlushing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestEnableFlushing");
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestReadIdent ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestReadIdent");

   if ( !mCurrentJob ) return;

   DIA_TR_INF("--- Sending RDBI Request To Service Engine ...");

   static const tU16 length = 4;
   static tU8 data[length] = { length, 0x22, 0x00, 0x00 };
   data[2] = (tU8) (mCurrentJob->mDID >> 8);
   data[3] = (tU8) (mCurrentJob->mDID >> 0);

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = DIA_NEW dia_MessageBufferUDS(data,length, onReadIdentResponse, dia_MessageBuffer::holds_request_internal); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used

//-----------------------------------------------------------------------------

void
EngineFlash::onReadIdentResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("EngineFlash::onReadIdentResponse");

   DIA_TR_SM("UDS msg TX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   EngineFlash* pEngine = getInstanceOfFactoryPluginPDXFlashing()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = DIA_NEW tU8[pEngine->mResponseLength];

      //copy message data to the buffer
      ::memcpy(pEngine->mpResponseData, data, length);

      pEngine->postEvent(dia_EngineFlashFSM::evResponseReceived);
   }

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestSessionChange ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestSessionChange");

   DIA_TR_INF("--- Sending Session Control Request To Service Engine (Session = 0x60)");

   static const tU16 length = 3;
   static tU8 data[length] = {length, 0x10, 0x00};
   data[2] = mSession;

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = DIA_NEW dia_MessageBufferUDS(data, 3, onSessionChangeResponse, dia_MessageBuffer::holds_request_internal); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used

//-----------------------------------------------------------------------------

void
EngineFlash::onSessionChangeResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("EngineFlash::onSessionChangeResponse");

   DIA_TR_SM("UDS msg TX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   EngineFlash* pEngine = getInstanceOfFactoryPluginPDXFlashing()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = DIA_NEW tU8[pEngine->mResponseLength];

      //copy message data to the buffer
      ::memcpy(pEngine->mpResponseData, data, length);

      pEngine->postEvent((pEngine->isPositiveResponseReceived()) ? dia_EngineFlashFSM::evProcessed : dia_EngineFlashFSM::evProcessingFailed);
   }

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}


//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestTransferData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestTransferData");

   if ( !mCurrentJob ) return;

   DIA_TR_INF("--- Sending Data Transfer Request To Service Engine ...");

   std::vector<tU8> data;
   data.push_back(0x00); // place holder, will be set later
   data.push_back(0x36);
   data.push_back((tU8) mBlockSequenceNumber);
   data.insert(data.end(), mBlockData[mBlockSequenceNumber-1].begin(), mBlockData[mBlockSequenceNumber-1].end());
   tU8 dataLength = (data.size() > 1) ? ((tU8) (data.size()-1)) : 0;
   data[0] = dataLength;

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = DIA_NEW dia_MessageBufferUDS(data.data(), (tU16) data.size(), onTransferDataResponse, dia_MessageBuffer::holds_request_internal); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via FlashEngine: Length: %zu Data: %s", data.size(), dia::utils::bin2str(data.data(),(int) data.size(),' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used

//-----------------------------------------------------------------------------

void
EngineFlash::onTransferDataResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("EngineFlash::onTransferDataResponse");

   DIA_TR_SM("UDS msg TX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   EngineFlash* pEngine = getInstanceOfFactoryPluginPDXFlashing()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = DIA_NEW tU8[pEngine->mResponseLength];
      ::memcpy(pEngine->mpResponseData, data, length);
      pEngine->postEvent(dia_EngineFlashFSM::evResponseReceived);
   }

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}


//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestTransferExit ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestTransferExit");

   if ( !mCurrentJob ) return;

   DIA_TR_INF("--- Sending Transfer Exit Request To Service Engine ...");

   std::vector<tU8> data;
   data.push_back(0x01); // place holder, will be set later
   data.push_back(0x37);

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = DIA_NEW dia_MessageBufferUDS(data.data(),(tU16) data.size(), onTransferExitResponse, dia_MessageBuffer::holds_request_internal); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via FlashEngine: Length: %zu Data: %s", data.size(), dia::utils::bin2str(data.data(),(int) data.size(),' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used

//-----------------------------------------------------------------------------

void
EngineFlash::onTransferExitResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("EngineFlash::onTransferExitResponse");

   DIA_TR_SM("UDS msg TX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   EngineFlash* pEngine = getInstanceOfFactoryPluginPDXFlashing()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = DIA_NEW tU8[pEngine->mResponseLength];
      ::memcpy(pEngine->mpResponseData, data, length);
      pEngine->postEvent(dia_EngineFlashFSM::evResponseReceived);
   }

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}
//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSendRequestWDBI ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSendRequestWDBI");

   if ( !mCurrentJob ) return;

   DIA_TR_INF("--- Sending WDBI Request To Service Engine ...");

   std::vector<tU8> data;
   data.push_back(0x00); // place holder, will be set later
   data.push_back(0x2E);
   data.push_back((tU8) (mCurrentJob->mDID >> 8));
   data.push_back((tU8) (mCurrentJob->mDID >> 0));
   data.insert(data.end(), mCurrentJob->mData.begin(), mCurrentJob->mData.end());
   tU8 dataLength = (data.size() > 1) ? ((tU8) (data.size()-1)) : 0;
   data[0] = dataLength;

   // create an object containing the UDS msg buffer
   dia_MessageBufferUDS* pMsgUDS = DIA_NEW dia_MessageBufferUDS(data.data(),(tU16) data.size(), onWDBIResponse, dia_MessageBuffer::holds_request_internal); //lint !e429: custodial pointer is freed by after engine has processed the message

   DIA_TR_SM("UDS msg RX via FlashEngine: Length: %zu Data: %s", data.size(), dia::utils::bin2str(data.data(),(int) data.size(),' ').c_str());

   // create and send event ReqRx containing a ptr to the reqest msg buffer
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventReqRx(pMsgUDS)); //lint !e429: custodial pointer is freed by after engine has processed the message

   // make lint happy
   pMsgUDS = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'poMsgBuffer'. --> lifetime is controlled by diagnostic session
} //lint !e438 Warning: last value assigned to variable 'poMsgBuffer' not used

//-----------------------------------------------------------------------------

void
EngineFlash::onWDBIResponse ( const tU8 data[], tU16 length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("EngineFlash::onWDBIResponse");

   DIA_TR_SM("UDS msg TX via FlashEngine: Length: %d Data: %s", length, dia::utils::bin2str(data,length,' ').c_str());

   EngineFlash* pEngine = getInstanceOfFactoryPluginPDXFlashing()->getEngine();

   if ( pEngine )
   {
      pEngine->mResponseLength = length;
      pEngine->mpResponseData  = DIA_NEW tU8[pEngine->mResponseLength];

      //copy message data to the buffer
      ::memcpy(pEngine->mpResponseData, data, length);

      pEngine->postEvent((pEngine->isPositiveResponseReceived()) ? dia_EngineFlashFSM::evProcessed : dia_EngineFlashFSM::evProcessingFailed);
   }

   // sent trigger to the UDS state machine that the response was sent
   getInstanceOfApplication()->postMessage(DIA_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
}

//-----------------------------------------------------------------------------

tU16
EngineFlash::getBlockSizeFromResponse ( void ) const
{
   // return invalid block size
   if ( !mpResponseData ) return DIA_C_U16_INVALID_BLOCK_SIZE;
   if ( mResponseLength < 4 ) return DIA_C_U16_INVALID_BLOCK_SIZE;

   return tU16((mpResponseData[2] << 8) + mpResponseData[3]);
}

//-----------------------------------------------------------------------------

tU16
EngineFlash::calcBlockSize ( void ) const
{
   const tU16 DIA_C_U16_CRC_SIZE_IN_BYTES = 2;

   tU16 blockSize = getBlockSizeFromResponse();
   if ( (blockSize != DIA_C_U16_INVALID_BLOCK_SIZE) && mConfig.getCRCCalculationMode() && (blockSize >= DIA_C_U16_CRC_SIZE_IN_BYTES) )
   {
      blockSize = tU16(blockSize + DIA_C_U16_CRC_SIZE_IN_BYTES);
   }

   return blockSize;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmSplitData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmSplitData");

   if ( !mpResponseData ) return;

   mBlockSizeInBytes = calcBlockSize();
   DIA_TR_INF("      +--> mBlockSizeInBytes = %d",mBlockSizeInBytes);

   mBlockSequenceNumber = 0;
   mBlockData.clear();

   if ( mFlashData.size() )
   {
      bool isDone   = false;
      size_t position = 0;
      tU16 i = 0;

      while ( !isDone )
      {
         size_t remainingLength = mFlashData.size() - position;
         size_t blockSize = mBlockSizeInBytes;
         if ( remainingLength > mBlockSizeInBytes )
         {
            std::vector<tU8> dataVec;
            dataVec.insert(dataVec.end(), &mFlashData[position], &mFlashData[position+mBlockSizeInBytes]);

            mBlockData.push_back(dataVec);
            position += mBlockSizeInBytes;
         }
         else
         {
            std::vector<tU8> dataVec;
            dataVec.insert(dataVec.end(), &mFlashData[position], &mFlashData[position+remainingLength]);
            mBlockData.push_back(dataVec);
            isDone = true;
            position = mFlashData.size();
            blockSize = remainingLength;
         }
         DIA_TR_INF("      +--> Created Block #%02d (size is %zu bytes)",i,blockSize);
         DIA_TR_INF("      +--> Block Data: '%s'", dia::utils::bin2str(mBlockData[i].data(),(int) mBlockData[i].size(),' ').c_str());
         i++;
      }
   }
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStartTimerDownload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmStartTimerDownload");
   mIsTimerRunning = true;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStartTimerExpectedIdent ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmStartTimerExpectedIdent");
   mIsTimerRunning = true;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStartTimerFlushing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmStartTimerFlushing");
   mIsTimerRunning = true;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStoreActiveSession ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmRestoreActiveSession");

   dia_EngineServer* pEngine = 0;
   if ( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) != DIA_SUCCESS )
   {
      DIA_TR_ERR(" --- FAILED TO GET ACTIVE DIAGNOSTIC SESSION !!!!");
      return;
   }

   mSessionToBeRestored = pEngine->getActiveSession()->getID();
   DIA_TR_INF(" --- ACTIVE DIAGNOSTIC SESSION IS 0x%02x ", mSessionToBeRestored);
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStartTimerSession ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmStartTimerSession");
   mIsTimerRunning = true;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStartTimerWDBI ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmStartTimerWDBI");
   mIsTimerRunning = true;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmStopTimer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmStopTimer");
   mIsTimerRunning = false;
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmUnpackPDXContainer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmUnpackPDXContainer");

   DIA_TR_INF("##### UNPACKING PDX CONTAINER '%s' (FOLDER = '%s')",((mPDXContainer.empty()) ? "EMPY-STRING" : mPDXContainer.c_str()),((mPDXFolder.empty()) ? "EMPY-STRING" : mPDXFolder.c_str()));
   postEvent((unpackPDXContainer(mPDXContainer) == DIA_SUCCESS) ? dia_EngineFlashFSM::evUnpacked : dia_EngineFlashFSM::evUnpackingFailed);
}

//-----------------------------------------------------------------------------

void
EngineFlash::vFsmFinalizeFlashProcess ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::vFsmFinalizeFlashProcess");

   if ( mPDXContainers.size() != 0 )
   {
      DIA_TR_INF("#####################################################################");
      DIA_TR_INF("#                                                                   #");
      DIA_TR_INF("##### FURTHER PDX CONTAINER FOUND. INITIATING NEW FLASH PROCESS #####");
      DIA_TR_INF("#                                                                   #");
      DIA_TR_INF("#####################################################################");
      postEvent(dia_EngineFlashFSM::evStart);
   }
   else
   {
      (void)dia_setProperty(DIA_PROP_ENGINE_PDX_FLASH_FINISH, (tU8)TRUE);
      dia_TestController* pTestController = getInstanceOfTestController();
      if ( pTestController )
      {
    	  DIA_TR_INF("--- dia_EventThread::vWaitThread => Ending PDXFlashingTest ...");
    	  (void) pTestController->runTests(DIA_EN_TESTCONDITION_PDX_FLASHING_END);
      }
      else
      {
    	  DIA_TR_ERR("!!! dia_EventThread::vWaitThread => ERROR: pTestController == NULL");
      }
   }
}

//-----------------------------------------------------------------------------

bool
EngineFlash::bFsmHasCatalogFlashContainer ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::bFsmHasCatalogFlashContainer");

   mHasCatalogFlashContainer = false;

   if ( mpPDXCatalog )
   {
      const std::list<odx::PDXABlock*>& listOfPDXABlocks = mpPDXCatalog->getABlocks();
      if ( listOfPDXABlocks.size() )
      {
         odx::PDXABlock* pPDXABlock = listOfPDXABlocks.front();
         if ( pPDXABlock )
         {
            const std::list<odx::PDXFile*>& files = pPDXABlock->getFiles();
            if ( files.size() )
            {
               std::list<odx::PDXFile*>::const_iterator cIter = files.begin();
               for ( ; cIter != files.end(); ++cIter )
               {
                  const odx::PDXFile* pFile = (*cIter);
                  DIA_TR_INF("  +--> Found File '%s' in PDXCatalog; Suffix '%s'", pFile->getFileName().c_str(), pFile->getSuffix().c_str());
                  if ( pFile->getSuffix() == "odx-f" )
                  {
                     DIA_TR_INF("    +--> ODX-F FILE DETECTED !!!");
                     mPDXFlashContainerFile = pFile;
                     mHasCatalogFlashContainer = true;
                  }
               }
            }
         }
      }
   }

   return mHasCatalogFlashContainer;
}

//-----------------------------------------------------------------------------

bool
EngineFlash::bFsmIsDataTransferComplete ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::bFsmIsDataTransferComplete");
   return mIsDataTransferComplete;
}

//-----------------------------------------------------------------------------

bool
EngineFlash::bFsmIsPDXContainerAvailable ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::bFsmIsPDXContainerAvailable");
   return mIsPDXContainerAvailable;
}

//-----------------------------------------------------------------------------

bool
EngineFlash::bFsmNeedToDisableFlushing ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::bFsmNeedToDisableFlushing");

   return mConfig.getFlushControlMode();
}

//-----------------------------------------------------------------------------

bool
EngineFlash::bFsmWasTimerStarted ( void* /*pArg*/ )
{
   ScopeTrace oTrace("EngineFlash::bFsmWasTimerStarted");

   return mIsTimerRunning;
}

//-------------------------------------------------------------------------

void
EngineFlash::vOnTimerElapsed ( dia_TimerID id )
{
   tU32 timerID = (tU32) id;

   DIA_TR_INF("### EngineFlash::vOnTimerElapsed --- ID = %d ###", timerID);

   if ( timerID == mEngineTimerID )
   {
      postEvent(dia_EngineFlashFSM::evTimeout);
   }
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
EngineFlash::unpackPDXContainer ( const std::string& pdxFileName )
{
   ScopeTrace oTrace("dia::EngineFlash::unpackPDXContainer(std::string&)");

   if ( mPDXFolder.empty() ) return DIA_FAILED;
   if ( pdxFileName.empty() ) return DIA_FAILED;

   DIA_TR_INF("mPDXFolder  = %s",mPDXFolder.c_str());
   DIA_TR_INF("pdxFileName = %s",pdxFileName.c_str());

   std::string destFolder("/tmp");
   size_t found = pdxFileName.find_last_of(".");
   std::string strippedName(pdxFileName.substr(0,found));
   destFolder += '/';
   destFolder += strippedName;

   std::string cmdString1 = "rm -rf " + destFolder; // command used to retrieve the list of shared libraries in the given folder
   DIA_TR_INF("  +--> Executing command '%s'",cmdString1.c_str());
   int ret = system(cmdString1.c_str());
   if ( WIFSIGNALED(ret) && ((WTERMSIG(ret) == SIGINT) || (WTERMSIG(ret) == SIGQUIT)) )
   {
      DIA_TR_INF("  +--> COMMAND EXECUTION FAILED (REMOVE EXISTING PDX FOLDER !!!");
      return DIA_FAILED;
   }

   std::string cmdString2 = "mkdir " + destFolder; // command used to retrieve the list of shared libraries in the given folder
   DIA_TR_INF("  +--> Executing command '%s'",cmdString2.c_str());
   ret = system(cmdString2.c_str());
   if ( WIFSIGNALED(ret) && ((WTERMSIG(ret) == SIGINT) || (WTERMSIG(ret) == SIGQUIT)) )
   {
      DIA_TR_INF("  +--> COMMAND EXECUTION FAILED (CREATING FOLDER FOR INFLATED PDX CONTENT!!!");
      return DIA_FAILED;
   }

   DIA_TR_INF("mPDXFolder  = '%s'. size=%zu", mPDXFolder.c_str(), mPDXFolder.size());
   DIA_TR_INF("pdxFileName = '%s'. size=%zu", pdxFileName.c_str(), pdxFileName.size());
   DIA_TR_INF("destFolder  = '%s'. size=%zu", destFolder.c_str(), destFolder.size());

   char cmd[2000];
   (void) snprintf(cmd, sizeof(cmd), "unzip %s/%s -d %s 2>&1",  mPDXFolder.c_str(), pdxFileName.c_str(), destFolder.c_str());
   DIA_TR_INF("  +--> Executing command (snprintf) '%s'",cmd);
   ret = system(cmd);

   if ( WIFSIGNALED(ret) && ((WTERMSIG(ret) == SIGINT) || (WTERMSIG(ret) == SIGQUIT)) )
   {
      DIA_TR_INF("  +--> UNZIP COMMAND EXECUTION FAILED !!!");
      return DIA_FAILED;
   }

   mPDXFolderUnpacked = destFolder;

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

void
EngineFlash::removePDXContainer ( void )
{
   ScopeTrace oTrace("dia::EngineFlash::removePDXContainer()");

   if ( mPDXContainer.empty() ) return;

   DIA_TR_INF("  +--> mPDXFolder    = '%s'",mPDXFolder.c_str());
   DIA_TR_INF("  +--> mPDXContainer = '%s'",mPDXContainer.c_str());

   char cmd[2000];
   (void) snprintf(cmd, sizeof(cmd), "rm -f %s/%s", mPDXFolder.c_str(), mPDXContainer.c_str());
   DIA_TR_INF("  +--> Removing '%s/%s'",mPDXFolder.c_str(), mPDXContainer.c_str());
   std::string cmdString(cmd);

   if ( executeSystemCommand(cmdString) != DIA_SUCCESS )
   {
      DIA_TR_INF("  +--> FAILED TO REMOVE EXISTING PDX CONTAINER '%s' !!!", mPDXContainer.c_str());
   }

   mPDXContainer.clear();
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
EngineFlash::executeSystemCommand ( const std::string& cmdString )
{
   ScopeTrace oTrace("dia::EngineFlash::executeSystemCommand()");

   if ( cmdString.empty() ) return DIA_E_EMPTY_STRING;

   DIA_TR_INF("  +--> Executing command '%s'",cmdString.c_str());
   int resultCode = system(cmdString.c_str());
   if ( WIFSIGNALED(resultCode) && ((WTERMSIG(resultCode) == SIGINT) || (WTERMSIG(resultCode) == SIGQUIT)) )
   {
      DIA_TR_INF("  +--> COMMAND EXECUTION FAILED !!!");
      return DIA_FAILED;
   }

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

bool
EngineFlash::isPositiveResponseReceived ( void ) const
{
   return ((mResponseLength >= 1) && (mpResponseData[0] != 0x7F)) ? true : false;
}

}
