/*!
 * \file       dia_UploadDownloadStrategyFiles.h
 *
 * \brief      Upload and download strategy used for service $38 (RequestFileTransfer)
 *
 * \details    Abstract upload and download strategy used for service $34 (RequestDownload) and $35 (RequestUpload)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY_FILES__
#define __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY_FILES__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY__
#include <common/framework/datadownload/dia_UploadDownloadStrategy.h>
#endif

#ifndef __INCLUDED_DIA_FILETRANSFERFSM__
#include <common/framework/fsm/generated/dia_FileTransferFSM.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SESSION_LISTENER__
#include <common/interfaces/dia_ISessionListener.h>
#endif

namespace dia {

class UploadDownloadStrategyFiles
   : public UploadDownloadStrategy,
     protected dia_FileTransferFSM::FsmBehaviour,
     public dia_ISessionListenerImpl
{
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(UploadDownloadStrategyFiles);
protected:
   //! mode of operation as defined in the UDS specification
   enum enModeOfOperation
   {
      DIA_EN_MODE_OF_OPERATION_UNKNOWN = 0,
      DIA_EN_MODE_OF_OPERATION_ADD_FILE,
      DIA_EN_MODE_OF_OPERATION_DELETE_FILE,
      DIA_EN_MODE_OF_OPERATION_REPLACE_FILE,
      DIA_EN_MODE_OF_OPERATION_READ_FILE,
      DIA_EN_MODE_OF_OPERATION_READ_DIR,
      DIA_EN_MODE_OF_OPERATION_COUNT
   };

public:
   //! class constructor
   UploadDownloadStrategyFiles ( const std::string& name );
   //! class constructor
   UploadDownloadStrategyFiles ( const char* name );
   //! class destructor
   virtual ~UploadDownloadStrategyFiles ( void );

   //! we use a two level construction process
   virtual tDiaResult setup ( void );
   virtual tDiaResult tearDown ( void );

   virtual tU32 getID ( void );

   virtual tDiaResult requestDownload ( std::vector<tU8>& /*requestData*/ )  { return DIA_E_NOT_SUPPORTED; }
   virtual tDiaResult requestFileTransfer ( std::vector<tU8>& requestData );
   virtual tDiaResult requestUpload ( std::vector<tU8>& /*requestData*/ )  { return DIA_E_NOT_SUPPORTED; }
   virtual tDiaResult transferData ( std::vector<tU8>& requestData, std::vector<tU8>& responseData );
   virtual tDiaResult transferExit ( std::vector<tU8>& requestData );

   //! called by the session control object to notify about a session change
   virtual void vOnSessionChanged ( tU8 newSession, tU8 oldSession );

protected:
   //! deprecated default class constructor
   UploadDownloadStrategyFiles ( void );

   //! method used for posting events to the state machine
   tDiaResult acceptEvent ( dia_FileTransferFSM::FsmEvent event, void* pArg );

   //! validate the data of an incoming RequestDownload request
   virtual tDiaResult validateDataRequestFileTransfer ( std::vector<tU8>& data );
   //! validate the data of an incoming transfer data request
   virtual tDiaResult validateDataTransferData ( std::vector<tU8>& data );
   //! validate the data of an incoming transfer data request
   virtual tDiaResult validateDataTransferExit ( std::vector<tU8>& data );
   //! check if format identifier is ok
   virtual tDiaResult validateDataFormatIdentifier ( std::vector<tU8>& data );
   //! ckeck length of file size parameter
   virtual tDiaResult validatefileSizeParameterLength ( std::vector<tU8>& data );
   //! extract the file path and file name from data
   virtual void readFilePathAndName ( std::vector<tU8>& data );

   tDiaResult triggerFileTransfer ( void );
   void triggerReset ( void );

   // FSM actions
   virtual void vFsmCalcCrcRam ( void* pArg );
   virtual void vFsmCheckForCompletion ( void* pArg );
   virtual void vFsmDeleteFile ( void* pArg );
   virtual void vFsmDownloadData ( void* pArg );
   virtual void vFsmPostprocessDownload ( void* pArg );
   virtual void vFsmPostprocessUpload ( void* pArg );
   virtual void vFsmPrepareDirRead ( void* pArg );
   virtual void vFsmPrepareFileAdd ( void* pArg );
   virtual void vFsmPrepareFileDelete ( void* pArg );
   virtual void vFsmPrepareFileRead ( void* pArg );
   virtual void vFsmPrepareFileReplace ( void* pArg );
   virtual void vFsmReset ( void* pArg );
   virtual void vFsmSaveFile ( void* pArg );
   virtual void vFsmSendDownloadTransferDataResponse ( void* pArg );
   virtual void vFsmSendErrorAlreadyRequested ( void* pArg );
   virtual void vFsmSendErrorInvalidData ( void* pArg );
   virtual void vFsmSendErrorInvalidDataTransExit ( void* pArg );
   virtual void vFsmSendErrorInvalidFormat ( void* pArg );
   virtual void vFsmSendErrorInvalidLength ( void* pArg );
   virtual void vFsmSendErrorInvalidRequest ( void* pArg );
   virtual void vFsmSendErrorInvalidSequence ( void* pArg );
   virtual void vFsmSendErrorNoPermission ( void* pArg );
   virtual void vFsmSendErrorRequestRejectedDirRead ( void* pArg );
   virtual void vFsmSendErrorRequestRejectedFileAdd ( void* pArg );
   virtual void vFsmSendErrorRequestRejectedFileDelete ( void* pArg );
   virtual void vFsmSendErrorRequestRejectedFileRead ( void* pArg );
   virtual void vFsmSendErrorRequestRejectedFileReplace ( void* pArg );
   virtual void vFsmSendRequestDirReadResponse ( void* pArg );
   virtual void vFsmSendRequestFileAddResponse ( void* pArg );
   virtual void vFsmSendRequestFileDeleteResponse ( void* pArg );
   virtual void vFsmSendRequestFileReadResponse ( void* pArg );
   virtual void vFsmSendRequestFileReplaceResponse ( void* pArg );
   virtual void vFsmSendTransExitResponse ( void* pArg );
   virtual void vFsmSendUploadTransferDataResponse ( void* pArg );
   virtual void vFsmSetDownloadActive ( void* pArg );
   virtual void vFsmSetUploadActive ( void* pArg );
   virtual void vFsmUploadData ( void* pArg );
   virtual void vFsmValidateFileTransferRequest ( void* pArg );
   virtual void vFsmValidateTransferredData ( void* pArg );

   // FSM guards
   virtual bool areDirReadConditionsOK ( void* pArg );
   virtual bool areFileAddConditionsOK ( void* pArg );
   virtual bool areFileDeleteConditionsOK ( void* pArg );
   virtual bool areFileReadConditionsOK ( void* pArg );
   virtual bool areFileReplaceConditionsOK ( void* pArg );
   virtual bool bFsmIsCrcOk ( void* pArg );
   virtual bool bFsmIsDataValid ( void* pArg );

   tDiaResult checkTransferredDownloadData ( tU16 bsn, tU8 /*data*/[], tU16 /*length*/ );
   bool isDataTransferComplete ( void ) const;
   tDiaResult downloadData ( tU16 bsn, tU8 data[], tU16 length );
   virtual const tU8* getDataTransferred ( void ) const { return mpByteBufferDownload; }

protected:
   //! pointer to the FSM object
   dia_FileTransferFSM::Fsm* mpFSM;
   //! error code
   tDiaResult mErrorCode;
   //! mode of operation, could be addFile, deleteFile, replaceFile, readFile, readDir
   enModeOfOperation mModeOfOperation;
   //! length of the file path and name in bytes (format is ASCII)
   size_t mFilePathAndNameLength;
   //! path and name of the file
   std::string mFilePathAndName;
   //!
   tU8 mFileSizeParameterLength;
   //!
   tU8 mDataFormatIdentifier;

   tU32 mSizeInBytes;

   //! block sequence number
   tU16 mBSN;

   //! flag to indicate that a data transfer is in progress
   bool mIsDataTransferActive;
   //! flag to indicate that a data transfer is in progress
   bool mIsDataTransferComplete;

   //
   // download related attributes
   //

   //! pointer to the byte buffer that keeps the downloaded data
   tU8* mpByteBufferDownload;
   //! pointer to the byte buffer that keeps the downloaded data
   tU16 mWritePosition;
   //! pointer to the byte buffer that keeps the downloaded data
   tU16 mNumOfBytesReceived;

   //
   // upload related attributes
   //

   //! pointer to the byte buffer that keeps the downloaded data
   tU16 mReadPosition;
   //! pointer to the byte buffer that keeps the downloaded data
   tU16 mNumOfBytesTransmitted;
};

}

#endif /* __INCLUDED_DIA_UPLOAD_DOWNLOAD_STRATEGY_FILES__ */
