/*!
 * \file       dia_UploadDownloadStrategyDatasets.cpp
 *
 * \brief      Abstract upload and download strategy used for service $34 (RequestDownload) and $35 (RequestUpload)
 *
 * \details    Abstract upload and download strategy used for service $34 (RequestDownload) and $35 (RequestUpload)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include "dia_UploadDownloadStrategyDatasets.h"

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include "common/framework/datadownload/dia_DataDownloadManager.h"
#endif

#ifndef __INCLUDED_DATA_DOWNLOAD_MANAGER_PLUGIN__
#include "common/framework/datadownload/dia_DataDownloadManagerPlugin.h"
#endif

#ifndef __INCLUDED_DIA_DATASET__
#include "common/framework/datadownload/dia_Dataset.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

namespace dia {

//-----------------------------------------------------------------------------

UploadDownloadStrategyDatasets::UploadDownloadStrategyDatasets ( const std::string& name, tU8 fmtID, tU8 addrFmtID )
   : UploadDownloadStrategy(name),
     mFormatIdentifier(fmtID),
     mAddressFormatIdentifier(addrFmtID),
     mpFSM(0),
     mErrorCode(DIA_E_NOERROR),
     mIsCrcOk(false),
     mIsDataValid(false),
     mRequestedStartAddress(0),
     mRequestedMemSize(0),
     mRequestedDataset(0)
{
   ScopeTrace trc("dia::UploadDownloadStrategyDatasets::UploadDownloadStrategyDatasets");

   // create the state machine object
   if ( !(dia_DataDownloadFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
   }
}

//-----------------------------------------------------------------------------

UploadDownloadStrategyDatasets::UploadDownloadStrategyDatasets ( const char* name, tU8 fmtID, tU8 addrFmtID )
   : UploadDownloadStrategy(name),
     mFormatIdentifier(fmtID),
     mAddressFormatIdentifier(addrFmtID),
     mpFSM(0),
     mErrorCode(DIA_E_NOERROR),
     mIsCrcOk(false),
     mIsDataValid(false),
     mRequestedStartAddress(0),
     mRequestedMemSize(0),
     mRequestedDataset(0)
{
   ScopeTrace trc("dia::UploadDownloadStrategyDatasets::UploadDownloadStrategyDatasets");

   // create the state machine object
   if ( !(dia_DataDownloadFSM::Fsm::createFSM(&mpFSM,this)) )
   {
      DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
   }
}

//-----------------------------------------------------------------------------

UploadDownloadStrategyDatasets::~UploadDownloadStrategyDatasets ( void )
{
   _BP_TRY_BEGIN
   {
      delete mpFSM;
      mpFSM = 0;
      mRequestedDataset = 0;
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia::UploadDownloadStrategyDatasets::~UploadDownloadStrategyDatasets !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tU32
UploadDownloadStrategyDatasets::getID ( void ) //const
{
   return (((tU32) mAddressFormatIdentifier) | (((tU32) mFormatIdentifier) << 8));
}

//------------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyDatasets::triggerDownload ( void )
{
   mErrorCode = DIA_SUCCESS;
   if ( mpFSM ) mpFSM->acceptEvent(dia_DataDownloadFSM::evRequestDownload,0);

   return mErrorCode;
}

//------------------------------------------------------------------------------

tDiaResult
UploadDownloadStrategyDatasets::triggerUpload ( void )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::triggerUpload()");

   mErrorCode = DIA_SUCCESS;
   if ( mpFSM ) mpFSM->acceptEvent(dia_DataDownloadFSM::evRequestUpload,0);

   return mErrorCode;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::triggerReset ( void )
{
   if ( mpFSM ) mpFSM->acceptEvent(dia_DataDownloadFSM::evIdle,0);
   mErrorCode = DIA_E_NOERROR;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_InvalidAddress ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_InvalidAddress()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeReqDnlInvalidAddress() : DIA_E_OUT_OF_RANGE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_InvalidLength ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_InvalidLength()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeReqDnlInvalidLength() : DIA_E_OUT_OF_RANGE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_InvalidFormat ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_InvalidFormat()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeReqDnlInvalidFormat() : DIA_E_OUT_OF_RANGE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_AlreadyRequested ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_AlreadyRequested()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeReqDnlAlreadyRequested() : DIA_E_CONDITIONS_NOT_CORRECT;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_InvalidData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_InvalidData()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeTransDataInvalidData() : DIA_E_GENERAL_PROGRAMMING_FAILURE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_TransExitInvalidData ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_TransExitInvalidData()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeTransExitInvalidData() : DIA_E_GENERAL_PROGRAMMING_FAILURE;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendErr_InvalidSequence ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendErr_InvalidSequence()");

   mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeTransDataInvalidSequence() : DIA_E_SEQUENCE_ERROR;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::validateDownloadRequest ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::validateDownloadRequest()");

   if ( ! mpFSM ) return;

   mRequestedDataset = 0;
   tDiaResult res = getInstanceOfDataDownloadManager()->queryDataset(mRequestedStartAddress,&mRequestedDataset);

   if ( (DIA_SUCCESS!=res) || (!mRequestedDataset) )
   {
      DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::validateDownloadRequest res=0x%08X %s", res, (mRequestedDataset ? "": "mRequestedDataset is NULL") );
      mpFSM->acceptEvent(dia_DataDownloadFSM::evInvalidAddr,0);
      return;
   }

   if ( mRequestedDataset->hasVariableLength() )
   {
      if ( mRequestedMemSize > mRequestedDataset->getSize() )
      {
         DIA_TR_ERR("### dia::UploadDownloadStrategyDatasets::validateDownloadRequest size=%d mRequestedMemSize=%d (max exceeded)", mRequestedDataset->getSize(), mRequestedMemSize );
         mpFSM->acceptEvent(dia_DataDownloadFSM::evInvalidLength,0);
         return;
      }
   }
   else
   {
      if ( mRequestedMemSize != mRequestedDataset->getSize() )
      {
         DIA_TR_ERR("### dia::UploadDownloadStrategyDatasets::validateDownloadRequest size=%d mRequestedMemSize=%d", mRequestedDataset->getSize(), mRequestedMemSize );
         mpFSM->acceptEvent(dia_DataDownloadFSM::evInvalidLength,0);
         return;
      }
   }

   mRequestedDataset->setRequestedSize((tU32) mRequestedMemSize);

   mRequestedDataset->setCallback(this);

   mpFSM->acceptEvent(dia_DataDownloadFSM::evRequestDownloadValid,0);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::validateUploadRequest ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::validateUploadRequest()");

   if ( ! mpFSM ) return;

   mRequestedDataset = 0;
   if ( (getInstanceOfDataDownloadManager()->queryDataset(mRequestedStartAddress,&mRequestedDataset) != DIA_SUCCESS) || (!mRequestedDataset) )
   {
      mpFSM->acceptEvent(dia_DataDownloadFSM::evInvalidAddr,0);
      return;
   }

   if ( mRequestedDataset->hasVariableLength() )
   {
      if ( mRequestedMemSize > mRequestedDataset->getSize() )
      {
         DIA_TR_ERR("### dia::UploadDownloadStrategyDatasets::validateUploadRequest size=%d mRequestedMemSize=%d (max exceeded)", mRequestedDataset->getSize(), mRequestedMemSize );
         mpFSM->acceptEvent(dia_DataDownloadFSM::evInvalidLength,0);
         return;
      }
   }
   else
   {
      if ( mRequestedMemSize != mRequestedDataset->getSize() )
      {
         DIA_TR_ERR("### dia::UploadDownloadStrategyDatasets::validateUploadRequest size=%d mRequestedMemSize=%d", mRequestedDataset->getSize(), mRequestedMemSize );
         mpFSM->acceptEvent(dia_DataDownloadFSM::evInvalidLength,0);
         return;
      }
   }

   mpFSM->acceptEvent(dia_DataDownloadFSM::evRequestUploadValid,0);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::prepareDownload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::prepareDownload()");

   if ( !mpPlugin || !mRequestedDataset )
   {
      DIA_TR_ERR("### prepareDownload DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s mRequestedDataset=%s", (!mpPlugin? "NULL":"OK"), (!mRequestedDataset? "NULL":"OK"));

      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   mErrorCode = mpPlugin->prepareDownload(*mRequestedDataset);

   if (DIA_SUCCESS!=mErrorCode)
   {
      DIA_TR_ERR("### prepareDownload mErrorCode is set to 0x%08X", mErrorCode);
   }

   mIsCrcOk   = false;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::onInitializeDataDownload( void )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::onInitializeDataDownload");

   if ( !mpFSM )
   {
      DIA_ASSERT_ALWAYS();
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   mpFSM->acceptEvent(dia_DataDownloadFSM::evRequestDownloadReady, 0);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::prepareUpload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::prepareUpload()");

   if ( !mpPlugin || !mRequestedDataset )
   {
      DIA_TR_ERR("### prepareUpload DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s mRequestedDataset=%s", (!mpPlugin? "NULL":"OK"), (!mRequestedDataset? "NULL":"OK"));

      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   mErrorCode = mpPlugin->prepareUpload(*mRequestedDataset);

   if (DIA_SUCCESS!=mErrorCode)
   {
      DIA_TR_ERR("### prepareUpload mErrorCode is set to 0x%08X", mErrorCode);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::postprocessDownload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::postprocessDownload()");

   if ( !mpPlugin || !mRequestedDataset )
   {
      DIA_TR_ERR("### postprocessDownload DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s mRequestedDataset=%s", (!mpPlugin? "NULL":"OK"), (!mRequestedDataset? "NULL":"OK"));

      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   if (DIA_SUCCESS==mErrorCode)
   {
      mErrorCode = mpPlugin->postprocessDownload(*mRequestedDataset);

      if (DIA_SUCCESS!=mErrorCode)
      {
         DIA_TR_ERR("### postprocessDownload mErrorCode is set to 0x%08X", mErrorCode);
      }
   }
   else
   {
      DIA_TR_ERR("### postprocessDownload ERROR Previous error was set in mErrorCode=0x%08X.", mErrorCode);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::onFinalizeDataDownload( void )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::onFinalizeDataDownload");

   if ( !mpFSM || !mRequestedDataset )
   {
      DIA_ASSERT_ALWAYS();
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   mRequestedDataset->setCallback(nullptr);

   mpFSM->acceptEvent(dia_DataDownloadFSM::evDataTransferExitReady, 0);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::postprocessUpload ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::postprocessUpload()");

   if ( !mpPlugin || !mRequestedDataset )
   {
      DIA_TR_ERR("### postprocessUpload DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s mRequestedDataset=%s", (!mpPlugin? "NULL":"OK"), (!mRequestedDataset? "NULL":"OK"));

      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   if (DIA_SUCCESS==mErrorCode)
   {
      mErrorCode = mpPlugin->postprocessUpload(*mRequestedDataset);

      if (DIA_SUCCESS!=mErrorCode)
      {
         DIA_TR_ERR("### postprocessUpload mErrorCode is set to 0x%08X", mErrorCode);
      }
   }
   else
   {
      DIA_TR_ERR("### postprocessUpload ERROR Previous error was set in mErrorCode=0x%08X.", mErrorCode);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendReqDownloadResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendReqDnlResponse()");
   mResponseData.clear();
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockByteNum()  << 4));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() >> 8));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() & 0x00FF));

   dia_IDataDownloadListener* pListener = nullptr;
   if ( (DIA_SUCCESS == querySysAdapterListener<dia_IDataDownloadListener>(&pListener)) && pListener )
   {
      pListener->vOnRequestDownloadUpdate(mResponseData);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendReqUploadResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendReqUploadResponse()");
   mResponseData.clear();
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockByteNum()  << 4));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() >> 8));
   mResponseData.push_back((tU8) (getInstanceOfDataDownloadManager()->getDataBlockMaxBytes() & 0x00FF));
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::saveDataset ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::saveDataset()");

   if ( !mRequestedDataset )
   {
      DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::saveDataset ERROR mRequestedDataset is NULL");
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   mErrorCode = mRequestedDataset->sync();

   if (DIA_SUCCESS!=mErrorCode)
   {
      DIA_TR_ERR("### saveDataset mErrorCode is set to 0x%08X", mErrorCode);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendTransExitResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendTransExitResponse()");

   dia_IDataDownloadListener* pListener = nullptr;
   if ( (DIA_SUCCESS == querySysAdapterListener<dia_IDataDownloadListener>(&pListener)) && pListener )
   {
      pListener->vOnTransferExitUpdate();
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::checkForCompletion ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::checkForCompletion()");

   if ( !mRequestedDataset )
   {
      DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::checkForCompletion ERROR mRequestedDataset is NULL");
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   if ( mRequestedDataset->isDataTransferComplete() )
   {
      if ( !mpFSM )
      {
         DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::checkForCompletion ERROR mpFSM is NULL");
         mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
         return;
      }

      mpFSM->acceptEvent(dia_DataDownloadFSM::evDataTransferComplete,0);
   }
   else
   {
      DIA_TR_INF("dia::UploadDownloadStrategyDatasets::checkForCompletion nothing to be done.");
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::validateTransferredData ( void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::validateTransferredData()");

   if ( ! mpFSM || !pArg || !mRequestedDataset )
   {
      DIA_TR_ERR("### validateTransferredData DIA_E_GENERAL_PROGRAMMING_FAILURE mpFSM=%s pArg=%s", (!mpFSM? "NULL":"OK"), (!pArg? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   std::vector<tU8> dataVec = *((std::vector<tU8>*) pArg);

   if ( dataVec.size() < 3 )
   {
      DIA_TR_INF("dia_DataDownloadManager::validateTransferredData: No Data Available !!");
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   // at this point we know that byte buffer has at least 3 bytes
   tU16 bsn   =  dataVec[1];
   tU8* pData = &dataVec[0];
   tU8* pDataStripped = &pData[2]; //lint !e416 Warning: Likely creation of out-of-bounds pointer (2 beyond end of data) by operator '['

   // validate the received data and store validation result for further processing
   mIsDataValid = (mRequestedDataset->checkTransferredDownloadData(bsn, pDataStripped, (tU16) (dataVec.size()-2)) == DIA_SUCCESS) ? true : false;

   // data is validated so we can continue processing
   mpFSM->acceptEvent(dia_DataDownloadFSM::evDataTransferValidated,pArg);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::calcCrcRam ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::calcCrcRam()");

   if ( !mRequestedDataset )
   {
      DIA_TR_ERR("dia::UploadDownloadStrategyDatasets::calcCrcRam ERROR mRequestedDataset is NULL");
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   mIsCrcOk = mRequestedDataset->isCrcOK();
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::downloadData ( void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::downloadData()");

   if ( ! mpFSM || !pArg || !mRequestedDataset )
   {
      DIA_TR_ERR("### downloadData DIA_E_GENERAL_PROGRAMMING_FAILURE mpFSM=%s pArg=%s", (!mpFSM? "NULL":"OK"), (!pArg? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   std::vector<tU8> dataVec = *((std::vector<tU8>*) pArg);

   if ( dataVec.size() >= 3 )
   {
      // at this point we know that byte buffer has at least 3 bytes
      tU16 bsn   =  dataVec[1];
      tU8* pData = &dataVec[0];
      tU8* pDataStripped = &pData[2]; //lint !e416 Warning: Likely creation of out-of-bounds pointer (2 beyond end of data) by operator '['

      // data was already validated so we start can safely perform the download at this point
      mErrorCode = mRequestedDataset->downloadData(bsn, pDataStripped, (tU16) (dataVec.size()-2));

      if (DIA_SUCCESS!=mErrorCode)
      {
         DIA_TR_ERR("### downloadData mErrorCode is set to 0x%08X", mErrorCode);
      }

      // data is validated so we can continue processing
      mpFSM->acceptEvent(dia_DataDownloadFSM::evDataTransferDone,pArg);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendDownloadTransferDataResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendTransferDataResponse()");
   mErrorCode = DIA_SUCCESS;
}

void
UploadDownloadStrategyDatasets::retransmitTransferDataResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::retransmitTransferDataResponse()");
   mErrorCode = DIA_SUCCESS;
   /* nothing else to be done here, last resonse to transfer-data is still stored in mResponseData */
}


//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::uploadData ( void* pArg )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::uploadData()");

   if ( ! mpFSM || !pArg || !mRequestedDataset )
   {
      DIA_TR_ERR("### uploadData DIA_E_GENERAL_PROGRAMMING_FAILURE mpFSM=%s pArg=%s", (!mpFSM? "NULL":"OK"), (!pArg? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
   }

   std::vector<tU8> dataVec = *((std::vector<tU8>*) pArg);

   tU16 bsn = dataVec[1];

   // data was already validated so we start can safely perform the download at this point
   mErrorCode = mRequestedDataset->uploadData(bsn,mResponseData);

   if (DIA_SUCCESS!=mErrorCode)
   {
      DIA_TR_ERR("### uploadData mErrorCode is set to 0x%08X", mErrorCode);
   }
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::sendUploadTransferDataResponse ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::sendUploadTransferDataResponse()");

   //   mErrorCode = DIA_SUCCESS;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyDatasets::isCrcOk ( void* /*pArg*/ )
{
   return mIsCrcOk;
}

bool
UploadDownloadStrategyDatasets::isRetransmission ( void* pArg ) {
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::isRetransmission()");

   if ( ! mpFSM || !pArg || !mRequestedDataset )
   {
      DIA_TR_ERR("### isRetransmission DIA_E_GENERAL_PROGRAMMING_FAILURE mpFSM=%s pArg=%s", (!mpFSM? "NULL":"OK"), (!pArg? "NULL":"OK"));
      return false;
   }

   std::vector<tU8> dataVec = *((std::vector<tU8>*) pArg);

   if ( dataVec.size() < 2 )
   {
      DIA_TR_INF("dia_DataDownloadManager::isRetransmission: No Data Available !!");
      return false;
   }

   // at this point we know that byte buffer has at least 3 bytes
   tU16 bsn   =  dataVec[1];

   bool res = mRequestedDataset->isRetransmission(bsn);
   DIA_TR_INF("dia_DataDownloadManager::isRetransmission() bsn=%u res=%u", bsn, res);
   return res;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyDatasets::isDataValid ( void* /*pArg*/ )
{
   return mIsDataValid;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyDatasets::areDownloadConditionsOk ( void* /*pArg*/ )
{
   return true;
}

//------------------------------------------------------------------------------

bool
UploadDownloadStrategyDatasets::areUploadConditionsOk ( void* /*pArg*/ )
{
   return true;
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::setDownloadActivated ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::setDownloadActivated()");
   getInstanceOfDataDownloadManager()->setOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_DOWNLOAD);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::setUploadActivated ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::setUploadActivated()");
   getInstanceOfDataDownloadManager()->setOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_UPLOAD);
}

//------------------------------------------------------------------------------

void
UploadDownloadStrategyDatasets::reset ( void* /*pArg*/ )
{
   ScopeTrace oTrace("dia::UploadDownloadStrategyDatasets::reset()");

   if ( mRequestedDataset )
   {
      mRequestedDataset->setCallback(nullptr);
      mRequestedDataset->initializeDataDownload();
   }
   else
   {
      DIA_TR_ERR("### reset mRequestedDataset is NULL");
   }

   getInstanceOfDataDownloadManager()->setOperationMode(DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_IDLE);
}

}


