/*
 * dia_Dataset.cpp
 *
 *  Created on: 11.01.2014
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_DATASET__
#include "common/framework/datadownload/dia_Dataset.h"
#endif

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include "common/framework/datadownload/dia_DataDownloadManager.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_CRC_CALCULATOR__
#include "common/framework/utils/dia_CRCCalculator.h"
#endif

#define DIA_RECEIVE_BUFF_LEN  (4096U)

static tU8  startBitMask[8] = {
      0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F, 0xFF
};

static tU8   stopBitMask[8] = {
      0xFF, 0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01
};

using namespace dia;

//-----------------------------------------------------------------------------

dia_Dataset::dia_Dataset( tCString name, tU32 startAddr, tU32 length, tU32 propID )
   : mName(name),
     mStartAddr(startAddr),
     mSizeInBytes(length),
     mSizeInBytesRequested(length),
     mSettingsBitmask(DIA_C_U16_DATASET_CFG_UPDOWNLOADABLE),
     mPropID(propID),
     mCRC(0),
     mIsDirty(false),
     mIsLoaded(false),
     mProgAttempts(0),
     mpByteBuffer(0),
     mITC(0),
     mBSN(DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER),
     mIsDataTransferActive(false),
     mIsDataTransferComplete(false),
     mpCallback(nullptr),
     mpByteBufferDownload(0),
     mWritePosition(0),
     mNumOfBytesReceived(0),
     mReadPosition(0),
     mNumOfBytesTransmitted(0)
{}

//-----------------------------------------------------------------------------

dia_Dataset::dia_Dataset( const dia_DatasetConfig& cfgObj )
   : mName(cfgObj.mName),
     mStartAddr(cfgObj.mAddrStart),
     mSizeInBytes(cfgObj.mSizeInBytes),
     mSizeInBytesRequested(cfgObj.mSizeInBytes),
     mSettingsBitmask(cfgObj.mSettings),
     mPropID(cfgObj.mPropID),
     mCRC(0),
     mIsDirty(false),
     mIsLoaded(false),
     mProgAttempts(0),
     mpByteBuffer(0),
     mITC(cfgObj.mITC),
     mBSN(DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER),
     mIsDataTransferActive(false),
     mIsDataTransferComplete(false),
     mpCallback(nullptr),
     mpByteBufferDownload(0),
     mWritePosition(0),
     mNumOfBytesReceived(0),
     mReadPosition(0),
     mNumOfBytesTransmitted(0)
{}

//-----------------------------------------------------------------------------

dia_Dataset::~dia_Dataset( void )
{
   mName  = 0;
   if ( mpByteBuffer )
   {
      OSAL_DELETE [] mpByteBuffer;
      mpByteBuffer = 0;
   }

   if ( mpByteBufferDownload )
   {
      OSAL_DELETE [] mpByteBufferDownload;
      mpByteBufferDownload = 0;
   }
}

//-----------------------------------------------------------------------------

tCString
dia_Dataset::getName ( void ) const
{
   return mName;
}

//-----------------------------------------------------------------------------

tU32
dia_Dataset::getLogicalAddressStart ( void ) const
{
   return mStartAddr;
}

//-----------------------------------------------------------------------------

tU32
dia_Dataset::getLogicalAddressEnd ( void ) const
{
   return (mStartAddr + mSizeInBytes - 1);
}

//-----------------------------------------------------------------------------

tU32
dia_Dataset::getSize ( void ) const
{
   // if the dataset is configured to have variable length this value denotes the maximum size
   return mSizeInBytes;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::adjustSettings ( tU16 newSettings )
{
   mSettingsBitmask = newSettings;
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tU16
dia_Dataset::getSettings ( void ) const
{
   return mSettingsBitmask;
}

//-----------------------------------------------------------------------------

void
dia_Dataset::incrProgrammingAttemptCounter ( void )
{
   if ( mProgAttempts < 0xFFFF ) mProgAttempts++;
}

//-----------------------------------------------------------------------------

void
dia_Dataset::clearProgrammingAttemptCounter ( void )
{
   mProgAttempts = 0;
}

//-----------------------------------------------------------------------------

void
dia_Dataset::setProgrammingAttemptCounter ( tU16 value )
{
   mProgAttempts = value;
}

//-----------------------------------------------------------------------------

tU16
dia_Dataset::getProgrammingAttemptCounter ( void ) const
{
   return mProgAttempts;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::getData ( tU8 data[], tU32 length )
{
   ScopeTrace oTrace("dia_Dataset::getData(data,length)");

   if ( hasVariableLength() )
   {
      // length must be in the defined range
      if ( length > mSizeInBytes )
      {
         DIA_TR_INF("DIA_E_DATASET_INVALID_LENGTH length=%d mSizeInBytes=%d (max)", length, mSizeInBytes);
         return DIA_E_DATASET_INVALID_LENGTH;
      }
   }
   else
   {
      // length must exactly match the number of data set bytes
      if ( length != mSizeInBytes )
      {
         DIA_TR_INF("DIA_E_DATASET_INVALID_LENGTH length=%d mSizeInBytes=%d", length, mSizeInBytes);
         return DIA_E_DATASET_INVALID_LENGTH;
      }
   }

   // load the data set bytes into RAM if they were not already loaded
   if ( (!isLoaded()) && (load() != DIA_SUCCESS) )
   {
      DIA_TR_INF("dia_Dataset::getData returned DIA_E_DATASET_LOAD_ERROR");
      return DIA_E_DATASET_LOAD_ERROR;
   }
   // if loaded the data buffer pointer should not be equal to NULL
   if ( !mpByteBuffer )
   {
      DIA_TR_INF("dia_Dataset::getData returned DIA_E_DATASET_INVALID_DATA_POINTER");
      return DIA_E_DATASET_INVALID_DATA_POINTER;
   }

   // copy all data set bytes in one step
   ::memcpy(data,mpByteBuffer,length);

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::getData ( tU8 data[], tU32 length, tU32 startByte, tU32 stopByte )
{
   ScopeTrace oTrace("dia_Dataset::getData(data,length,startByte,stopByte)");

   tU32 rangeInBytes = 0;
   tDiaResult retCode = validateRange(length,startByte,stopByte,&rangeInBytes);

   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("dia_Dataset::getData validateRange failed with code=0x%08X", retCode);
      return retCode;
   }
   if ( (!isLoaded()) && (load() != DIA_SUCCESS) )
   {
      DIA_TR_INF("dia_Dataset::getData returned DIA_E_DATASET_LOAD_ERROR");
      return DIA_E_DATASET_LOAD_ERROR;
   }
   if ( !mpByteBuffer )
   {
      DIA_TR_INF("dia_Dataset::getData returned DIA_E_DATASET_INVALID_DATA_POINTER");
      return DIA_E_DATASET_INVALID_DATA_POINTER;
   }

   ::memcpy(data,&mpByteBuffer[startByte],rangeInBytes);
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::getData ( tU32* pValue, tU32 startByte, tU16 startBit, tU32 stopByte, tU16 stopBit )
{
   ScopeTrace oTrace("dia_Dataset::getData(tU32*,tU16,tU8,tU16,tU8)");

   tU32 rangeInBytes = 0;
   if ( !pValue ) return DIA_E_INVALID_POINTER;
   if ( (startBit > 7) || (stopBit > 7) ) return DIA_E_DATASET_INVALID_DATA_RANGE;
   tDiaResult retCode = validateRange(sizeof(tU32),startByte,stopByte,&rangeInBytes);
   if ( retCode != DIA_SUCCESS ) return retCode;
   if ( (!isLoaded()) && (load() != DIA_SUCCESS) ) return DIA_E_DATASET_LOAD_ERROR;
   if ( !mpByteBuffer ) return DIA_E_DATASET_INVALID_DATA_POINTER;

   tU32 value  = (tU32) (mpByteBuffer[stopByte] & stopBitMask[stopBit]);
   if ( startByte == stopByte )
   {
      value &= (tU32) (startBitMask[startBit]);
   }
   else
   {
      tU32 bitPos = stopBitMask[stopBit] + 1;
      for ( auto i=stopByte-1; i>startByte; --i )
      {
         value |= ((tU32) (mpByteBuffer[i])) << bitPos;
         bitPos += 8;
      }
      value |= ((tU32) (mpByteBuffer[startByte] & startBitMask[startBit])) << bitPos;
   }

   *pValue = value;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::getData ( tU8* pValue, tU32 startByte, tU16 startBit, tU32 stopByte, tU16 stopBit )
{
   ScopeTrace oTrace("dia_Dataset::getData(tU8*,tU16,tU8,tU16,tU8)");

   tU32 rangeInBytes = 0;
   if ( !pValue ) return DIA_E_INVALID_POINTER;
   if ( (startBit > 7) || (stopBit > 7) ) return DIA_E_DATASET_INVALID_DATA_RANGE;
   tDiaResult retCode = validateRange(sizeof(tU8),startByte,stopByte,&rangeInBytes);
   if ( retCode != DIA_SUCCESS ) return retCode;
   if ( (!isLoaded()) && (load() != DIA_SUCCESS) ) return DIA_E_DATASET_LOAD_ERROR;
   if ( !mpByteBuffer ) return DIA_E_DATASET_INVALID_DATA_POINTER;

   tU8 value = mpByteBuffer[stopByte] & stopBitMask[stopBit];
   if ( startByte == stopByte )
   {
      value &= startBitMask[startBit];
   }
   else
   {
      int bitPos = stopBitMask[stopBit] + 1;
      value = tU8(value | ((mpByteBuffer[startByte] & startBitMask[startBit]) << bitPos));
   }

   *pValue = value;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::setData ( const tU8 data[], tU32 length, tBool autoSync )
{
   ScopeTrace oTrace("dia_Dataset::setData()");

   if ( hasVariableLength() )
   {
      // length must be in the defined range
      if ( length > mSizeInBytes )
      {
         DIA_TR_INF("DIA_E_DATASET_INVALID_LENGTH length=%u mSizeInBytes=%u (max)", length, mSizeInBytes);
         return DIA_E_DATASET_INVALID_LENGTH;
      }
   }
   else
   {
      // length must exactly match the number of data set bytes
      if ( length != mSizeInBytes )
      {
         DIA_TR_INF("DIA_E_DATASET_INVALID_LENGTH length=%u mSizeInBytes=%u", length, mSizeInBytes);
         return DIA_E_DATASET_INVALID_LENGTH;
      }
   }

   //if ( (!isLoaded()) && (load() != DIA_SUCCESS) ) return DIA_E_DATASET_LOAD_ERROR;
   if ( !mpByteBuffer )
   {
      mpByteBuffer = OSAL_NEW tU8[length];
      if ( !mpByteBuffer ) return DIA_E_DATASET_INVALID_DATA_POINTER;
      DIA_TR_INF("dia_Dataset::setData: mpByteBuffer=0x%p created for %u bytes", mpByteBuffer, length);
   }
   else
   {
      DIA_TR_ERR("dia_Dataset::setData: mpByteBuffer=0x%p used for %u bytes", mpByteBuffer, length);
   }

   ::memcpy(mpByteBuffer,data,length);
   mIsDirty = true;

   if ( autoSync ) {
      return store(length);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::load ( void )
{
   ScopeTrace oTrace("dia_Dataset::load()");

   if ( !mpByteBuffer )
   {
      mpByteBuffer = OSAL_NEW tU8[mSizeInBytes];
      if ( !mpByteBuffer ) return DIA_E_DATASET_INVALID_DATA_POINTER;
   }

   size_t sizeInBytes = mSizeInBytes;
   DIA_TR_INF("dia_Dataset::load() Max. number of bytes that can be read : %zu", sizeInBytes);
   tDiaResult retCode = dia_getProperty(mPropID,mpByteBuffer,&sizeInBytes);
   DIA_TR_INF("dia_Dataset::load() Actual Size of the property : %zu", sizeInBytes);
   if ( retCode != DIA_SUCCESS )
   {
      return retCode;
   }
   if ( (!hasVariableLength()) && (sizeInBytes != mSizeInBytes) )
   {
      return DIA_FAILED;
   }
   if ( mSizeInBytes > sizeInBytes ) 
   {
      mSizeInBytes = static_cast<tU32>(sizeInBytes);
   }
   DIA_TR_INF("dia_Dataset::load() Number of bytes that were read : %d", mSizeInBytes);

   mIsDirty  = false; // data is in sync
   mIsLoaded = true;  // data is loaded

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::store ( void )
{
   ScopeTrace oTrace("dia_Dataset::store()");
   return store(mSizeInBytes);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::store ( tU32 length )
{
   ScopeTrace oTrace("dia_Dataset::store(length)");

   if ( !mpByteBuffer ) return DIA_E_DATASET_INVALID_DATA_POINTER;

   if ( mIsDirty )
   {
      tDiaResult retCode = dia_setProperty(mPropID,mpByteBuffer, length);
      if ( retCode != DIA_SUCCESS ) return DIA_E_DATASET_STORE_ERROR;
      mIsDirty = false; // data is in sync now
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

bool
dia_Dataset::isCrcOK ( void )
{
   return ( calcCRC() == DIA_SUCCESS ) ? true : false;

}

//-----------------------------------------------------------------------------

bool
dia_Dataset::isRetransmission(tU16 bsn) {
   return (mBSN !=DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER &&   bsn == (tU8)(mBSN));
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::calcCRC ( void )
{
   ScopeTrace oTrace("dia_Dataset::calcCRC()");

   bool calcResult = false;

   mCRC = 0;

   if (mSettingsBitmask & DIA_C_U16_DATASET_CFG_NO_CRC_CHECK)
   {
      DIA_TR_INF("CRC NOT CHECKED DUE TO DATASET CONFIG!!");
      calcResult = true;
   }
   else
   {
      if ( (NULL!=mpByteBufferDownload) && (mNumOfBytesReceived == mSizeInBytes) )
      {
         // data was completely transferred

         // extract CRC value from transferred data
         tU16 dataCRC = 0;
         if (mNumOfBytesReceived>=2)
         {
            /* if number of received data is 0 or 1, then there are no checksum in this data (checksum=0) */
            dataCRC = tU16((mpByteBufferDownload[mNumOfBytesReceived - 2]) << 8);
            dataCRC = tU16(dataCRC | (mpByteBufferDownload[mNumOfBytesReceived - 1]));
         }

         // calculate checksum
         dia_eCRCFormat fmt = DIA_EN_CRC_FORMAT_BIG_ENDIAN;
         if ( mSettingsBitmask & DIA_C_U16_DATASET_CFG_CRC_FMT_LITTLE_ENDIAN )
         {
            fmt = DIA_EN_CRC_FORMAT_LITTLE_ENDIAN;
         }

         dia_CRCCalculator* pCrcCalc = getInstanceOfCRCCalculator();

         if (NULL!=pCrcCalc)
         {
            calcResult = pCrcCalc->calcAndCompareCrcCCITT(mpByteBufferDownload,tU16(mNumOfBytesReceived-2),dataCRC,fmt);
         }
         else
         {
            DIA_TR_INF("dia_Dataset::isCrcOK pCrcCalc NULL PTR!");
         }

         if ( calcResult )
         {
            mCRC = dataCRC;
            DIA_TR_INF("CRC MATCHED !!");
         }
         else
         {
            DIA_TR_INF("CRC NOT MATCHED OR NOT CALCULATED!!");
         }
      }
   }

   return ( calcResult ) ? DIA_SUCCESS : DIA_FAILED;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::validateRange ( tU32 length, tU32 startByte, tU32 stopByte, tU32* pRangeInBytes )
{
   ScopeTrace oTrace("dia_Dataset::validateRange()");

   if ( startByte >= mSizeInBytes )
   {
      DIA_TR_INF("startByte=%d mSizeInBytes=%d DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE", startByte, mSizeInBytes);
      return DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE;
   }
   if ( startByte > stopByte )
   {
      DIA_TR_INF("startByte=%d stopByte=%d DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE", startByte, stopByte);
      return DIA_E_DATASET_INVALID_DATA_RANGE_START_BYTE;
   }

   if ( stopByte >= mSizeInBytes )
   {
      DIA_TR_INF("stopByte=%d mSizeInBytes=%d DIA_E_DATASET_INVALID_DATA_RANGE_STOP_BYTE", stopByte, mSizeInBytes);
      return DIA_E_DATASET_INVALID_DATA_RANGE_STOP_BYTE;
   }
   tU32 rangeInBytes = tU32((stopByte - startByte) + 1);
   if ( length < rangeInBytes )
   {
      DIA_TR_INF("length=%d rangeInBytes=%d DIA_E_DATASET_INVALID_LENGTH", length, rangeInBytes);
      return DIA_E_DATASET_INVALID_LENGTH;
   }

   if ( pRangeInBytes ) *pRangeInBytes = rangeInBytes;
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

bool
dia_Dataset::isDownloadable ( void ) const
{
   return (mSettingsBitmask & DIA_C_U16_DATASET_CFG_DOWNLOADABLE) ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_Dataset::isUploadable ( void ) const
{
   return (mSettingsBitmask & DIA_C_U16_DATASET_CFG_UPLOADABLE) ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_Dataset::hasVariableLength ( void ) const
{
   return (mSettingsBitmask & DIA_C_U16_DATASET_CFG_VARIABLE_LENGTH) ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_Dataset::isDataTransferActive ( void ) const
{
   return mIsDataTransferActive;
}

//-----------------------------------------------------------------------------

bool
dia_Dataset::isDataTransferComplete ( void ) const
{
   return (mIsDataTransferActive) ? mIsDataTransferComplete : false;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::initializeDataDownload ( void )
{
   ScopeTrace oTrace("dia_Dataset::initializeDataDownload()");

   if ( mpByteBufferDownload )
   {
      DIA_DBG_TR_INF("dia_Dataset::initializeDataDownload mpByteBufferDownload=0x%08X", mpByteBufferDownload);
      OSAL_DELETE [] mpByteBufferDownload;
      mpByteBufferDownload = 0;
   }

   mpByteBufferDownload = OSAL_NEW tU8[mSizeInBytes];
   if ( !mpByteBufferDownload )
   {
      DIA_DBG_TR_ERR("dia_Dataset::initializeDataDownload returned DIA_FAILED");
      return DIA_FAILED;
   }

   mWritePosition = 0;
   mNumOfBytesReceived = 0;
   mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
   mIsDataTransferActive = true;
   mIsDataTransferComplete = false;

   if (mpCallback) mpCallback->onInitializeDataDownload();

   DIA_DBG_TR_INF("dia_Dataset::initializeDataDownload returned DIA_SUCCESS");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::initializeDataUpload ( void )
{
   ScopeTrace oTrace("dia_Dataset::initializeDataUpload()");

   if ( (!isLoaded()) && (load() != DIA_SUCCESS) )
   {
      DIA_DBG_TR_ERR("dia_Dataset::initializeDataUpload returned DIA_E_DATASET_LOAD_ERROR");
      return DIA_E_DATASET_LOAD_ERROR;
   }

   mReadPosition = 0;
   mNumOfBytesTransmitted = 0;
   mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
   mIsDataTransferActive = true;
   mIsDataTransferComplete = false;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::finalizeDataDownload ( void )
{
   ScopeTrace oTrace("dia_Dataset::finalizeDataDownload()");

   mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
   mIsDataTransferActive = false;

   if ( mpByteBufferDownload )
   {
      OSAL_DELETE [] mpByteBufferDownload;
      mpByteBufferDownload = 0;
   }

   if ( mpByteBuffer )
   {
      OSAL_DELETE [] mpByteBuffer;
      mpByteBuffer = 0;
   }
   mIsLoaded = false;

   if (mpCallback) mpCallback->onFinalizeDataDownload();

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::finalizeDataUpload ( void )
{
   ScopeTrace oTrace("dia_Dataset::finalizeDataUpload()");

   mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
   mIsDataTransferActive = false;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::downloadData ( tU16 bsn, tU8 data[], tU32 length )
{
   ScopeTrace oTrace("dia_Dataset::downloadData(bsn,data,length)");

   dia_enDownloadManagerOperationMode operatingMode = getInstanceOfDataDownloadManager()->getOperationMode();

   if ( operatingMode != DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_DOWNLOAD )
   {
      return DIA_E_DATASET_INVALID_OPERATION_MODE;
   }

   // download mode is activated
   tU32 numBytesToWrite = ( (mWritePosition + length) >= mSizeInBytes ) ? tU32(mSizeInBytes - mNumOfBytesReceived) : length;

   DIA_TR_INF("##### numBytesToWrite = %d #####", numBytesToWrite);

   tDiaResult retCode = checkTransferredDownloadData(bsn,data,numBytesToWrite);
   if ( retCode != DIA_SUCCESS ) return retCode;

   ::memcpy(&mpByteBufferDownload[mWritePosition],data,numBytesToWrite);

   mWritePosition = tU32(mWritePosition + numBytesToWrite);
   mNumOfBytesReceived = tU32(mNumOfBytesReceived + numBytesToWrite);
   mBSN = bsn;

   DIA_TR_INF("##### mWritePosition = %d #####", mWritePosition);
   DIA_TR_INF("##### mNumOfBytesReceived = %d #####", mNumOfBytesReceived);
   DIA_TR_INF("##### mSizeInBytes = %d #####", mSizeInBytes);
   DIA_TR_INF("##### mSizeInBytesRequested = %d #####", mSizeInBytesRequested);

   if ( mNumOfBytesReceived == mSizeInBytesRequested )
   {
      DIA_TR_INF("##### DATASET DOWNLOAD COMPLETED #####");
      mIsDataTransferComplete = true;
   }

   DIA_TR_INF("##### mIsDataTransferComplete = %s #####", (mIsDataTransferComplete) ? "true" : "false");

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::uploadData ( tU16 bsn, std::vector<tU8>& data )
{
   ScopeTrace oTrace("dia_Dataset::uploadData(bsn, data)");

   dia_enDownloadManagerOperationMode operatingMode = getInstanceOfDataDownloadManager()->getOperationMode();

   if ( operatingMode != DIA_EN_DOWNLOADMANAGER_OPERATION_MODE_UPLOAD )
   {
      DIA_TR_INF("dia_Dataset::uploadData returned DIA_E_DATASET_INVALID_OPERATION_MODE");
      return DIA_E_DATASET_INVALID_OPERATION_MODE;
   }

   // upload mode is activated

   tDiaResult retCode = checkTransferredUploadData(bsn);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("dia_Dataset::uploadData checkTransferredUploadData FAILED code=%d", retCode);
      return retCode;
   }

   tU32 remainingBytes = tU32(mSizeInBytes - mNumOfBytesTransmitted);
   tU32 numBytesToRead = getInstanceOfDataDownloadManager()->getDataBlockMaxBytes();
   DIA_TR_INF("dia_Dataset::uploadData numBytesToRead=%u", numBytesToRead);
   if ( remainingBytes < numBytesToRead ) {
      numBytesToRead = remainingBytes;
   }

   tU8 dataBlock[DIA_RECEIVE_BUFF_LEN] = { 0 };
   retCode = getData(dataBlock, DIA_RECEIVE_BUFF_LEN, mReadPosition, tU16(mReadPosition + numBytesToRead - 1));
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_INF("dia_Dataset::uploadData returned DIA_E_DATASET_DATA_UPLOAD_FAILED");
      return DIA_E_DATASET_DATA_UPLOAD_FAILED;
   }

   data.clear();
   for ( tU32 i=0; i<numBytesToRead; i++ ) data.push_back(dataBlock[i]);

   mReadPosition = tU32(mReadPosition + numBytesToRead);
   mNumOfBytesTransmitted = tU32(mNumOfBytesTransmitted + numBytesToRead);
   mBSN = bsn;

   if ( mNumOfBytesTransmitted == mSizeInBytes) mIsDataTransferComplete = true;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::checkTransferredDownloadData ( tU16 bsn, tU8 /*data*/[], tU32 /*length*/ )
{
   ScopeTrace oTrace("dia_Dataset::checkTransferredData(bsn,data,length)");
   if ( isDataTransferComplete() ) {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_ALREADY_COMPLETED");
      return DIA_E_DATASET_ALREADY_COMPLETED;
   }


   tU8 nbsn = tU8(mBSN + 1);

   if ( bsn != nbsn )
    {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER (received %u, expected %u)", bsn, (mBSN+1) );
      return DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER;
    }

   if ( mWritePosition >= mSizeInBytes )
   {
      DIA_TR_INF("checkTransferredDownloadData: DIA_E_DATASET_INVALID_DATA_RANGE (mWritePosition = %d)", mWritePosition);
      return DIA_E_DATASET_INVALID_DATA_RANGE;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::checkTransferredUploadData ( tU16 bsn )
{
   ScopeTrace oTrace("dia_Dataset::checkTransferredUploadData(bsn)");

   if ( isDataTransferComplete() ) {
      DIA_TR_INF("checkTransferredUploadData: DIA_E_DATASET_ALREADY_COMPLETED");
      return DIA_E_DATASET_ALREADY_COMPLETED;
   }


   tU8 nbsn = tU8(mBSN + 1);

   if ( bsn != nbsn )
   {      
      DIA_TR_INF("checkTransferredUploadData: DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER (BSN=%d)",bsn);
      return DIA_E_DATASET_INVALID_BLOCK_SEQUENCE_NUMBER;
   }

   if ( mReadPosition >= mSizeInBytes ) {
      DIA_TR_INF("checkTransferredUploadData: DIA_E_DATASET_INVALID_DATA_RANGE (mReadPosition = %d)", mReadPosition);
      return DIA_E_DATASET_INVALID_DATA_RANGE;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Dataset::sync ( void )
{
   ScopeTrace oTrace("dia_Dataset::sync()");

   if ( mNumOfBytesReceived != mSizeInBytesRequested ) return DIA_E_DATASET_INVALID_LENGTH;

   return setData(mpByteBufferDownload,mNumOfBytesReceived,true);
}
