/*!
 * \file       dia_ConfigManager.h
 *
 * \brief      Manager class that is used to read/write properties from several resources
 *
 * \details    Manager class that is used to read/write properties from several resources
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreConfig
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#define __INCLUDED_DIA_CONFIG_MANAGER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

// local includes
#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_NOTIFIER__
#include <common/framework/config/dia_ConfigNotifier.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_CLIENT_INTERFACE__
#include "common/framework/config/dia_IEngineClient.h"
#endif

#define DIA_C_U16_MAX_PROPERTY_SIZE     512

#define DIA_M_PROP_STR_GET(id,data,len) (dia_ConfigManager::getProperty(id,data,len))
#define DIA_M_PROP_VAL_GET(id,data,len) (dia_ConfigManager::getProperty(id,data))
#define DIA_M_PROP_SIZE_GET(id)         (dia_ConfigManager::getSize(id))

class dia_PropertyBag;

class dia_ConfigManager
{
    friend class dia_PropertyBag;

    DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_ConfigManager);

public:
    //! retrieve an interface pointer via the facade
    template<class _T> static tDiaResult queryInterface ( _T** pInterface );

    //! retrieve the size of the given property
    virtual size_t getSize ( tU32 propID );
    //! retrieve the size of the given property
    virtual size_t getSize ( const std::string& propName );

    //! retrieve a 1 byte property value
    virtual tDiaResult getProperty ( tU32 propID, bool& propValue );
    //! retrieve a 1 byte property value
    virtual tDiaResult getProperty ( const std::string& propName, bool& propValue );
    //! retrieve a 1 byte property value
    virtual tDiaResult getProperty ( tU32 propID, tU8& propValue );
    //! retrieve a 1 byte property value
    virtual tDiaResult getProperty ( const std::string& propName, tU8& propValue );
    //! retrieve a 2 byte property value
    virtual tDiaResult getProperty ( tU32 propID, tU16& propValue );
    //! retrieve a 2 byte property value
    virtual tDiaResult getProperty ( const std::string& propName, tU16& propValue );
    //! retrieve a 4byte property value
    virtual tDiaResult getProperty ( tU32 propID, tU32& propValue );
    //! retrieve a 4byte property value
    virtual tDiaResult getProperty ( const std::string& propName, tU32& propValue );
    //! retrieve a byte stream property value
    virtual tDiaResult getProperty ( tU32 propID, tU8 propValue[], size_t propLength );
    //! retrieve a byte stream property value
    virtual tDiaResult getProperty ( const std::string& propName, tU8 propValue[], size_t propLength );
    //! retrieve a byte stream property value
    virtual tDiaResult getProperty ( tU32 propID, tU8 propValue[], size_t* maxLength );
    //! retrieve a byte stream property value
    virtual tDiaResult getProperty ( const std::string& propName, tU8 propValue[], size_t* maxLength );
    //! retrieve a byte stream property value
    virtual tDiaResult getProperty ( tU32 propID, std::vector<tU8>& propValue );
    //! retrieve a byte stream property value
    virtual tDiaResult getProperty ( const std::string& propName, std::vector<tU8>& propValue );
    //! retrieve a property asynchronously
    virtual tDiaResult getProperty ( tU32 propID, dia_IEngineClient* callback);

    //! retrieve a 1 byte property value
    virtual tDiaResult setProperty ( tU32 propID, tU8 propValue );
    //! retrieve a 1 byte property value
    virtual tDiaResult setProperty ( const std::string& propName, tU8 propValue );
    //! retrieve a 2 byte property value
    virtual tDiaResult setProperty ( tU32 propID, tU16 propValue );
    //! retrieve a 2 byte property value
    virtual tDiaResult setProperty ( const std::string& propName, tU16 propValue );
    //! retrieve a 4 byte property value
    virtual tDiaResult setProperty ( tU32 propID, tU32 propValue );
    //! retrieve a 4 byte property value
    virtual tDiaResult setProperty ( const std::string& propName, tU32 propValue );
    //!
    virtual tDiaResult setProperty ( tU32 propID, tU8 propValue[], size_t propLength );
    //!
    virtual tDiaResult setProperty ( const std::string& propName, tU8 propValue[], size_t propLength );
    //! write the property data given in the vector
    virtual tDiaResult setProperty ( tU32 propID, const std::vector<tU8>& propData );
    //! write the property data given in the vector
    virtual tDiaResult setProperty ( const std::string& propName, const std::vector<tU8>& propData );
	//! set property (assign 1 byte value) asynchronously
    virtual tDiaResult setPropertyU8 ( tU32 propID, tU8 propValue, dia_IEngineClient* callback);

    //! configure the config manager with a given property bag (dependency inversion)
    tDiaResult addPropertyBag ( dia_PropertyBag* pPropBag );
    //! couple the given interface pointer with the given IID
    template<class _T> static tDiaResult setInterface ( _T* pInterface );

    //! load a single property
    tDiaResult loadProperty ( const dia_PropertyInfo& propData );
    //! load a set of properties
    tDiaResult loadProperties ( const dia_PropertyInfo propData[], tU16 numOfProps );
    //! retrieve the property information for a given property ID
    dia_PropertyInfo* getPropertyInfo ( tU32 propID );

    //! add a notifier
    tDiaResult addConfigNotifier ( dia_ConfigNotifier* pNotifier );
    //! remove a notifier
    tDiaResult removeConfigNotifier ( dia_ConfigNotifier* pNotifier );
    //! remove a notifier
    tDiaResult clearConfigNotifiers ( void );
    //! notify all notifiers registered for the given property
    tDiaResult notifyPropertUpdate ( tU32 propID );

    //force flush for all property bags
    tDiaResult forceFlush ( void ) const;

    //! retrieve an interface pointer via the facade
    tDiaResult queryPropertyBag ( dia_UID, dia_PropertyBag** ppPropBag );


protected:
    //! protected class constructor
    dia_ConfigManager ( void );
    //! protected class destructor
    virtual ~dia_ConfigManager ( void );

    //! setup method for the application
    virtual tDiaResult setup ( void );
    //! shutdown method for the application
    virtual tDiaResult tearDown ( void );

protected:
    //! repository of property bags
    std::map<dia_UID,dia_PropertyBag*> mPropertyBagRep;
    //! repository of configuration interface (e.g. used for direct read/write access)
    std::map<dia::IID,void*> mInterfaceRep;

    //! repository of properties
    std::map<tU32,dia_PropertyInfo> mPropertyRepository;
    //! repository of property notifiers
    std::map<tU32,std::list<dia_ConfigNotifier*> > mNotifierRepository;
};

//! retrieve the size of the given property
size_t dia_getPropertySize ( tU32 propID );
//! retrieve a 1 byte property value
tDiaResult dia_getProperty ( tU32 propID, bool& propValue );
//! retrieve a 1 byte property value
tDiaResult dia_getProperty ( const std::string& propName, bool& propValue );
//! retrieve a 1 byte property value
tDiaResult dia_getProperty ( tU32 propID, tU8& propValue );
//! retrieve a 1 byte property value
tDiaResult dia_getProperty ( const std::string& propName, tU8& propValue );
//! retrieve a 2 byte property value
tDiaResult dia_getProperty ( tU32 propID, tU16& propValue );
//! retrieve a 2 byte property value
tDiaResult dia_getProperty ( const std::string& propName, tU16& propValue );
//! retrieve a 4 byte property value
tDiaResult dia_getProperty ( tU32 propID, tU32& propValue );
//! retrieve a 4 byte property value
tDiaResult dia_getProperty ( const std::string& propName, tU32& propValue );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( tU32 propID, tU8 propValue[], size_t propLength );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( const std::string& propName, tU8 propValue[], size_t propLength );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( tU32 propID, tU8 propValue[], size_t* propLength );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( const std::string& propName, tU8 propValue[], size_t* propLength );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( tU32 propID, std::vector<tU8>& propValue );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( const std::string& propName, std::vector<tU8>& propValue );
//! retrieve a byte stream property value
tDiaResult dia_getProperty ( tU32 propID, dia_IEngineClient* callback );

//! boolean check value of the given property (only allowed for integer value properties)
tBool dia_checkPropertyValue ( tU32 propID );

//! set a 1 byte property value
tDiaResult dia_setProperty ( tU32 propID, tU8 propValue );
//! set a 2 byte property value
tDiaResult dia_setProperty ( tU32 propID, tU16 propValue );
//! set a 4 byte property value
tDiaResult dia_setProperty ( tU32 propID, tU32 propValue );
//! write a byte stream to property value
tDiaResult dia_setProperty ( tU32 propID, tU8 propValue[], size_t propLength );
//! write a byte stream to property value
tDiaResult dia_setProperty ( tU32 propID, const std::vector<tU8>& propData );
//! write a byte stream to property value
tDiaResult dia_setProperty ( tU32 propID, tU8 propValue, dia_IEngineClient* callback);

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
queryConfMgrInterface ( _T** pInterface )
{
    return dia_ConfigManager::queryInterface(pInterface);
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_ConfigManager::queryInterface ( _T** pInterface )
{
   if ( !pInterface ) return DIA_E_INVALID_POINTER;

   std::map<dia::IID,void*>::iterator iter = getInstance()->mInterfaceRep.find(_T::getIID());
   if ( iter != getInstance()->mInterfaceRep.end() )
   {
      *pInterface = static_cast<_T*>(getInstance()->mInterfaceRep[_T::getIID()]);
      return DIA_SUCCESS;
   }

   *pInterface = 0;
   return DIA_E_NOT_FOUND;
}

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_ConfigManager::setInterface ( _T* pInterface )
{
   if ( !pInterface ) return DIA_E_INVALID_POINTER;
   if ( pInterface->getIID() != _T::getIID() ) return DIA_E_INTERFACE_ID_MISMATCH;

   getInstance()->mInterfaceRep[_T::getIID()] = (void*) pInterface;

   return DIA_SUCCESS;
}

#endif
