/*!
 * \file       dia_Factory.h
 *
 * \brief      Generic macro definitions
 *
 * \details    Generic macro definitions
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_MACROS__
#define __INCLUDED_DIA_MACROS__


#define DECL_DEPRECATED_DEFAULT_CONSTRUCTOR(class)                                                 \
protected:                                                                                         \
    class(void)

#define DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(class)                              \
protected:                                                                                         \
   /**
    * \brief deprecated copy constructor
    */                                                                                             \
    class ( const class& obj ) = delete;                                                           \
                                                                                                   \
    /**
     * \brief deprecated assignment operator
     */                                                                                            \
    class& operator = ( const class& obj ) = delete;


#define DECL_DEFAULT_CONSTRUCTOR_AND_DESTRUCTOR(class)                                             \
public:                                                                                            \
    class(void);                                                                                   \
    virtual ~class(void)

#define DECL_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(class)                                         \
protected:                                                                                         \
   /**
    * \brief deprecated copy constructor
    */                                                                                             \
    class ( const class& obj );                                                                    \
                                                                                                   \
    /**
     * \brief deprecated assignment operator
     */                                                                                            \
    class& operator = ( const class& obj )

#define DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(class)                                           \
protected:                                                                                         \
   static class* mpInstance;                                                                       \
                                                                                                   \
public:                                                                                            \
   static class* getInstance ( void );                                                             \
   static void deleteInstance ( void )

#define DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR_WITH_SETUP_AND_TEARDOWN(class)                   \
protected:                                                                                         \
   static class* mpInstance;                                                                       \
                                                                                                   \
public:                                                                                            \
   static class* getInstance ( void );                                                             \
   static void deleteInstance ( void );                                                            \
                                                                                                   \
protected:                                                                                         \
   virtual tDiaResult setup ( void );                                                              \
   virtual tDiaResult tearDown ( void )

#define DECL_FACTORY_OBJECT_CONSTRUCTOR_AND_DESTRUCTOR(class)                                      \
protected:                                                                                         \
   static class* mpInstance;                                                                       \
                                                                                                   \
public:                                                                                            \
   static class* createObject ( void );                                                            \
   static tVoid destroyObject ( void )

/* -- LINT: Use if paramter should be neglected without LINT objection */
#define DIA_M_PARAMETER_INTENTIONALLY_UNUSED(x) (void)(x);
#define DIA_M_RETURNCODE_INTENTIONALLY_IGNORED(x) (void)(x);
#define DIA_M_LAST_ASSIGNED_VALUE_INTENTIONALLY_NOT_USED(x) (void)(x);

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_SINGLETON(className)                                                              \
className*                                                                                         \
className::mpInstance = 0;                                                                         \
                                                                                                   \
className*                                                                                         \
className::getInstance ( void )                                                                    \
{                                                                                                  \
   if ( !mpInstance )                                                                              \
   {                                                                                               \
      mpInstance = new className;                                                                  \
   }                                                                                               \
                                                                                                   \
   return mpInstance;                                                                              \
}                                                                                                  \
                                                                                                   \
void                                                                                               \
className::deleteInstance ( void )                                                                 \
{                                                                                                  \
   if ( mpInstance )                                                                               \
   {                                                                                               \
      delete mpInstance;                                                                           \
      mpInstance = 0;                                                                              \
   }                                                                                               \
}

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(className)                                      \
className*                                                                                         \
className::mpInstance = 0;                                                                         \
                                                                                                   \
className*                                                                                         \
className::getInstance ( void )                                                                    \
{                                                                                                  \
   if ( !mpInstance )                                                                              \
   {                                                                                               \
      mpInstance = new className;                                                                  \
      if ( mpInstance )                                                                            \
      {                                                                                            \
         if ( mpInstance->setup() != DIA_SUCCESS )                                                 \
         {                                                                                         \
            delete mpInstance;                                                                     \
            mpInstance = 0;                                                                        \
         }                                                                                         \
      }                                                                                            \
   }                                                                                               \
                                                                                                   \
   return mpInstance;                                                                              \
}                                                                                                  \
                                                                                                   \
void                                                                                               \
className::deleteInstance ( void )                                                                 \
{                                                                                                  \
   if ( mpInstance )                                                                               \
   {                                                                                               \
      (void) mpInstance->tearDown();                                                               \
      delete mpInstance;                                                                           \
      mpInstance = 0;                                                                              \
   }                                                                                               \
}

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_FACTORY_OBJECT(className)                                                         \
className*                                                                                         \
className::mpInstance = 0;                                                                         \
                                                                                                   \
className*                                                                                         \
className::createObject ( void )                                                                   \
{                                                                                                  \
   if ( !mpInstance )                                                                              \
   {                                                                                               \
      mpInstance = new className;                                                                  \
      if ( mpInstance )                                                                            \
      {                                                                                            \
         (void) mpInstance->setup();                                                               \
      }                                                                                            \
   }                                                                                               \
                                                                                                   \
   return mpInstance;                                                                              \
}                                                                                                  \
                                                                                                   \
void                                                                                               \
className::destroyObject ( void )                                                                  \
{                                                                                                  \
   if ( mpInstance )                                                                               \
   {                                                                                               \
      (void) mpInstance->tearDown();                                                               \
      delete mpInstance;                                                                           \
      mpInstance = 0;                                                                              \
   }                                                                                               \
}

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_LIST_REPOSITORY_TEAR_DOWN(repType,repName)                                        \
   if ( repName.size() )                                                                           \
   {                                                                                               \
      std::list<repType*>::iterator iter = repName.begin();                                        \
      for ( ; iter != repName.end(); ++iter )                                                      \
      {                                                                                            \
         delete (*iter);                                                                           \
      }                                                                                            \
      repName.clear();                                                                             \
   }

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_SET_REPOSITORY_TEAR_DOWN(repType,repName)                                        \
   if ( repName.size() )                                                                           \
   {                                                                                               \
      std::set<repType*>::iterator iter = repName.begin();                                        \
      for ( ; iter != repName.end(); ++iter )                                                      \
      {                                                                                            \
         delete (*iter);                                                                           \
      }                                                                                            \
      repName.clear();                                                                             \
   }

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_VECTOR_REPOSITORY_TEAR_DOWN(repType,repName)                                      \
   if ( repName.size() )                                                                           \
   {                                                                                               \
      std::vector<repType*>::iterator iter = repName.begin();                                      \
      for ( ; iter != repName.end(); ++iter )                                                      \
      {                                                                                            \
         delete (*iter);                                                                           \
      }                                                                                            \
      repName.clear();                                                                             \
   }

//-------------------------------------------------------------------------------------------------

#define DIA_IMPL_MAP_REPOSITORY_TEAR_DOWN(repType1,repType2,repName)                               \
   if ( repName.size() )                                                                           \
   {                                                                                               \
      std::map<repType1,repType2*>::iterator iter = repName.begin();                               \
      for ( ; iter != repName.end(); ++iter )                                                      \
      {                                                                                            \
         delete iter->second;                                                                      \
      }                                                                                            \
      repName.clear();                                                                             \
   }

//-------------------------------------------------------------------------------------------------

#define DIA_RETURN_METHOD_NOT_IMPLEMENTED(traceMessage)                                            \
   DIA_TR_WRN("##### NOT_IMPLEMENTED: %s #####",traceMessage);                                     \
   return DIA_E_NOT_IMPLEMENTED

//-------------------------------------------------------------------------------------------------

#define DIA_RETURN_PROPERTY_BASED_METHOD_NOT_IMPLEMENTED(propID,traceMessage)                      \
   DIA_TR_WRN("##### NOT_IMPLEMENTED (PROPID=0x%08x): %s #####",propID,traceMessage);              \
   return DIA_E_NOT_IMPLEMENTED

#endif /* DIA_MACROS_H_ */

#if __cplusplus>=201402L
#define DEPRECATED_FUNCTION(msg,func) [[deprecated(msg)]]func
#else
#define DEPRECATED_FUNCTION(msg,func) func __attribute__ ((deprecated(msg)))
#endif