/*!
 * \file       dia_Timer.cpp
 *
 * \brief      Platform specific implementation of a timer resource
 *
 * \details    Platform specific implementation of a timer resource
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_TIMER__
#include <common/framework/application/dia_Timer.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_TIMER_LISTENER__
#include <common/interfaces/dia_ITimerListener.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include <common/framework/application/dia_ThreadMonitor.h>
#endif

using namespace dia;

tU32 dia_Timer::mSerial = 0;

//-----------------------------------------------------------------------------

dia_Timer::dia_Timer ( void )
   : mID(mSerial++),
     mSyncObj(dia_composeStlString("Timer_%d_LK",(int) mID)),
     mIsTimerThreadRegistered(false)
{
   mName = dia_composeStlString("Timer_%d",(int) mID);
}

//-----------------------------------------------------------------------------

dia_Timer::~dia_Timer ( void )
{}

//-----------------------------------------------------------------------------

void
dia_Timer::vOnAlarm ( void )
{
   ScopeTrace trc("dia_Timer::vOnAlarm()");

   if ( !mIsTimerThreadRegistered )
   {
      (void) dia::registerThread(mName);
      mIsTimerThreadRegistered = true;
   }

   LockScope oLock(mSyncObj);

   std::list<dia_ITimerListener*> notificationList = mListenerRep;

   std::list<dia_ITimerListener*>::iterator listIter = notificationList.begin();
   for ( ; listIter != notificationList.end(); ++listIter ) {
        if ( *listIter ) (*listIter)->vOnTimerElapsed((dia_TimerID) mID);
    }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Timer::addTimerListener ( dia_ITimerListener* pListener )
{
   ScopeTrace trc("dia_Timer::addTimerListener()");

   LockScope oLock(mSyncObj);

   tDiaResult retCode = DIA_FAILED;

   std::list<dia_ITimerListener*>::iterator listIter = mListenerRep.begin();
   for ( ; listIter != mListenerRep.end(); listIter++ ) {
      if ( (*listIter) == pListener ) break;
   }

   if ( listIter == mListenerRep.end() )
   {
      mListenerRep.push_back(pListener);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_Timer::removeTimerListener ( const dia_ITimerListener* pListener )
{
   ScopeTrace trc("dia_Timer::removeTimerListener()");

   LockScope oLock(mSyncObj);

   tDiaResult retCode = DIA_FAILED;

   std::list<dia_ITimerListener*>::iterator listIter = mListenerRep.begin();
   for ( ; listIter != mListenerRep.end(); listIter++ )
   {
      if ( (*listIter) == pListener )
      {
         mListenerRep.erase(listIter);
         retCode = DIA_SUCCESS;
         break;
      }
   }

   return retCode;
}

