/*!
 * \file       dia_Socket.cpp
 *
 * \brief      Socket class
 *
 * \details    Socket class used to create a socket for INC communication with SCC.
 *             When using USE_DGRAM_SERVICE based socket implementation (for Gen4), 
 *             dgram-service wrapper functions ensure the assembly of large data into
 *             one recv() call. Otherwise, the assembly of large data into once recv() 
 *             call is assumed to be handled by the Kernel.
 *             This is specific to socket handling for INC communication with SCC alone
 *             and should not be used as a generic abstraction for socket communication.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2015-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <errno.h>
#include <string.h>

#include <common/framework/application/dia_Socket.h>
#include <common/framework/application/dia_SocketConfiguration.h>

namespace dia
{

#ifndef __DIA_UNIT_TESTING__

INCSocket* createObjectINCSocket ( SocketConfiguration& config )
{
   return new INCSocket(config);
}

#endif

//-------------------------------------------------------------------------------------------------

INCSocket::INCSocket ( SocketConfiguration& config )
   : mSocketHandle(0),
     mpConfig(&config),
     //mIsCreated(false),
     mIsConnected(false),
     mIsBound(false)
{}

//-------------------------------------------------------------------------------------------------

INCSocket::~INCSocket ( void )
{
   mpConfig = 0;
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::create ( void )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::create");

   if ( isCreated() ) return DIA_SUCCESS;
   if ( !mpConfig )   return DIA_E_SOCKET_CONFIG_NOT_AVAILABLE;

   // create socket for communication
   mSocketHandle = socket(mpConfig->getFamily(),mpConfig->getType(),mpConfig->getProtocol());
   if ( mSocketHandle < 0)
   {
      DIA_TR_ERR("dia::INCSocket::create: ERROR: DIA_E_SOCKET_CREATION_FAILED");
      return DIA_E_SOCKET_CREATION_FAILED;
   }

   DIA_TR_INF("dia::INCSocket::create: Socket with handle %d created", mSocketHandle);
   //mIsCreated = true;

#ifdef USE_DGRAM_SERVICE
   mDgram = dgram_init(mSocketHandle, DGRAM_MAX, NULL);
   if ( !mDgram )
   {
      (void) close();
      return DIA_E_SOCKET_DGRAM_INIT_FAILED;
   }
#endif

   return DIA_SUCCESS;
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::close ( void )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::close");

   if ( mpConfig )
   {
#ifdef USE_DGRAM_SERVICE
      dgram_exit(mDgram);
#endif
      (void)::close(mSocketHandle);
      mSocketHandle = 0;
   }

   return DIA_SUCCESS;
}

//-------------------------------------------------------------------------------------------------

bool
INCSocket::isReady ( void ) const
{
   bool ready = true;
   if (!isCreated())
   {
      DIA_TR_ERR("ERROR: socket not ready. mSocketHandle=%d", mSocketHandle);
      ready = false;
   }

   return ready;
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::connect ( void )
{
   dia_tclFnctTrace oTrace(__PRETTY_FUNCTION__);

   if ( !isCreated() ) return DIA_E_SOCKET_CREATION_BYPASSED;
   if ( !mpConfig )    return DIA_E_SOCKET_CONFIG_NOT_AVAILABLE;

   tDiaResult retCode = DIA_SUCCESS;

   struct addrinfo    hints, *infoptr=NULL, *loopptr=NULL;

   memset(&hints, 0, sizeof(hints));
   hints.ai_family=mpConfig->getFamilyLocal();
   hints.ai_socktype=mpConfig->getType();

   char portstr[16];
   snprintf(portstr, sizeof(portstr), "%d", mpConfig->getPortLocal());

   // get the Local Host for v850 and Bind to that Host, which is dynamically created at runtime by kernel in the file /etc/hosts
   int result=getaddrinfo(mpConfig->getHostNameLocal().c_str(), portstr, &hints, &infoptr);
   if(result || !infoptr)
      retCode=DIA_E_SOCKET_LOCAL_HOST_NOT_AVAILABLE;
   else
   {
      // Bind to a local port to recieve response
      for(loopptr=infoptr; loopptr; loopptr=loopptr->ai_next)
         if(!bind(mSocketHandle, loopptr->ai_addr, loopptr->ai_addrlen))
            break;
      if(!loopptr)
         retCode=DIA_E_SOCKET_LOCAL_BIND_FAILED;
   }
   freeaddrinfo(infoptr);
   infoptr=NULL;

   if(retCode!=DIA_SUCCESS)
   {
      DIA_TR_INF("### dia::INCSocket::connect FAILED (ERR = 0x%08x) ###",retCode);
      (void)close();
      return retCode;
   }



   memset(&hints, 0, sizeof(hints));
   hints.ai_family=mpConfig->getFamilyServer();
   hints.ai_socktype=mpConfig->getType();

   snprintf(portstr, sizeof(portstr), "%d", mpConfig->getPortServer());

   // get the remote hosts IP address and try to connect to its socket
   result=getaddrinfo(mpConfig->getHostNameServer().c_str(), portstr, &hints, &infoptr);
   if(result || !infoptr)
      retCode=DIA_E_SOCKET_SERVER_HOST_NOT_AVAILABLE;
   else
   {
      // Connect to port of the v850
      for(loopptr=infoptr; loopptr; loopptr=loopptr->ai_next)
         if(!::connect(mSocketHandle, loopptr->ai_addr, loopptr->ai_addrlen))
            break;
      if(!loopptr)
         retCode=DIA_E_SOCKET_CONNECT_FAILED;
   }
   freeaddrinfo(infoptr);
   infoptr=NULL;

   if(retCode!=DIA_SUCCESS)
   {
      DIA_TR_INF("### dia::INCSocket::connect FAILED (ERR = 0x%08x) ###",retCode);
      (void)close();
   }
   return retCode;
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::disconnect ( void )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::disconnect");
   return DIA_SUCCESS;
}


//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::writeData ( const tU8 data[], tU16 totalLength )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::writeData");

   if ( totalLength == 0 )
   {
      DIA_TR_INF("### dia::INCSocket::writeData FAILED (NO DATA. PROVIDED LENGTH IS ZERO)");
      return DIA_E_SOCKET_WRITE_DATA_FAILED_NO_DATA;
   }

   tU16 remainingLength = totalLength;
   tDiaResult retCode = DIA_SUCCESS;

   do {
#ifdef USE_DGRAM_SERVICE
      ssize_t writeReturn = dgram_send(mDgram, const_cast<tU8*>(data + (totalLength - remainingLength)), remainingLength);
#else
      ssize_t writeReturn = write(mSocketHandle, data + (totalLength - remainingLength), remainingLength);
#endif
      if (writeReturn == -1 && errno == EINTR) {
         continue;
      }
      else if (writeReturn == -1) {
         char * errorstr = strerror(errno);
         DIA_TR_INF("### dia::INCSocket::writeData FAILED (error=%s)", errorstr);
         retCode = DIA_E_SOCKET_WRITE_DATA_FAILED;
         break;
      }
      remainingLength = static_cast<tU16>(remainingLength - writeReturn);
      DIA_TR_INF("### dia::INCSocket::writeData %d bytes of %d total sent", totalLength - remainingLength, totalLength);
   } while (remainingLength > 0);

   return retCode;
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::writeData ( const std::vector<tU8>& data )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::writeData(const std::vector<tU8>&)");
   return writeData(&data[0],(tU16) data.size());
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::readData ( tU8 data[], tU16& length )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::readData");

   tDiaResult retCode = DIA_SUCCESS;

   ssize_t readReturn = 0;

   do {
#ifdef USE_DGRAM_SERVICE
      readReturn = dgram_recv(mDgram, data, length);
#else
      readReturn = recv(mSocketHandle, data, length, 0);
#endif
   } while (readReturn == -1 && errno == EINTR);

   if( readReturn == -1 ) {
      char * errorstr = strerror(errno);
      DIA_TR_ERR("### dia::INCSocket::readData FAILED (error=%s)", errorstr);
      retCode = DIA_E_SOCKET_READ_DATA_FAILED;
   }
   else
   {
      DIA_TR_INF("--- INCSocket::readData => Read %zd bytes", readReturn);
      length = (tU16) readReturn;
   }

   return retCode;
}

//-------------------------------------------------------------------------------------------------

tDiaResult
INCSocket::readData ( std::vector<tU8>& data )
{
   dia_tclFnctTrace oTrace("dia::INCSocket::readData(std::vector<tU8>&)");

   if ( !mpConfig ) return DIA_E_SOCKET_CONFIG_NOT_AVAILABLE;

   // retrieve the maximum size of a message we can receive on this socket
   tU16 numOfBytesRead = mpConfig->getMaxMsgSize();

   DIA_TR_INF("--- INCSocket: maxMsgSize is %d bytes", numOfBytesRead);

   tU8 recvBuffer[numOfBytesRead];
   // we pass in the maximum number of bytes that can be read. After the call numOfBytesRead will reflect the actual number of bytes read
   tDiaResult retCode = readData(&recvBuffer[0],numOfBytesRead);

   data.clear();
   if ( numOfBytesRead > 0 )
   {
      data.reserve(numOfBytesRead);
      data.insert(data.end(), &recvBuffer[0], &recvBuffer[numOfBytesRead]);
   }

   return retCode;
}

}

