/*!
 * \file       dia_SelfMsg.h
 *
 * \brief      This file contains generic declaration and definition to async send messages to self
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2019 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SELF_MSG__
#define __INCLUDED_DIA_SELF_MSG__

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACES__
#include "common/interfaces/dia_interfaces.h"
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include "common/framework/application/dia_Lock.h"
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include "common/framework/application/dia_LockScope.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#include <sstream>


namespace dia {

class RefCounter {
public:
   class RefCounterVal {
   private:
      RefCounterVal():
         mCnt(1),
         mLock(getLockName(&mCnt))
      {
         ScopeTrace oTrace("RefCounterVal::RefCounterVal()");
         DIA_TR_INF("RefCounterVal::RefCounterVal() lock=%p mCnt=%u", &mLock, mCnt);
      }
   public:
      virtual ~RefCounterVal() {
         DIA_TR_INF("RefCounterVal::~RefCounterVal() lock=%p mCnt=%u", &mLock, mCnt);
      }
      static RefCounterVal *create() {
         RefCounterVal *res=new RefCounterVal();
         DIA_TR_INF("RefCounterVal::create() lock=%p", &(res->mLock));

         return res;
      }
      void increment() {
         DIA_TR_INF("RefCounterVal::increment() lock=%p, old mCnt=%u", &mLock, mCnt);
         LockScope lockScope(mLock);
         ++mCnt;
      }

      void decrement() {
         DIA_TR_INF("RefCounterVal::decrement() lock=%p old mCnt=%u", &mLock, mCnt);
         LockScope lockScope(mLock);
         if (mCnt) {
            --mCnt;
         }
      }

      bool isOrphan() {
         DIA_TR_INF("RefCounterVal::isOrphan() lock=%p: %u", &mLock, mCnt == 0);

         return mCnt == 0;
      }

      uint32_t getVal() {
         DIA_TR_INF("RefCounterVal::getVal() lock=%p: mCnt=%u", &mLock, mCnt);
         return mCnt;
      }

      static std::string getLockName(uint32_t *pCnt) {
         std::ostringstream oss;
         std::string lockName;
         oss << "REFCOUNTER" << (uint64_t)pCnt;
         lockName=oss.str();
         return lockName;
      }
      
      uint32_t mCnt;
      Lock mLock;

   };


   RefCounter()
   {
      val=RefCounterVal::create();
   }

   ~RefCounter() {
      val->decrement();
      if (val->isOrphan()) {
         delete val;
      }
   }

   void increment() {val->increment();}
   void decrement() {val->decrement();}
   bool isOrphan() {
      return val->isOrphan();
   }
   RefCounterVal *getVal() {
      return val;
   }




private:
   RefCounterVal *val;
};


template<class CLSTARGET, class ARG>
class SelfMsg: public dia_tclDiagSession::tclEventIntMsgRxGeneric {
   

private:
   SelfMsg(RefCounter::RefCounterVal *refCounterVal, dia_FunctorImpl<CLSTARGET, ARG> *functor):
      tclEventIntMsgRxGeneric((dia_Functor *)functor),
      mRefCounterVal(refCounterVal)
   {
      ScopeTrace oTrace("SelfMsg::SelfMsg()");

   }

public:
   static tDiaResult createAndPost(CLSTARGET *target, RefCounter &refCounter, typename dia_FunctorImpl<CLSTARGET, ARG>::MemFuncPtr fn) {
      return createAndPost(target, refCounter.getVal(), fn);
   }   
   static tDiaResult createAndPost(CLSTARGET *target, RefCounter::RefCounterVal *refCounterVal, typename dia_FunctorImpl<CLSTARGET, ARG>::MemFuncPtr fn, ARG arg) {
      ScopeTrace oTrace("SelfMsg::createAndPost()");
      dia_FunctorImpl<CLSTARGET, ARG> *functor=new dia_FunctorImpl<CLSTARGET, ARG>(target, fn, arg);
      if (!functor) {
         return DIA_E_INVALID_POINTER;
      }
      SelfMsg *poThis =new SelfMsg(refCounterVal, functor);
      if (poThis != nullptr)
      {
         if (!poThis->mRefCounterVal->isOrphan())
         {
            poThis->mRefCounterVal->increment();
            return poThis->post();
         }
         DIA_TR_ERR("SelfMsg:createAndPost:  allocation failed Orphan!");
      }
      delete functor;
      delete poThis;
      return DIA_E_INVALID_POINTER;
   }

   
protected:
   virtual tDiaResult post() {
      ScopeTrace oTrace("SelfMsg::post()");
      getInstanceOfApplication()->postMessage(this);
      return DIA_SUCCESS;
   }


public:
   virtual tDiaResult execute ( void )
   {
      dia_tclFnctTrace trc("SelfMsg::execute()");
      
      if ( !mpArg ) return DIA_E_INVALID_POINTER;
      if (mRefCounterVal) {
          mRefCounterVal->decrement();
          if (mRefCounterVal->isOrphan()) {
             delete mRefCounterVal;
             return DIA_E_INVALID_POINTER;
          }
      }
      dia_Functor* pFunctor = static_cast<dia_Functor*>(mpArg);
      return (*pFunctor)();
   }


   ~SelfMsg() {
      ScopeTrace oTrace("SelfMsg::~SelfMsg()");

#if 0
      if (!mRefCounterVal) {
         return;
      }
      mRefCounterVal->decrement();
      if (mRefCounterVal->isOrphan()) {
         delete mRefCounterVal;
      }
#endif
   }
   
private:
   RefCounter::RefCounterVal *mRefCounterVal;
};

}
#endif
