/*!
 * \file       dia_SCCManager.h
 *
 * \brief      Manager class that monitors and changes the status of the SCC
 *
 * \details    Manager class that monitors and changes the status of the SCC
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SCC_MANAGER__
#define __INCLUDED_DIA_SCC_MANAGER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SCC_LISTENER__
#include <common/interfaces/dia_ISCCListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_TIMER_LISTENER__
#include "common/interfaces/dia_ITimerListener.h"
#endif

#ifndef __INCLUDED_DIA_SCCMANAGERFSM__
#include <common/framework/fsm/generated/dia_SCCManagerFSM.h>
#endif

namespace dia
{

const tU16 DIA_C_U16_MAX_RECONNECT_RETRIES_SCC = 0xFFFF;

class SCCManager
    : public ISCCListener,
      public dia_ITimerListener,
      public dia_SCCManagerFSM::FsmBehaviour
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(SCCManager);
   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(SCCManager);

public:
   //! received notification that connection to SCC was established
   virtual void onSccConnected ( void );
   //! received notification that connection to SCC was terminated
   virtual void onSccDisconnected ( void );
   //! received notification about the current component status
   virtual void onSccStatusUpdate  ( enSccComponentStatus status );

   //! received request to send the given status to SCC
   virtual void onSccStatusRequest ( enSccComponentStatus status );
   //! received request to reestablish connection, e.g. after a reset of the SCC
   virtual void onSccReconnect ( tU16 timeout, tU16 cycleTime, tU16 numOfRetries = DIA_C_U16_MAX_RECONNECT_RETRIES_SCC );

   virtual tDiaResult connectProxy ( ISCC* pScc );

   //! overloaded method from class dia_ITimerListener
   virtual void vOnTimerElapsed ( dia_TimerID id );

   virtual enSccComponentStatus getComponentStatus ( void ) const { return mComponentStatus; }

protected:
    //! protected because we are using the singleton pattern
    SCCManager ( void );
    //! protected because we are using the singleton pattern
    virtual ~SCCManager ( void );

    //! setup (create objects,...)
    tDiaResult setup ( void );
    //! tear down (destroy objects,...)
    //lint -sem(dia::SCCManager::tearDown,cleanup)
    tDiaResult tearDown ( void );

    tDiaResult acceptEvent ( dia_SCCManagerFSM::FsmEvent event, void* pArg );

    // FSM actions
    virtual void vFsmFinalize ( void* pArg );
    virtual void vFsmOnActive ( void* pArg );
    virtual void vFsmOnInactive ( void* pArg );
    virtual void vFsmOnInitializing ( void* pArg );
    virtual void vFsmOnReconnecting ( void* pArg );
    virtual void vFsmSendStatusActive ( void* pArg );
    virtual void vFsmSendStatusInactive ( void* pArg );
    virtual void vFsmStartReconnection ( void* pArg );
    virtual void vFsmRetryReconnect ( void* pArg );
    virtual void vFsmStartTimer ( void* pArg );
    virtual void vFsmStopTimer ( void* pArg );
    virtual void vFsmTriggerSystemReset ( void* pArg );

    // FSM guards
    virtual bool isActive ( void* pArg );
    virtual bool isActiveRequested ( void* pArg );
    virtual bool isInactive ( void* pArg );
    virtual bool isMaxRetriesReached ( void* pArg );

    //! received request to send the given status to SCC
    virtual void sendSccStatusRequest ( enSccComponentStatus status );

protected:
    //! pointer to the state machine
    dia_SCCManagerFSM::Fsm* mpFSM; //lint -sem(dia::dia_SCCManager::tearDown, cleanup)
    //! link to the SCC proxy
    ISCC* mpSccProxy; //lint -sem(dia::dia_SCCManager::tearDown, cleanup)
    //! current component status of SCC
    enSccComponentStatus mComponentStatus;
    //! requested component status from APP to SCC
    enSccComponentStatus mRequestedComponentStatus;
    //! internal error code
    tDiaResult mErrorCode;
    //! max number of retries
    tU16 mMaxReconnectRetries;
    //! retry counter
    tU16 mRetryCounter;
    //! timer used for reconnecting to SCC
    dia_Timer mReconnectTimer;
    //! 1st timeout value
    tU16 mTimeout;
    //! cycle time
    tU16 mCycleTime;
    //! flag to indicate if SCCManager is reconnecting to the INC channel
    bool mIsReconnecting;
};

}

dia::SCCManager* getInstanceOfSCCManager ( void );
void releaseInstanceOfSCCManager ( void );

#endif /* __INCLUDED_DIA_SCC_MANAGER__ */
