/*!
 * \file       dia_SCCManager.cpp
 *
 * \brief      Manager class that monitors and changes the status of the SCC
 *
 * \details    Manager class that monitors and changes the status of the SCC
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SCC_MANAGER__
#include <common/framework/application/dia_SCCManager.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SPM__
#include "common/interfaces/dia_ISpm.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia::SCCManager)

#ifndef __DIA_UNIT_TESTING__

dia::SCCManager*
getInstanceOfSCCManager ( void )
{
   return dia::SCCManager::getInstance();
}

void
releaseInstanceOfSCCManager ( void )
{
   dia::SCCManager::deleteInstance();
}

#endif

namespace dia
{

static const tU16 DIA_C_U16_SCCMANAGER_TIMEOUT_INFINITE = 0xFFFF;

//---------------------------------------------------------------------------------------------------------------------

SCCManager::SCCManager ( void )
   : mpFSM(0),
     mpSccProxy(0),
     mComponentStatus(DIA_EN_SCC_COMP_STATUS_UNKNOWN),
     mRequestedComponentStatus(DIA_EN_SCC_COMP_STATUS_UNKNOWN),
     mErrorCode(DIA_E_NO_ERROR),
     mMaxReconnectRetries(DIA_C_U16_MAX_RECONNECT_RETRIES_SCC),
     mRetryCounter(0),
     mTimeout(DIA_C_U16_SCCMANAGER_TIMEOUT_INFINITE),
     mCycleTime(0),
     mIsReconnecting(false)
{
   mReconnectTimer.s32Create();
}

//---------------------------------------------------------------------------------------------------------------------

SCCManager::~SCCManager ( void )
{
   _BP_TRY_BEGIN
   {
      (void) SCCManager::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: SCCManager::~SCCManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
SCCManager::setup ( void )
{
   dia_tclFnctTrace trc("SCCManager::setup");

   if ( !mpFSM )
   {
      // create the state machine object
      if ( !(dia_SCCManagerFSM::Fsm::createFSM(&mpFSM,this)) )
      {
         DIA_TR_INF( "### STATE MACHINE INITIALIZATION FAILED ###");
         return DIA_FAILED;
      }
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
SCCManager::tearDown ( void )
{
   if ( mpFSM )
   {
      OSAL_DELETE mpFSM;
      mpFSM = 0;
   }

   mpSccProxy = 0;

   mReconnectTimer.s32Delete();
   mReconnectTimer.removeTimerListener(this);

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
SCCManager::connectProxy ( ISCC* pProxy )
{
   if ( pProxy )
   {
      this->mpSccProxy = pProxy;
      return DIA_SUCCESS;
   }

   return DIA_FAILED;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::onSccReconnect ( tU16 timeout, tU16 cycleTime, tU16 numOfRetries  )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::onSccReconnect(tU16,tU16,tU16)");

   if ( mIsReconnecting )
   {
      DIA_TR_INF("##### SCCManager is already reconnecting ... #####");
      return;
   }

   mTimeout   = timeout;
   mCycleTime = cycleTime;
   mMaxReconnectRetries = numOfRetries;
   mRetryCounter = 0;
   mRequestedComponentStatus = DIA_EN_SCC_COMP_STATUS_UNKNOWN;

   (void) acceptEvent( dia_SCCManagerFSM::evReconnect,0);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::onSccConnected ( void )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::onSccConnected()");
   (void) acceptEvent( dia_SCCManagerFSM::evRequestStatusActive,0);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::onSccDisconnected ( void )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::onSccDisconnected()");
   (void) acceptEvent( dia_SCCManagerFSM::evRequestStatusInactive,0);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::onSccStatusRequest ( enSccComponentStatus status )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::onSccStatusRequest(enSccComponentStatus)");

   if ( status == DIA_EN_SCC_COMP_STATUS_ACTIVE )
   {
      (void) acceptEvent( dia_SCCManagerFSM::evRequestStatusActive,0);
   }
   else if ( status == DIA_EN_SCC_COMP_STATUS_INACTIVE )
   {
      (void) acceptEvent( dia_SCCManagerFSM::evRequestStatusInactive,0);
   }
   else
   {
      // nothing to to
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::sendSccStatusRequest ( enSccComponentStatus status )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::sendSccStatusRequest(status)");

   if ( !mpSccProxy )
   {
      DIA_TR_ERR("dia::SCCManager::sendSccStatusRequest => UNABLE TO SEND STATUS %d. SCC PROXY NOT AVAILABLE!", (int)status);
      return;
   }

   if ( mpSccProxy->sendComponentStatusToSCC(status) == DIA_SUCCESS )
   {
      DIA_TR_INF("dia::SCCManager::sendSccStatusRequest => sent status %d", (int)status);
      mRequestedComponentStatus = status;
   }
   else
   {
      DIA_TR_INF("dia::SCCManager::sendSccStatusRequest => FAILED TO SEND STATUS %d", (int)status);
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::onSccStatusUpdate ( enSccComponentStatus status )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::onSccStatusUpdate(enSccComponentStatus)");
   DIA_TR_INF("dia::SCCManager::onSccStatusUpdate => received SCC status %d", (int)status);
   mComponentStatus = status;
   (void) acceptEvent( dia_SCCManagerFSM::evStatusUpdate,0);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vOnTimerElapsed ( dia_TimerID /*id*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vOnTimerElapsed(dia_TimerID)");
   (void) acceptEvent( dia_SCCManagerFSM::evTimeout,0);
}

//-----------------------------------------------------------------------------

tDiaResult
SCCManager::acceptEvent ( dia_SCCManagerFSM::FsmEvent event, void* pArg )
{
   dia_tclFnctTrace oTrace("SCCManager::acceptEvent(dia_SCCManagerFSM::FsmEvent,void*)");

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;

   mErrorCode = DIA_E_NO_ERROR;

   DIA_TR_INF("SCCManager::acceptEvent - State (before): %s (Event = %s)", mpFSM->getStateName(), dia_SCCManagerFSM::getEventName(event));
   mpFSM->acceptEvent(event,pArg);
   DIA_TR_INF("SCCManager::acceptEvent - State (after) : %s", mpFSM->getStateName());

   return mErrorCode;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmFinalize ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmFinalize(void*)");
   mComponentStatus = DIA_EN_SCC_COMP_STATUS_UNKNOWN;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmOnActive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmOnActive(void*)");
   mComponentStatus = DIA_EN_SCC_COMP_STATUS_ACTIVE;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmOnInactive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmOnInactive(void*)");
   mComponentStatus = DIA_EN_SCC_COMP_STATUS_INACTIVE;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmOnInitializing ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmOnInitializing(void*)");
   mComponentStatus = DIA_EN_SCC_COMP_STATUS_UNKNOWN;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmOnReconnecting ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmOnReconnecting(void*)");
   mComponentStatus = DIA_EN_SCC_COMP_STATUS_UNKNOWN;
   mIsReconnecting = true;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmSendStatusActive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmSendStatusActive(void*)");
   //onSccStatusRequest(DIA_EN_SCC_COMP_STATUS_ACTIVE,true);
   sendSccStatusRequest(DIA_EN_SCC_COMP_STATUS_ACTIVE);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmSendStatusInactive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmSendStatusInactive(void*)");
   //onSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_INACTIVE,true);
   sendSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_INACTIVE);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmStartReconnection ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmStartReconnection(void*)");
   mRetryCounter = 0;
   mReconnectTimer.s32SetTime(0,0);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmRetryReconnect ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmRetryReconnect(void*)");
   mRetryCounter++;
   //onSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_ACTIVE,true);
   sendSccStatusRequest(dia::DIA_EN_SCC_COMP_STATUS_ACTIVE);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmStartTimer ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmStartTimer(void*)");
   mReconnectTimer.addTimerListener(this);
   mReconnectTimer.s32SetTime(0,0);
   mReconnectTimer.s32SetTime(1000, 1000);
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmStopTimer ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmStopTimer(void*)");
   mReconnectTimer.s32SetTime(0,0);
   mReconnectTimer.removeTimerListener(this);

   mIsReconnecting = false;
}

//---------------------------------------------------------------------------------------------------------------------

void
SCCManager::vFsmTriggerSystemReset ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::vFsmTriggerSystemReset(void*)");

   dia_ISpm* pInterface = 0;
   if ( (querySysAdapterInterface<dia_ISpm>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      if (pInterface->signalSystemReset(DIA_EN_SPM_CORE_RESTART_MODE_ECU_FAST) != DIA_SUCCESS)
      {
         DIA_TR_ERR("SCCManager::vFsmTriggerSystemReset --- signalSystemRestart SEND TO SPM SERVER FAILED");
      }
      else
      {
         DIA_TR_ERR("SCCManager::vFsmTriggerSystemReset --- signalSystemRestart SEND TO SPM SERVER SUCCESSFULLY");
      }
   }
   else
   {
      DIA_TR_ERR("SCCManager::vFsmTriggerSystemReset --- Unable to get interface to SPM system adapter.");
   }
}

//---------------------------------------------------------------------------------------------------------------------

bool
SCCManager::isActive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::isActive(void*)");
   return ( mComponentStatus == DIA_EN_SCC_COMP_STATUS_ACTIVE ) ? true : false;
}

//---------------------------------------------------------------------------------------------------------------------

bool
SCCManager::isActiveRequested ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::isActiveRequested(void*)");
   return ( mRequestedComponentStatus == DIA_EN_SCC_COMP_STATUS_ACTIVE ) ? true : false;
}

//---------------------------------------------------------------------------------------------------------------------

bool
SCCManager::isInactive ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::isInactive(void*)");
   return ( mComponentStatus == DIA_EN_SCC_COMP_STATUS_INACTIVE ) ? true : false;
}

//---------------------------------------------------------------------------------------------------------------------

bool
SCCManager::isMaxRetriesReached ( void* /*pArg*/ )
{
   dia_tclFnctTrace oTrace("dia::SCCManager::isMaxRetriesReached(void*)");
   return ( mRetryCounter == mMaxReconnectRetries ) ? true : false;
}

}
