/*!
 * \file       dia_Functor.h
 *
 * \brief      templates for functors
 *
 * \details    -
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2018 Robert Bosch Car Multimedia
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_FUNCTOR__
#define __INCLUDED_DIA_FUNCTOR__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Base class for all functor classes
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////




template<class RET>
class dia_FunctorRet
{
public:
   //! class constructor
   dia_FunctorRet ( void ) {}
   //! overloaded () operator. Via this operator an object can be used like a function
   virtual RET operator() ( void ) = 0;

   virtual RET execute() {
      return this->operator()();
   }

   //! class destructor
   virtual ~dia_FunctorRet ( void ) {}
};


using dia_Functor = dia_FunctorRet<tDiaResult>;
using dia_FunctorVoid = dia_FunctorRet<void>;


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions without arguments
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T>
class dia_FunctorNoArgs
   : public dia_Functor
{
   typedef tDiaResult (_T::*MemFuncPtr) ( /*dia_Functor**/ );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorNoArgs ( _T* pObj, MemFuncPtr pFuncPtr ) : mpObj(pObj), mpMemberFunc(pFuncPtr) {}
   //! class destructor
   virtual ~dia_FunctorNoArgs ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { return (*mpObj.*mpMemberFunc)(); }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorNoArgs ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions without arguments
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T>
class dia_FunctorNoArgsNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( /*dia_Functor**/ );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorNoArgsNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr ) : mpObj(pObj), mpMemberFunc(pFuncPtr) {}
   //! class destructor
   virtual ~dia_FunctorNoArgsNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorNoArgsNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with one argument
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// derived template class
template <class _T, typename _P1>
class dia_FunctorOneArg
   : public dia_Functor
{
   typedef tDiaResult (_T::*MemFuncPtr) ( _P1 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorOneArg ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1) {}
   //! class destructor
   virtual ~dia_FunctorOneArg ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { return (*mpObj.*mpMemberFunc)(mArg1); }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorOneArg ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with one argument
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// derived template class
template <class _T, typename _P1>
class dia_FunctorOneArgNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( _P1 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorOneArgNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1) {}
   //! class destructor
   virtual ~dia_FunctorOneArgNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorOneArgNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
};

// derived template class
template <class _T, typename _P1>
class dia_FunctorOneRefArgNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( const _P1& );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorOneRefArgNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, _P1& arg1 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1) {}
   //! class destructor
   virtual ~dia_FunctorOneRefArgNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorOneRefArgNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with one argument of template type
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// derived template class
template <class _T, class _T2, template <typename, typename> class _T3 >
class dia_FunctorOneTemplateArgNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( const _T3<_T2, std::allocator<_T2> >& );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorOneTemplateArgNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, const _T3<_T2,std::allocator<_T2> >& arg1 )
      : mpObj(pObj),
        mpMemberFunc(pFuncPtr),
        mArg1(arg1)
   {}

   //! class destructor
   virtual ~dia_FunctorOneTemplateArgNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorOneTemplateArgNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _T3<_T2, std::allocator<_T2> > mArg1;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with two arguments
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T, typename _P1, typename _P2>
class dia_FunctorTwoArgs
   : public dia_Functor
{
   typedef tDiaResult (_T::*MemFuncPtr) ( _P1, _P2 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorTwoArgs ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2) {}
   //! class destructor
   virtual ~dia_FunctorTwoArgs ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { return (*mpObj.*mpMemberFunc)(mArg1,mArg2); }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorTwoArgs ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with one argument
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// derived template class
template <class _T, typename _P1, typename _P2>
class dia_FunctorTwoArgsNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( _P1, _P2 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorTwoArgsNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2) {}
   //! class destructor
   virtual ~dia_FunctorTwoArgsNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1,mArg2); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorTwoArgsNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with three arguments
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T, typename _P1, typename _P2, typename _P3>
class dia_FunctorThreeArgs
   : public dia_Functor
{
   typedef tDiaResult (_T::*MemFuncPtr) ( _P1, _P2, _P3 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorThreeArgs ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2, _P3 arg3 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2), mArg3(arg3) {}
   //! class destructor
   virtual ~dia_FunctorThreeArgs ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { return (*mpObj.*mpMemberFunc)(mArg1,mArg2,mArg3); }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorThreeArgs ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
    //!
    _P3 mArg3;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with three arguments no return value
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// derived template class
template <class _T, typename _P1, typename _P2, typename _P3>
class dia_FunctorThreeArgsNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( _P1, _P2, _P3 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorThreeArgsNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2, _P3 arg3 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2), mArg3(arg3) {}
   //! class destructor
   virtual ~dia_FunctorThreeArgsNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1,mArg2,mArg3); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorThreeArgsNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
    //!
    _P3 mArg3;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with four arguments
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T, typename _P1, typename _P2, typename _P3, typename _P4>
class dia_FunctorFourArgs
   : public dia_Functor
{
   typedef tDiaResult (_T::*MemFuncPtr) ( _P1, _P2, _P3, _P4 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorFourArgs ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2, _P3 arg3, _P4 arg4 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2), mArg3(arg3), mArg4(arg4) {}
   //! class destructor
   virtual ~dia_FunctorFourArgs ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { return (*mpObj.*mpMemberFunc)(mArg1,mArg2,mArg3,mArg4); }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorFourArgs ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
    //!
    _P3 mArg3;
    //!
    _P4 mArg4;
};
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with four arguments no return value
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T, typename _P1, typename _P2, typename _P3, typename _P4>
class dia_FunctorFourArgsNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( _P1, _P2, _P3, _P4 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorFourArgsNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2, _P3 arg3, _P4 arg4 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2), mArg3(arg3), mArg4(arg4) {}
   //! class destructor
   virtual ~dia_FunctorFourArgsNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1,mArg2,mArg3,mArg4); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorFourArgsNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
    //!
    _P3 mArg3;
    //!
    _P4 mArg4;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with five arguments
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

template <class _T, typename _P1, typename _P2, typename _P3, typename _P4, typename _P5>
class dia_FunctorFiveArgs
   : public dia_Functor
{
   typedef tDiaResult (_T::*MemFuncPtr) ( _P1, _P2, _P3, _P4, _P5 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorFiveArgs ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2, _P3 arg3, _P4 arg4, _P5 arg5 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2), mArg3(arg3), mArg4(arg4), mArg5(arg5) {}
   //! class destructor
   virtual ~dia_FunctorFiveArgs ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { return (*mpObj.*mpMemberFunc)(mArg1,mArg2,mArg3,mArg4,mArg5); }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorFiveArgs ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
    //!
    _P3 mArg3;
    //!
    _P4 mArg4;
    //!
    _P5 mArg5;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class for member functions with Five arguments no return value
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//derived class
template <class _T, typename _P1, typename _P2, typename _P3, typename _P4, typename _P5>
class dia_FunctorFiveArgsNoReturnValue
   : public dia_Functor
{
   typedef void (_T::*MemFuncPtr) ( _P1, _P2, _P3, _P4, _P5 );

public:
   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorFiveArgsNoReturnValue ( _T* pObj, MemFuncPtr pFuncPtr, _P1 arg1, _P2 arg2, _P3 arg3, _P4 arg4, _P5 arg5 ) : mpObj(pObj), mpMemberFunc(pFuncPtr), mArg1(arg1), mArg2(arg2), mArg3(arg3), mArg4(arg4), mArg5(arg5) {}
   //! class destructor
   virtual ~dia_FunctorFiveArgsNoReturnValue ( void ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual tDiaResult operator() ( void ) { (*mpObj.*mpMemberFunc)(mArg1,mArg2,mArg3,mArg4,mArg5); return DIA_SUCCESS; }

protected:
    //! deprecated default constructor without implementation
   dia_FunctorFiveArgsNoReturnValue ( void );

private:
    //! pointer to object
    _T* mpObj;
    //! function pointer
    MemFuncPtr mpMemberFunc;
    //!
    _P1 mArg1;
    //!
    _P2 mArg2;
    //!
    _P3 mArg3;
    //!
    _P4 mArg4;
    //!
    _P5 mArg5;
};


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Functor class ...
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// helpers to put arg-list to touple and back.
namespace dia_FunctorHelpers {

template <typename... T>
using tuple_with_removed_refs = std::tuple<typename std::remove_reference<T>::type...>;

template<size_t ...> struct seq {};

template<size_t N, size_t ...S> struct gens : gens<N-1, N-1, S...> {};

template<size_t ...S> struct gens<0, S...>{ typedef seq<S...> type; };

}

/*
  functor that stores that arguments given by ARGS into a touple.
  When executing, elements of stored touple will be used a arg-list for called member-method.
 */
template <class RET, class CLS, typename... ARGS>
class dia_FunctorRetImpl
   : public dia_FunctorRet<RET>
{

public:
   typedef RET (CLS::*MemFuncPtr) ( ARGS... );

   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorRetImpl ( CLS* obj, MemFuncPtr method, ARGS... args ) : mObj(obj), mMethod(method), mTuple(args...) {}
   //! class destructor
   virtual ~dia_FunctorRetImpl ( ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual RET operator() ( ) {  
      return execute();
   }

   virtual RET execute() {
      return static_cast<RET>(callFunc(typename dia_FunctorHelpers::gens<sizeof...(ARGS)>::type()));
   }

protected:

private:
    //! pointer to object
    CLS* mObj;
    //! function pointer
   MemFuncPtr mMethod;;
    //! argument which has to be passed when calling the function
   std::tuple<ARGS...> mTuple;


   template<std::size_t... S>
   RET callFunc(dia_FunctorHelpers::seq<S...>) {
      return  (mObj->*mMethod)(std::get<S>(mTuple) ...);
   }
      
};

/*
  as dia_FunctorRetImpl, but arguments with reference-type will be stored by value.
 */
template <class RET, class CLS, typename... ARGS>
class dia_FunctorRetCpImpl
   : public dia_FunctorRet<RET>
{

public:
   typedef RET (CLS::*MemFuncPtr) ( ARGS... );

   //! constructor - takes pointer to an object and pointer to a member function
   dia_FunctorRetCpImpl ( CLS* obj, MemFuncPtr method, ARGS... args ) : mObj(obj), mMethod(method), mTuple(args...) {}
   //! class destructor
   virtual ~dia_FunctorRetCpImpl ( ) {}

   //! override operator "()" in order to call the member function with stored argument
   virtual RET operator() ( ) {  
      return static_cast<RET>(execute());
   }

   virtual RET execute() {
      return callFunc(typename dia_FunctorHelpers::gens<sizeof...(ARGS)>::type());
   }

protected:

private:
    //! pointer to object
    CLS* mObj;
    //! function pointer
   MemFuncPtr mMethod;;
    //! argument which has to be passed when calling the function
   dia_FunctorHelpers::tuple_with_removed_refs<ARGS...> mTuple; 


   template<std::size_t... S>
   RET callFunc(dia_FunctorHelpers::seq<S...>) {
      return  (mObj->*mMethod)(std::get<S>(mTuple) ...);
   }
      
};

// shortcuts for dia_FunctorImpl and dia_FunctorCpImpl with fixed return-types
template <class CLS, typename... ARGS>
using dia_FunctorImpl = dia_FunctorRetImpl<tDiaResult, CLS, ARGS...>;

template <class CLS, typename... ARGS>
using dia_FunctorCpImpl = dia_FunctorRetCpImpl<tDiaResult, CLS, ARGS...>;

template <class CLS, typename... ARGS>
using dia_FunctorVoidImpl = dia_FunctorRetImpl<void, CLS, ARGS...>;

template <class CLS, typename... ARGS>
using dia_FunctorVoidCpImpl = dia_FunctorRetCpImpl<void, CLS, ARGS...>;


#endif
