/*!
 * \file       dia_Factory.h
 *
 * \brief      TBD
 *
 * \details    TBD
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_FACTORY__
#define __INCLUDED_DIA_FACTORY__


#ifndef __INCLUDED_DIA_COMMON_APPLICATION__
#include <common/framework/application/dia_common_application.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include <common/framework/config/dia_common_config.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_ENGINE__
#include <common/framework/engine/dia_common_engine.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include <common/framework/sysadapters/dia_SystemAdapterFacade.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN__
#include <common/framework/sysadapters/dia_SystemAdapterServicePlugin.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN_DIAGLIB__
#include <common/framework/sysadapters/dia_SystemAdapterServicePluginDiaglib.h>
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

class dia_ITimer;

enum enDiaType
{
    DIA_E_TYPE_UNKNOWN = 0,
    DIA_E_TYPE_CUSTOMER,
    DIA_E_TYPE_PRODUCTION,
    DIA_E_TYPE_COUNT
};

enum enDefaultType
{
    DIA_E_DEFTYPE_UNKNOWN = 0,
    DIA_E_DEFTYPE_TEF,
    DIA_E_DEFTYPE_USER,
    DIA_E_DEFTYPE_TEF_NO_LOCK,
    DIA_E_DEFTYPE_COUNT
};

enum dia_enInitLevel
{
    DIA_EN_INITLEVEL_UNKNOWN = 0,
    DIA_EN_INITLEVEL_0,
    DIA_EN_INITLEVEL_1,
    DIA_EN_INITLEVEL_2,
    DIA_EN_INITLEVEL_3,
    DIA_EN_INITLEVEL_4,
    DIA_EN_INITLEVEL_5,
    DIA_EN_INITLEVEL_6,
    DIA_EN_INITLEVEL_7,
    DIA_EN_INITLEVEL_8,
    DIA_EN_INITLEVEL_9,
    DIA_EN_INITLEVEL_10,
    DIA_EN_INITLEVEL_ALL,
    DIA_EN_INITLEVEL_COUNT
};

enum dia_enDriverType
{
   DIA_EN_DRVTYPE_UNKNOWN = 0,
   DIA_EN_DRVTYPE_KDS,
   DIA_EN_DRVTYPE_TRACE,
   DIA_EN_DRVTYPE_COUNT
};

struct dia_NRCConfig
{
   //! internal diagnosis error code to be mapped to NRC
   tDiaResult mErrCode;
   //! associated NRC
   tU8 mNRC;
};

const dia_NRCConfig
aCommonNRCMapping[] =
{
   { DIA_E_CONDITIONS_NOT_CORRECT                          , DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT },
   { DIA_E_OUT_OF_RANGE                                    , DIA_E_U8_UDS_OUT_OF_RANGE },
   { DIA_E_SEQUENCE_ERROR                                  , DIA_E_U8_UDS_SEQUENCE_ERROR },
   { DIA_E_BUSY_REPEAT                                     , DIA_E_U8_UDS_BUSY_REPEAT },
   { DIA_E_GENERAL_REJECT                                  , DIA_E_U8_UDS_GENERAL_REJECT },
   { DIA_E_GENERAL_PROGRAMMING_FAILURE                     , DIA_E_U8_UDS_GENERAL_PROGRAMMING_FAILURE },
   { DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT        , DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT },
   { DIA_E_ROUTINE_NOT_COMPLETE                            , DIA_E_U8_UDS_ROUTINE_NOT_COMPLETE },
   { DIA_E_SECURITY_ACCESS_DENIED                          , DIA_E_U8_UDS_SECURITY_ACCESS_DENIED },
   { DIA_E_INVALID_SECURITY_KEY                            , DIA_E_U8_UDS_INVALID_KEY },
   { DIA_E_EXCEED_NUMBER_OF_ATTEMPTS                       , DIA_E_U8_UDS_EXCEED_NUMBER_OF_ATTEMPTS },
   { DIA_E_REQUIRED_TIME_DELAY_NOT_EXPIRED                 , DIA_E_U8_UDS_REQUIRED_TIME_DELAY_NOT_EXPIRED },
   { DIA_E_SERVICE_NOT_SUPPORTED                           , DIA_E_U8_UDS_SERVICE_NOT_SUPPORTED },
   { DIA_E_SERVICE_NOT_SUPPORTED_IN_ACTIVE_SESSION         , DIA_E_U8_UDS_SERVICE_NOT_SUPPORTED_IN_ACTIVE_SESSION },
   { DIA_E_SUBFUNCTION_NOT_SUPPORTED                       , DIA_E_U8_UDS_SUBFUNCTION_NOT_SUPPORTED },
   { DIA_E_SUBFUNCTION_NOT_SUPPORTED_IN_ACTIVE_SESSION     , DIA_E_U8_UDS_SUBFUNCTION_NOT_SUPPORTED_IN_ACTIVE_SESSION },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED                    , DIA_E_U8_UDS_UPLOAD_DOWNLOAD_NOT_ACCEPTED },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_DIR_READ           , DIA_E_U8_UDS_UPLOAD_DOWNLOAD_NOT_ACCEPTED },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_ADD           , DIA_E_U8_UDS_UPLOAD_DOWNLOAD_NOT_ACCEPTED },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_DELETE        , DIA_E_U8_UDS_UPLOAD_DOWNLOAD_NOT_ACCEPTED },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_READ          , DIA_E_U8_UDS_UPLOAD_DOWNLOAD_NOT_ACCEPTED },
   { DIA_E_UPLOAD_DOWNLOAD_NOT_ACCEPTED_FILE_REPLACE       , DIA_E_U8_UDS_UPLOAD_DOWNLOAD_NOT_ACCEPTED },
   { DIA_E_WRONG_BLOCK_SEQUENCE_NUMBER                     , DIA_E_U8_UDS_WRONG_BLOCK_SEQUENCE_NUMBER },
   { DIA_E_TRANSFER_DATA_SUSPENDED                         , DIA_E_U8_UDS_TRANSFER_DATA_SUSPENDED },
   { DIA_E_ENGINE_IS_RUNNING                               , DIA_E_U8_UDS_ENGINE_IS_RUNNING },
   { DIA_E_VEHICLE_SPEED_TOO_HIGH                          , DIA_E_U8_UDS_VEHICLE_SPEED_TOO_HIGH },
   { DIA_E_VOLTAGE_TOO_LOW                                 , DIA_E_U8_UDS_VOLTAGE_TOO_LOW },
   { DIA_E_VOLTAGE_TOO_HIGH                                , DIA_E_U8_UDS_VOLTAGE_TOO_HIGH  },
   { DIA_E_ACCESS_SERVICE_NOT_SUPPORTED                    , DIA_E_U8_UDS_SERVICE_NOT_SUPPORTED },
   { DIA_E_ACCESS_SERVICE_NOT_SUPPORTED_BY_SESSION         , DIA_E_U8_UDS_SERVICE_NOT_SUPPORTED_IN_ACTIVE_SESSION },
   { DIA_E_ACCESS_DID_NOT_SUPPORTED                        , DIA_E_U8_UDS_OUT_OF_RANGE },
   { DIA_E_ACCESS_DID_NOT_SUPPORTED_BY_SESSION             , DIA_E_U8_UDS_OUT_OF_RANGE },
   { DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED                    , DIA_E_U8_UDS_SUBFUNCTION_NOT_SUPPORTED },
   { DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_SESSION         , DIA_E_U8_UDS_SUBFUNCTION_NOT_SUPPORTED_IN_ACTIVE_SESSION },
   { DIA_E_ACCESS_SUBFUNC_NOT_SUPPORTED_BY_CALIBRATION     , DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT },
   { DIA_E_ACCESS_REJECTED_BY_SECURITY_LEVEL               , DIA_E_U8_UDS_SECURITY_ACCESS_DENIED },
   { DIA_E_ACCESS_INVALID_FORMAT                           , DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT },
   { DIA_E_ACCESS_GENERAL_PROGRAMMING_FAILURE              , DIA_E_U8_UDS_GENERAL_PROGRAMMING_FAILURE },
   { DIA_E_SERVICE_HANDLER_NOT_AVAILABLE                   , DIA_E_U8_UDS_OUT_OF_RANGE },
   { DIA_E_IOCTRL_PRECONDITIONS_NOT_MATCHED                , DIA_E_U8_UDS_SEQUENCE_ERROR },
   { DIA_E_UPLOAD_DOWNLOAD_REJECTED_DUE_TO_POWER_LEVEL     , DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT },
   { DIA_E_UPLOAD_DOWNLOAD_REQUEST_NO_STRATEGY             , DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT },
   { DIA_E_UPLOAD_DOWNLOAD_TRANSFER_NO_STRATEGY            , DIA_E_U8_UDS_SEQUENCE_ERROR },
   { DIA_E_UPLOAD_DOWNLOAD_EXIT_NO_STRATEGY                , DIA_E_U8_UDS_SEQUENCE_ERROR },
   { DIA_E_ROUTINE_SUBFUNCTION_NOT_SUPPORTED               , DIA_E_U8_UDS_SUBFUNCTION_NOT_SUPPORTED },
   { DIA_E_FAILURE_PREVENTS_EXECUTION_OF_REQUESTED_ACTION  , DIA_E_U8_UDS_FAILURE_PREVENTS_EXECUTION_OF_REQUESTED_ACTION}
};

class dia_EngineServer;
class dia_Session;
class dia_Application;
class dia_SystemAdapter;
class dia_IDriver;
class dia_FactoryPlugin;

#ifdef __STARTUP_REFACTORING_ENABLE__
namespace dia {
class Bundle;
class BundleConfiguration;
}
#endif
class dia_Factory
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_Factory);

public:
   //! setup (create objects,...)
   virtual tDiaResult setup ( dia_enInitLevel level ) = 0;
   //! tear down (destroy objects,...)
   virtual tDiaResult shutdown ( dia_enInitLevel level ) = 0;


   // create diagnostic service handlers for the given type of diagnosis (customer or production
   virtual tDiaResult makeServiceHandlers ( enDiaType /*type*/ ) { return DIA_SUCCESS; }

#ifndef __DIA_UNIT_TESTING__
   //! create threads used for processing messages
   virtual tDiaResult makeActiveObjects ( dia_Application& /*app*/ ) { return DIA_E_NOT_IMPLEMENTED; }
#endif
   //! create a project specific session object for the given session ID
   virtual dia_Session* makeSession ( tU8 sessionID, dia_EngineServer& rEngine );

   //! create the client handler instances
   virtual tDiaResult makeSystemAdapters ( void ) = 0;

   //!
   virtual dia_IDriver* makeDriver ( dia_enDriverType driverID ) = 0;
   //!
   virtual dia_ITimer*  makeTimer ( void ) = 0;

   //! map a diagnosis error code to a UDS negative response code (NRC)
   virtual tU8 makeNRC ( tDiaResult retCode );

   //! load a single NRC
   virtual tDiaResult loadNRCMapping ( const dia_NRCConfig& mapping );
   //! load a set of NRCs
   virtual tDiaResult loadNRCMappings ( const dia_NRCConfig mapping[], tU16 numOfItems );
   //! clear set of NRCs
   virtual void clearNRCmappings(void);

   //! couple the given interface pointer with the given IID
   template<class _T> tDiaResult assignInterfaceImplementation ( _T* pInterfaceImp );

   virtual dia_enInitLevel getInitializationLevel ( void ) const { return mActiveLevel; }
   virtual void setInitializationLevel ( dia_enInitLevel level ) { mActiveLevel = level; }

   virtual tDiaResult addFactoryPlugin ( dia_FactoryPlugin* pPlugin );

   virtual tDiaResult requestInitializationLevel ( dia_enInitLevel level );

   virtual tDiaResult queryFactoryPlugin ( dia_UID uid, dia_FactoryPlugin** ppPlugin );
   virtual tDiaResult queryFactoryPlugin ( tCString name, dia_FactoryPlugin** ppPlugin );
#ifdef __STARTUP_REFACTORING_ENABLE__
   virtual tDiaResult makeInitializationLevels(void);
   virtual dia::Bundle* makeBundle ( const std::string& bundleName ) = 0;
#endif

protected:
   //! default class constructor
   dia_Factory ( void );
   //! class destructor
   virtual ~dia_Factory ( void );

   #ifdef __STARTUP_REFACTORING_ENABLE__
   std::list<dia::Bundle*> mBundleRepo;
   #endif
   //------------------------------------------------------------------------------

#ifndef __DIA_UNIT_TESTING__
   dia_SystemAdapter* createSystemAdapter ( tCString name, tU16 clientAppID ) const;
   dia_SystemAdapterServicePlugin* createSystemAdapterServicePlugin ( tCString name, tU16 srvID, tU16 srvMajVersion, tU16 srvMinVersion, dia_SystemAdapter* adapter ) const;
   dia_SystemAdapterServicePluginDiaglib* createSystemAdapterServicePluginDiaglib ( tCString name, tU16 srvMajVersion, tU16 srvMinVersion, dia_SystemAdapter* adapter ) const;

   template<class _T> _T* createSystemAdapterFeature ( tCString name, dia_SystemAdapterServicePlugin* pSrvPlugin );
   template<class _T> _T* createSystemAdapterFeatureDiaglib ( tCString name, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin );

   template<class _T> _T* createSystemAdapterSysSetFeature ( tCString name, tU32 sysSetId, tU32 sysSetTypes, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin );
   template<class _T> _T* createSystemAdapterDefSetFeature ( tCString name, tU32 sysSetId, tU32 sysSetTypes, dia_SystemAdapterServicePlugin* pSrvPlugin, tU8 defsetGroup, tU16 methFID, tU16 propFID=0xFFFF );
   template<class _T> _T* createSystemAdapterIOControlFeature ( tCString name, dia_SystemAdapterServicePlugin* pSrvPlugin );
   template<class _T> _T* createSystemAdapterRemoteControlFeature ( tCString name, dia_eRemoteControlPluginType id, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin ); // obsolete, don't use anymore
   template<class _T> _T* createSystemAdapterRemoteControlFeature ( tCString name, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin );
   template<class _T> _T* createSystemAdapterConfigUpdateFeature ( tCString name, tU32 propID, tU32 sysSetId, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin );

   //! helper method used to delete all known system adapters
   virtual tDiaResult deleteSystemAdapters ( void );
   //! helper method used to unregister all known interface implementations
   virtual tDiaResult unregisterInterfaces ( void );
#endif

   //! setup of sub factories
   virtual tDiaResult setupFactoryPlugins ( dia_enInitLevel level );
   virtual tDiaResult teardownFactoryPlugins ( dia_enInitLevel level );

protected:
   // synchronization lock object.
   dia_Lock mSyncObj;
   //! current initialization level of the diagnosis application
   dia_enInitLevel mActiveLevel;
   //! list of routine control objects
   std::list<dia_FactoryPlugin*> mSubFactoryRep;
   //!
   std::map<dia_enDriverType,dia_IDriver*> mDriverMap;
   //! repository of properties
   std::map<tDiaResult,dia_NRCConfig> mNRCRepository;
};

//----------------------------------------------------------------------------------------

template<class _T> tDiaResult
dia_Factory::assignInterfaceImplementation ( _T* pInterfaceImp )
{
   if ( !pInterfaceImp ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = getInstanceOfSystemAdapterFacade()->setInterface<_T>(pInterfaceImp);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("FAILED TO ASSIGN INSTANCE OF INTERFACE (ERRCODE = 0x%08x, IID = 0x%08x)",retCode,_T::getUID());
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

template<class _T, class _L, class _N>
tDiaResult
assignInterfaceAndNotifier ( _T* pInterface )
{
   if ( !pInterface ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = getInstanceOfSystemAdapterFacade()->setInterface<_T>(pInterface);
   if ( retCode == DIA_SUCCESS )
   {
      _N* pNotifier = new _N;
      retCode = setSysAdapterInterfaceNotifier<_L,_N>(pNotifier,pNotifier);
      if ( retCode != DIA_SUCCESS )
      {
         DIA_TR_ERR("FAILED TO INSTALL NOTIFIER FOR INTERFACE LISTENER (ERRCODE = 0x%08x, IID = 0x%08x)",retCode,_L::getUID());
      }
   }
   else
   {
      DIA_TR_ERR("FAILED TO ASSIGN INSTANCE OF INTERFACE (ERRCODE = 0x%08x, IID = 0x%08x)",retCode,_T::getUID());
   }

   return retCode;
}

//----------------------------------------------------------------------------------------

template<class _L, class _N>
tDiaResult
assignNotifier ( void )
{
   _N* pNotifier = new _N;
   tDiaResult retCode = setSysAdapterInterfaceNotifier<_L,_N>(pNotifier,pNotifier);
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("FAILED TO INSTALL NOTIFIER FOR INTERFACE LISTENER (ERRCODE = 0x%08x, IID = 0x%08x)",retCode,_L::getUID());
   }

   return retCode;
}

//------------------------------------------------------------------------------

#ifndef __DIA_UNIT_TESTING__

template<class _T>
_T*
dia_Factory::createSystemAdapterFeature ( tCString name, dia_SystemAdapterServicePlugin* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,*pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterFeatureDiaglib ( tCString name, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,*pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterSysSetFeature ( tCString name, tU32 sysSetId, tU32 sysSetTypes, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,sysSetId,sysSetTypes,*pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterDefSetFeature (
        tCString name,
        tU32 sysSetId,
        tU32 sysSetTypes,
        dia_SystemAdapterServicePlugin* pSrvPlugin,
        tU8  defsetGroup,
        tU16 methFID,
        tU16 propFID
 )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,sysSetId,sysSetTypes,*pSrvPlugin,defsetGroup,methFID,propFID);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterRemoteControlFeature ( tCString name, dia_eRemoteControlPluginType id, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,id,*pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterRemoteControlFeature ( tCString name, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,*pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterConfigUpdateFeature ( tCString name, tU32 propID, tU32 sysSetId, dia_SystemAdapterServicePluginDiaglib* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name,propID,sysSetId,*pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}


//------------------------------------------------------------------------------

template<class _T>
_T*
dia_Factory::createSystemAdapterIOControlFeature ( tCString name, dia_SystemAdapterServicePlugin* pSrvPlugin )
{
   _T* pFeature = 0;

   if ( pSrvPlugin )
   {
      pFeature = new _T(name, *pSrvPlugin);
      if ( pFeature )
      {
         pSrvPlugin->addFeature(*pFeature);
      }
   }

   return pFeature;
}

#endif // __DIA_UNIT_TESTING__

//#############################################################################
//#
//# factory methods declarations
//#
//#############################################################################

dia_Factory* getInstanceOfFactory ( void );

//-----------------------------------------------------------------------------

void releaseInstanceOfFactory ( void );

#endif /* __INCLUDED_DIA_FACTORY__ */
