#include <string>
#include <stdint.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

#ifndef VARIANT_S_FTR_DISABLE_REGISTRY
#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"
#endif


#include <openssl/sha.h>

#include "dia_EcuUid.h"

#define DIA_ECU_UID_REG_PATH "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS"
#define DIA_ECU_UID_STR_REG_KEY "ECU_UID_STR"
#define DIA_ECU_UID_REG_KEY "ECU_UID"

#define MAX_ECU_UID_STR_LEN 0x100


#define MAX_HWUID_SIZE 100
#define MAX_PLATFORM_SIZE 10
#define MAX_TYPE_SIZE 10
#define MAX_UUID_SIZE ( MAX_HWUID_SIZE + MAX_PLATFORM_SIZE + MAX_TYPE_SIZE + 2)


static void binToHex(uint8_t *bin, uint32_t numBytes, std::string &res) {
   char const *nibbleToHex= {"0123456789abcdef"};
   fprintf(stderr, "binToHex:numBytes=%u\n", numBytes);
   uint8_t *tmp=bin; 
   uint8_t *end=bin + numBytes;
   char *hexBuf=(char *)alloca(numBytes * 2 + 1);
   char *to=hexBuf;
   for (; tmp<end;++tmp) {
      uint8_t byte=*tmp;
      *to++=nibbleToHex[byte >> 4];
      *to++=nibbleToHex[byte & 0x0F];
   }
   *to=0;
   fprintf(stderr, "binToHex:hex=%s\n", hexBuf);
   res+=hexBuf;
}


static bool getFirstLineFromFile(char const *fileName, std::string &res) {
   FILE *fp;
   ssize_t numBytes;
   char *buf=0;
   int allOk=0;
   fprintf(stderr, "getFirstLineFromFile:%s\n", fileName);
   do {
      fp=fopen(fileName, "r");
      if (!fp) {
         fprintf(stderr, "getFirstLineFromFile:could not open %s\n", fileName);
         return 0;
      }
      size_t num=0;
      numBytes=getline(&buf, &num, fp);
      if (numBytes <=0) {
         fprintf(stderr, "getFirstLineFromFile:getline failed numBytes=%i\n", (int32_t)numBytes);
         break;
      }
      fprintf(stderr, "getFirstLineFromFile:getline numBytes=%i\n", (int32_t)numBytes);
      fprintf(stderr, "getFirstLineFromFile:getline num=%u\n", (uint32_t)num);
      char *tmp=buf;
      size_t i=0;
      for (i=0; i<(size_t)numBytes; ++i) {
         if (*tmp==' ' || *tmp=='\n' || *tmp=='\r' || !*tmp) {
            *tmp=0;
            fprintf(stderr, "add LAST\n");
            break;
         }
         allOk=1;
         ++tmp;
      }
      
      
   } while (0);
   
   if (allOk) {
      res+=buf;
      free(buf);
   }
   
   fprintf(stderr, "getFirstLineFromFile:allOk=%u res='%s'\n", allOk, res.c_str());
   
   return allOk;
}


#ifdef VARIANT_S_FTR_DISABLE_REGISTRY
static bool writeRegistry(const char *regPath, const char *regKey, const uint8_t * /*val */ , uint32_t len) {
   fprintf(stderr, "writeRegistry() skipping in pc-build regPath='%s' regKey=%s len=%u'\n'", regPath, regKey, len);
   return true;
}
#else
static bool writeRegistry(const char *regPath, const char *regKey, const uint8_t *val, uint32_t len) {
   bool res=false;
   fprintf(stderr, "writeRegistry regPath=%s regKey=%s len=%u\n", regPath, regKey, len);
   reg_tclRegKey oReg;

   if (oReg.bOpen(regPath)) {
      // Reading Data from Registry
      if (oReg.bSetString(regKey, (char *)val, len)) {
         res=true;
      }
      oReg.vClose();
   }
   return res;
}
#endif

static bool writeRegistry(const char *regPath, const char *regKey, std::string const &val) {
   uint32_t numBytes = (uint32_t)val.length() + 1;
   fprintf(stderr, "writeRegistry regPath=%s regKey=%s, val=%s\n", regPath, regKey, val.c_str());
   bool res=writeRegistry(regPath, regKey, (uint8_t const *)val.c_str(), numBytes);
   return res;
}


static const char *getPlatForm() {
#if defined(GEN3ARM)
   return "GEN3ARM";
#elif defined(GEN3X86)
   return "GEN3X86";
#elif defined(GEN4INTEL)
   return "GEN4INTEL";
#elif defined(GEN4LSIM)
   return "GEN4LSIM";
#elif defined(GEN4ARM)
   return "GEN4ARM";
#elif defined(_LINUXX86MAKE_)
   return "LINUXX86";
#elif defined(_LINUXX86_64_)
   return "LINUXX86_64";
#else
// unknown env.
#error
#endif
 }



static bool getEcuUid(std::string &res) {
   bool allOk=false;
   char const *source=0;
   char const *platForm=0;
   std::string hwUid;
   do {
      platForm=getPlatForm();
      if (!platForm) {
         fprintf(stderr, "could not get platform\n");
         break;
      }
#if defined(GEN3ARM) || defined(GEN4ARM) || defined(GEN4INTEL)
      else if (getFirstLineFromFile("/sys/block/mmcblk1/device/cid", hwUid)) {
         source="CID";
      }
#elif defined(GEN3X86) || defined(GEN4LSIM) || defined(_LINUXX86_64_) || defined(_LINUXX86MAKE_)
      else if (getFirstLineFromFile("/sys/class/dmi/id/product_uuid", hwUid)) {
         source="DMI";
      }
#endif
      else {
         fprintf(stderr, "No way to find hwUid for known platform :%s\n", platForm);
         break;
      }
      
      res+=platForm;
      res+="_";
      res+=source;
      res+="_";
      res+=hwUid;
      allOk=true;
      
   } while (0);
   
   return allOk;
}

 
extern "C" {

int dia_storeEcuUidToRegistry() {
   std::string ecuUid;
   if (!getEcuUid(ecuUid)) {
      fprintf(stderr, "getEcuUid failed");
      return -1;
   }
   writeRegistry(DIA_ECU_UID_REG_PATH, DIA_ECU_UID_STR_REG_KEY, ecuUid);
   uint8_t shaHash[SHA_DIGEST_LENGTH];
   SHA1((uint8_t const *)ecuUid.c_str(), ecuUid.length(), shaHash);
   std::string shaHex;
   binToHex(shaHash, SHA_DIGEST_LENGTH, shaHex);
   fprintf(stderr, "dia_storeEcuUidToRegistry:hex='%s'\n", shaHex.c_str());
   writeRegistry(DIA_ECU_UID_REG_PATH, DIA_ECU_UID_REG_KEY, shaHex.c_str());
   fprintf(stdout, "%s\n", shaHex.c_str());
   int fd=open("/tmp/ecu_uid", O_CREAT | O_WRONLY, S_IRUSR, S_IWUSR);
   if (fd >= 0)  {
     //Coverity Fix CID563850, CID563849
     (void)write(fd, shaHex.c_str(), shaHex.length()); 
     (void)close(fd);
   } 
   return 0;
}

}
