/*
 * dia_CondVar.cpp
 *
 *  Created on: 22.05.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_CONDVAR__
#include <common/framework/application/dia_CondVar.h>
#endif

//------------------------------------------------------------------------------

dia_CondVar::dia_CondVar ( const tChar* pName )
   : mName(pName),
     mEventMask(0),
#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR
     mEvent(OSAL_C_INVALID_HANDLE)
#else
     mWaitMask(DIA_C_CONDVAR_EVENT_BROADCAST)
#endif
{
#ifdef __DIA_UNIT_TESTING__
   DIA_TR_INF("dia_CondVar::dia_CondVar \"%s\" this=0x%08X", mName.c_str(), this);
#endif

#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR
   // create the event
   tS32 result = OSAL_s32EventCreate(mName.c_str(),&mEvent);

   if ( result == OSAL_ERROR )
   {
      mEvent = OSAL_C_INVALID_HANDLE;
   }

   DIA_ASSERT(OSAL_ERROR != result);

#else
   int ret = pthread_cond_init(&mCondVar, NULL);
   DIA_ASSERT(0==ret);
   ret = pthread_mutexattr_init(&mAttr);
   DIA_ASSERT(0==ret);
   ret = pthread_mutexattr_settype(&mAttr, PTHREAD_MUTEX_RECURSIVE);
   DIA_ASSERT(0==ret);
   ret = pthread_mutex_init(&mMutex, &mAttr);
   DIA_ASSERT(0==ret);
#endif
}

//------------------------------------------------------------------------------

dia_CondVar::~dia_CondVar ( void )
{
#ifdef __DIA_UNIT_TESTING__
   DIA_TR_INF("dia_CondVar::~dia_CondVar \"%s\" this=0x%08X", mName.c_str(), this);
#endif

   _BP_TRY_BEGIN
   {
#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR
      DIA_ASSERT( OSAL_ERROR != OSAL_s32EventClose(mEvent) );
      DIA_ASSERT( OSAL_ERROR != OSAL_s32EventDelete(mName.c_str())); //Function c_str() may throw exception '...' in destructor.
#else
      DIA_ASSERT( 0==pthread_cond_destroy(&mCondVar) );
      DIA_ASSERT( 0==pthread_mutex_destroy(&mMutex) );
      DIA_ASSERT( 0==pthread_mutexattr_destroy(&mAttr) );
#endif
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_CondVar::~dia_CondVar !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------


tDiaResult
dia_CondVar::wait ( tU32 condMask, tU32 timeoutMsec )
{
   DIA_TR_INF("dia_CondVar::wait condMask 0x%08x, EventMask 0x%08x, Timeout 0x%08x (decimal %d)", condMask, mEventMask, timeoutMsec, timeoutMsec);

   tDiaResult retVal = DIA_SUCCESS;

#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR

   DIA_ASSERT(OSAL_C_INVALID_HANDLE != mEvent);

   OSAL_tEventMask hEvRequest = 0;

   tS32 result = OSAL_s32EventWait (
      mEvent,
      condMask | DIA_C_CONDVAR_EVENT_BROADCAST,
      OSAL_EN_EVENTMASK_OR,
      timeoutMsec,
      &hEvRequest
      );

   if ( OSAL_OK != result )
   {
      retVal = (OSAL_u32ErrorCode() == OSAL_E_TIMEOUT) ? DIA_E_CONDVAR_TIMEOUT : DIA_E_CONDVAR_FAILED;
   }
   else
   {
      DIA_TR_INF("Wait for CondVar passed!");

      if ( hEvRequest & DIA_C_CONDVAR_EVENT_BROADCAST )
      {
         retVal = DIA_E_CONDVAR_BROADCAST;
         if ( OSAL_s32EventPost(mEvent, ~DIA_C_CONDVAR_EVENT_BROADCAST, OSAL_EN_EVENTMASK_AND) != OSAL_OK)
         {
            DIA_ASSERT_ALWAYS();
         }
      }
   }
#else

   DIA_TR_INF("dia_CondVar::wait (%s, %p) CONDVAR_AS_PTHREAD_CONDVAR", mName.c_str(), this);

   if ( (mEventMask & condMask) != 0 )
   {
      DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_SUCCESS", mName.c_str(), this);
      // at least one condition is true
      retVal = DIA_SUCCESS;
   }
   else
   {
      mWaitMask = condMask | DIA_C_CONDVAR_EVENT_BROADCAST;

      DIA_ASSERT( 0==pthread_mutex_lock(&mMutex) );

      if ( timeoutMsec != DIA_C_CONDVAR_TIMEOUT_FOREVER )
      {
         DIA_TR_INF("dia_CondVar::wait (%s, %p) timeoutMsec != DIA_C_CONDVAR_TIMEOUT_FOREVER", mName.c_str(), this);
         timespec timeSpec;
         timeSpec.tv_sec  = (long) timeoutMsec / 1000;
         timeSpec.tv_nsec = (long) (timeoutMsec % 1000) * 1000 /*usec*/ * 1000 /*nsec*/;

         DIA_TR_INF("timeSpec tv_sec=%d tv_nsec=%d", timeSpec.tv_sec, timeSpec.tv_nsec );

         int rc = pthread_cond_timedwait(&mCondVar, &mMutex, &timeSpec);
         DIA_TR_INF("CondVarWait::wait (%s, %p) pthread_cond_timedwait returned %d", mName.c_str(), this, rc);

         switch (rc)
         {
            case 0:
            {
               /* OK, nothing to do */
            }
            break;

            case ETIMEDOUT:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_TIMEOUT", mName.c_str(), this);
               retVal = DIA_E_CONDVAR_TIMEOUT;
            }
            break;

            case EINVAL:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_FAILED (EINVAL)", mName.c_str(), this);
               retVal = DIA_E_CONDVAR_FAILED;
            }
            break;

            case EPERM:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_FAILED (EPERM)", mName.c_str(), this);
               retVal = DIA_E_CONDVAR_FAILED;
            }
            break;

            default:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_FAILED (unknown error dec %d)", mName.c_str(), this, rc);
               retVal = DIA_E_CONDVAR_FAILED;
            }
            break;
         }
      }
      else
      {
         int retCode =  pthread_cond_wait(&mCondVar,&mMutex);
         DIA_TR_INF("CondVarWait::wait pthread_cond_wait (%s, %p) returned %d", mName.c_str(), this, retCode );

         switch (retCode)
         {
            case 0:
            {
               /* OK, nothing to do */
            }
            break;

            case ETIMEDOUT:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_TIMEOUT", this, mName.c_str());
               retVal = DIA_E_CONDVAR_TIMEOUT;
            }
            break;

            case EINVAL:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_FAILED (EINVAL)", this, mName.c_str());
               retVal = DIA_E_CONDVAR_FAILED;
            }
            break;

            case EPERM:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_FAILED (EPERM)", this, mName.c_str());
               retVal = DIA_E_CONDVAR_FAILED;
            }
            break;

            default:
            {
               DIA_TR_INF("dia_CondVar::wait (%s, %p) returned DIA_E_CONDVAR_FAILED (unknown error dec %d)", mName.c_str(), this, retCode);
               retVal = DIA_E_CONDVAR_FAILED;
            }
            break;
         }
      }

      DIA_ASSERT( 0==pthread_mutex_unlock(&mMutex) );

      if ( mEventMask & DIA_C_CONDVAR_EVENT_BROADCAST )
      {
         mEventMask &= (~DIA_C_CONDVAR_EVENT_BROADCAST);
      }
   }
#endif

   DIA_TR_INF("CondVarWait::wait (%s, %p) returned 0x%08X", mName.c_str(), this, retVal);
   return retVal;
}

#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR
tDiaResult
dia_CondVar::resetBroadcast ( void )
{
    if ( OSAL_s32EventPost(mEvent, ~DIA_C_CONDVAR_EVENT_BROADCAST, OSAL_EN_EVENTMASK_AND) != OSAL_OK)
    {
       DIA_ASSERT_ALWAYS();
    }
    return DIA_SUCCESS;
}
#endif
//------------------------------------------------------------------------------

tDiaResult
dia_CondVar::signal ( tU32 condMask, tBool state )
{
   tDiaResult retCode = DIA_SUCCESS;

   DIA_TR_INF("dia_CondVar::signal (%s, %p) condMask 0x%08x, state %s", mName.c_str(), this, condMask, (state ? "TRUE" : "FALSE"));

#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR

   if ( mEvent == OSAL_C_INVALID_HANDLE )
   {
      DIA_TR_INF("dia_CondVar::signal (%s) returned DIA_E_CONDVAR_FAILED", mName.c_str());
      return DIA_E_CONDVAR_FAILED;
   }

   if ( state )
   {
      mEventMask |= condMask;

      if ( OSAL_s32EventPost(mEvent,mEventMask,OSAL_EN_EVENTMASK_OR) != OSAL_OK )
      {
         retCode = DIA_E_CONDVAR_FAILED;
      }
   }
   else
   {
      mEventMask &= (~condMask);

      if ( OSAL_s32EventPost(mEvent, ~condMask, OSAL_EN_EVENTMASK_AND) != OSAL_OK )
      {
         retCode = DIA_E_CONDVAR_FAILED;
      }
   }

#else
   if ( state )
   {
      mEventMask |= condMask;
   }
   else
   {
      mEventMask &= (~condMask);
   }

   DIA_TR_INF("dia_CondVar::signal (%s, %p) mEventMask=0x%08X", mName.c_str(), this, mEventMask);

   if ( ( mEventMask & mWaitMask ) != 0 )
   {
      int rc = pthread_cond_signal(&mCondVar);
      DIA_TR_INF("dia_CondVar::signal (%s) pthread_cond_signal returned %d", mName.c_str(), rc);
   }
#endif

   DIA_TR_INF("dia_CondVar::signal (%s) returned 0x%08X", mName.c_str(), retCode);
   return retCode;
}

//------------------------------------------------------------------------------

tBool
dia_CondVar::checkSignal ( tU32 condMask ) const
{
   return ( mEventMask & condMask ) ? TRUE : FALSE;
}

//------------------------------------------------------------------------------

#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR
tDiaResult
dia_CondVar::broadcast ( void ) const
#else
tDiaResult
dia_CondVar::broadcast ( void )
#endif
{
   tDiaResult retCode = DIA_SUCCESS;

   DIA_TR_INF("dia_CondVar::broadcast (%s, %p)", mName.c_str(), this);

#ifndef VARIANT_S_FTR_ENABLE_CONDVAR_AS_PTHREAD_CONDVAR

   if ( mEvent == OSAL_C_INVALID_HANDLE ) return DIA_E_CONDVAR_FAILED;

   if ( OSAL_s32EventPost(mEvent,DIA_C_CONDVAR_EVENT_BROADCAST,OSAL_EN_EVENTMASK_OR) != OSAL_OK )
   {
      retCode = DIA_E_CONDVAR_FAILED;
   }

#else
   mEventMask |= DIA_C_CONDVAR_EVENT_BROADCAST;
   int rc = pthread_cond_broadcast(&mCondVar);
   if (0!=rc)
   {
      retCode = DIA_FAILED;
      DIA_TR_INF("dia_CondVar::broadcast (%s, %p) pthread_cond_broadcast returned %d", mName.c_str(), this, rc);
   }
#endif

   DIA_TR_INF("dia_CondVar::broadcast (%s, %p) returned 0x%08X", mName.c_str(), this, retCode);

   return retCode;
}

