/*!
 * \file       dia_CommandController.h
 *
 * \brief      Manager class that manages different kinds of command plugins and commands
 *
 * \details    Manager class that manages and maintains different kinds of diagnostic command plugins and commands
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#define __INCLUDED_DIA_COMMAND_CONTROLLER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER_PLUGIN__
#include <common/framework/application/dia_CommandControllerPlugin.h>
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

namespace dia {

class Command;

}

class dia_CommandControllerPlugin;

class dia_CommandController
{
   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_CommandController);

   friend class dia_CommandControllerPlugin;

public:
   //! add a plugin
   virtual tDiaResult addCommandControllerPlugin ( dia_CommandControllerPlugin* pPlugin );
   //! remove a plugin
   virtual tDiaResult removeCommandControllerPlugin ( dia_enCmdCtrlPluginType type );
   //! return number of registered plugins
   virtual tU16 numberOfCommandControllerPlugins ( void ) const;

   //! delegate a new command to by executed via the command controller
   virtual tDiaResult addCommand ( dia::Command* pCmd );
   //! return number of pending commands
   virtual tU16 numberOfCommands ( void ) const;

   //! observer pattern that notifies about an update of the given predicate
   virtual void update ( dia_UID predicateID, const std::vector<tU32>& args );

protected:
   //! class constructor
   dia_CommandController ( void );
   //! class destructor
   virtual ~dia_CommandController ( void );

   //! retrieve list of pending commands. will be accessed from plugins
   virtual std::list<dia::Command*>& getCommands ( void );
   //! push a command to the front of the pending list
   virtual void pushFront ( dia::Command* pCmd );
   //! push a command to the end of the pending list
   virtual void pushBack ( dia::Command* pCmd );

   //! clear the content of all repositories
   virtual void destroy ( void );

protected:
   //! synchronization object used to synchronize concurrent access from multiple threads
   mutable dia::Lock mSyncObj;
   //! list of commands managed by the controller
   std::list<dia::Command*> mCommandRep;
   //! map of plugins
   std::map<dia_enCmdCtrlPluginType,dia_CommandControllerPlugin*> mPluginRep;
};

#endif
