/*!
 * \file       dia_BashCommand.cpp
 *
 * \brief      Wrapper class to execute a bash command
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUtilis
 *
 * \copyright  (c) 2015-2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef DIA_BASHCOMMAND_H_
#include "common/framework/application/dia_BashCommand.h"
#endif

#include <errno.h>      //needed by errno

#define DIA_BASHCMD_MAX_COMMAND_LENGTH             (256)
#define DIA_BASHCMD_MAX_BUFF_LEN_FOR_LINE          (256)
#define DIA_BASHCMD_ERROR_CODE_OF_PCLOSE           ((int)-1)
#define DIA_BASHCMD_ERROR_NO_COMMAND               ((tU8)0xFF)
#define DIA_BASHCMD_ERROR_COMMAND_TOO_LONG         ((tU8)0xFE)
#define DIA_BASHCMD_ERROR_COMMAND_CREATION_FAILED  ((tU8)0xFD)
#define DIA_BASHCMD_ERROR_COMMAND_FAILED           ((tU8)0xFC)
#define DIA_BASHCMD_SCRIPT_RETURN_VALUE_KEY        "SCRIPT_RETURN_VALUE="
#define DIA_BASHCMD_RET_VAL_NOK                    ((tU32)1)

tU8 dia_BashCommand::executeCommand(const std::string& cmd)
{
   return executeCommand(cmd.c_str());
}

tU8 dia_BashCommand::executeCommand(const char* cmd)
{
   dia_tclFnctTrace trc("dia_BashCommand::executeCommand");

   FILE *ptr;
   char commandToBeExecuted[DIA_BASHCMD_MAX_COMMAND_LENGTH] = {0};
   tU32 retValScript = DIA_BASHCMD_RET_VAL_NOK;

   if (NULL==cmd)
   {
      DIA_TR_INF("dia_BashCommand::executeCommand - No expected parameters.");
      return DIA_BASHCMD_ERROR_NO_COMMAND;
   }

   if (strnlen(cmd, DIA_BASHCMD_MAX_COMMAND_LENGTH)>(DIA_BASHCMD_MAX_COMMAND_LENGTH-(sizeof(DIA_BASHCMD_SCRIPT_RETURN_VALUE_KEY)-1)))
   {
      DIA_TR_INF("dia_BashCommand::executeCommand - Too long command.");
      return DIA_BASHCMD_ERROR_COMMAND_TOO_LONG;
   }

   int cx = snprintf ( commandToBeExecuted, DIA_BASHCMD_MAX_COMMAND_LENGTH, "%s 2>&1; echo \"%s$?\"", cmd, DIA_BASHCMD_SCRIPT_RETURN_VALUE_KEY );

   if (cx<0)
   {
      DIA_TR_INF("dia_BashCommand::executeCommand - sprintf failed. cx=%d", cx);
      return DIA_BASHCMD_ERROR_COMMAND_CREATION_FAILED;
   }
   else if (cx>=DIA_BASHCMD_MAX_COMMAND_LENGTH)
   {
      DIA_TR_INF("dia_BashCommand::executeCommand - sprintf failed. cmd too long. cx=%d", cx);
      return DIA_BASHCMD_ERROR_COMMAND_FAILED;
   }

   DIA_TR_INF( "cmd = %s", commandToBeExecuted);

   if ( NULL!=(ptr = popen(commandToBeExecuted, "r")))
   {
      DIA_TR_INF("dia_BashCommand::executeCommand - popen successful");

      char buff[DIA_BASHCMD_MAX_BUFF_LEN_FOR_LINE];
      int i = 0;  //only for tracing
      while (buff==fgets(buff, DIA_BASHCMD_MAX_BUFF_LEN_FOR_LINE, ptr))
      {
         /* remove trailing white space from string */
         size_t len = strnlen(buff, DIA_BASHCMD_MAX_BUFF_LEN_FOR_LINE);
         if (len>0 && len<sizeof(buff))
         {
            char* pToTerminated = buff +  len -1;

            while (0!=isspace(pToTerminated[0]) && (len>0))
            {
               //DIA_TR_INF("Removed one letter: 0x%02X", pToTerminated[0]);
               pToTerminated[0] = 0;
               pToTerminated--;
               len--;
            }

            DIA_TR_INF("#Line[%02d]=%s", i, buff );
         }
         else
         {
            DIA_TR_INF("ERROR: len=%zu is not less than %zu", len, sizeof(buff));
            DIA_TR_INF("#Line[%02d]=%s", i, buff );
         }

         /* skip empty strings */
         if (len>0)
         {
            /* get return value from executed script */
            char* p = strstr( buff, (const char *)DIA_BASHCMD_SCRIPT_RETURN_VALUE_KEY);
            if (NULL!=p)
            {
               p += (sizeof(DIA_BASHCMD_SCRIPT_RETURN_VALUE_KEY) - 1);

               DIA_TR_INF("return value = %s", p);

               retValScript = 0;
               while (((p-buff)<DIA_BASHCMD_MAX_BUFF_LEN_FOR_LINE) && isdigit(p[0]))
               {
                  retValScript = (retValScript*10) + ((tU32)p[0] - '0');
                  p++;
               }

               DIA_TR_INF("retValScript = %d", retValScript);
            }
         }
         i++;
      }

      int pCloseReturnValue = pclose(ptr);
      DIA_TR_INF("dia_BashCommand::executeCommand pCloseReturnValue = 0x%08X", pCloseReturnValue);

      if (DIA_BASHCMD_ERROR_CODE_OF_PCLOSE==pCloseReturnValue)
      {
         DIA_TR_INF( "dia_BashCommand::executeCommand pclose return -1 with errno = 0x%08X", errno);
      }
      else if (WIFEXITED(pCloseReturnValue))
      {
         //status was returned for a child process that terminated normally
         DIA_TR_INF("dia_BashCommand::executeCommand 8-bits return code of pclose = %d", WEXITSTATUS(pCloseReturnValue));
      }
   }
   else
   {
      DIA_TR_INF( "ERROR: popen return NULL");
   }

   return static_cast<tU8>((retValScript & 0x000000FF));
}
