/*!
 * \file       dia_ApplicationLauncher.cpp
 *
 * \brief      Class dia::ApplicationLauncher is responsible for launching (initializing and
 *             starting) the diagnostics application and other diagnostics bundles (collections
 *             of files representing a certain feature or group of features)
 *
 * \details    dia::ApplicationLauncher is responsible for launching the diagnostics application
 *             and other diagnostics bundles
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the communication of its
 * contents to others without express authorization is prohibited. Offenders will be held liable
 * for the payment of damages. All rights reserved in the event of the grant of a patent, utility
 * model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_APPLICATION_LAUNCHER__
#include <common/framework/application/dia_ApplicationLauncher.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_BUNDLE__
#include <common/framework/application/dia_Bundle.h>
#endif

#ifndef __INCLUDED_DIA_BUNDLE_CONFIGURATION__
#include <common/framework/application/dia_BundleConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_INITIALIZATION_LEVEL__
#include <common/framework/application/dia_InitializationLevel.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include "common/framework/application/dia_ThreadMonitor.h"
#endif

//#ifndef __DIA_UNIT_TESTING__
//   (void) registerThread(eThreadID_Main);
//#endif

#ifndef __DIA_UNIT_TESTING__

dia::ApplicationLauncher*
getInstanceOfApplicationLauncher ( void )
{
   return dia::ApplicationLauncher::getInstance();
}

void
releaseInstanceOfApplicationLauncher ( void )
{
   dia::ApplicationLauncher::deleteInstance();
}
#endif

namespace dia {

const std::string appBundleName = "DIA_MAIN_APPLICATION";

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(ApplicationLauncher)

//------------------------------------------------------------------------------

ApplicationLauncher::ApplicationLauncher ( void )
   : syncObj("dia::ApplicationLauncher_LK"),
     mIsAppAlreadyStarted(false),
     pBundle (0)
{
   registerThread("DIA::MAIN");
}

//------------------------------------------------------------------------------

ApplicationLauncher::~ApplicationLauncher ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
ApplicationLauncher::setup ( void )
{
   ScopeTrace oTrace("ApplicationLauncher::setup");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
ApplicationLauncher::tearDown ( void )
{
   ScopeTrace oTrace("ApplicationLauncher::tearDown");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
ApplicationLauncher::launchApplication ( void )
{
   ScopeTrace oTrace("ApplicationLauncher::launchApplication");

   LockScope oLock(syncObj);

   if ( mIsAppAlreadyStarted ) return DIA_E_ALREADY_STARTED;

   tDiaResult retCode = DIA_FAILED;

   dia_Factory* pFactory = getInstanceOfFactory();
   if ( pFactory )
   {
      retCode = pFactory->makeInitializationLevels();
      if ( retCode == DIA_SUCCESS )
      {
         retCode = launchBundle(appBundleName);
         if ( retCode == DIA_SUCCESS )
         {
            mIsAppAlreadyStarted = true;
         }
         else
         {
            DIA_TR_ERR("##### FAILED TO START BUNDLE \"%s\" (ERR=0x%08x)", appBundleName.c_str(),retCode);
         }
      }
	  else
	  {
		  DIA_TR_ERR("ApplicationLauncher::launchApplication - makeInitializationLevels() FAILED with retCode = %d!", retCode);
	  }
   }
   else
   {
	   DIA_TR_ERR("ApplicationLauncher::launchApplication - getInstanceOfFactory() FAILED!");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
ApplicationLauncher::launchBundle ( const std::string& bundleName )
{
   ScopeTrace oTrace("ApplicationLauncher::launchBundle(bundleName)");

   LockScope oLock(syncObj);

   tDiaResult retCode = DIA_FAILED;

   dia_Factory* pFactory = getInstanceOfFactory();
   if ( pFactory )
   {
   	pBundle = pFactory->makeBundle(bundleName);
      if ( !pBundle )
      {
         DIA_TR_ERR("##### FAILED TO CREATE BUNDLE \"%s\" #####", bundleName.c_str());
         return DIA_FAILED;
      }

      if ( pBundle->isStarted() )
      {
         DIA_TR_ERR("##### BUNDLE \"%s\" ALREADY STARTED. RETURNING... #####", pBundle->getName());

         return DIA_E_ALREADY_STARTED;
      }

      // we have a bundle object and if its configuration is ok we can proceed with initializing the bundle
      if ( pBundle->isConfigurationValid() )
      {
		 retCode = pBundle->initializeBundle();
		 if (retCode == DIA_SUCCESS)
		 {
         // as the bundle might have an internal hierarchy we initialize it level by level
         const std::list<InitializationLevel*>& initLevels = InitializationLevel::getInitializationLevels();
         retCode = pBundle->initializeBundle(initLevels);
         if ( retCode == DIA_SUCCESS )
         {
            retCode = pBundle->startBundle();
            if ( retCode != DIA_SUCCESS )
            {
               DIA_TR_ERR("##### FAILED TO START BUNDLE \"%s\" (ERR=0x%08x)", pBundle->getName(),retCode);
            }
         }
			 else
			 {
				 DIA_TR_ERR("##### FAILED TO initlialize level of BUNDLE \"%s\" (ERR=0x%08x)", pBundle->getName(), retCode);
			 }
		 }
		 else
		 {
			 DIA_TR_ERR("##### FAILED TO initialize BUNDLE \"%s\" (ERR=0x%08x)", pBundle->getName(), retCode);
		 }
      }
      else
      {
         DIA_TR_ERR("##### BUNDLE \"%s\" CONFIGURATION PROBLEM DETECTED #####", pBundle->getName());
         return DIA_FAILED;
      }

   }
   else
   {
	   DIA_TR_ERR("ApplicationLauncher::pFactory is NULL!!");
   }
   return retCode;
}

}

