/*!
 * \file       dia_AppController.h
 *
 * \brief      Responsible for application states, run levels, voltage monitoring, ...
 *
 * \details    Responsible for application states, run levels, voltage monitoring, ...
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#define __INCLUDED_DIA_APPCONTROLLER__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_APPLICATION_STATE_CONTROL__
#include <common/interfaces/dia_IApplicationStateControl.h>
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

// included for backward compatibility as thread monitoring part was moved to another class
#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include "common/framework/application/dia_ThreadMonitor.h"
#endif

#ifndef __INCLUDE_DIA_OPMODE__
#include "common/framework/application/dia_OpMode.h"
#endif

//#define DIA_C_S32_INVALID_SYSTEM_THREAD_ID ((tS32) 0xFFFFFFFF)
class dia_IAppStateListener;
class dia_IRunLevelListener;
class dia_tclFunctor;

enum eAppCtrlEvent
{
   eAppCtrlEvent_Unknown = 0,
   eAppCtrlEvent_InternalSetupDone,
   eAppCtrlEvent_StateChangeRequested,
   eAppCtrlEvent_StateNormalReached,
   eAppCtrlEvent_StateOffReached,
   eAppCtrlEvent_StatePauseReached,
   eAppCtrlEvent_StateReceiveReadyReached,
   eAppCtrlEvent_VoltageLowCritical,
   eAppCtrlEvent_VoltageLow,
   eAppCtrlEvent_VoltageNormal,
   eAppCtrlEvent_VoltageHigh,
   eAppCtrlEvent_VoltageHighCritical,
   eAppCtrlEvent_ECUOnOffReset,
   eAppCtrlEvent_Count
};

//enum eAppThreadID
//{
//   eThreadID_Unknown = 0,
//   eThreadID_Main,
//   eThreadID_Body,
//   eThreadID_EngineUDS,
//   eThreadID_EngineRUNIN,
//   eThreadID_EnginePDXFlash,
//   eThreadID_EngineInternalTester,
//   eThreadID_EngineMCNET,
//   eThreadID_EventLoop,
//   eThreadID_DriverCAN,
//   eThreadID_DriverProd,
//   eThreadID_DataLoggerRequest,
//   eThreadID_VehicleSettings,
//   eThreadID_Count
//};

enum dia_enResetMode
{
   eResetMode_Unknown = 0,
   eResetMode_Inactive,
   eResetMode_Active,
   eResetMode_Count
};

typedef void (*CBFunc_RunLevel) ( void* pArg );
typedef void (*CBFunc_ECUReset) ( void* pArg );
typedef void (*CBFunc) ( void* pArg );

//struct dia_ThreadInfo
//{
//   dia_ThreadInfo ( eAppThreadID ID, OSAL_tThreadID sysID ) : mID(ID), mSysID(sysID) {}
//
//   eAppThreadID   mID;
//   OSAL_tThreadID mSysID;
//
//protected:
//   //! deprecated default constructor without implementation
//   dia_ThreadInfo ();
//};

//namespace dia {
//
//class ThreadInfo
//   : public ObjectWithUID
//{
//   DECL_DEPRECATED_COPYCONSTRUCTOR_AND_ASSIGNMENTOPERATOR(ThreadInfo);
//
//public:
//   ThreadInfo ( const std::string& threadName );
//   ThreadInfo ( const std::string& threadName, const std::string& threadShortName );
//   virtual ~ThreadInfo ( void );
//
//   const std::string& getShortName ( void ) const { return mShortName; }
//   int getSystemThreadID ( void ) const { return mSystemThreadID; }
//
//protected:
//   ThreadInfo ( void );
//
//protected:
//   std::string mShortName;
//   int mSystemThreadID;
//   static int mSerial;
//};
//
//}

//class dia_IDataProviderThreads
//{
//public:
//   //! register thread information (ensure that subclasse use the same default parameter --> lint !!)
//   virtual tDiaResult registerThread ( eAppThreadID ID, OSAL_tThreadID sysID = DIA_C_S32_INVALID_SYSTEM_THREAD_ID ) = 0; //lint !e1735: checked that default param is equal to base class default param
//
//   //! unregister the thread information for the calling thread
//   virtual tDiaResult unregisterThread ( void ) = 0;
//   //! unregister the thread information for the thread with the given ID
//   virtual tDiaResult unregisterThread ( eAppThreadID ID ) = 0;
//
//   //! get the identifier for the calling thread
//   virtual eAppThreadID getCurrentThreadID ( void ) const = 0;
//
//protected:
//   //! class destructor
//   virtual ~dia_IDataProviderThreads () {}
//};

class dia_AppController
   : //public dia_IDataProviderThreads,
     public dia::IApplicationStateControl
{
   struct CBData
   {
      CBData ( void );

      CBFunc mCBFunc;
      void* mCBArg;
   };

   DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_AppController);

public:
   //! setup method for the application
   virtual tDiaResult setup ( void );
   //! shutdown method for the application
   virtual tDiaResult shutdown ( void );
   //! check if component is properly initialized
   virtual bool isSetup ( void ) const { return mIsSetup; }

   //! trigger application control event
   virtual void vOnEvent ( eAppCtrlEvent event, intptr_t par1=0, intptr_t par2=0 );

   ///////////////////////////////////////////////////////////////////////////
   //
   // Overloaded methods from interface dia::IApplicationStateControl
   //
   ///////////////////////////////////////////////////////////////////////////

   //! request a change of the application state
   virtual tDiaResult changeApplicationState ( dia::ApplicationStateID sourceStateID, dia::ApplicationStateID destinationStateID );
   virtual bool isApplicationStateChangeInProgress () {return mIsApplicationStateChangeInProgress;}

   ///////////////////////////////////////////////////////////////////////////
   //
   // Application State Management
   //
   ///////////////////////////////////////////////////////////////////////////

   //! add the given application state listener to the repository of application state listeners
   tDiaResult addAppStateListener ( dia_IAppStateListener* pListener );
   //! remove the given application state listener from the repository of application state listeners
   tDiaResult removeAppStateListener ( const dia_IAppStateListener* pListener );

   //! notification that the given application state listener has reached the given application state
   void onApplicationStateChanged ( dia::ApplicationStateID appStateID, dia_IAppStateListener* pListener );
   //! notification that the complete diagnostics application (that means all application state listeners) has reached the given application state
   void onApplicationStateChanged ( dia::ApplicationStateID appStateID );

   //! retrievel the current application state
   tU32 getAppState ( void ) { return mAppState; }

   ///////////////////////////////////////////////////////////////////////////
   //
   // Runlevel management
   //
   ///////////////////////////////////////////////////////////////////////////

   //! add the given listener to the listener repository
   virtual tDiaResult addRunlevelListener ( dia_IRunLevelListener* pListener );
   //! remove the given listener
   virtual tDiaResult removeRunlevelListener ( const dia_IRunLevelListener* pListener );
   //! retrieve the current run level
   virtual dia_enRunlevel getRunLevel ( void ) const;
   //! set a new run level
   void setRunLevel ( dia_enRunlevel newLevel );

   ///////////////////////////////////////////////////////////////////////////
   //
   // OpMode management
   //
   ///////////////////////////////////////////////////////////////////////////

   tU8 getCurrOpMode (void) {return dia::OpMode::getMapping(dia::OpMode::getActiveOpMode());}//Does not look good!!! Through Listener Impl?

#ifndef __DIA_UNIT_TESTING__
   //! clients  register their callback functor so that they will get notification when the specified runlevel is reached
   tBool RegisterListener ( dia_tclFunctor* pFunctor, dia_enRunlevel level );
#endif
   //! install a callback function that is called when the specified runlevel is reached
   void setCallback ( dia_enRunlevel level, CBFunc_RunLevel pFunc, void* pArg );

   ///////////////////////////////////////////////////////////////////////////
   //
   // Power Management
   //
   ///////////////////////////////////////////////////////////////////////////

   //! retrieve the current power level
   dia_enPowerLevel getPowerLevel ( void ) const;
   //! check if we are currently operating in normal power mode
   bool isPowerModeNormal ( void ) const;

   ///////////////////////////////////////////////////////////////////////////
   //
   // Reset Management
   //
   ///////////////////////////////////////////////////////////////////////////

    //! retrieve the current reset mode
    dia_enResetMode getResetMode ( void ) const;
   //! install a callback function that is called when an ECUReset is finished
   void setECUResetCallback ( CBFunc_ECUReset pFunc, void* pArg );

    ///////////////////////////////////////////////////////////////////////////
    //
    // Thread Management
    //
    ///////////////////////////////////////////////////////////////////////////

   // implementation for the interface dia_IDataProviderThreads

//   //! register thread information (verified: same default params as defined by base class)
//   virtual tDiaResult registerThread ( eAppThreadID ID, OSAL_tThreadID sysID = DIA_C_S32_INVALID_SYSTEM_THREAD_ID ); //lint !e1735: checked that default param is equal to base class default param
//   //! unregister the thread information for the calling thread
//   virtual tDiaResult unregisterThread ( void );
//   //! unregister the thread information for the thread with the given ID
//   virtual tDiaResult unregisterThread ( eAppThreadID ID );
//   //! get the identifier for the calling thread
//   virtual eAppThreadID getCurrentThreadID ( void ) const;

   // deprecated method. retained for backward compatibility
   virtual tDiaResult registerThread ( eAppThreadID ) const { return DIA_SUCCESS; /*return dia::getInstanceOfThreadMonitor()->registerThread(ID);*/ }


protected:
   //! protected class constructor
   dia_AppController ( void );
   //! protected class destructor
   virtual ~dia_AppController ( void );

   //! clear the content of all repositories
   virtual void destroy ( void );

   //! initiate an application state change of the diagnostics application
   virtual void startApplicationStateChange ( dia::ApplicationStateID srcStateID, dia::ApplicationStateID destStateID );
   //! process the application state change (this method will be executed in the context of the worker thread)
   virtual void processApplicationStateChangeRequest ( dia::ApplicationStateID srcStateID, dia::ApplicationStateID destStateID );

   //! handle a runlevel change
   void vOnRunlevelChanged ( void );

   //! handle a finished ECU reset
   void vOnECUResetFinished ( void );

#ifndef __DIA_UNIT_TESTING__
   void vTraceOutEvent(eAppCtrlEvent enEvent) const;
#endif

   ////////////////////////////////////////////////////////////////////////////////////////////////
   //
   // DEPRECATED METHODS. DO NOT USE THESE METHODS EXCEPT FOR BACKWARD COMPATIBILITY REASONS
   //
   ////////////////////////////////////////////////////////////////////////////////////////////////

   tBool bAddRunlevelListener ( dia_IRunLevelListener* pListener );
   tBool bRemoveRunlevelListener ( const dia_IRunLevelListener* pListener );

protected:
   //! current run level
   dia_enRunlevel mLevel;
   //! current power level
   dia_enPowerLevel mPowerLevel;
   //! current application state
   dia::ApplicationStateID mAppState;
   //! reset mode that indicates if a reset is ongoing or not
   dia_enResetMode mResetMode;

   //! runlevel callbacks
   CBData mCallbackRep[25];
   //! ecu reset callback
   CBData mCallbackECUReset;

   std::list<dia_IRunLevelListener*> mRunLevelListenerRep;
   std::list<dia_IAppStateListener*> mAppStateListenerRep;
   std::list<dia_IAppStateListener*> mRemainingAppStateListenerRep;

#ifndef __DIA_UNIT_TESTING__
   typedef std::vector<dia_tclFunctor*> tVectorFunctorPtr;
   typedef tVectorFunctorPtr::iterator tVectorFunctorItr;
   //! array of functor pointer vectors
   tVectorFunctorPtr mFtorPtrVector[DIA_EN_RUNLEVEL_COUNT];
   //! notifies the registered clients when the runlevel changes
   void vNotifyRegisterdClients ( dia_enRunlevel );

//   // repository with thread information
//   dia_ThreadInfo* mThreadInfoRep[eThreadID_Count];
#endif

   ///////////////////////

//#ifndef __DIA_UNIT_TESTING__
   bool mIsApplicationStateChangeInProgress;
   dia::ApplicationStateID mApplicationStateRequested;

//#endif
   bool mIsSetup;

   //! synchronization object used to synchronize concurrent access from multiple threads
   mutable dia_Lock syncObj;
};

//-----------------------------------------------------------------------------

//#ifndef __DIA_UNIT_TESTING__
//eAppThreadID dia_getCurrentThreadID ( void );
//#endif
//
//namespace dia {
//
//tDiaResult registerThread ( eAppThreadID ID, OSAL_tThreadID sysID = DIA_C_S32_INVALID_SYSTEM_THREAD_ID );
//tDiaResult registerThread ( const std::string& name, const std::string& shortName );
//int getCurrentThreadID ( void );
//const std::string& getCurrentThreadName ( void );
//const ThreadInfo& getCurrentThreadInfo ( void );
//
//}

#endif /* DIA_APPCONTROLLER_H_ */
