/*!
 * \file       dia_AppController.cpp
 *
 * \brief      Responsible for application states, run levels, voltage monitoring, ...
 *
 * \details    Responsible for application states, run levels, voltage monitoring, ...
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_RUNLEVEL_LISTENER__
#include <common/interfaces/dia_IRunLevelListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_APPSTATE_LISTENER__
#include <common/interfaces/dia_IAppStateListener.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_APPLICATION_STATE_CONTROL_LISTENER__
#include <common/interfaces/dia_IApplicationStateControlListener.h>
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include <common/framework/application/dia_Application.h>
#endif

#ifndef __INCLUDED_DIA_TEST_CONTROLLER__
#include <common/framework/test/dia_TestController.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include <common/framework/sysadapters/dia_SystemAdapterFacade.h>
#endif

#ifndef __DIA_UNIT_TESTING__

#ifndef __INCLUDED_DIA_MAIN__
#include <common/depricated/dia_main.h>
#endif

#ifndef DIA_TCLFUNCTOR_H
#include "common/depricated/dia_tclfunctor.h"
#endif

#endif

#ifndef __INCLUDED_DIA_SERVICE_TRACKER__
#include <common/framework/platform/cca/dia_ServiceTracker.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

#include <unistd.h>        //lint !e537  kaa1hi: repeatedly included header file without standard include guard
#include <pthread.h>       //lint !e537  kaa1hi: repeatedly included header file without standard include guard

using namespace dia;

DIA_IMPL_SINGLETON(dia_AppController)

#ifndef __DIA_UNIT_TESTING__

dia_AppController*
getInstanceOfAppController ( void )
{
   return dia_AppController::getInstance();
}

void
releaseInstanceOfAppController ( void )
{
   dia_AppController::deleteInstance();
}

#endif

static std::string strStateMapping[DIA_C_U32_APP_STATE_COUNT] = {
   "AMT_C_U32_STATE_UNKNOWN",
   "AMT_C_U32_STATE_UNINITALIZED",
   "AMT_C_U32_STATE_INITIALIZED",
   "AMT_C_U32_STATE_NORMAL",
   "AMT_C_U32_STATE_DIAGNOSIS",
   "AMT_C_U32_STATE_PAUSE",
   "AMT_C_U32_STATE_PREPARE_DOWNLOAD",
   "AMT_C_U32_STATE_RECEIVE_READY",
   "AMT_C_U32_STATE_OFF"
};

//------------------------------------------------------------------------------

dia_AppController::CBData::CBData ( void )
   : mCBFunc(0),
     mCBArg(0)
{}

//------------------------------------------------------------------------------

dia_AppController::dia_AppController ( void )
   : mLevel(DIA_EN_RUNLEVEL_UNDEFINED),
     mPowerLevel(ePowerNormal),
     mAppState(DIA_C_U32_APP_STATE_OFF),
     mResetMode(eResetMode_Inactive),
     mIsApplicationStateChangeInProgress(false),
     mApplicationStateRequested(DIA_C_U32_APP_STATE_OFF),
     mIsSetup(false),
     syncObj("dia_AppController_LK")
{
#if 0 // trace at this location has caused stack overflow
   //Stack overflow with tr_class_diagnostics_base 8 due to recursive call to dia_AppController::getInstance() to get the thread id
   dia_tclFnctTrace oTrace("dia_AppController::dia_AppController");
#endif

//#ifndef __DIA_UNIT_TESTING__
//   (void) ::memset(mThreadInfoRep,0,eThreadID_Count*sizeof(dia_ThreadInfo*));
//#endif
}

//------------------------------------------------------------------------------

dia_AppController::~dia_AppController ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_AppController::setup ( void )
{
   ScopeTrace oTrace("dia_AppController::setup");

//#ifndef __DIA_UNIT_TESTING__
//   (void) registerThread(eThreadID_Main);
//#endif
   mIsSetup = true;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_AppController::shutdown ( void )
{
   ScopeTrace oTrace("dia_AppController::shutdown");

   mIsSetup = false;
   return DIA_SUCCESS;
}


//------------------------------------------------------------------------------

void
dia_AppController::vOnEvent ( eAppCtrlEvent event, intptr_t , intptr_t  )
{
#ifndef __DIA_UNIT_TESTING__
   vTraceOutEvent(event);
#endif

   switch ( event )
   {
   case eAppCtrlEvent_InternalSetupDone:
      {
         setRunLevel(DIA_EN_RUNLEVEL_1);
         (void) getInstanceOfTestController()->runTests(DIA_EN_TESTCONDITION_RUN_LEVEL_1_STARTUP);
         vOnRunlevelChanged();
      }
      break;

   case eAppCtrlEvent_StateNormalReached:
      {
         // set the run level
         setRunLevel(DIA_EN_RUNLEVEL_2);

         // reset the reset mode
         if ( mResetMode == eResetMode_Active ) vOnECUResetFinished();
         mResetMode = eResetMode_Inactive;

         (void) getInstanceOfTestController()->runTests(DIA_EN_TESTCONDITION_RUN_LEVEL_2);
         vOnRunlevelChanged();
      }
      break;

   case eAppCtrlEvent_StateOffReached:
      {
         if ( mResetMode != eResetMode_Active )
         {
            setRunLevel(DIA_EN_RUNLEVEL_1);
            vOnRunlevelChanged();
         }
      }
      break;

   case eAppCtrlEvent_VoltageLow:
      mPowerLevel = ePowerLow;
      break;

   case eAppCtrlEvent_VoltageNormal:
      mPowerLevel = ePowerNormal;
      break;

   case eAppCtrlEvent_VoltageHigh:
      mPowerLevel = ePowerHigh;
      break;

   case eAppCtrlEvent_ECUOnOffReset:
      {
         // set the reset mode
         mResetMode = eResetMode_Active;

         // we switch immediately to runlevel one, so that server interaction will be prevented during the reset
         setRunLevel(DIA_EN_RUNLEVEL_1);
         (void) getInstanceOfTestController()->runTests(DIA_EN_TESTCONDITION_RUN_LEVEL_1_SHUTDOWN);
         vOnRunlevelChanged();
      }
      break;

   default:
      break;
   }  //lint !e788: not all items intentionally used within defaulted switch
}

//-----------------------------------------------------------------------------

#ifndef __DIA_UNIT_TESTING__
void
dia_AppController::vTraceOutEvent ( eAppCtrlEvent enEvent ) const
{
   switch ( enEvent )
   {
   case eAppCtrlEvent_InternalSetupDone:
       DIA_TR_SM("##### RUNLEVEL 1 (ON) #####");
       break;

   case eAppCtrlEvent_StateNormalReached:
      DIA_TR_SM("##### RUNLEVEL 2 (ON) #####");
       if ( mResetMode == eResetMode_Active )
       {
          DIA_TR_SM("##### ECU Reset (On-Off) - FINISHED ######");
       }
       break;

   case eAppCtrlEvent_StateOffReached:
       if ( mResetMode != eResetMode_Active )
       {
          DIA_TR_SM("##### RUNLEVEL 1 (OFF) #####");
           dia_ServiceTracker* pSrvTracker = getInstanceOfServiceTracker();
           if ( pSrvTracker )
           {
              pSrvTracker->reset();
           }
       }
       break;

   case eAppCtrlEvent_StatePauseReached:
      DIA_TR_SM("##### SYSTEM STATE PAUSE #####");
       break;

   case eAppCtrlEvent_VoltageLow:
      DIA_TR_SM("##### LOW POWER MODE #####");
       break;

   case eAppCtrlEvent_VoltageNormal:
      DIA_TR_SM("##### NORMAL POWER MODE #####");
       break;

   case eAppCtrlEvent_VoltageHigh:
      DIA_TR_SM("##### HIGH POWER MODE #####");
       break;

   case eAppCtrlEvent_ECUOnOffReset:
      DIA_TR_SM("##### ECU Reset (On-Off) #####");
      DIA_TR_SM("##### RUNLEVEL 1 (forced by reset) #####");
      break;

   default:
      break;
   } //lint !e788: not all items intentionally used within defaulted switch
}
#endif

//------------------------------------------------------------------------------

dia_enRunlevel
dia_AppController::getRunLevel ( void ) const
{
   DIA_TR_INF("dia_AppController::getRunLevel: %d", mLevel);
   return mLevel;
}

//------------------------------------------------------------------------------

dia_enPowerLevel
dia_AppController::getPowerLevel ( void ) const
{
    return mPowerLevel;
}

//------------------------------------------------------------------------------

bool
dia_AppController::isPowerModeNormal ( void ) const
{
   return (mPowerLevel == ePowerNormal) ? true : false;
}

//------------------------------------------------------------------------------

dia_enResetMode
dia_AppController::getResetMode ( void ) const
{
    return mResetMode;
}

//------------------------------------------------------------------------------

void
dia_AppController::setCallback ( dia_enRunlevel level, CBFunc_RunLevel pFunc, void* pArg )
{
   if ( level < DIA_EN_RUNLEVEL_COUNT )
   {
      mCallbackRep[level].mCBFunc = pFunc;
      mCallbackRep[level].mCBArg  = pArg;
   }
}

//------------------------------------------------------------------------------
#ifndef __DIA_UNIT_TESTING__
tBool
dia_AppController::RegisterListener ( dia_tclFunctor* pFunctor, dia_enRunlevel level)
{
   ScopeTrace oTrace("dia_AppController::RegisterListener");

   tBool bRetVal = FALSE;

   if ( level < DIA_EN_RUNLEVEL_COUNT && pFunctor!=OSAL_NULL )
   {
      mFtorPtrVector[level].push_back(pFunctor) ;
      bRetVal = TRUE;
   }

   return bRetVal;
}
#endif

//------------------------------------------------------------------------------

#ifndef __DIA_UNIT_TESTING__
void
dia_AppController::vNotifyRegisterdClients ( dia_enRunlevel level)
{
   ScopeTrace oTrace("dia_AppController::vNotifyRegisterdClients");

    tVectorFunctorItr mItr = mFtorPtrVector[level].begin();

    if(mItr!=mFtorPtrVector[level].end())
    {
        for(;mItr < mFtorPtrVector[level].end();mItr++)
        {
            if((*mItr)!=OSAL_NULL)
            {
                dia_tclFunctor* pFunctor = *mItr;
                //calls the overloaded operator
                (*pFunctor)();

                //check the notification mode
                if((*pFunctor).bIsSingleShot())
                {
                    //(*mItr) = OSAL_NULL;
                    //or remove the element from vector?
                    mItr = mFtorPtrVector[level].erase(mItr); //Coverity Fix(CID:14899)
                    //free the memory
                    OSAL_DELETE(pFunctor);
                }
            }
        }

    }
}
#endif

//------------------------------------------------------------------------------

void
dia_AppController::vOnRunlevelChanged ( void )
{
   if ( mCallbackRep[mLevel].mCBFunc )
   {
      // call the callback function
      (mCallbackRep[mLevel].mCBFunc)(mCallbackRep[mLevel].mCBArg);

      // at the moment the callback function is called only once. Afterwards it is deleted
      mCallbackRep[mLevel].mCBFunc = 0;
      mCallbackRep[mLevel].mCBArg  = 0;
   }
}

//------------------------------------------------------------------------------

void
dia_AppController::setECUResetCallback ( CBFunc_ECUReset pFunc, void* pArg )
{
   mCallbackECUReset.mCBFunc = pFunc;
   mCallbackECUReset.mCBArg  = pArg;
}

//------------------------------------------------------------------------------

void
dia_AppController::vOnECUResetFinished ( void )
{
   if ( mCallbackECUReset.mCBFunc )
   {
      // call the callback function
      (mCallbackECUReset.mCBFunc)(mCallbackECUReset.mCBArg);

      // at the moment the callback function is called only once. Afterwards it is deleted
      mCallbackECUReset.mCBFunc = 0;
      mCallbackECUReset.mCBArg  = 0;
   }
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_AppController::addRunlevelListener ( dia_IRunLevelListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::addRunlevelListener(dia_IRunLevelListener*)");

    tDiaResult retCode = DIA_FAILED;

    std::list<dia_IRunLevelListener*>::iterator listIter = mRunLevelListenerRep.begin();
    for ( ; listIter != mRunLevelListenerRep.end(); listIter++ )
    {
        if ( (*listIter) == pListener )
        {
            // already registered
            break;
        }
    }

    if ( listIter == mRunLevelListenerRep.end() )
    {
        mRunLevelListenerRep.push_back(pListener);
        retCode = DIA_SUCCESS;
    }

    return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
dia_AppController::removeRunlevelListener ( const dia_IRunLevelListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::bRemoveListener()");

    tDiaResult retCode = DIA_FAILED;

    std::list<dia_IRunLevelListener*>::iterator listIter = mRunLevelListenerRep.begin();
    for ( ; listIter != mRunLevelListenerRep.end(); listIter++ )
    {
        if ( (*listIter) == pListener )
        {
            mRunLevelListenerRep.erase(listIter);
            retCode = DIA_SUCCESS;
            break;
        }
    }

    return retCode;
}

//---------------------------------------------------------------------------------------------------------------------

tBool
dia_AppController::bAddRunlevelListener ( dia_IRunLevelListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::bAddRunlevelListener(dia_IRunLevelListener*)");
    return (addRunlevelListener(pListener) == DIA_SUCCESS) ? TRUE : FALSE;
}

//---------------------------------------------------------------------------------------------------------------------

tBool
dia_AppController::bRemoveRunlevelListener ( const dia_IRunLevelListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::bRemoveListener()");
    return (bRemoveRunlevelListener(pListener) == DIA_SUCCESS) ? TRUE : FALSE;
}

//-----------------------------------------------------------------------------

void
dia_AppController::setRunLevel ( dia_enRunlevel newLevel )
{
   // store old level for later usage
   dia_enRunlevel oldLevel = mLevel;
   // set the new level
   mLevel = newLevel;
   DIA_TR_INF("dia_AppController::setRunLevel: %d", newLevel);

   std::list<dia_IRunLevelListener*>::iterator listIter = mRunLevelListenerRep.begin();
    for ( ; listIter != mRunLevelListenerRep.end(); listIter++ )
    {
        (*listIter)->vOnRunLevelChanged(mLevel,oldLevel);
    }
}

//-----------------------------------------------------------------------------

//tDiaResult
//dia_AppController::registerThread ( eAppThreadID appThreadID, OSAL_tThreadID sysThreadID )
//{
//#ifndef __DIA_UNIT_TESTING__
//    tDiaResult retCode = DIA_SUCCESS;
//
//    if ( appThreadID <= eThreadID_Unknown || appThreadID >= eThreadID_Count )
//    {
//        return DIA_E_THREAD_ID_INVALID;
//    }
//
//    OSAL_tThreadID osalID = sysThreadID;
//    if ( osalID == DIA_C_S32_INVALID_SYSTEM_THREAD_ID )
//    {
//        osalID = OSAL_ThreadWhoAmI();
//    }
//
//    if ( mThreadInfoRep[appThreadID] )
//    {
//        retCode = DIA_E_THREAD_ALREADY_REGISTERED;
//    }
//
//    if ( retCode == DIA_SUCCESS )
//    {
//        mThreadInfoRep[appThreadID] = OSAL_NEW dia_ThreadInfo(appThreadID,osalID);
//    }
//
//    DIA_TR_INF("registerThread(appThreadID = %d, sysThreadID = %x, pthreadID = %x)", appThreadID, sysThreadID, pthread_self());
//
//    return retCode;
//#else
//    return DIA_SUCCESS;
//#endif
//}

//-----------------------------------------------------------------------------

//tDiaResult
//dia_AppController::unregisterThread ( void )
//{
//    return unregisterThread(getCurrentThreadID());
//}

//-----------------------------------------------------------------------------

//tDiaResult
//dia_AppController::unregisterThread ( eAppThreadID appThreadID )
//{
//#ifndef __DIA_UNIT_TESTING__
//    tDiaResult retCode = DIA_SUCCESS;
//
//    if ( appThreadID >= eThreadID_Count )
//    {
//        return DIA_E_THREAD_ID_INVALID;
//    }
//
//    if ( mThreadInfoRep[appThreadID] )
//    {
//        OSAL_DELETE mThreadInfoRep[appThreadID];
//        mThreadInfoRep[appThreadID] = 0;
//    }
//
//    return retCode;
//#else
//    return DIA_SUCCESS;
//#endif
//}

//-----------------------------------------------------------------------------

//eAppThreadID
//dia_AppController::getCurrentThreadID ( void ) const
//{
//#ifndef __DIA_UNIT_TESTING__
//    eAppThreadID threadID = eThreadID_Unknown;
//
//    OSAL_tThreadID osalID = OSAL_ThreadWhoAmI();
//
//    for ( tU32 i=0; i<eThreadID_Count; i++ )
//    {
//        if ( mThreadInfoRep[i] && mThreadInfoRep[i]->mSysID == osalID )
//        {
//           threadID = mThreadInfoRep[i]->mID;
//            break;
//        }
//    }
//
//    return threadID;
//#else
//    return eThreadID_Unknown;
//#endif
//}

//-----------------------------------------------------------------------------

//eAppThreadID
//dia_getCurrentThreadID ( void )
//{
//#ifndef __DIA_UNIT_TESTING__
//    return (getInstanceOfAppController()->isSetup()) ? getInstanceOfAppController()->getCurrentThreadID() : eThreadID_Unknown;
//#else
//    return eThreadID_Unknown;
//#endif
//}

/*!
 *
 * This method is called to trigger an application state change. It is checked if a state change operation is really
 * required or if the requested state is already active.
 *
 * \param[in]  srcStateID     source state ID of the application change request
 * \param[in]  destStateID    destination state ID of the application change request
 *
 * \return     DIA_SUCCESS
 *
 */

tDiaResult
dia_AppController::changeApplicationState ( dia::ApplicationStateID srcStateID, dia::ApplicationStateID destStateID )
{
   ScopeTrace oTrace("dia_AppController::changeApplicationState");

   LockScope lock(syncObj);

   DIA_TR_SM("--- RECEIVED REQUEST TO CHANGE APPLICATION STATE FROM '%s' to '%s' !!!",strStateMapping[srcStateID].c_str(),strStateMapping[destStateID].c_str());

   bool isNewStateRequested = false;

   // Analyse the state we have to go to and select that state
   switch ( destStateID )
   {
   case DIA_C_U32_APP_STATE_NORMAL:
      {
         // --- Entered AppSate DIA_C_U32_APP_STATE_NORMAL ");
         DIA_TR_INF("Entered AppSate DIA_C_U32_APP_STATE_NORMAL");

         // Check the Old State if it was DIA_C_U32_APP_STATE_NORMAL nothing to do
         if ( mAppState /*srcStateID*/ != DIA_C_U32_APP_STATE_NORMAL )
         {
            // If no state switch is already in progress this is a NEW state change request
            // Do stuff we have to do when we switch to NORMAL
            if ( !mIsApplicationStateChangeInProgress ) isNewStateRequested = true;

#ifndef __DIA_UNIT_TESTING__
            diagnostics_tclApp::getInstance()->vServiceAvailabilityChanged(CCA_C_U16_SRV_DIAGDEBUG, AMT_C_U8_SVCSTATE_AVAILABLE);
#endif
         }
      }
      break;

   case DIA_C_U32_APP_STATE_PAUSE:
      {
         // --- Entered App state DIA_C_U32_APP_STATE_PAUSE
         DIA_TR_INF("Entered AppSate DIA_C_U32_APP_STATE_PAUSE");

         // Check the Old State if it was DIA_C_U32_APP_STATE_PAUSE nothing to do
         if ( mAppState /*srcStateID*/ != DIA_C_U32_APP_STATE_PAUSE )
         {
            // Do stuff we have to do when we switch to DIA_C_U32_APP_STATE_PAUSE
            if ( !mIsApplicationStateChangeInProgress ) isNewStateRequested = true;
         }
      }
      break;

   case DIA_C_U32_APP_STATE_OFF:
      {
         // --- Entered App State Statement DIA_C_U32_APP_STATE_OFF
         DIA_TR_INF("Entered AppSate DIA_C_U32_APP_STATE_OFF");

         // Check the Old State if it was DIA_C_U32_APP_STATE_OFF nothing to do
         if ( mAppState /*srcStateID*/ != DIA_C_U32_APP_STATE_OFF )
         {
            // Do stuff we have to do when we switch to DIA_C_U32_APP_STATE_OFF
            if ( !mIsApplicationStateChangeInProgress ) isNewStateRequested = true;
         }
      }
      break;

   default:
      {
         // Unable to find Initialized state..Entered Default
         DIA_TR_INF("Unable to find Initialized state..Entered Default");
      }
      break;
   }

   // (1) Check if the new state requested flag is set if it is then this is a new Sate Transition and we must
   //     inform all the client handlers of this chnage
   //
   // (2) If the new state flag is not set then check, whether a state change is already in progress
   //     if it is call the frame work and apply for more time in which to complete the state change
   //
   // (3) if not (1) or (2) then confirm with teh frame work what state we are in
   //
   if ( isNewStateRequested )
   {
      DIA_TR_SM("--- Starting to change application state ---");
      startApplicationStateChange(srcStateID, destStateID);
   }
   else if ( mIsApplicationStateChangeInProgress )
   {
      dia::IApplicationStateControlListener* pListener = 0;
      if ((queryListener<dia::IApplicationStateControlListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         pListener->onApplicationStateChangePending(srcStateID,destStateID);
      }
   }
   else
   {
      dia::IApplicationStateControlListener* pListener = 0;
      if ((queryListener<dia::IApplicationStateControlListener>(&pListener) == DIA_SUCCESS) && pListener)
      {
         pListener->onApplicationStateChanged(destStateID);
      }
   }

   return DIA_SUCCESS;
}

/*!
 *
 * This method is called to really change the state of the application. To change the application state a three
 * step approach is used similar to system settings. Every application state listener is informed that it should
 * prepare itself for the application state change. For example system adapters will rejects requests to communi-
 * cate with their associated application during an application state change. The preparation is executed in a
 * synchronous way. After preparation is done 2nd step is triggered by sending a message to the worker threads
 * queue.
 *
 * \param[in]  srcStateID     source state ID of the application change request
 * \param[in]  destStateID    destination state ID of the application change request
 *
 * \return     DIA_SUCCESS
 *
 */

//-----------------------------------------------------------------------------

void
dia_AppController::startApplicationStateChange ( dia::ApplicationStateID srcStateID, dia::ApplicationStateID destStateID )
{
   ScopeTrace oTrace("dia_AppController::startApplicationStateChange");

   LockScope lock(syncObj);

   // for backward compatibility we retain changing the state here as it was done before, but principally it should be
   // changed after the state change is completed
   mAppState = destStateID;
   // preserve the new application state
   mApplicationStateRequested = destStateID;
   // set a flag to indicate that we are now busy with performing the state change
   mIsApplicationStateChangeInProgress = true;

   // before rolling out the application state change we request application state listeners to prepare themselves, e.g.
   // system adapters will reject sending messages to other applications during a state change
   std::list<dia_IAppStateListener*>::iterator listIter = mAppStateListenerRep.begin();
   for ( ; listIter != mAppStateListenerRep.end(); listIter++ )
   {
      dia_IAppStateListener* pListener = (*listIter);
      if ( !pListener ) continue;
      mRemainingAppStateListenerRep.push_back(pListener);
      pListener->onApplicationStateChangePrepare(srcStateID,destStateID);
   }

   if ( mRemainingAppStateListenerRep.size() )
   {
      // execution of the application state change will be done in the context of the diagnostics worker thread
#ifndef __DIA_UNIT_TESTING__
      getInstanceOfApplication()->postMessage (
            DIA_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                  DIA_NEW dia_FunctorTwoArgsNoReturnValue< dia_AppController,dia::ApplicationStateID,dia::ApplicationStateID >(this,&dia_AppController::processApplicationStateChangeRequest,srcStateID,destStateID)
            )
      );
#endif
   }
   else
   {
      // we are done
      onApplicationStateChanged(destStateID);
   }
}

//-----------------------------------------------------------------------------

void
dia_AppController::processApplicationStateChangeRequest ( dia::ApplicationStateID srcStateID, dia::ApplicationStateID destStateID )
{
   ScopeTrace oTrace("dia_AppController::processApplicationStateChangeRequest()");

   LockScope lock(syncObj);

   std::list<dia_IAppStateListener*>::iterator listIter = mAppStateListenerRep.begin();
   for ( listIter=mAppStateListenerRep.begin(); listIter != mAppStateListenerRep.end(); listIter++ )
   {
      dia_IAppStateListener* pListener = (*listIter);
      if ( pListener ) pListener->onApplicationStateChange(srcStateID,destStateID);
   }
}

//-----------------------------------------------------------------------------

void
dia_AppController::onApplicationStateChanged ( dia::ApplicationStateID appStateID, dia_IAppStateListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::onApplicationStateChanged(appStateID,pListener)");

   LockScope lock(syncObj);

   DIA_TR_INF("--- RECEIVED APPSTATE NOTIFICATION FROM %p : STATE %d !!!",pListener,appStateID);

   if ( !pListener ) {
      DIA_TR_INF("dia_AppController::onApplicationStateChanged: INVALID LISTENER POINTER!!!!");
      return;
   }

   std::list<dia_IAppStateListener*>::iterator listIter = mRemainingAppStateListenerRep.begin();
   for ( ; listIter != mRemainingAppStateListenerRep.end(); listIter++ ) {
      if ( (*listIter) == pListener ) break;
   }

   if ( listIter != mRemainingAppStateListenerRep.end() ) {
      DIA_TR_INF("dia_tclAppController::onApplicationStateChanged: REMOVING APP_STATE_LISTENER FROM NOTIFICATION LIST !!!!");
      mRemainingAppStateListenerRep.erase(listIter);
   }

   // Check to see if all listeners have processed the requested transition
   if ( mRemainingAppStateListenerRep.size() == 0 )
   {
      onApplicationStateChanged(appStateID);
   }
   else
   {
      DIA_TR_INF("AppState change not yet completed (ID = 0x%02x)",appStateID);
   }
}

//-----------------------------------------------------------------------------

void
dia_AppController::onApplicationStateChanged ( dia::ApplicationStateID appStateID )
{
   ScopeTrace oTrace("dia_AppController::onApplicationStateChanged(appStateID)");

   LockScope lock(syncObj);

   DIA_TR_INF("AppState change completed (ID = 0x%02x)",appStateID);

   mIsApplicationStateChangeInProgress = false;
   std::list<dia_IAppStateListener*>::iterator listIter = mAppStateListenerRep.begin();
   for ( listIter=mAppStateListenerRep.begin(); listIter != mAppStateListenerRep.end(); ++listIter )
   {
      dia_IAppStateListener* pListener = (*listIter);
      if ( pListener ) pListener->onApplicationStateChangeFinalize(appStateID);
   }

   // set the switch in progress flag to FALSE

   dia::IApplicationStateControlListener* pListener = 0;
   if ((queryListener<dia::IApplicationStateControlListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      DIA_TR_INF("Notify about diagnostics AppState change to state 0x%02x",appStateID);
      pListener->onApplicationStateChanged(appStateID);
   }
   else
   {
      DIA_TR_INF("No listeners registered for AppState change");
   }

   switch ( appStateID )
   {
   case DIA_C_U32_APP_STATE_NORMAL:
      {
         vOnEvent(eAppCtrlEvent_StateNormalReached);
         (void) getInstanceOfTestController()->runTests(DIA_EN_TESTCONDITION_STARTUP_DIAGNOSIS_NORMAL_STATE);
      }
      break;

   case DIA_C_U32_APP_STATE_PAUSE:
      vOnEvent(eAppCtrlEvent_StatePauseReached);
      break;

   case DIA_C_U32_APP_STATE_OFF:
      vOnEvent(eAppCtrlEvent_StateOffReached);
      break;

   default:
      break;
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_AppController::addAppStateListener ( dia_IAppStateListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::addAppStateListener()");

   LockScope lock(syncObj);

   if ( !pListener ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::list<dia_IAppStateListener*>::iterator listIter = mAppStateListenerRep.begin();
   for ( ; listIter != mAppStateListenerRep.end(); listIter++ )
   {
      if ( (*listIter) == pListener ) break;
   }

   if ( listIter == mAppStateListenerRep.end() )
   {
      mAppStateListenerRep.push_back(pListener);
      retCode = DIA_SUCCESS;
   }

    return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_AppController::removeAppStateListener ( const dia_IAppStateListener* pListener )
{
   ScopeTrace oTrace("dia_AppController::removeAppStateListener()");

   LockScope lock(syncObj);

    tDiaResult retCode = DIA_E_NOT_FOUND;

    std::list<dia_IAppStateListener*>::iterator listIter = mAppStateListenerRep.begin();
    for ( ; listIter != mAppStateListenerRep.end(); listIter++ )
    {
        if ( (*listIter) == pListener )
        {
            mAppStateListenerRep.erase(listIter);
            retCode = DIA_SUCCESS;
            break;
        }
    }

    return retCode;
}

//-----------------------------------------------------------------------------

void
dia_AppController::destroy ( void )
{
// @TODO
//   std::list<dia_Command*>::iterator listIter = mCommandRep.begin();
//   for ( ; listIter != mCommandRep.end(); listIter++ ) OSAL_DELETE (*listIter);
//   mCommandRep.clear();
//
//   std::map<dia_enCmdCtrlPluginType,dia_CommandControllerPlugin*>::iterator mapIter = mPluginRep.begin();
//   for ( ; mapIter != mPluginRep.end(); mapIter++ ) OSAL_DELETE mapIter->second;
//   mPluginRep.clear();
}

