/*
 * dia_ActiveObject.cpp
 *
 *  Created on: 22.05.2012
 *      Author: gib2hi
 */

#include "common/framework/application/dia_ActiveObject.h"

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include "common/framework/application/dia_ThreadMonitor.h"
#endif

using namespace dia;

//-----------------------------------------------------------------------------

dia_ActiveObject::dia_ActiveObject ( tCString name, tU32 propPrio, tU32 propStackSize )
   : mThreadName(name),
     mPropThreadPrio(propPrio),
     mPropThreadStackSize(propStackSize),
#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
     mThreadID(0)
#else
     mThreadID(OSAL_ERROR)
#endif
{
#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   int retVal = pthread_attr_init(&mThreadAttr); // Coverity Fix(CID:139743)
   DIA_ASSERT(0 == retVal);
   DIA_TR_INF("dia_ActiveObject::dia_ActiveObject pthread_attr_init returned %d", retVal);

   struct sched_param schedParams;
   retVal = pthread_attr_getschedparam(&mThreadAttr, &schedParams);
   if (0!=retVal)    DIA_TR_ERR( "### ERROR pthread_attr_getschedparam returned %d", retVal);
   DIA_TR_INF("dia_ActiveObject::dia_ActiveObject sched param = %d", schedParams.sched_priority );

   size_t actStackSize = 0;
   retVal = pthread_attr_getstacksize(&mThreadAttr, &actStackSize);
   if (0!=retVal)    DIA_TR_ERR( "pthread_attr_getstacksize returned %d", retVal);
   DIA_TR_INF("dia_ActiveObject::dia_ActiveObject actStackSize = %zu", actStackSize );
#endif
}

//-----------------------------------------------------------------------------

dia_ActiveObject::~dia_ActiveObject ( void )
{
//   mName = 0;
#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   DIA_TR_INF("dia_ActiveObject::~dia_ActiveObject pthread_attr_destroy");
   DIA_ASSERT(0==pthread_attr_destroy(&mThreadAttr));
#endif
   mThreadName = 0;
}

//-----------------------------------------------------------------------------

#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
void*
#else
void
#endif
dia_ActiveObject::vThreadEntrypoint ( void* pActiveObj )
{
   ScopeTrace trc("dia_ActiveObject::vThreadEntrypoint");

   if ( pActiveObj )
   {
      dia_ActiveObject* pObj = (dia_ActiveObject*) pActiveObj;

      DIA_TR_INF("dia_ActiveObject::vThreadEntrypoint(name = %s, pObject = %p)",pObj->mThreadName,pObj);

      registerThread(pObj->mThreadName);
      pObj->vThreadInitialize();
      pObj->vThreadEntrypointObject();
      pObj->vThreadFinalize();
   }

#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   return 0;
#endif
}

//-----------------------------------------------------------------------------

void
dia_ActiveObject::vThreadInitialize ( void )
{
   ScopeTrace trc("dia_ActiveObject::vThreadInitialize");
}

//-----------------------------------------------------------------------------

void
dia_ActiveObject:: vThreadFinalize ( void )
{
   ScopeTrace trc("dia_ActiveObject::vThreadFinalize");
   // Now everything cleared up --- Exit this thread of execution
#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   pthread_exit(NULL);
#else
   OSAL_vThreadExit();
#endif
}

//-----------------------------------------------------------------------------



tDiaResult
dia_ActiveObject::startThread ( void )
{
   ScopeTrace trc("dia_ActiveObject::startThread");

   tU32 threadPrio = getThreadPriority();
   tU32 stackSize  = getStackSize();

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   OSAL_trThreadAttribute rThAttr = {
         const_cast<tString>(mThreadName),
         threadPrio,
         static_cast<tS32>(stackSize),
         dia_ActiveObject::vThreadEntrypoint,
         (void*) this
   };

    // And spawn it from the OS
   mThreadID = OSAL_ThreadSpawn(&rThAttr);

   DIA_TR_INF("dia_ActiveObject::startThread OSAL_ThreadSpawn returned %s", (mThreadID != OSAL_ERROR) ? "DIA_SUCCESS" : "DIA_FAILED" );

   return ( mThreadID != OSAL_ERROR ) ? DIA_SUCCESS : DIA_FAILED;
#else
   struct sched_param schedParams;
   size_t actStackSize = 0;
   size_t expStackSize = (size_t) stackSize;
   int retVal;
   //set stack size
   retVal = pthread_attr_setstacksize(&mThreadAttr, expStackSize);
   if (0!=retVal)    DIA_TR_ERR( "pthread_attr_setstacksize returned %d", retVal);

   //set schedule priority
   schedParams.sched_priority = (int) threadPrio;
   retVal = pthread_attr_getschedparam(&mThreadAttr, &schedParams);
   if (0!=retVal)    DIA_TR_ERR( "pthread_attr_getschedparam returned %d", retVal);

   //check if stack size setting was successful
   retVal = pthread_attr_getstacksize(&mThreadAttr, &actStackSize);
   if (0!=retVal)    DIA_TR_ERR( "pthread_attr_getstacksize returned %d", retVal);

   DIA_TR_INF( "dia_ActiveObject::startThread expStackSize=%zu actStackSize=%zu schedParams.sched_priority=%d", expStackSize, actStackSize, schedParams.sched_priority);

   if (expStackSize!=actStackSize)
   {
      DIA_TR_ERR( "#### dia_ActiveObject::startThread Set size of stack failed ####");
   }

   int retCode = pthread_create(&mThreadID, &mThreadAttr, &(dia_ActiveObject::vThreadEntrypoint), this);

   if (0!=retCode)
   {
      //the contents of mThreadID are undefined in case of failure
      mThreadID = 0;

      DIA_TR_ERR("#### pthread_create returned %d ####", retCode);
   }
   else
   {
      DIA_TR_INF("Thread created with mThreadID=0x%08X", (uint32_t)mThreadID);
   }
   return ( retCode == 0 ) ? DIA_SUCCESS : DIA_FAILED;
#endif

}

//-----------------------------------------------------------------------------

tDiaResult
dia_ActiveObject::terminateThread ( void )
{
   DIA_TR_INF( "dia_ActiveObject::terminateThread mThreadID=0x%08X", (uint32_t)mThreadID);

#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   if (0!=mThreadID)
   {
      int ret = pthread_join(mThreadID,NULL);
      if (0!=ret)
      {
         DIA_TR_ERR( "##### ERROR #####  pthread_join returned %d", ret);
         DIA_ASSERT_ALWAYS();
      }
      else
      {
         DIA_TR_INF( "dia_ActiveObject::terminateThread mThreadID=0x%08X pthread_join OK", (uint32_t)mThreadID);
      }

      mThreadID = 0;
   }
#endif

   return DIA_SUCCESS;
}


//-----------------------------------------------------------------------------

tU32
dia_ActiveObject::getThreadPriority ( void ) const
{
   tU32 threadPrio = DIA_C_U32_PLATFORM_DEFAULT_THREAD_PRIORITY;

   if ( mPropThreadPrio != DIA_PROP_INVALID_IDENTIFIER )
   {
      if ( dia_getProperty(mPropThreadPrio,threadPrio) != DIA_SUCCESS )
      {
         threadPrio = DIA_C_U32_PLATFORM_DEFAULT_THREAD_PRIORITY;
         DIA_TR_INF( "### UNABLE TO RETRIEVE THREAD PRIORITY. ASSUMING PLATFORM DEFAULT VALUE (0x%08x) ###",threadPrio);
      }
   }

   DIA_TR_INF("dia_ActiveObject::getThreadPriority: Thread Priority = %u (TID = 0x%08x)", threadPrio, (uint32_t)mThreadID);

   return threadPrio;
}

//-----------------------------------------------------------------------------

tU32
dia_ActiveObject::getStackSize ( void ) const
{
   tU32 stackSize = DIA_C_U32_PLATFORM_DEFAULT_THREAD_STACK_SIZE;

   if ( mPropThreadStackSize != DIA_PROP_INVALID_IDENTIFIER )
   {
      if ( dia_getProperty(mPropThreadStackSize,stackSize) != DIA_SUCCESS )
      {
         stackSize = DIA_C_U32_PLATFORM_DEFAULT_THREAD_STACK_SIZE;
         DIA_TR_INF( "### UNABLE TO RETRIEVE THREAD STACK SIZE. ASSUMING PLATFORM DEFAULT VALUE (0x%08x) ###",stackSize);
      }
   }

   DIA_TR_INF("dia_ActiveObject::getStackSize: Thread Stack Size = %u (TID = 0x%08x)", stackSize, (uint32_t)mThreadID);

   return stackSize;
}

