/************************************************************************
* FILE:         tclLock.h
* PROJECT:      Ford MFD
* SW-COMPONENT: diaglib
*----------------------------------------------------------------------
*
* DESCRIPTION:  lock synchronisation utility class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2006 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                     | Modification
* 28.03.08  | gib2hi                     | initial version
* 01.04.09  | eva2hi                     | adapted for use in diaglib
*
*************************************************************************/

#ifndef DIAGLIB_LOCK_H
#define DIAGLIB_LOCK_H

#ifndef __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_MEMORY
#include "osal_if.h" //lint !e451 Warning: Header file repeatedly included but does not have a standard include guard
#endif

#include <string>

using namespace std;

/**
* Lock utility class.
* @note
* Implements lock behaviour in terms of an underlying Signal object.
*
* Example: Implementing a critical section.
* @code
*   // class containing critical section code
*   Class A
*   {
*       Lock l_;
*       int var_;
*   public:
*       A() : l_( Lock::eUnlocked, "critical section a" ), var_(0)
*       {
*       }
*
*       void IncVar()
*       {
*           l_.s32Lock();
*           ++var_;
*           l_.bUnlock();
*       }
*
*       void DecVar()
*       {
*           l_.s32Lock();
*           --var_;
*           l_.bUnlock();
*       }
*   };
* @endcode
*
* Counted semaphore wrapper class.
* Abstracts the functionality of a semaphore and implements
* it in terms of the underlying OS/API.
*
* @see
* http://www.cs.utexas.edu/users/EWD/transcriptions/EWD00xx/EWD54.html
*
* @todo
* - Fix semaphore max count, not implemented in OSAL call.
*/

namespace diaglib {

class tclLock
{
public:
   //! Creation type manifest constants.
    enum eLockState { eLocked, eUnlocked };

    //! class constructor, created locked or unlocked (default) and with (an optional) name
    tclLock (tCString name, eLockState mode = eUnlocked);
    //! class destructor
    virtual ~tclLock ( tVoid );

    //! unlock the synchronization object
    virtual tBool bUnlock ( tVoid );
    //! lock the synchronization object
    virtual tS32  s32Lock ( tU32 timeout = OSAL_C_TIMEOUT_FOREVER );  //lint !e1735: checked that default param is equal to base class default param

    // Do not use default constructor!!! Only public because of lint
    tclLock(): mName(""), mHandle(OSAL_C_INVALID_HANDLE) { NORMAL_M_ASSERT_ALWAYS(); };

protected:
   //! deprecated default constructor (no implementation)
   //! deprecated copy constructor (no implementation)
   tclLock( const tclLock& );
   //! deprecated assignment operator (no implementation)
   tclLock& operator = ( const tclLock& );

   //! name of the lock object
   string  mName;
   //! handle of the lock object
    OSAL_tSemHandle mHandle;
};
}
#endif /*DIAGLIB_LOCK_H*/
