/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Interface Manager - Diagnosis Library - Introduced in MFD Nav
*
* \todo add comments (doxygen)
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif
#define DIAGLIB_FILE_NUMBER F_DIAGLIB_IF_MANAGER

#ifndef __INCLUDED_DIAGLIB_COMMON__
#include "diaglib_common.h"
#endif

#define ET_TRACE_INFO_ON
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifndef __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_MEMORY
#include "osal_if.h"
#endif


#define DIAGLIB_STATIC_IF_DEFINES
#include "tclInterfaceManager.h"
#include "tclServiceHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CCA
#include "CCA/tclServiceHandlerCCA.h"
#include "CCA/tclCCAServiceProxy.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CSFW
#include "CSFW/tclServiceHandlerCSFW.h"
#include "CSFW/tclCSFWServiceProxy.h"
#endif

#include "handler/tclRoutineControlHandler.h"
#include "handler/tclDiagDataHandler.h"
#include "handler/tclIoControlHandler.h"
#include "handler/tclSystemSetHandler.h"
#include "tclDiagnosisIF.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ET_DIAGLIB_GENERAL
#include "trcGenProj/Header/tclInterfaceManager.cpp.trc.h"
#endif

#include "Trace.h"

namespace diaglib {

/***********************************************************************//**
 * \brief Constructor
 *
 * Initializes all members with default values
 * 
 * \return n/a
 *//************************************************************************/
tclInterfaceManager::tclInterfaceManager ()
{
   diaglib_FuncTrace oTrace(I_DIAGLIB_F_CONSTRUCTOR);

   poServiceHandler        = OSAL_NULL;
   poRoutineControlHandler = OSAL_NULL;
   poDiagDataHandler       = OSAL_NULL;
   poIoControlHandler      = OSAL_NULL;
   poSystemSetHandler      = OSAL_NULL;

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CSFW
   poCSFWServiceProxy      = OSAL_NULL;
#endif
#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CCA
   poCCAServiceProxy       = OSAL_NULL;
#endif

   enProtocol = EN_PROTOCOL_UNDEFINED;
   bDeleted = FALSE;
}

/***********************************************************************//**
 * \brief Initialisation of InterfaceManager
 *
 * Sets up the whole environment for one component. This component will then
 * be in the responsibility of this InterfaceManager object until it deinits
 * the diaglib using the Factory. But it is a 1 to 1 relationship between
 * InterfaceManager and Component.
 *
 * \param[in] enRequestedProtocol Enumerations stating the protocol o be used
 * \param[in] poApplicationContext Void Pointer to application of component
 * 
 * \return none
 *//************************************************************************/
tVoid tclInterfaceManager::vInit (
                                    tenSupportedProtocols enRequestedProtocol,
                                    tVoid* poApplicationContext
                                 )
{
   diaglib_FuncTrace oTrace(I_DIAGLIB_F_INIT);
   enProtocol = enRequestedProtocol;

   if(OSAL_NULL == poApplicationContext)
   {
      ETG_TRACE_USR2_THR(( "!!! tclInterfaceManager::tclvInitServiceDiaglib => ERROR: OSAL_NULL == poApplicationContext"));
      //DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
   }

   /*
   Create the correct service handler depending on the requested protocol.
   */
   switch(enProtocol)
   {
#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CCA
      case EN_PROTOCOL_CCA:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CCA_FRAMEWORK_REQUESTED);
         poServiceHandler = OSAL_NEW tclServiceHandlerCCA(poApplicationContext);
      break;
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CSFW
      case EN_PROTOCOL_CSFW:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CSFW_FRAMEWORK_REQUESTED);
         poServiceHandler = OSAL_NEW tclServiceHandlerCSFW(poApplicationContext);
      break;
#endif

      default:
         ETG_TRACE_ERR_THR(( "!!! tclInterfaceManager::tclvInitServiceDiaglib => ERROR: UNKNOWN_PROTOCOL_REQUESTED"));
         //DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNKNOWN_PROTOCOL_REQUESTED );
      return;
   }

   if(OSAL_NULL == poServiceHandler)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_CANT_ALLOC_MEMORY_ON_HEAP );
      FATAL_M_ASSERT_ALWAYS();
   }
}

/***********************************************************************//**
 * \brief Destructor
 *
 * Tests if all members have already been invalidated. If not it calls the
 * function that does so and traces out a warning.
 *
 * \return n/a
 *//************************************************************************/
tclInterfaceManager::~tclInterfaceManager()
{
    _BP_TRY_BEGIN
    {
        vDeleteServiceHandlerAndInterfaces();

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CCA
        poCCAServiceProxy = OSAL_NULL;
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CSFW
        poCSFWServiceProxy = OSAL_NULL;
#endif

        poRoutineControlHandler = OSAL_NULL;
        poDiagDataHandler = OSAL_NULL;
        poIoControlHandler = OSAL_NULL;
        poSystemSetHandler = OSAL_NULL;
        DIAGLIB_DELETE_IF_NOT_NULL(poServiceHandler); // Lint does not understand that it was deleted before.
    }
    _BP_CATCH_ALL
    {

    }
    _BP_CATCH_END
}

/***********************************************************************//**
 * \brief Returns the main interface for calling component
 *
 * If framework has not been deleted, this function returns a pointer to the
 * main interface. This happens to be the InterfaceManager object, so we
 * return the this pointer.
 *
 * \return tclDiagnosisIF* Pointer to main interface
 *//************************************************************************/
tclDiagnosisIF* tclInterfaceManager::poGetMainInterface()
{
   diaglib_FuncTrace oTrace(I_DIAGLIB_F_GETMAININTERFACE);

   if(TRUE == bDeleted)
   {
      ETG_TRACE_ERR_THR(( "!!! tclInterfaceManager::poGetMainInterface => ERROR: FRAMEWORK_HAS_BEEN_DELETED"));
      return OSAL_NULL;
   }

   return this;
}

/***********************************************************************//**
 * \brief Returns the requested interface for the calling component
 *
 * If framework has not been deleted, this function returns a pointer to the
 * requested interface. If the implementation of the requested interface is
 * not yet existing it will be created. The interface manager is actually the
 * owner of all interfaces. So he is the one that creates and deletes them.
 *
 * \param[in] enInterfaceId Enumeration stating the requested interface
 *
 * \return tclDiagnosisIF* Pointer to requested interface. 
 *//************************************************************************/
tclDiagnosisIF* tclInterfaceManager::poGetInterface(tenDiagHandlerInterfaces enInterfaceId)
{
   tclDiagnosisIF* poInterface = OSAL_NULL;

   diaglib_FuncTrace oTrace(I_DIAGLIB_F_GETINTERFACE);

   /*
   It is an error condition if this is called when the framework has been deleted, or
   when the pointer to the service handler is invalid.
   */
   if(TRUE == bDeleted)
   {
      ETG_TRACE_ERR_THR(( "!!! tclInterfaceManager::poGetInterface => ERROR: FRAMEWORK_HAS_BEEN_DELETED"));
      return OSAL_NULL;
   }

   if(OSAL_NULL == poServiceHandler)
   {
      ETG_TRACE_ERR_THR(( "!!! tclInterfaceManager::poGetInterface => ERROR: OSAL_NULL == poServiceHandler"));
      //DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return OSAL_NULL;
   }

   /*
   Further action depending on requested interface. 
      *  If it is a protocol interface, we check if this framework uses the requested protocol. If
         not, this is an error condition.
      *  If it is a diagnosis interface, we just check if the interface implementation (handler)
         already exists. If not we create it
      *  An unknown interface is an error condition
   */
   switch(enInterfaceId)
   {
      case EN_INTERFACE_DIAGNOSIS:
         poInterface = poGetMainInterface();
      break;

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CCA
      case EN_INTERFACE_CCA_SERVICE:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CCA_INTERFACE_REQUESTED);
         if(EN_PROTOCOL_CCA != enProtocol)
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_INTERFACE_UNSUPPORTED );
            return OSAL_NULL;
         }

         if(OSAL_NULL == poCCAServiceProxy)
         {
            poCCAServiceProxy = OSAL_NEW tclCCAServiceProxy( dynamic_cast<tclServiceHandlerCCA*>(poServiceHandler), this );
         }

         poInterface = poCCAServiceProxy;
      break;
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CSFW
      case EN_INTERFACE_CSFW_SERVICE:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CSFW_INTERFACE_REQUESTED);
         if(EN_PROTOCOL_CSFW != enProtocol)
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_INTERFACE_UNSUPPORTED);
            return OSAL_NULL;
         }

         if(OSAL_NULL == poCSFWServiceProxy)
         {
            poCSFWServiceProxy = OSAL_NEW tclCSFWServiceProxy( static_cast<tclServiceHandlerCSFW*>(poServiceHandler), this );
         }

         poInterface = poCSFWServiceProxy;
      break;
#endif

      case EN_INTERFACE_SYSTEMSET:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_SYSTEMSET_INTERFACE_REQUESTED );
         if(OSAL_NULL == poSystemSetHandler)
         {
            poSystemSetHandler = OSAL_NEW tclSystemSetHandler( poServiceHandler, this );
         }

         poInterface = poSystemSetHandler;
      break;

      case EN_INTERFACE_IO_CONTROL:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_IOCTRL_INTERFACE_REQUESTED);
         if(OSAL_NULL == poIoControlHandler)
         {
            poIoControlHandler = OSAL_NEW tclIoControlHandler( poServiceHandler, this );
         }

         poInterface = poIoControlHandler;
      break;

      case EN_INTERFACE_ROUTINE_CONTROL:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ROUTINECTRL_INTERFACE_REQUESTED);
         if(OSAL_NULL == poRoutineControlHandler)
         {
            poRoutineControlHandler = OSAL_NEW tclRoutineControlHandler( poServiceHandler, this );
         }

         poInterface = poRoutineControlHandler;
      break;

      case EN_INTERFACE_DIAGNOSIS_DATA:
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_DIAGDATA_INTERFACE_REQUESTED);
         if(OSAL_NULL == poDiagDataHandler)
         {
            poDiagDataHandler = OSAL_NEW tclDiagDataHandler( poServiceHandler, this );
         }

         poInterface = poDiagDataHandler;
      break;

      default:
         DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNKNOWN_INTERFACE_REQUESTED );
         return OSAL_NULL;
   }

   /*
   This is definetly a FATAL, Pointer can only be NULL when a "new" command failed.
   */
   if(OSAL_NULL == poInterface)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_CANT_ALLOC_MEMORY_ON_HEAP);
      FATAL_M_ASSERT_ALWAYS();
   }

   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATED_REQUESTED_INTERFACE );

   return poInterface;
}

/***********************************************************************//**
 * \brief Frees and invalidates all owned object-pointer
 *
 * Checks every owned pointer if valid and deletes it. Then all pointers
 * are invalidated.
 *
 * \return none
 *//************************************************************************/
tVoid tclInterfaceManager::vDeleteServiceHandlerAndInterfaces()
{
   diaglib_FuncTrace oTrace(I_DIAGLIB_F_DELSERVICEHANDLERANDIF);

   if(TRUE == bDeleted)
   {
      ETG_TRACE_ERR_THR(( "!!! tclInterfaceManager::vDeleteServiceHandlerAndInterfaces => ERROR: FRAMEWORK_HAS_BEEN_DELETED"));
      //DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_FRAMEWORK_HAS_BEEN_DELETED );
      return;
   }

   bDeleted = TRUE;

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CCA
   DIAGLIB_DELETE_IF_NOT_NULL(poCCAServiceProxy);
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAGLIB_CSFW
   DIAGLIB_DELETE_IF_NOT_NULL(poCSFWServiceProxy);
#endif
   DIAGLIB_DELETE_IF_NOT_NULL(poRoutineControlHandler);
   DIAGLIB_DELETE_IF_NOT_NULL(poDiagDataHandler);
   DIAGLIB_DELETE_IF_NOT_NULL(poIoControlHandler);
   DIAGLIB_DELETE_IF_NOT_NULL(poSystemSetHandler);
   DIAGLIB_DELETE_IF_NOT_NULL(poServiceHandler);

   enProtocol = EN_PROTOCOL_UNDEFINED;

   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_FRAMEWORK_INSTANCE_DELETED);

   return;
}

}
