/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Diagnosis Interface Factory - Diagnosis library - Introduced in MFD Nav
*
* (c) 2008 Blaupunkt Werke GmbH
* \Todo 2008-07-10 Add documentation, sequence diagrams ...
*//****************************************************************************/

#ifndef DIAGLIB_DIAGNOSIS_INTERFACE_FACTORY_H
#define DIAGLIB_DIAGNOSIS_INTERFACE_FACTORY_H

#ifndef __INCLUDED_DIAGLIB_COMMON__
#include "diaglib_common.h"
#endif

#include "Enumerations.h"  //lint !e537 Info 537 , eliminating lint warnings Repeated include file


namespace diaglib {

class tclDiagnosisIF;   //lint !e830 Info 830 , eliminating lint warnings Location cited in prior message

/**************************************************************************//**
* \brief Diagnosis Interface Factory
*
* The Diagnosis Interface Factory is the "entry point" to the library for
* components that need to communicate with diagnosis. It is implemented as a
* singleton, so first thing a component needs to do is to get a handle on the
* instance. After this has been done one can create a DiagnosisInterface to
* establish communication with diagnosis. Through this interface the component
* can get themselfes additional interface, which can be used for different
* purposes. Deletion of an once created communication "framework" also works
* through this factory.
*
* \ingroup        diaglibAPI
*//****************************************************************************/
class tclDiagnosisInterfaceFactory
{
private:
   static tclDiagnosisInterfaceFactory* poInstance;

protected:
   // Internal use
   tclDiagnosisInterfaceFactory();
   virtual ~tclDiagnosisInterfaceFactory();

public:
   /***********************************************************************//**
   * \brief Retrieve instance of this singleton class
   *
   * Returns the singleton instance of this factory and creates it if it did
   * not exist before.
   *
   * \return         Pointer to singleton instance of this factory.
   *//*************************************************************************/
   static tclDiagnosisInterfaceFactory* poGetInstance();

   /***********************************************************************//**
   * \brief Establish communication and return interface
   *
   * Call this function once per component that has to provide diagnostic
   * services. A pointer to the main interface of the exclusively for this
   * component created framework is returned. See the following sequence diagram
   * to get a better picture of the initialization of the framework:
   * \image html seqInitFramework.png
   *
   * \param[in]      enProtocol  Protocol identifier of protocol that should be
   *                 used.
   * \param[in]      poApplicationContext Pointer to application under which
   *                 the internal service should be executed. Must be of type
   *
   *
   * \return         Pointer to main interface.
   *//*************************************************************************/
   tclDiagnosisIF* poCreateDiagnosisInterface(tenSupportedProtocols enProtocol, tVoid* poApplicationContext) const;

   /***********************************************************************//**
   * \brief Destroy communication and free memory used by framework
   *
   * Call this function once per component shortly before going down. It will
   * free all memory ressources used by comunication framework for exactly this
   * one component. Use any interface that was issued to your component as a
   * parameter to this function. <b> And most important: Don't use any of your
   * interface pointers after calling this function, as they will be pointing
   * to freed memory. </b> So only call this function immediately before your
   * component is going to stop anyway. See the following sequence diagram
   * to get a better picture of the deletion process of the framework:
   * \image html seqDelFramework.png
   *
   * \param[in]      poInterface  One of the interfaces issued to you by diaglib
   *
   * \return         boolean that indicates if deleting was successful.
   *//*************************************************************************/
   tBool bDeleteConnectionAndFreeInterfaces(tclDiagnosisIF* poInterface) const;

   /***********************************************************************//**
   * \brief Get the version of the diaglib
   *
   * Returns the version of the diaglib implementation.
   *
   * \param[out]     u8MajorVersion Major version of diaglib
   * \param[out]     u8MinorVersion Minor version of diaglib
   *
   * \return         boolean that indicates if valid version number is provided
   *//*************************************************************************/
   tBool bGetVersion(tU8& u8MajorVersion, tU8& u8MinorVersion) const;
};
}
#endif // DIAGLIB_DIAGNOSIS_INTERFACE_FACTORY_H


