/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Service Handler CCA - Diagnosis Library - Introduced in MFD Nav
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif
#define DIAGLIB_FILE_NUMBER F_DIAGLIB_CCA_SERVICE_HANDLER

#ifndef __INCLUDED_DIAGLIB_COMMON__
#define __INCLUDED_DIAGLIB_COMMON__
#include "../diaglib_common.h"
#endif

#define ET_TRACE_INFO_ON
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard



// MIDW_FI DEFINES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_FUNCTIONIDS

// FI DEFINES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

// If the Service ID is not defined in ccaservice.h use the default one
#ifndef CCA_C_U16_SRV_DIAGLIB
#define CCA_C_U16_SRV_DIAGLIB 0x0071
#endif

#include "tclCCAHelper.h"
#include "../Enumerations.h"
#include "tclServiceHandlerCCA.h"
#include "../tclMessageContextManager.h"
#include "tclMessageContextCCA.h"
#include "../tclRoutineControlListenerIF.h"

#include "../Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ET_DIAGLIB_ROUTINECONTROL
#include "trcGenProj/Header/tclRoutineControlCCA.cpp.trc.h"
#endif

namespace diaglib {

midw_fi_tcl_e8_RoutineStatus RoutineStatus2Fi(tenRoutineCtrlResult enRoutineStatus)
{
   midw_fi_tcl_e8_RoutineStatus result;
   switch (enRoutineStatus)
   {
      case EN_ROUTINE_CONTROL_OK:
         result.enType = midw_fi_tcl_e8_RoutineStatus::FI_EN_ROUTINE_CONTROL_OK;
         break;
      case EN_ROUTINE_CONTROL_NOT_OK:
         result.enType = midw_fi_tcl_e8_RoutineStatus::FI_EN_ROUTINE_CONTROL_NOT_OK;
         break;
      default:
         result.enType = midw_fi_tcl_e8_RoutineStatus::FI_EN_ROUTINE_CONTROL_NOT_OK;
         DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_ROUTINECONTROL, E_DIAGLIB_UNEXPECTED_ENUM_VALUE);
         break;
   }
   return result;
}

/***********************************************************************//**
 * \brief Send routine control MethodResult Message
 *
 * Construct and send CCA message by doing the following steps:
 * \li Retrieve the message context using the provided Context pointer
 * \li Extract messaging data out of the MessageContext
 * \li Convert diaglib internal types to CCA specific midw_fi types
 * \li Put everything into a CCA message object
 * \li Send it using the application message send functionality
 * As there is no need to report success or failure on a finish message, only
 * the MessageContext handle is a parameter to this message.
 *
 * \param[in] enRoutineStatus - Status of routine control request
 * \param[in] oRoutineResult - Result list of routine control or error code
 * \param[in] poMessageContext - MessageContext handle of incoming message
 *
 * \return  boolean if sending of response was successful or failed
 *//************************************************************************/
tBool tclServiceHandlerCCA::bSendRoutineCtrlMethodResult (
                                                tenRoutineCtrlResult enRoutineStatus,
                                                const tclParameterVector& oRoutineResult,
                                                tContext context
                                              )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_ROUTINE_CTRL_MR);

   midw_diaglibfi_tclMsgGetRoutineResultMethodResult oGRRMethordResult;
   midw_diaglibfi_tclMsgRoutineControlMethodResult oRCMethordResult;
   fi_tclVisitorMessage* poResultMsg = OSAL_NULL;

   tclMessageContextCCA* poMessageContextCca = OSAL_NULL;
   tBool bResult = TRUE;

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_GET_BACK_CTXT_FOR_HANDLE , static_cast<tU32>(context) );

   if(
       tclMessageContextManager::bGetContextBack<tclMessageContextCCA>(context, &poMessageContextCca) == FALSE ||
       tclMessageContext::EN_MESSAGE_ROUTINE_CONTROL != poMessageContextCca->getMessageGroup()
     )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_ROUTINECONTROL, E_DIAGLIB_CANT_SEND_MSG_DUE_TO_CTXT);
      bResult = FALSE;
   }
   else
   {
      // Fill the CCA result message
      if(poMessageContextCca->u16GetFunctionId() == MIDW_DIAGLIBFI_C_U16_ROUTINECONTROL)
      {
         oRCMethordResult.RoutineId                = poMessageContextCca->getId();         
         oRCMethordResult.RoutineStatus = RoutineStatus2Fi(enRoutineStatus);
         tclCCAHelper::vFillCcaValueListFromVector(oRoutineResult, oRCMethordResult.RoutineResult );

         poResultMsg = OSAL_NEW fi_tclVisitorMessage( oRCMethordResult );
      }
      else if(poMessageContextCca->u16GetFunctionId() == MIDW_DIAGLIBFI_C_U16_GETROUTINERESULT)
      {
         oGRRMethordResult.RoutineId                = poMessageContextCca->getId();
         oGRRMethordResult.RoutineStatus     = RoutineStatus2Fi(enRoutineStatus);
         tclCCAHelper::vFillCcaValueListFromVector(oRoutineResult, oGRRMethordResult.RoutineResult );

         poResultMsg = OSAL_NEW fi_tclVisitorMessage( oGRRMethordResult );
      }
      else
      {
         DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_ROUTINECONTROL, E_DIAGLIB_MSG_CONTEXT_ERROR);
         bResult = FALSE;
      }

      if(OSAL_NULL == poResultMsg)
      {
         DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_ROUTINECONTROL, E_DIAGLIB_CANT_ALLOC_MEMORY_ON_HEAP);
         FATAL_M_ASSERT_ALWAYS();
         bResult = FALSE; // lint sucks
      }

      if(bResult == TRUE && poResultMsg != OSAL_NULL)
      {
         tclCCAHelper::vInitMethodResultServiceData((*poResultMsg), poMessageContextCca, u16OurAppId);

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_SENDING_MESSAGE_ID, poMessageContextCca->getId());
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_SENDING_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));
         bResult = bSendMessage((*poResultMsg));
      }
      else
      {
         bResult = FALSE;
      }
   }

   if(OSAL_NULL != poResultMsg)
      OSAL_DELETE poResultMsg;

   if(OSAL_NULL != poMessageContextCca)
      OSAL_DELETE poMessageContextCca;

   return bResult;
}

/***********************************************************************//**
 * \brief Send Get routine result MethodResult Message
 *
 * Construct and send CCA message by doing the following steps:
 * \li Retrieve the message context using the provided Context pointer
 * \li Extract messaging data out of the MessageContext
 * \li Convert diaglib internal types to CCA specific midw_fi types
 * \li Put everything into a CCA message object
 * \li Send it using the application message send functionality
 * As there is no need to report success or failure on a finish message, only
 * the MessageContext handle is a parameter to this message.
 *
 * \param[in] enRoutineStatus - Status of routine control request
 * \param[in] oRoutineResult - List of routine result or error code
 * \param[in] poMessageContext - MessageContext handle of incoming message
 *
 * \return  boolean if sending of response was successful or failed
 *//************************************************************************/
tBool tclServiceHandlerCCA::bSendGetRoutineResultMethodResult (
                                                     tenRoutineCtrlResult enRoutineStatus,
                                                     const tclParameterVector& oRoutineResult,
                                                     tContext context
                                                   )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_GET_ROUTINE_RESULT_MR);
   return bSendRoutineCtrlMethodResult(enRoutineStatus, oRoutineResult, context);
}


/***********************************************************************//**
 * \brief Register the routine control listener
 *
 * Register the pointer to the diaglib INTERNAL listener for routine control
 * requests.
 *
 * \param[in] poListener - Pointer to Listener
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vRegisterRoutineControlListener (
                                                   tclRoutineControlListenerIF* poListener
                                                 )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_REGISTER_ROUTINE_CTRL_LISTENER);
   if(OSAL_NULL == poListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   poRoutineControlListener = poListener;

}

/***********************************************************************//**
 * \brief Handles incoming Routine Control messages
 *
 * On incoming request the following steps are executed
 * \li Check whether Message and registered listener are valid
 * \li On correct OpCode use midw_fi types to extract payload
 * \li Convert message payload to diaglib internal types
 * \li Create a message context to be able to send back a response
 * \li Call the registered listener using diaglib internal types
 *
 * \param[in] poMessage The incoming message
 *
 * \todo Implement!
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vHandleRoutineControl     ( amt_tclServiceData* poMessage )
{
   ETG_TRACE_USR3_THR(( "--> tclServiceHandlerCCA::vHandleRoutineControl"));
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_HANDLE_ROUTINE_CTRL);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_INCOMING_MESSAGE_ADV_START, static_cast<tU32>(u16OurAppId));

   if(OSAL_NULL == poMessage || OSAL_NULL == poRoutineControlListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_ROUTINECONTROL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   tclMessageContextCCA* poMsgContext;
   tclParameterVector oParameters;
   oParameters.clear();
   tContext context;

   switch (poMessage->u8GetOpCode())
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
      {
         ETG_TRACE_COMP_THR(( "--- tclServiceHandlerCCA::vHandleRoutineControl => AMT_C_U8_CCAMSG_OPCODE_METHODSTART"));
         midw_diaglibfi_tclMsgRoutineControlMethodStart oMsgData;
         // Create concrete msg object
         fi_tclVisitorMessage oMsg( poMessage );
         if (oMsg.s32GetData(oMsgData) == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS ();
         }

         // Cast CCA types to internal types
         tclCCAHelper::vFillVectorFromCcaValueList(oMsgData.RoutineParameterList, oParameters);

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_RECEIVED_MESSAGE_ID, oMsgData.RoutineId);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_RECEIVED_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

         // Create Message context and store it in MessageContext Manager
         poMsgContext = OSAL_NEW tclMessageContextCCA(
                                                   poMessage,
                                                   tclMessageContext::EN_MESSAGE_ROUTINE_CONTROL,
                                                   oMsgData.RoutineId
                                                );

         context = tclMessageContextManager::poAddContext(poMsgContext);
         poRoutineControlListener->vOnRoutineControlStart(
                                                            oMsgData.RoutineId,
                                                            oMsgData.RoutineBitMask.u32Value,
                                                            oParameters,
                                                            context
                                                         );


         break;
      }

      case AMT_C_U8_CCAMSG_OPCODE_METHODABORT:
      {
         ETG_TRACE_COMP_THR(( "--- tclServiceHandlerCCA::vHandleRoutineControl => AMT_C_U8_CCAMSG_OPCODE_METHODABORT"));
         midw_diaglibfi_tclMsgRoutineControlMethodAbort oMsgData;
         // Create concrete msg object
         fi_tclVisitorMessage oMsg( poMessage );
         if (oMsg.s32GetData(oMsgData) == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS ();
         }

         // Cast CCA types to internal types

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_RECEIVED_MESSAGE_ID, oMsgData.RoutineId);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_RECEIVED_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

         // Create Message context and store it in MessageContext Manager
         poMsgContext = OSAL_NEW tclMessageContextCCA(
                                                   poMessage,
                                                   tclMessageContext::EN_MESSAGE_ROUTINE_CONTROL,
                                                   oMsgData.RoutineId
                                                );

         context = tclMessageContextManager::poAddContext(poMsgContext);
         poRoutineControlListener->vOnRoutineControlAbort(
                                                            oMsgData.RoutineId,
                                                            context
                                                         );
         break;
      }

      default:
      {
         ETG_TRACE_ERR_THR(( "!!! tclServiceHandlerCCA::vHandleRoutineControl => ERROR: unknown OpCode = %d ",poMessage->u8GetOpCode()));
         vSendError ( poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         break;
      }
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_INCOMING_MESSAGE_ADV_STOP, static_cast<tU32>(u16OurAppId));
   ETG_TRACE_USR3_THR(( "<-- tclServiceHandlerCCA::vHandleRoutineControl"));
   return;
}

/***********************************************************************//**
 * \brief Handles incoming Get Routine Result messages
 *
 * On incoming request the following steps are executed
 * \li Check whether Message and registered listener are valid
 * \li On correct OpCode use midw_fi types to extract payload
 * \li Convert message payload to diaglib internal types
 * \li Create a message context to be able to send back a response
 * \li Call the registered listener using diaglib internal types
 *
 * \param[in] poMessage The incoming message
 *
 * \todo Implement!
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vHandleGetRoutineResult   ( amt_tclServiceData* poMessage )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_HANDLE_GET_ROUTINE_RESULT);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_INCOMING_MESSAGE_ADV_START, static_cast<tU32>(u16OurAppId));

   if(OSAL_NULL == poMessage || OSAL_NULL == poRoutineControlListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_ROUTINECONTROL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   tclMessageContextCCA* poMsgContext;
   midw_diaglibfi_tclMsgGetRoutineResultMethodStart oMsgData;
   tContext context;

   switch (poMessage->u8GetOpCode())
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
      {
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_MESSAGE_TYPE_GET_ROUTINE_RES_MS);
         // Create concrete msg object
         fi_tclVisitorMessage oMsg( poMessage );
         if (oMsg.s32GetData(oMsgData) == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS ();
         }

         // Cast CCA types to internal types

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_RECEIVED_MESSAGE_ID, oMsgData.RoutineId);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_RECEIVED_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

         // Create Message context and store it in MessageContext Manager
         poMsgContext = OSAL_NEW tclMessageContextCCA(
                                                   poMessage,
                                                   tclMessageContext::EN_MESSAGE_ROUTINE_CONTROL,
                                                   oMsgData.RoutineId
                                                );

         context = tclMessageContextManager::poAddContext(poMsgContext);
         poRoutineControlListener->vOnGetRoutineResult   (
                                                            oMsgData.RoutineId,
                                                            context
                                                         );


         break;
      }

      default:
      {
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, W_DIAGLIB_OPCODE_NOT_SUPPORTED , poMessage->u8GetOpCode() );
         vSendError ( poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         break;
      }
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_ROUTINECONTROL, I_DIAGLIB_INCOMING_MESSAGE_ADV_STOP, static_cast<tU32>(u16OurAppId));

   return;
}

}
