/************************************************************************
 * FILE:         dia_tclService.cpp
 * PROJECT:      ARION Prototyp
 * SW-COMPONENT: diagnostics
 *----------------------------------------------------------------------
 *
 * DESCRIPTION: CCA Server component for the ETAS test tool
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2004 Robert Bosch GmbH, Hildesheim
 * HISTORY:
 * Date      | Author             | Modification
 * 17.10.04  | CM-DI/ESA2 Wersuhn | initial version
 * 06.06.05  | 3SOFT  Drenkhahn   | converted for ETAS
 * 05.06.14  | NBS3KOR            | Configured AM_FM settings category to start with system set.
 * 17.03.16  | BSU9KOR            | Extended displayTestPatternMapping array to support 3 new screens(AIVI-20477)
 * 25.01.17  | HLD5KOR            | Added Handler to support DiagConfigItem property for managed apps(AIVI-26969)
 *************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_ERROR_LOG__
#include "common/interfaces/dia_IErrorLog.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE__
#include "common/depricated/dia_tclService.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#include "common/interfaces/dia_ITouchPanelListener.h"
#include "common/interfaces/dia_IDisplayTestListener.h"
#include "common/interfaces/dia_INavigationListener.h"

#ifndef __INCLUDED_DIA_INTERFACE_HMILANGUAGE_LISTENER__
#include "common/interfaces/dia_IHMILanguageListener.h"
#endif

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLOGFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLOGFI_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGNOSTICS_HMI
#include "trcGenProj/Header/dia_tclService.cpp.trc.h"
#endif

#define U8RCTRLSOURCE_TEF       (tU8)0x01

#include "project/framework/config/dia_defsProjectConfig.h"

using namespace dia;

// MESSAGE MAP:
BEGIN_MSG_MAP(dia_tclService, ahl_tclBaseWork)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_DIAGREQUEST, vHandleNativeDiagnosisRequest)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_STARTROUTECALCULATION, vHandleStartRouteCalculation)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_ROUTECALCULATIONRESULT, vHandleRouteCalculationResult)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SETLOCALESTART, vSetLocaleStart)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SETLOCALERESULT, vSetLocaleResult)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_GETLOCALE, vGetLocale)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SETDISPLAY, vSetDisplay)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_RESETSECURITY, vResetSecurity)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SETDEFAULTSETTINGS, vSetDefaultSettings)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SETFACTORYSETTINGSSTATUS, vSetFactorySettingsStatus)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SPEEDLOCK, vCarConfig)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_REMOTECONTROL, vHandleRemoteControlMessage)

#if defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_NISSAN_LCN) || defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_NISSAN_LCN_NAR) || defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_MC_BASE)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_SPDLOCK, vCarConfig)
#endif

ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY, vHandleKdsEntryProperty)
ON_MESSAGE(MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM, vHandleDiagConfigItem)

END_MSG_MAP()

/*************************************************************************
 *
 * FUNCTION: dia_tclService::dia_tclService(diagnostics_tclApp*
 * poGWMainApp)
 *
 * DESCRIPTION: constructor, creates object dia_tclService - object
 *
 * PARAMETER: diagnostics_tclApp* poGWMainApp: main - object of this application
 *
 * RETURNVALUE: none
 *
 *************************************************************************/

dia_tclService::dia_tclService(diagnostics_tclApp* poDiagnosisMainApp)
   : dia_RemoteControlPlugin("RemotePluginHMI",DIA_EN_REMOTE_PLUGIN_HMI),
     menSysSetState(EN_USERSYSSET_IDLE),
     m_poGWMain(poDiagnosisMainApp),
     m_u16ClientAppID(0),
     m_u16ClientSubAppID(0),
     m_u16RegisterID(0),
     m_u8ProtocolId(0),
     m_u16CmdCounter(0),
     m_u16SubID(0),
     m_eCurrentTest(),
     m_bEnableProperty(FALSE),
     m_bRunSysSet(FALSE)
{
   // constructor calls the constructor of the base classes ahl_tclBaseWork,
   // mm_tclMessageTarget impicitely
   m_oDefSetMessage.u16AppID = 0;
   m_oDefSetMessage.u16CmdCounter = 0;
   m_oDefSetMessage.u16RegisterID = 0;
   m_oDefSetMessage.u16FunctionID = 0;
   m_oDefSetMessage.u16SubID = 0;
}

dia_tclService::~dia_tclService()
{
   m_poGWMain = OSAL_NULL;
}

/************************************************************************
 *
 * FUNCTION: tVoid dia_tclService::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
 *
 * DESCRIPTION: handles state - change messages form the spm
 *
 * PARAMETER: old state, new state
 *
 * RETURNVALUE: void
 *
 *************************************************************************/
tVoid dia_tclService::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState) const
{
   // switching to the same state ?
   if (u32OldAppState != u32AppState) {
      switch (u32AppState) {
         case DIA_C_U32_APP_STATE_NORMAL:
         case DIA_C_U32_APP_STATE_DIAGNOSIS:
         case DIA_C_U32_APP_STATE_PAUSE:
            /* +++
             here you can react on changes of the power state
             ( if not already done in diagnostics_tclApp )
             +++ */
            break;

         case DIA_C_U32_APP_STATE_OFF:
            /* +++
             here you can react on changes of the power state
             ( if not already done in diagnostics_tclApp )
             +++ */

            break;

         default:
            /* +++
             unkown power state:
             enter error handling here ( trace or assert )
             +++ */
            break;
      }
   }
}

//-----------------------------------------------------------------------------
// handler functions
//-----------------------------------------------------------------------------

void
dia_tclService::vHandleNativeDiagnosisRequest ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace oTrace("dia_tclService::vHandleNativeDiagnosisRequest()");

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain) return;

   // the opcode tells what we should do
   switch ( poMessage->u8GetOpCode() )
   {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
      case AMT_C_U8_CCAMSG_OPCODE_GET:
      case AMT_C_U8_CCAMSG_OPCODE_SET:
         {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
            m_poGWMain->enPostMessage(&oErrorMessage);
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         {
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diagnosticsfi_tclMsgDiagRequestMethodStart oRequestData;
            (tVoid) oMsg.s32GetData(oRequestData);

            // save info for reply (tbd: refactor to be able to support multiple clients -> require setup of lookup table)
            m_u16ClientAppID    = poMessage->u16GetSourceAppID();
            m_u16ClientSubAppID = poMessage->u16GetSourceSubID();
            m_u16RegisterID     = poMessage->u16GetRegisterID();
            m_u16CmdCounter     = poMessage->u16GetCmdCounter();
            m_u16SubID          = poMessage->u16GetSourceSubID();

            dia_IInternalTester* pInterface = OSAL_NULL;
            querySysAdapterInterface<dia_IInternalTester>(&pInterface);
            if ( pInterface )
            {
               // tbd: check protID

               if ( pInterface->processInternalTesterRequest(oRequestData.diagData.itemList,*this, (tCookieType)oRequestData.cookie) != DIA_SUCCESS )
               {
                  DIA_TR_ERR("FAILED TO FORWARD MESSAGE TO INTERNAL TESTER");
               }
            }
            else
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
               m_poGWMain->enPostMessage(&oErrorMessage);
            }
         }
         break;

      default:
         // unkown opcode: just delete message
         break;
   }

   (tVoid) poMessage->bDelete();
}

//-----------------------------------------------------------------------------

void
dia_tclService::onInternalTesterResponse ( std::vector<tU8>& response, tCookieType cookie )
{
   dia_tclFnctTrace trc("dia_tclService::onInternalTesterResponse");

   // create result object

   midw_diagnosticsfi_tclMsgDiagRequestMethodResult oResultData;
   oResultData.protID.enType = midw_fi_tcl_DiagProtocol::FI_EN_UDS;
   oResultData.diagData.itemList = response;
   oResultData.cookie = (tU32)VoidCast<tU32>(cookie);

   fi_tclVisitorMessage oResultMsg(oResultData);
   oResultMsg.vInitServiceData (
         DIA_USED_APP_ID,                       // source
         m_u16ClientAppID,                      // Target
         AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,     // StreamType
         0,                                     // StreamCounter
         m_u16RegisterID,                       // RegisterID
         m_u16CmdCounter,                       // nCmdCounter,
         DIA_USED_SRV_ID,                       // nServiceID,
         MIDW_DIAGNOSTICSFI_C_U16_DIAGREQUEST,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
         AMT_C_U16_DEFAULT_NULL,
         m_u16SubID
         );

   DIA_ASSERT(NULL != m_poGWMain);

   // send it
   if ( NULL != m_poGWMain )
   {
      if ( m_poGWMain->enPostMessage(&oResultMsg) != AIL_EN_N_NO_ERROR )
      {
         (tVoid) oResultMsg.bDelete();
      }
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSetLocaleStart
//
//  DESCRIPTION: handle a SetLocaleStart message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSetLocaleStart(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace trc("dia_tclService::vSetLocaleStart()");

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;
   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgSetLocaleStartStatus oStatusData;
            vSendStatusMsg(poMessage, oStatusData);
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_SET:
      {
         DIA_TR_INF("dia_tclService::vSetLocaleStart OPCODE_SET --- 1/2.");

         dia_IHMILanguageListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_IHMILanguageListener>(&pListener);
         if (pListener) {
            DIA_TR_INF("dia_tclService::vSetLocaleStart OPCODE_SET --- 2/2.");

            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diagnosticsfi_tclMsgSetLocaleStartSet oData;
            (tVoid) oMsg.s32GetData(oData);

            tDiaResult sourceSetResult = (oData.LocaleSwitched == TRUE ) ? DIA_SUCCESS : DIA_FAILED;

            DIA_TR_INF("dia_tclService::vSetLocaleStart OPCODE_SET --- result = 0x%08x.", sourceSetResult);

            // inform all interested listeners
            pListener->vOnSetHMILanguage(sourceSetResult);
         }
      }
      break;

      default:
         // unknown opcode: just delete message
         (tVoid) poMessage->bDelete();
         break;
   }
}

//////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSetLocaleStartStatus
//
//  DESCRIPTION: sends a SetLocaleStartStatus status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////

tBool
dia_tclService::bSendSetLocaleStartStatus(tU8 u8Country, tU8 u8Language, tU8 u8LangSubsys, tU8 u8Store) const
{
   dia_tclFnctTrace trc("dia_tclService::bSendSetLocaleStartStatus()");

   // create status result object
   midw_diagnosticsfi_tclMsgSetLocaleStartStatus oStatusData;
   oStatusData.CountryLanguage.Country.enType = static_cast<midw_fi_tcl_e8_Country::tenType>(u8Country);
   oStatusData.CountryLanguage.Language.enType = static_cast<midw_fi_tcl_e8_Language::tenType>(u8Language);
   oStatusData.CountryLanguage.Subsystem.enType = static_cast<midw_fi_tcl_e8_LanguageSubsystem::tenType>(u8LangSubsys);
   oStatusData.Store.enType = static_cast<midw_fi_tcl_e8_Store::tenType>(u8Store);
   vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_SETLOCALESTART);

   return true;
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSetLocaleResult
//
//  DESCRIPTION: handle a SetLocaleResult message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSetLocaleResult(amt_tclServiceData* poMessage)
{
   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;
   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {

            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgSetLocaleResultStatus oStatusData;
            oStatusData.SetLocaleResultRequested = false;
            vSendStatusMsg(poMessage, oStatusData);
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
         (tVoid) poMessage->bDelete();
         break;

      default:
         // unknown opcode: just delete message
         (tVoid) poMessage->bDelete();
         break;
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSetLocaleResultStatus
//
//  DESCRIPTION: sends a SetLocaleResult status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendSetLocaleResultStatus(tBool /*bRequest*/) const
{
   return true;
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vGetLocale
//
//  DESCRIPTION: handle a GetLocale message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vGetLocale(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace oTrace("dia_tclService::vGetLocale()");
   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;
   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgGetLocaleStatus oStatusData;
            // send dummy status data
            oStatusData.GetLocaleRequested = false;
            vSendStatusMsg(poMessage, oStatusData);
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         dia_IHMILanguageListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_IHMILanguageListener>(&pListener);
         if (pListener) {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_diagnosticsfi_tclMsgGetLocaleSet oData;
            (tVoid) oResultMsg.s32GetData(oData);
            DIA_TR_INF("dia_tclService::vGetLocale - curr CountryLanguage = %d .",oData.CountryLanguage.Language.enType);

            dia_eCMLanguage currLanguage = DIA_EN_CM_LANGUAGE_UNKNOWN;
            switch(oData.CountryLanguage.Language.enType){
               case midw_fi_tcl_e8_Language::FI_EN_GERMAN:
                  currLanguage = DIA_EN_CM_LANGUAGE_GERMAN;
                  DIA_TR_INF("dia_tclService::vGetLocale - vOnGetHMILanguage currLang = LANGUAGE_GERMAN .");
                  break;
               case midw_fi_tcl_e8_Language::FI_EN_ENGLISHMETRIC:
                  currLanguage = DIA_EN_CM_LANGUAGE_ENGLISH;
                  DIA_TR_INF("dia_tclService::vGetLocale - vOnGetHMILanguage currLang = LANGUAGE_ENGLISH .");
                  break;
               case midw_fi_tcl_e8_Language::FI_EN_PORTUGESE:
                  currLanguage = DIA_EN_CM_LANGUAGE_PORTUGESE;
                  DIA_TR_INF("dia_tclService::vGetLocale - vOnGetHMILanguage currLang = LANGUAGE_PORTUGESE .");
                  break;
               case midw_fi_tcl_e8_Language::FI_EN_CHINESE:
                  currLanguage = DIA_EN_CM_LANGUAGE_CHINESE;
                  DIA_TR_INF("dia_tclService::vGetLocale - vOnGetHMILanguage currLang = LANGUAGE_CHINESE .");
                  break;
               default:
                  DIA_TR_ERR("dia_tclService::vGetLocale - vOnGetHMILanguage CURRENT LANGUAGE UNKNOWN !");
                  break;
            }

            if (currLanguage != DIA_EN_CM_LANGUAGE_UNKNOWN) {
               pListener->vOnGetHMILanguage(currLanguage);
            }
         }
      }
         break;

      default:
         // unknown opcode: just delete message
         DIA_TR_ERR("dia_tclService::vGetLocale UNKNOWN OPCODE = 0x%02x", poMessage->u8GetOpCode());
         (tVoid) poMessage->bDelete();
         break;
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendGetLocaleStatus
//
//  DESCRIPTION: sends a GetLocale status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendGetLocaleStatus(tBool bRequest) const
{
   // create status result object
   midw_diagnosticsfi_tclMsgGetLocaleStatus oStatusData;
   oStatusData.GetLocaleRequested = bRequest;
   vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_GETLOCALE);
   return true;
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSetDisplay
//
//  DESCRIPTION: handle a SetDisplay message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSetDisplay(amt_tclServiceData* poMessage)
{
   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;
   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgSetDisplayStatus oStatusData;
            // send dummy status data
            vSendStatusMsg(poMessage, oStatusData);
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         DIA_TR_INF("vSetDisplay --> AMT_C_U8_CCAMSG_OPCODE_SET !!");

         dia_IDisplayTestListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_IDisplayTestListener>(&pListener);
         if (pListener) {
            DIA_TR_INF(" A valid listener exists for SetDisplay");

            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diagnosticsfi_tclMsgSetDisplaySet oData;
            (tVoid) oMsg.s32GetData(oData);

            tDiaResult SetDisplayResult = (oData.DisplayTestSuccess == TRUE ) ? DIA_SUCCESS : DIA_FAILED;

            DIA_TR_INF("vSetDisplay --- result = %0x08x", SetDisplayResult);

            // inform all interested listeners
            pListener->vOnSetDisplayTestPatternDone(SetDisplayResult);
         }
      }
         break;

      default:
         // unknown opcode: just delete message
         (tVoid) poMessage->bDelete();
         break;
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSetDisplayStatus
//
//  DESCRIPTION: sends a SetDisplay status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendSetDisplayStatus(tU8 /*u8TestScreen*/, tCString /*szLine1*/, tCString /*szLine2*/, tCString /*szLine3*/, tCString /*szLine4*/, tCString /*szLine5*/, tCString /*szLine6*/
) const
{
   return true;
}

////////////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSetRemoteControl
//
//  DESCRIPTION: handle a SetRemoteControl message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
////////////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vHandleRemoteControlMessage(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace oTrace("dia_tclService::vHandleRemoteControlMessage()"); //lint !e1788 ok --> scoped trace

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;

   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG: {
         DIA_TR_INF("vHandleRemoteControlMessage --> AMT_C_U8_CCAMSG_OPCODE_UPREG !!");
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            DIA_TR_INF("vHandleRemoteControlMessage --> vSendStatusMsg !!");

            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgRemoteControlSet oStatusData;
            // send dummy status data
            vSendStatusMsg(poMessage, oStatusData);
         }
      }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: {
         DIA_TR_INF("vHandleRemoteControlMessage --> AMT_C_U8_CCAMSG_OPCODE_RELUPREG !!");
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
      }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         DIA_TR_INF("vHandleRemoteControlMessage --> AMT_C_U8_CCAMSG_OPCODE_SET !!");

         fi_tclVisitorMessage oResultMsg(poMessage);
         midw_diagnosticsfi_tclMsgRemoteControlSet oRemoteCtrlResp;
         (tVoid) oResultMsg.s32GetData(oRemoteCtrlResp);

         dia_eRemoteCtrlMode mode = (oRemoteCtrlResp.cmd.enType == midw_fi_tcl_e8_RCtrlCommand::FI_EN_RCTRL_CMD_ACTIVATE)
         ? DIA_EN_REMOTE_CONTROL_MODE_ACTIVE : DIA_EN_REMOTE_CONTROL_MODE_NOT_ACTIVE;

         if (oRemoteCtrlResp.status == TRUE) {
            vOnRemoteControlUpdate(mode);
         } else {
            DIA_TR_INF("##### FAILED TO SET REQUESTED REMOTE CONTROL MODE #####");
         }
      }
         break;

      default: {
         // unknown opcode: just delete message
         DIA_TR_INF("vHandleRemoteControlMessage --> AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED (0x%02x) !!", poMessage->u8GetOpCode());
         // create and send error  message
         amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         m_poGWMain->enPostMessage(&oErrorMessage);
      }
         break;
   }

   (tVoid) poMessage->bDelete();
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSetRemoteControlStatus
//
//  DESCRIPTION: sends a RemoteControl status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//

tDiaResult
dia_tclService::requestRemoteControlMode(dia_eRemoteCtrlMode mode, dia_RemoteControlSettings settings)
{
   dia_tclFnctTrace oTrace("dia_tclService::requestRemoteControlMode()"); //lint !e1788 ok --> scoped trace

   DIA_TR_INF("requestRemoteControlMode():mode=%d, settings=%d.", mode, settings);

   if ( !(settings & DIA_C_U32_REMOTE_CTRL_SETTINGS_NO_HMI) )
   {
      // create status result object
      midw_diagnosticsfi_tclMsgRemoteControlStatus oStatusData;
      oStatusData.src.enType = static_cast<midw_fi_tcl_e8_RCtrlSource::tenType>(U8RCTRLSOURCE_TEF);
      if (mode == DIA_EN_REMOTE_CONTROL_MODE_ACTIVE)
      {
         oStatusData.cmd.enType = static_cast<midw_fi_tcl_e8_RCtrlCommand::tenType>(0x01);
         DIA_TR_INF(">> REQUESTING HMI REMOTE CONTROL ACTIVATION <<");
      }
      else
      {
         oStatusData.cmd.enType = static_cast<midw_fi_tcl_e8_RCtrlCommand::tenType>(0x00);
         DIA_TR_INF(">> REQUESTING HMI REMOTE CONTROL DEACTIVATION <<");
      }
      oStatusData.signal.enType = static_cast<midw_fi_tcl_e8_RCtrlSignal::tenType>(0xFF);
      oStatusData.status = TRUE;

      DIA_TR_INF(">> UPDATE REGISTERED CLIENTS <<");

      vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_REMOTECONTROL);
   }
   else
   {
      DIA_TR_INF("requestRemoteControlMode: HMI has been skipped.");

      vOnRemoteControlUpdate(mode);
   }

   return DIA_SUCCESS;
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vClearDestinationMemory
//
//  DESCRIPTION: handle a ClearDestinationMemory message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vClearDestinationMemory(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendClearDestinationMemoryStatus
//
//  DESCRIPTION: sends a ClearDestinationMemory status message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendClearDestinationMemoryStatus(amt_tclServiceData const* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendClearDestinationMemoryStatus
//
//  DESCRIPTION: sends a ClearDestinationMemory status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////

tBool
dia_tclService::bSendClearDestinationMemoryStatus(tBool /*bRequest*/) const
{
   return true;
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vGetDestinationMemorySize
//
//  DESCRIPTION: handle a GetDestinationMemorySize message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vGetDestinationMemorySize(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendGetDestinationMemorySizeStatus
//
//  DESCRIPTION: sends a GetDestinationMemorySize status message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendGetDestinationMemorySizeStatus(amt_tclServiceData const* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendGetDestinationMemorySizeStatus
//
//  DESCRIPTION: sends a GetDestinationMemorySize status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendGetDestinationMemorySizeStatus(tBool /*bRequest*/) const
{
   return true;
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSimulateKeyPress
//
//  DESCRIPTION: handle a SimulateKeyPress message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSimulateKeyPress(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendSimulateKeyPressStatus
//
//  DESCRIPTION: sends a SimulateKeyPress status message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendSimulateKeyPressStatus(amt_tclServiceData const* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSimulateKeyPressStatus
//
//  DESCRIPTION: sends a SimulateKeyPress status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendSimulateKeyPressStatus(tU8 /*u8KeyCode*/, tS8 /*s8EncoderSteps*/) const
{
   return true;
}
///////////////////////////////////////////////////////////////////////////
//
// FUNCTION:   dia_tclService::bUpreg
//
// DESCRIPTION:   handle an UpReg request
//
// PARAMETER:  upreg request message
//
// RETURNVALUE:   true  : Client reqistration successful
//                false : Client registration failed
//
///////////////////////////////////////////////////////////////////////////
//
bool
dia_tclService::bUpreg(amt_tclServiceData const* poMessage)
{
   dia_tclFnctTrace trc("dia_tclService::bUpreg");

   if ( (!m_poGWMain) || (!(m_poGWMain->m_poNotTable)) || (!poMessage) ) return false;

   tBool bRegOpStatus = FALSE;

   switch(poMessage->u16GetFunctionID())
   {
      case MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY:
      {
         fi_tclVisitorMessage oInVisitorMsg((amt_tclServiceData*) const_cast<amt_tclServiceData*>(poMessage));
         midw_diagnosticsfi_tclMsgKdsEntryUpReg oKdsEntryUpreg;
         (void) oInVisitorMsg.s32GetData(oKdsEntryUpreg);

         tU32 u32CcaKdsType = (tU32) oKdsEntryUpreg.KdsEntry;

         tU32 u32Key = ((tU32)poMessage->u16GetSourceAppID() << 16) | (0x0000FFFF & u32CcaKdsType);

         tMapKdsEntryNotify[u32Key].u16AppID      = poMessage->u16GetSourceAppID();
         tMapKdsEntryNotify[u32Key].u16CmdCounter = poMessage->u16GetCmdCounter();
         tMapKdsEntryNotify[u32Key].u16RegisterID = poMessage->u16GetRegisterID();
         tMapKdsEntryNotify[u32Key].u32CcaSubId   = u32CcaKdsType;

         dia_IKDS* pKDS = 0;
         if (queryConfMgrInterface<dia_IKDS>(&pKDS) == DIA_SUCCESS)
         {
            pKDS->addKDSListener((tU16) u32CcaKdsType,this);
         }

         bRegOpStatus = TRUE;
         break;
      }

      case MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM:
      {
         fi_tclVisitorMessage oInVisitorMsg((amt_tclServiceData*) const_cast<amt_tclServiceData*>(poMessage));
         midw_diagnosticsfi_tclMsgDiagConfigItemUpReg oDiagConfigItemUpreg;
         (void) oInVisitorMsg.s32GetData(oDiagConfigItemUpreg);

         tU32 u32CcaConfigItemType = (tU32) oDiagConfigItemUpreg.ConfigItem;

         tMapConfigItemEntryNotify[u32CcaConfigItemType].u16AppID      = poMessage->u16GetSourceAppID();
         tMapConfigItemEntryNotify[u32CcaConfigItemType].u16CmdCounter = poMessage->u16GetCmdCounter();
         tMapConfigItemEntryNotify[u32CcaConfigItemType].u16RegisterID = poMessage->u16GetRegisterID();
         tMapConfigItemEntryNotify[u32CcaConfigItemType].u32CcaSubId   = u32CcaConfigItemType;

         dia_IConfigItem* pConfigItem = 0;
         if (queryConfMgrInterface<dia_IConfigItem>(&pConfigItem) == DIA_SUCCESS)
         {
            pConfigItem->addConfigItemListener(u32CcaConfigItemType,this);
         }

         bRegOpStatus = TRUE;
         break;
      }

      default:
      {
         ahl_bEnterCritical(diagnostics_tclApp::m_hNotTableSem);

         // Add the client to notification table
            bRegOpStatus = m_poGWMain->m_poNotTable->bAddNotification (
                  poMessage->u16GetFunctionID(),
                  poMessage->u16GetSourceAppID(),
                  poMessage->u16GetRegisterID(),
                  1,
                  poMessage->u16GetCmdCounter(),
               poMessage->u16GetSourceSubID());

         ahl_bReleaseCritical(diagnostics_tclApp::m_hNotTableSem);
         break;
      }
   }

   return (bRegOpStatus == TRUE) ? true : false;
}

///////////////////////////////////////////////////////////////////////////
//
// FUNCTION:   dia_tclService::bRelUpreg
//
// DESCRIPTION:   Removes the sender of the message from the NotifcationTable
//
// PARAMETER:  Relupreg request message
//
// RETURNVALUE:   true  : Client dereqistration successful
//                false : Client deregistration failed
//
///////////////////////////////////////////////////////////////////////////
//
bool
dia_tclService::bRelUpreg(amt_tclServiceData const* poMessage)
{
   if ( (!m_poGWMain) || (!(m_poGWMain->m_poNotTable)) || (!poMessage) ) return false;

   tBool bRegOpStatus = FALSE;

   if ( (MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY != poMessage->u16GetFunctionID()) || (MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM != poMessage->u16GetFunctionID()) )
   {
       ahl_bEnterCritical(diagnostics_tclApp::m_hNotTableSem);

   // Delete the client from notification table
      bRegOpStatus = m_poGWMain->m_poNotTable->bRemoveNotification (
            poMessage->u16GetFunctionID(),
            poMessage->u16GetSourceAppID(),
            poMessage->u16GetRegisterID(),
            1,
            poMessage->u16GetCmdCounter(),
            poMessage->u16GetSourceSubID()
            );

   ahl_bReleaseCritical(diagnostics_tclApp::m_hNotTableSem);
   }
   else if(MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY == poMessage->u16GetFunctionID())
   {
      fi_tclVisitorMessage oInVisitorMsg((amt_tclServiceData*) const_cast<amt_tclServiceData*>(poMessage));
      midw_diagnosticsfi_tclMsgKdsEntryRelUpReg oKdsEntryRelUpreg;
      oInVisitorMsg.s32GetData(oKdsEntryRelUpreg);

      tU32 u32CcaKdsType = (tU32) oKdsEntryRelUpreg.KdsEntry;
      tU32 u32Key = ((tU32)poMessage->u16GetSourceAppID() << 16) | (0x0000FFFF & u32CcaKdsType);
      tMapKdsEntryNotify.erase(u32Key);

      dia_IKDS* pKDS = 0;
      if (queryConfMgrInterface<dia_IKDS>(&pKDS) == DIA_SUCCESS)
      {
         pKDS->removeKDSListener((tU16) u32CcaKdsType,this);
      }

      bRegOpStatus = TRUE;
   }
   else
   {
      fi_tclVisitorMessage oInVisitorMsg((amt_tclServiceData*) const_cast<amt_tclServiceData*>(poMessage));
      midw_diagnosticsfi_tclMsgDiagConfigItemRelUpReg oDiagConfigItemRelUpreg;
      oInVisitorMsg.s32GetData(oDiagConfigItemRelUpreg);

      tU32 u32CcaConfigItemType = (tU32) oDiagConfigItemRelUpreg.ConfigItem;
      tMapConfigItemEntryNotify.erase(u32CcaConfigItemType);

      dia_IConfigItem* pConfigItem = 0;
      if (queryConfMgrInterface<dia_IConfigItem>(&pConfigItem) == DIA_SUCCESS)
      {
         pConfigItem->removeConfigItemListener(u32CcaConfigItemType,this);
      }

      bRegOpStatus = TRUE;
   }

   return (bRegOpStatus == TRUE) ? true : false;
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vReadDiaglog
//
//  DESCRIPTION: handle a ReadDiaglog message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vReadDiaglog(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vClearDiaglog
//
//  DESCRIPTION: handle a ClearDiaglog message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vClearDiaglog(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vStartDiagnosticTest
//
//  DESCRIPTION: handle a StartDiagnosticTest message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vStartDiagnosticTest(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSetDefaultSettings
//
//  DESCRIPTION: handle a SetDefaultSettings message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSystemSetFailedParameterIds(std::vector<tU16>& /*vectorDiff*/)
{
   // Intentionally not implemented
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vEnableTimeSlot
//
//  DESCRIPTION: handle a EnableTimeSlot message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vEnableTimeSlot(amt_tclServiceData* /*poMessage*/) const
{
}
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vUpdateAllClients
//
//  DESCRIPTION: update all clients with a property status
//
//  PARAMETER:   oFIData - property data
//               u16FuncId - property ID
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vUpdateAllClients(fi_tclTypeBase const& oFIData, tU16 u16FuncId) const
{
   dia_tclFnctTrace oTrace("dia_tclService::vUpdateAllClients()"); //lint !e1788 ok --> scoped trace

   if (!m_poGWMain) {
      return;
   }

   ahl_bEnterCritical(diagnostics_tclApp::m_hNotTableSem);

   // Run through the table to find all registered clients
   ahl_tNotification* pNot = m_poGWMain->m_poNotTable->poGetNotificationList(u16FuncId);
   for (; pNot != OSAL_NULL; pNot = pNot->pNext) {
      // construct result message
      fi_tclVisitorMessage oResultMsg(oFIData);
      // initialise result message
      oResultMsg.vInitServiceData(DIA_USED_APP_ID, // source
            pNot->u16AppID, // Target
            AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
            0, // StreamCounter
            pNot->u16RegisterID, // RegisterID
            pNot->u16CmdCounter, // nCmdCounter,
            DIA_USED_SRV_ID, // nServiceID,
            u16FuncId, // function ID
            AMT_C_U8_CCAMSG_OPCODE_STATUS, AMT_C_U16_DEFAULT_NULL, pNot->u16SubID);

      // send it
      if (m_poGWMain->enPostMessage(&oResultMsg) != AIL_EN_N_NO_ERROR) {
         (tVoid) oResultMsg.bDelete();
      }
   }
   ahl_bReleaseCritical(diagnostics_tclApp::m_hNotTableSem);
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendStatusMsg
//
//  DESCRIPTION: sends a CCA Status message
//
//  PARAMETER:   poMessage - incoming request
//                oFIData - property data
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendStatusMsg(amt_tclServiceData const* poMessage, fi_tclTypeBase const& oFIData) const
{
   // Check the validity of incoming data
   if (NULL == poMessage) {
      return;
   }
   // initialise result message
   vSendMsg(poMessage, oFIData, AMT_C_U8_CCAMSG_OPCODE_STATUS);
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendResultMsg
//
//  DESCRIPTION: sends a CCA MethodResult message
//
//  PARAMETER:   poMessage - incoming request
//                oFIData - property data
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendResultMsg(amt_tclServiceData const* poMessage, fi_tclTypeBase const& oFIData) const
{
   // Check the validity of incoming data
   if (NULL == poMessage) {
      return;
   }
   // initialise result message
   vSendMsg(poMessage, oFIData, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendReadDiaglogResult
//
//  DESCRIPTION: sends a ReadDiaglog CCA MethodResult message
//
//  PARAMETER:   poMessage - incoming request
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendReadDiaglogResult(amt_tclServiceData* /*poMessage*/) const
{
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendClearDiaglogResult
//
//  DESCRIPTION: sends a ClearDiaglog CCA MethodResult message
//
//  PARAMETER:   poMessage - incoming request
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendClearDiaglogResult(amt_tclServiceData* /*poMessage*/) const
{
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendStartDiagnosticTestResult
//
//  DESCRIPTION: sends a PerformDiagnosticTest CCA MethodResult message
//
//  PARAMETER:   tBool - result of test
//
//  RETURNVALUE: void
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendStartDiagnosticTestResult(bool /*bSuccess*/) const
{
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vDiagnoticTestCallback
//
//  DESCRIPTION: callback for StartDiagnosticTest
//
//  PARAMETER:   tBool - result of test
//
//  RETURNVALUE: void
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vDiagnoticTestCallback(bool /*bTestResult*/) const
{
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vStopDiagnosticTest
//
//  DESCRIPTION: handle a StopDiagnosticTest message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vStopDiagnosticTest(amt_tclServiceData* /*poMessage*/) const
{
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendStopDiagnosticTestResult
//
//  DESCRIPTION: sends a StopDiagnosticTest CCA MethodResult message
//
//  PARAMETER:   tBool - result of test
//
//  RETURNVALUE: void
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendStopDiagnosticTestResult(bool /*bSuccess*/) const
{
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vResetSecurity
//
//  DESCRIPTION: handle a ResetSecurity CCA message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vResetSecurity(amt_tclServiceData* poMessage)
{
   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;
   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgResetSecurityStatus oStatusData;
            // send dummy status data
            vSendStatusMsg(poMessage, oStatusData);
         }
         (tVoid) poMessage->bDelete();
         break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
         (tVoid) poMessage->bDelete();
         break;

      default:
         // unknown opcode: just delete message
         (tVoid) poMessage->bDelete();
         break;
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendResetSecurityStatus
//
//  DESCRIPTION: sends a ResetSecurity status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendResetSecurityStatus(tBool /*bReset*/) const
{
   return true;
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::vSendMsg
//
//  DESCRIPTION: sends a CCA message
//
//  PARAMETER:   poMessage - incoming request
//                oFIData - property data
//                u8OpCode - CCA op code
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSendMsg(amt_tclServiceData const* poMessage, fi_tclTypeBase const& oFIData, tU8 u8OpCode) const
{
   // Check the validity of incoming data
   if (NULL == poMessage) {
      return;
   }
   // construct result message
   fi_tclVisitorMessage oResultMsg(oFIData);
   // initialise result message
   oResultMsg.vInitServiceData(DIA_USED_APP_ID, // source
         poMessage->u16GetSourceAppID(), // Target
         AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
         0, // StreamCounter
         poMessage->u16GetRegisterID(), // RegisterID
         poMessage->u16GetCmdCounter(), // nCmdCounter,
         DIA_USED_SRV_ID, // nServiceID,
         poMessage->u16GetFunctionID(), // function ID
         u8OpCode, AMT_C_U16_DEFAULT_NULL, poMessage->u16GetSourceSubID());
   // send it
   if (NULL != m_poGWMain) {
      if (m_poGWMain->enPostMessage(&oResultMsg) != AIL_EN_N_NO_ERROR)
      {
         (tVoid) oResultMsg.bDelete();
      }
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vCarConfig
//
//  DESCRIPTION: handle a Car Config property get message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vCarConfig(amt_tclServiceData* poMessage)
{
   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;

   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
      {
         // create and send error  message
         amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         m_poGWMain->enPostMessage(&oErrorMessage);
      }
         break;

      default:
         // unknown opcode: just delete message
         break;
   }

   (tVoid) poMessage->bDelete();
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSendSetSpeedLockStatus
//
//  DESCRIPTION: send a SetSpeedLock  request
//
//  PARAMETER:   status data
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
tVoid
dia_tclService::vSendSetSpeedLockStatus(const ahl_tNotification* pNot, const tBool Status)
{
   dia_tclFnctTrace trc("dia_tclService::vSendSetSpeedLockStatus"); //lint !e1788 ok --> scoped trace

#if defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_NISSAN_LCN) || defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_NISSAN_LCN_NAR) || defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_MC_BASE)

   if(m_poGWMain != NULL)
   {
      midw_diagnosticsfi_tclMsgSpdLockStatus oServerRes;
      oServerRes.SpeedLockStatus = Status;

      fi_tclVisitorMessage oMyStatusMessage(oServerRes);

      // initialise result message
      oMyStatusMessage.vInitServiceData(
            DIA_USED_APP_ID,// source
            pNot->u16AppID,// Target
            AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,// StreamType
            0,// StreamCounter
            pNot->u16RegisterID,// RegisterID
            pNot->u16CmdCounter,// nCmdCounter,
            DIA_USED_SRV_ID,// nServiceID,
            MIDW_DIAGNOSTICSFI_C_U16_SPDLOCK,// function ID
            AMT_C_U8_CCAMSG_OPCODE_STATUS,
            AMT_C_U16_DEFAULT_NULL,
            pNot->u16SubID
      );

      if( AIL_EN_N_NO_ERROR != m_poGWMain->enPostMessage(&oMyStatusMessage))
      {
         // Error in sending the message
         if (!oMyStatusMessage.bDelete())
         {
            // Error in deleting the message
         }
      }
      else
      {
      }
   }
#else
   //Avoids Lint if the variable is not used in the Project
   DIA_PARAMETER_INTENTIONALLY_UNUSED(pNot);
   DIA_PARAMETER_INTENTIONALLY_UNUSED(Status);
#endif
} //lint !e1762 // Can not be made const, as in some projects it modifies the class

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSendGetSpeedLockStatus
//
//  DESCRIPTION: send a GetSpeedLock  request
//
//  PARAMETER:   status data
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
tVoid
dia_tclService::vSendGetSpeedLockStatus(const ahl_tNotification* pNot, const midw_fi_tcl_e16_SpeedLock& SpeedLock, const tU8 SpeedThreshold)
{
   dia_tclFnctTrace trc("dia_tclService::vSendGetSpeedLockStatus"); //lint !e1788 ok --> scoped trace

#if defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_NISSAN_LCN) || defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_NISSAN_LCN_NAR) || defined(VARIANT_S_FTR_ENABLE_PROJECTBUILD_MC_BASE)

   if(m_poGWMain != NULL)
   {
      midw_diagnosticsfi_tclMsgSpeedLockStatus oMyData;

      oMyData.SpeedLockStatus.enType = SpeedLock.enType;
      oMyData.SpeedLockThreshold = SpeedThreshold;

      //Create sender message.
      fi_tclVisitorMessage oMyStatusMessage(oMyData);

      //Now add targetting data
      oMyStatusMessage.vInitServiceData(
            DIA_USED_APP_ID,
            pNot->u16AppID,
            AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,// StreamType
            0,// StreamCounter
            pNot->u16RegisterID ,
            pNot->u16CmdCounter,
            DIA_USED_SRV_ID,
            MIDW_DIAGNOSTICSFI_C_U16_SPEEDLOCK,
            AMT_C_U8_CCAMSG_OPCODE_STATUS);
      // place FC_SECURITY_TRCL_HSI_STATUS_RETURN here

      if( AIL_EN_N_NO_ERROR != m_poGWMain->enPostMessage(&oMyStatusMessage))
      {
         // Error in sending the message
         if (!oMyStatusMessage.bDelete())
         {
            // Error in deleting the message
         }
      }
      else
      {
      }

   }
#else
//Avoids Lint if the variable is not used in the Project
   DIA_PARAMETER_INTENTIONALLY_UNUSED(pNot);
   DIA_PARAMETER_INTENTIONALLY_UNUSED(SpeedLock);
   DIA_PARAMETER_INTENTIONALLY_UNUSED(SpeedThreshold);
#endif
} //lint !e1762 // Can not be made const, as in some projects it modifies the class

//---------------------------------------------------------------------------

tBool
dia_tclService::bCodingValueUpdate(tVoid) const
{
   DIA_TR_ERR("---> dia_tclService::bCodingValueUpdate");

   return TRUE;
}

//------------------------------------------------------------------------------

tBool
dia_tclService::bEvaluateCodingUpdateResult(amt_tclServiceData* /*poMessage*/) const
{
   DIA_TR_ERR("---> dia_tclService::bEvaluateCodingUpdateResult");

   tBool bResult = FALSE;
//
//   // Get method result data into fi type
//    fi_tclVisitorMessage oMsg(poMessage);
//    midw_diagnosticsfi_tclMsgSetFactorySettingsStatusSet oData;
//    oMsg.s32GetData(oData);
//
//   if ( oData.DefSetStateGroup == midw_fi_tcl_e8_DefSetGroups::FI_EN_DEFSET_GROUP__MAINTENANCE )
//   {
//      if ( oData.DefSetStateMode.enType == midw_fi_tcl_e8_DefSetMode::FI_EN_DEFSET_SRC_CODING )
//      {
//         if ( oData.DefSetStateSuccessful == TRUE ) {
//            bResult = TRUE;
//         }
//      }
//   }
//
//   // we no longer need the message
////   poMessage->bDelete();

   return bResult;
}

//------------------------------------------------------------------------------
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::startRouteCalculation
//
//  DESCRIPTION: sends a StartRouteCalculation status message
//
//  PARAMETER:   see FI parameter data
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//------------------------------------------------------------------------------
tDiaResult
dia_tclService::startRouteCalculation(tS32 s32CurPosLong, tS32 s32CurPosLat, tS32 s32DestLong, tS32 s32DestLat, tU8 u8RouteIndex, tU8 u8RouteOptim, tU8 u8RouteCmd)
{
   dia_tclFnctTrace oTrace("dia_tclService::startRouteCalculation()"); //lint !e1788 ok --> scoped trace

   DIA_TR_INF("dia_tclService::startRouteCalculation - Send StartRoute command:");
   DIA_TR_INF("startRouteCalculation Start       Lang: 0x%08x Lat: 0x%08x.", s32CurPosLong, s32CurPosLat);
   DIA_TR_INF("startRouteCalculation Destination Lang: 0x%08x Lat: 0x%08x.", s32DestLong, s32DestLat);
   DIA_TR_INF("startRouteCalculation Route Index: %d, RouteOptim: %d, RouteCmd: %d.", u8RouteIndex, u8RouteOptim, u8RouteCmd);

   // fill FI class with parameter data
   midw_diagnosticsfi_tclMsgStartRouteCalculationStatus oStatusData;
   oStatusData.RouteDetails.CurrentPositionLongitude = s32CurPosLong;
   oStatusData.RouteDetails.CurrentPositionLatitude = s32CurPosLat;
   oStatusData.RouteDetails.DestinationLongitude = s32DestLong;
   oStatusData.RouteDetails.DestinationLatitude = s32DestLat;
   oStatusData.RouteDetails.RouteIndex = u8RouteIndex;
   oStatusData.RouteDetails.RouteOptimisation = u8RouteOptim;
   oStatusData.RouteDetails.Command.enType = static_cast<midw_fi_tcl_e8_RouteCalculationCommand::tenType>(u8RouteCmd);
   // send status message to clients
   vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_STARTROUTECALCULATION);

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::requestRouteCalcResultStatus
//
//  DESCRIPTION: sends a RouteCalculationresult status message
//
//  PARAMETER:   tVoid
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
//------------------------------------------------------------------------------
tDiaResult
dia_tclService::requestRouteCalcResultStatus(tVoid)
{
   dia_tclFnctTrace oTrace("--> dia_tclService::requestRouteCalcResultStatus"); //lint !e1788 ok --> scoped trace
   // create status result object
   midw_diagnosticsfi_tclMsgRouteCalculationResultStatus oStatusData;
   oStatusData.CalculationResultRequested = TRUE;
   vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_ROUTECALCULATIONRESULT);
   DIA_TR_INF("<-- dia_tclService::requestRouteCalcResultStatus");
   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_tclService::startTouchPanelAlignment(tVoid)
{
   dia_tclFnctTrace oTrace("dia_tclService::startTouchPanelAlignment"); //lint !e1788 ok --> scoped trace

#if 0 //STC2HI: unavailable in GMG3
   midw_diagnosticsfi_tclMsgStartTouchCalibrationStatus oStatusData;
   oStatusData.StartTouchCalibration = TRUE;
   vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_STARTTOUCHCALIBRATION);
#endif
   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_tclService::setDisplayPatternID(dia_eDisplayPatternID ePatternID)
{
   static midw_fi_tcl_e8_TestScreens::tenType displayTestPatternMapping[DIA_EN_DISPLAY_PATTERN_COUNT] = {
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_NOTESTSCREEN,        // DIA_EN_DISPLAY_PATTERN_UNKNOWN
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_NOTESTSCREEN,        // DIA_EN_DISPLAY_PATTERN_NONE
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_PLAINWHITE,          // DIA_EN_DISPLAY_PATTERN_WHITE
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_PLAINBLACK,          // DIA_EN_DISPLAY_PATTERN_BLACK
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_PLAINRED,            // DIA_EN_DISPLAY_PATTERN_RED
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_PLAINGREEN,          // DIA_EN_DISPLAY_PATTERN_GREEN
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_PLAINBLUE,           // DIA_EN_DISPLAY_PATTERN_BLUE
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_COLOURED,            // DIA_EN_DISPLAY_PATTERN_COLOURED
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_COLOURGRADIATION,    // DIA_EN_DISPLAY_PATTERN_COLOURED_GRADIATION
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_DIAGNOSISMASK,       // DIA_EN_DISPLAY_PATTERN_DIAGNOSIS_MASK
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_LATTICE,             // DIA_EN_DISPLAY_PATTERN_LATTICE
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_LEFTWHITERIGHTBLACK, // DIA_EN_DISPLAY_PATTERN_WHITE_BLACK
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_LEFTBLACKRIGHTWHITE, // DIA_EN_DISPLAY_PATTERN_BLACK_WHITE
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_PIXELWISECHESSBOARD, // DIA_EN_DISPLAY_PATTERN_CHESSBOARD
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_NOTESTSCREEN,        // DIA_EN_DISPLAY_PATTERN_CHESSBOARD_INVERT
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_NOTESTSCREEN,        // DIA_EN_DISPLAY_PATTERN_STRIPS
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_NOTESTSCREEN,        // DIA_EN_DISPLAY_PATTERN_GRAY64
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_SCREEN_7,            // DIA_EN_DISPLAY_PATTERN_FLICKER_TEST,
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_SCREEN_8,            // DIA_EN_DISPLAY_PATTERN_TEST_MARK,
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_NOTESTSCREEN,        // DIA_EN_DISPLAY_PATTERN_TEF_SPECIAL
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_TEFSPECIAL1,         // DIA_EN_DISPLAY_PATTERN_TEF_SPECIAL1
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_TEFSPECIAL2,          // DIA_EN_DISPLAY_PATTERN_TEF_SPECIAL2
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_RGB_NOK,				//DIA_EN_DISPLAY_PATTERN_RGB_NOK
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_RGB_OK,				//DIA_EN_DISPLAY_PATTERN_RGB_OK
      (midw_fi_tcl_e8_TestScreens::tenType) midw_fi_tcl_e8_TestScreens::FI_EN_RGB_REPDARK				//DIA_EN_DISPLAY_PATTERN_RGB_REPDARK
   };

   tDiaResult retVal = DIA_FAILED;

   if (ePatternID < DIA_EN_DISPLAY_PATTERN_COUNT)
   {
      midw_diagnosticsfi_tclMsgSetDisplayStatus oStatusData;
      oStatusData.DisplayData.TestScreen.enType = displayTestPatternMapping[ePatternID];

      midw_fi_tclString oFiString;
      oStatusData.DisplayData.DisplayStrings.StringList.resize(6);

      if (midw_fi_tcl_e8_TestScreens::FI_EN_DIAGNOSISMASK == displayTestPatternMapping[ePatternID]) {
         tCString szLine1 = "Best ";
         tCString szLine2 = "Wishes ";
         tCString szLine3 = "From ";
         tCString szLine4 = "Dynamic ";
         tCString szLine5 = "Diagnostics ";
         tCString szLine6 = "Team";
         oStatusData.DisplayData.DisplayStrings.StringList[0].bSet(szLine1, midw_fi_tclString::FI_EN_UTF8);
         oStatusData.DisplayData.DisplayStrings.StringList[1].bSet(szLine2, midw_fi_tclString::FI_EN_UTF8);
         oStatusData.DisplayData.DisplayStrings.StringList[2].bSet(szLine3, midw_fi_tclString::FI_EN_UTF8);
         oStatusData.DisplayData.DisplayStrings.StringList[3].bSet(szLine4, midw_fi_tclString::FI_EN_UTF8);
         oStatusData.DisplayData.DisplayStrings.StringList[4].bSet(szLine5, midw_fi_tclString::FI_EN_UTF8);
         oStatusData.DisplayData.DisplayStrings.StringList[5].bSet(szLine6, midw_fi_tclString::FI_EN_UTF8);
      }

      vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_SETDISPLAY);
      retVal = DIA_SUCCESS;
   }

   return retVal;
}

//------------------------------------------------------------------------------

tVoid
dia_tclService::vHandleTouchPanelCalibrationResult(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace oTrace("dia_tclService::vHandleTouchPanelCalibrationResult"); //lint !e1788 ok --> scoped trace

   if (!poMessage)
      return;
   if (!m_poGWMain)
      return;

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

#if 0 //STC2HI: unavailable in GMG3
   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG: {
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgStartTouchCalibrationStatus oStatusData;
            vSendStatusMsg(poMessage, oStatusData);
         }
      }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: {
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
      }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         DIA_TR_INF("vHandleTouchPanelCalibrationResult --- 1");

         dia_ITouchPanelListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ITouchPanelListener>(&pListener);
         if (pListener) {
            DIA_TR_INF("vHandleTouchPanelCalibrationResult --- 2");

            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diagnosticsfi_tclMsgStartTouchCalibrationSet oData;
            (tVoid) oMsg.s32GetData(oData);

            tDiaResult calibrationResult = (oData.TouchCalibrationSuccess == TRUE ) ? DIA_SUCCESS : DIA_FAILED;

            DIA_TR_INF("vHandleTouchPanelCalibrationResult --- result = %0x08x", calibrationResult);

            // inform all interested listeners
            pListener->vOnTouchPanelAlignmentDone(calibrationResult);
         }
      }
         break;

      default:
         // unknown opcode: just delete message
         DIA_TR_INF("vHandleTouchPanelCalibrationResult --> UNKNOWN MESSAGE !!");
         break;
   }
#endif //STC2HI: unavailable in GMG3

   (tVoid) poMessage->bDelete();
}

//------------------------------------------------------------------------------
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vHandleStartRouteCalculation
//
//  DESCRIPTION: handle a StartRouteCalculation message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//------------------------------------------------------------------------------
tVoid
dia_tclService::vHandleStartRouteCalculation(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace oTrace("dia_tclService::vHandleStartRouteCalculation()"); //lint !e1788 ok --> scoped trace

   if (!poMessage)
      return;
   if (!m_poGWMain)
      return;

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG: {
         // Removed trace as OPCode is know from trace-class SMALL
         // register client in notification array
         // if unsuccessful
         if (!bUpreg(poMessage)) {
            DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation - UPREG failed");
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.

            // create status result object
            midw_diagnosticsfi_tclMsgStartRouteCalculationStatus oStatusData;
            // send dummy status data
            vSendStatusMsg(poMessage, oStatusData);
         }
      }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: {
         // Removed trace as OPCode is know from trace-class SMALL
         if (!bRelUpreg(poMessage)) {
            DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation - RELUPREG failed");
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
      }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation - SET");

         dia_INavigationListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_INavigationListener>(&pListener);
         if (pListener) {
            DIA_TR_INF(" A valid listener exists for StartRouteCalculation");

            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diagnosticsfi_tclMsgStartRouteCalculationSet oData;
            (tVoid) oMsg.s32GetData(oData);

            DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation --- CalculationStarted2 = %d", oData.CalculationStarted);
            tDiaResult routeCalcStartResult = (oData.CalculationStarted == TRUE ) ? DIA_SUCCESS : DIA_FAILED;
            DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation --- result2 = %d", routeCalcStartResult);

            // inform all interested listeners
            pListener->vOnRouteCalculationStarted(routeCalcStartResult);
         }
      }
         break;
      case AMT_C_U8_CCAMSG_OPCODE_GET:
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART: {
         DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation - Unsupported OpCode received");
         // create and send error  message
         amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         m_poGWMain->enPostMessage(&oErrorMessage);
      }
         break;

      default:
         DIA_TR_INF("dia_tclService::vHandleStartRouteCalculation - Unknown OpCode received");
         break;
   }
   (tVoid) poMessage->bDelete();
   DIA_TR_INF(("<-- dia_tclService::vHandleStartRouteCalculation"));
}

//------------------------------------------------------------------------------
///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vHandleRouteCalculationResult
//
//  DESCRIPTION: handle a RouteCalculationResult message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
//------------------------------------------------------------------------------
tVoid
dia_tclService::vHandleRouteCalculationResult(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace oTrace("dia_tclService::vHandleRouteCalculationResult"); //lint !e1788 ok --> scoped trace

   if (!poMessage)
      return;
   if (!m_poGWMain)
      return;

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG: {
         // Removed trace as OPCode is know from trace-class SMALL
         // register client in notification array
         // if unsuccessful
         if (!bUpreg(poMessage)) {
            DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult - UPREG failed");
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Upreg successful : send a status message.

            // create status result object
            midw_diagnosticsfi_tclMsgRouteCalculationResultStatus oStatusData;
            oStatusData.CalculationResultRequested = false;
            // send dummy status data
            vSendStatusMsg(poMessage, oStatusData);
         }
      }
         break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: {
         // Removed trace as OPCode is know from trace-class SMALL
         if (!bRelUpreg(poMessage)) {
            DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult - RELUPREG failed");
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
         }
      }
         break;
      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult - SET");

         dia_INavigationListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_INavigationListener>(&pListener);
         if (pListener) {
            DIA_TR_INF(" A valid listener exists for requestRouteCalcResultStatus");

            // Get method result data into fi type
            fi_tclVisitorMessage oMsg(poMessage);
            midw_diagnosticsfi_tclMsgRouteCalculationResultSet oData;
            (tVoid) oMsg.s32GetData(oData);

            tU8 bRouteCalculationStatus = (tU8) (oData.CalculationResult.RouteCalculationStatus.enType);
            tU32 bRouteLength = oData.CalculationResult.RouteLength;
            tU32 bRouteDuration = oData.CalculationResult.RouteDuration;

            DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult --- Status = %d", bRouteCalculationStatus);
            DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult --- RouteLength = %d", bRouteLength);
            DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult --- bRouteDuration = %d", bRouteDuration);

            // inform all interested listeners
            pListener->vOnRouteCalculationStatus(bRouteCalculationStatus, bRouteLength, bRouteDuration);
         }

      }
         break;
      case AMT_C_U8_CCAMSG_OPCODE_GET:
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART: {
         DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult - Unsupported OpCode received");
         // create and send error  message
         amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         m_poGWMain->enPostMessage(&oErrorMessage);
      }
         break;

      default:
         DIA_TR_INF("dia_tclService::vHandleRouteCalculationResult - Unknown OpCode received");
         break;
   }
   (tVoid) poMessage->bDelete();
   DIA_TR_INF("<-- dia_tclService::vHandleRouteCalculationResult");
}

//------------------------------------------------------------------------------

tDiaResult
dia_tclService::setHMILanguage(dia_eCMLanguage u8Country)
{
   dia_tclFnctTrace trc("dia_tclService::setHMILanguage()");
   tDiaResult retCode = DIA_FAILED;

   midw_fi_tcl_e8_Language::tenType fi_language = midw_fi_tcl_e8_Language::FI_EN_UNDEFINED;
   switch (u8Country) {
      case DIA_EN_CM_LANGUAGE_ENGLISH:
         fi_language = midw_fi_tcl_e8_Language::FI_EN_ENGLISHMETRIC;
         DIA_TR_INF("dia_tclService::setHMILanguage - FI_EN_ENGLISHMETRIC.");
         break;
      case DIA_EN_CM_LANGUAGE_GERMAN:
         fi_language = midw_fi_tcl_e8_Language::FI_EN_GERMAN;
         DIA_TR_INF("dia_tclService::setHMILanguage - FI_EN_GERMAN.");
         break;
      case DIA_EN_CM_LANGUAGE_PORTUGESE:
         fi_language = midw_fi_tcl_e8_Language::FI_EN_PORTUGESE;
         DIA_TR_INF("dia_tclService::setHMILanguage - FI_EN_PORTUGESE.");
         break;
      case DIA_EN_CM_LANGUAGE_CHINESE:
         fi_language = midw_fi_tcl_e8_Language::FI_EN_CHINESE;
         DIA_TR_INF("dia_tclService::setHMILanguage - FI_EN_CHINESE.");
         break;
      default:
         DIA_TR_ERR("dia_tclService::setHMILanguage - UNKNOWN Language ID = 0x%02x", u8Country);
         break;
   }

   if (fi_language != midw_fi_tcl_e8_Language::FI_EN_UNDEFINED) {
      if (bSendSetLocaleStartStatus(midw_fi_tcl_e8_Country::FI_EN_UNDEFINED, fi_language, midw_fi_tcl_e8_LanguageSubsystem::FI_EN_HMI, midw_fi_tcl_e8_Store::FI_EN_VOLATILE) == TRUE) {
         retCode = DIA_SUCCESS;
      } else {
         DIA_TR_ERR("dia_tclService::setHMILanguage - bSendSetLocaleStartStatus failed.");
      }
   } else {
      DIA_TR_ERR("dia_tclService::setHMILanguage - midw_fi_tcl_e8_Language::FI_EN_UNDEFINED.");
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_tclService::getHMILanguage(tVoid)
{
   dia_tclFnctTrace trc("dia_tclService::getHMILanguage()");
   tDiaResult retCode = DIA_FAILED;

   if (bSendGetLocaleStatus(TRUE) == TRUE) {
      retCode = DIA_SUCCESS;
      DIA_TR_INF("dia_tclService::setHMILanguage - bSendGetLocaleStatus success.");
   } else {
      DIA_TR_ERR("dia_tclService::setHMILanguage - bSendGetLocaleStatus failed.");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vOnKDSUpdate ( tU16 kdsKey, tU8 kdsData[], tU16 length )
{
   dia_tclFnctTrace trc("dia_tclService::vOnKDSUpdate()");

   DIA_TR_INF("INFORM REGISTERED LISTENERS ABOUT UPDATE OF KDS KEY 0x%04x", kdsKey);

   if ( tMapKdsEntryNotify.size() )
   {
      std::map<tU32,TNotificationKds>::iterator iter = tMapKdsEntryNotify.begin();
      for ( ; iter != tMapKdsEntryNotify.end(); iter++)
      {
         if ( kdsKey == ((tU16) iter->first) )
         {
            ahl_tNotification keyNotification;
            keyNotification.u16AppID = iter->second.u16AppID;
            keyNotification.u16RegisterID = iter->second.u16RegisterID;
            keyNotification.u16CmdCounter = iter->second.u16CmdCounter;
            vSendKdsStatusEntry(&keyNotification, iter->second.u32CcaSubId, kdsData, length);
         }
      }
   }
   else
   {
      DIA_TR_INF("NO KDS LISTENERS REGISTERED !!!");
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vHandleKdsEntryProperty ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace trc("dia_tclService::vHandleKdsEntryProperty()");

   if ( m_poGWMain )
   {
      switch (poMessage->u8GetOpCode())
      {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:   /* ---   upreg: register client  --- */
         {
            if( !bUpreg(poMessage) )
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
               m_poGWMain->enPostMessage(&oErrorMessage);
            }
            else
            {
               if ( poMessage->u16GetFunctionID() == MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY )
               {
                  vSendKdsStatusEntry(poMessage);
               }
            }
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: /* ---   upreg: unregister client  --- */
         {
            if( !bRelUpreg(poMessage) )
            {
               // create and send error  message
               amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
               m_poGWMain->enPostMessage(&oErrorMessage);
            }
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_GET: /* --- the client asks for some data --- */
         {
            if ( poMessage->u16GetFunctionID() == MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY )
            {
               vSendKdsStatusEntry(poMessage);
            }
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_SET: /* --- the client wants to set a value --- */
         break;

      default:
         {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_INVALID_OPCODE);
            m_poGWMain->enPostMessage(&oErrorMessage);
         }
         break;
      }
   }

   if ( poMessage ) poMessage->bDelete();
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vSendKdsStatusEntry ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace trc("dia_tclService::vSendKdsStatusEntry(poMessage)");

   if (m_poGWMain != NULL)
   {
      fi_tclVisitorMessage oInVisitorMsg(poMessage);
      midw_diagnosticsfi_tclMsgKdsEntryGet oKdsEntryGet;
      (void) oInVisitorMsg.s32GetData(oKdsEntryGet);

      tU32 u32CcaKdsType = (tU32) oKdsEntryGet.KdsEntry;

      ahl_tNotification notification;
      notification.u16AppID = poMessage->u16GetSourceAppID();
      notification.u16RegisterID = poMessage->u16GetRegisterID();
      notification.u16CmdCounter = poMessage->u16GetCmdCounter();
      notification.u16SubID = poMessage->u16GetSourceSubID();

      vSendKdsStatusEntry(&notification, u32CcaKdsType);
   }

}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vSendKdsStatusEntry ( const ahl_tNotification* pNot, tU32 u32KdsEntry )
{
   dia_tclFnctTrace trc("dia_tclService::vSendKdsStatusEntry(const ahl_tNotification*,tU32)");

   if (m_poGWMain != NULL)
   {
      midw_diagnosticsfi_tclMsgKdsEntryStatus  oMyData;

      // Reading Data from KDS
      tU32 u32CcaKdsType = u32KdsEntry;
      tU16 u16Entry      = (tU16)(0x0000ffff & u32CcaKdsType); //get lower 16bit
      tU8  ku8datalength = (tU8) (u32CcaKdsType >> 16); //get higher 16bit

      if ( !ku8datalength )
      {
         vSendError(pNot->u16AppID, pNot->u16RegisterID, pNot->u16CmdCounter, MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY);
         return;
      }

      tU8 au8KdsData[256] = {0};
      (tVoid) ::memset(au8KdsData, 0, 256);

      dia_IKDS* pKDS = 0;
      if (queryConfMgrInterface<dia_IKDS>(&pKDS) == DIA_SUCCESS)
      {
         DIA_TR_INF("dia_tclService::vSendKdsStatusEntry - Trying to read KDS entry 0x%x with length 0x%x", u16Entry, ku8datalength);
         if (pKDS->readKDS(u16Entry, au8KdsData, ku8datalength) == DIA_SUCCESS)
         {
            DIA_TR_INF("dia_SrvHandler_ReadDataByMemory::vProcessRequest - Start reading Data.");

            for ( tU8 i=0; i<ku8datalength; i++ ) {
               oMyData.KdsDataList.itemList.push_back(au8KdsData[i]);
            }
            oMyData.KdsEntry = u32CcaKdsType;

            //Create some sender message.
            fi_tclVisitorMessage oMyStatusMessage(oMyData);

            //Now add targetting data
            oMyStatusMessage.vInitServiceData(
                  CCA_C_U16_APP_DIAGDEBUG,
                  pNot->u16AppID,
                  AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                  0,                                  // StreamCounter
                  pNot->u16RegisterID ,
                  pNot->u16CmdCounter,
                  CCA_C_U16_SRV_DIAGDEBUG,
                  MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS
                  );

            // Reply to requester
            if( AIL_EN_N_NO_ERROR != m_poGWMain->enPostMessage(&oMyStatusMessage))
            {
               // Error in sending the message
               if (!oMyStatusMessage.bDelete())
               {
                  // Error in deleting the message
               }
            }
         }
         else
         {
            vSendError(pNot->u16AppID,pNot->u16RegisterID,pNot->u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY);
         }
      }
      else
      {
         DIA_TR_ERR("dia_SrvHandler_ReadDataByMemory::vProcessRequest - Requesting ConfigManager queryConfMgrInterface FAILED.");
         vSendError(pNot->u16AppID,pNot->u16RegisterID,pNot->u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY);
      }
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vSendKdsStatusEntry ( const ahl_tNotification* pNot, tU32 u32KdsEntry, const tU8 kdsData[], tU16 length )
{
   dia_tclFnctTrace trc("dia_tclService::vSendKdsStatusEntry(const ahl_tNotification*,tU32, tU8[], tU16)");

   if (m_poGWMain != NULL)
   {
      midw_diagnosticsfi_tclMsgKdsEntryStatus  oMyData;

      // Reading Data from KDS
      tU32 u32CcaKdsType = u32KdsEntry;
      tU8  ku8datalength = (tU8) (u32CcaKdsType >> 16); //get higher 16bit

      if ( !ku8datalength )
      {
         vSendError(pNot->u16AppID, pNot->u16RegisterID, pNot->u16CmdCounter, MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY);
         return;
      }

      if ( length == ku8datalength )
      {
         for ( tU8 i=0; i<length; i++ ) oMyData.KdsDataList.itemList.push_back(kdsData[i]);
      }
      oMyData.KdsEntry = u32CcaKdsType;

      //Create some sender message.
      fi_tclVisitorMessage oMyStatusMessage(oMyData);

      //Now add targetting data
      oMyStatusMessage.vInitServiceData(
            CCA_C_U16_APP_DIAGDEBUG,
            pNot->u16AppID,
            AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
            0,                                  // StreamCounter
            pNot->u16RegisterID ,
            pNot->u16CmdCounter,
            CCA_C_U16_SRV_DIAGDEBUG,
            MIDW_DIAGNOSTICSFI_C_U16_KDSENTRY,
            AMT_C_U8_CCAMSG_OPCODE_STATUS
            );

      // Reply to requester
      if( AIL_EN_N_NO_ERROR != m_poGWMain->enPostMessage(&oMyStatusMessage))
      {
         // Error in sending the message
         if (!oMyStatusMessage.bDelete())
         {
            // Error in deleting the message
         }
      }
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vHandleDiagConfigItem ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace trc("dia_tclService::vHandleDiagConfigItem()");
   if ( m_poGWMain )
   {
      switch (poMessage->u8GetOpCode())
      {
      case AMT_C_U8_CCAMSG_OPCODE_GET: /* --- the client asks for some data --- */
      {
         fi_tclVisitorMessage oInVisitorMsg(poMessage);
         midw_diagnosticsfi_tclMsgDiagConfigItemGet oDiagConfigItemGet;
         if (OSAL_OK == oInVisitorMsg.s32GetData(oDiagConfigItemGet))
         {
            tU32 u32CcaConfigItemType = (tU32) oDiagConfigItemGet.ConfigItem;
            DIA_TR_INF("dia_tclService::vHandleDiagConfigItem - Trying to read DiagConfigItem entry = 0x%x",u32CcaConfigItemType);

            ahl_tNotification notification = {0,0,0,0};
            notification.u16AppID = poMessage->u16GetSourceAppID();
            notification.u16RegisterID = poMessage->u16GetRegisterID();
            notification.u16CmdCounter = poMessage->u16GetCmdCounter();
            notification.u16SubID = poMessage->u16GetSourceSubID();

            vSendDiagConfigItemEntry(&notification, u32CcaConfigItemType);
         }
         else
         {
            DIA_ASSERT_ALWAYS();
         }
      }
      break;

      case AMT_C_U8_CCAMSG_OPCODE_SET: /* --- the client asks for some data --- */
      {
#ifdef VARIANT_S_FTR_ENABLE_DIAGCONFIGITEM_OPCODE_SET
         fi_tclVisitorMessage oInVisitorMsg(poMessage);
         midw_diagnosticsfi_tclMsgDiagConfigItemSet oDiagConfigItemSet;
         if (OSAL_OK == oInVisitorMsg.s32GetData(oDiagConfigItemSet))
         {
            ahl_tNotification notification;
            notification.u16AppID = poMessage->u16GetSourceAppID();
            notification.u16RegisterID = poMessage->u16GetRegisterID();
            notification.u16CmdCounter = poMessage->u16GetCmdCounter();
            notification.u16SubID = poMessage->u16GetSourceSubID();

            tU32 u32CcaConfigItemType = (tU32) oDiagConfigItemSet.ConfigItem;
            DIA_TR_INF("dia_tclService::vHandleDiagConfigItem - Trying to write DiagConfigItem entry = 0x%x",u32CcaConfigItemType);

            dia_IConfigItem* pConfigItem = 0;
            if ((queryConfMgrInterface<dia_IConfigItem>(&pConfigItem) == DIA_SUCCESS) && pConfigItem)
            {
               tDiaResult retCode = pConfigItem->setConfigItem(u32CcaConfigItemType, oDiagConfigItemSet.ConfigItemContent.itemList);
               if (retCode == DIA_SUCCESS)
               {
                  vSendDiagConfigItemEntry(&notification, u32CcaConfigItemType);
               }
               else if (retCode == DIA_E_TEMPORARY_NOT_AVAILABLE)
               {
                  DIA_TR_INF("dia_tclService::vHandleDiagConfigItem - Start writing Data asynchronously.");

                  ahl_tNotification* pCookie = OSAL_NEW ahl_tNotification;
                  *pCookie = notification;
                  retCode = pConfigItem->setConfigItem(u32CcaConfigItemType, oDiagConfigItemSet.ConfigItemContent.itemList, this, pCookie);
                  if (retCode != DIA_SUCCESS)
                  {
                      DIA_TR_ERR("dia_tclService::vHandleDiagConfigItem - Requesting ConfigItem Not asynchronously available.");
                  }

               }
               else
               {
                   DIA_TR_ERR("dia_tclService::vHandleDiagConfigItem - Requesting ConfigItem Not available.");
                   vSendError(notification.u16AppID,notification.u16RegisterID,notification.u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
               }
            }
            else
            {
               DIA_TR_ERR("dia_tclService::vHandleDiagConfigItem - Requesting ConfigManager queryInterface FAILED.");
               vSendError(notification.u16AppID,notification.u16RegisterID,notification.u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
            }
         }
         else
         {
            DIA_ASSERT_ALWAYS();
         }
#else
         DIA_TR_ERR("dia_tclService::vHandleDiagConfigItem - AMT_C_U8_CCAMSG_OPCODE_SET NOT SUPPORTED!");
         vSendError(poMessage->u16GetSourceAppID(), poMessage->u16GetRegisterID(), poMessage->u16GetCmdCounter(), MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
#endif
      }
      break;
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:   /* ---   upreg: register client  --- */
      {
         if( !bUpreg(poMessage) )
         {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
         }
         else
         {
            fi_tclVisitorMessage oInVisitorMsg(poMessage);
            midw_diagnosticsfi_tclMsgDiagConfigItemUpReg oDiagConfigItemUpReg;
            if (OSAL_OK == oInVisitorMsg.s32GetData(oDiagConfigItemUpReg))
            {
               tU32 u32CcaConfigItemType = (tU32) oDiagConfigItemUpReg.ConfigItem;
               DIA_TR_INF("dia_tclService::vHandleDiagConfigItem - Trying to read DiagConfigItem entry = 0x%x",u32CcaConfigItemType);

               ahl_tNotification notification = {0,0,0,0};
               notification.u16AppID = poMessage->u16GetSourceAppID();
               notification.u16RegisterID = poMessage->u16GetRegisterID();
               notification.u16CmdCounter = poMessage->u16GetCmdCounter();
               notification.u16SubID = poMessage->u16GetSourceSubID();

               vSendDiagConfigItemEntry(&notification, u32CcaConfigItemType);
            }
            else
            {
               DIA_ASSERT_ALWAYS();
            }
         }

      }
      break;
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: /* ---   unreg: unregister client  --- */
      {
         if( !bRelUpreg(poMessage) )
         {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
         }
         else
         {
            DIA_ASSERT_ALWAYS();
         }
      }
      break;
      default :
      {
         // create and send error  message
         amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_INVALID_OPCODE);
         m_poGWMain->enPostMessage(&oErrorMessage);
      }
      break;
      }
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vSendDiagConfigItemEntry ( const ahl_tNotification* pNot, tU32 u32ConfigItem )
{
   dia_tclFnctTrace trc("dia_tclService::vSendDiagConfigItemEntry(const ahl_tNotification*,tU32)");

   if (m_poGWMain != NULL)
   {
      // Reading Data from DiagConfigItem
      tU32 retcode;
      tU32 u32CcaConfigItemType = u32ConfigItem;
      tU8* configItemData = NULL;
      tU16 configItemDatalength;

      dia_IConfigItem* pConfigItem = 0;
      if ((queryConfMgrInterface<dia_IConfigItem>(&pConfigItem) == DIA_SUCCESS) && pConfigItem)
      {
         DIA_TR_INF("dia_tclService::vSendDiagConfigItemEntry - Trying to read ConfigItem entry 0x%x", u32CcaConfigItemType);

         tDiaResult retCode = pConfigItem->getConfigItem(u32CcaConfigItemType,&configItemData,&configItemDatalength);

         if (retCode == DIA_SUCCESS)
         {
            DIA_TR_INF("dia_tclService::vSendDiagConfigItemEntry - Start reading Data.");

            //DIA_TR_INF("CCA msg TX via TTFis: Length: %d Data: %s", configItemDatalength, dia::utils::bin2str(configItemData,configItemDatalength,' ').c_str());
            //DIA_TR_INF("dia_tclService::vSendDiagConfigItemEntry - Data 0x%x ", *configItemData);

            midw_diagnosticsfi_tclMsgDiagConfigItemStatus oMyData;

            for ( tU16 i=0; i<configItemDatalength; i++ )
            {
               oMyData.ConfigItemContent.itemList.push_back(*configItemData++);
            }
            oMyData.ConfigItem = u32CcaConfigItemType;
            oMyData.ConfigItemValid = TRUE;

            //Create some sender message.
            fi_tclVisitorMessage oMyStatusMessage(oMyData);

            //Now add targetting data
            oMyStatusMessage.vInitServiceData(
                        CCA_C_U16_APP_DIAGDEBUG,
                        pNot->u16AppID,
                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                        0,                                  // StreamCounter
                        pNot->u16RegisterID ,
                        pNot->u16CmdCounter,
                        CCA_C_U16_SRV_DIAGDEBUG,
                        MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM,
                        AMT_C_U8_CCAMSG_OPCODE_STATUS
                        );

            // Reply to requester
            if( AIL_EN_N_NO_ERROR != (retcode = m_poGWMain->enPostMessage(&oMyStatusMessage)))
            {
               DIA_TR_INF("dia_tclService::vSendDiagConfigItemEntry Reply to Requester : %d",retcode);
                // Error in sending the message
                if (!oMyStatusMessage.bDelete())
                {
                   // Error in deleting the message
                }
            }
            else
            {
               DIA_TR_INF("dia_tclService::vSendDiagConfigItemEntry Reply to Requester else : %d",retcode);
                //vSendError(pNot->u16AppID,pNot->u16RegisterID,pNot->u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
            }
         }
         else if (retCode == DIA_E_TEMPORARY_NOT_AVAILABLE)
         {
            DIA_TR_INF("dia_tclService::vSendDiagConfigItemEntry - Start reading Data asynchronously.");

            ahl_tNotification* pCookie = OSAL_NEW ahl_tNotification;
            *pCookie = *pNot;
            retCode = pConfigItem->getConfigItem(u32CcaConfigItemType, this, pCookie);
            if (retCode != DIA_SUCCESS)
            {
                DIA_TR_ERR("dia_tclService::vSendDiagConfigItemEntry - Requesting ConfigItem Not asynchronously available.");
            }

         }
         else
         {
             DIA_TR_ERR("dia_tclService::vSendDiagConfigItemEntry - Requesting ConfigItem Not available.");
             vSendError(pNot->u16AppID,pNot->u16RegisterID,pNot->u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
         }
      }
      else
      {
         DIA_TR_ERR("dia_tclService::vSendDiagConfigItemEntry - Requesting ConfigManager queryInterface FAILED.");
         vSendError(pNot->u16AppID,pNot->u16RegisterID,pNot->u16CmdCounter,MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
      }

    }
}

//-----------------------------------------------------------------------------
tVoid dia_tclService::vSendError (
      tU16 u16TargetID,
      tU16 u16RegisterID,
      tU16 u16CmdCounter,
      tU16 u16Fid
      )
{
   if ( m_poGWMain )
   {
      gm_tclU8Message oErrorMsg (
            CCA_C_U16_APP_DIAGDEBUG,
            u16TargetID,
            u16RegisterID,
            u16CmdCounter,
            CCA_C_U16_SRV_DIAGDEBUG,
            u16Fid,
            AMT_C_U8_CCAMSG_OPCODE_ERROR
            );
      m_poGWMain->enPostMessage(&oErrorMsg);
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_tclService::vSetDefaultSettings(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace trc("dia_tclService::vSetDefaultSettings"); //lint !e1788 ok --> scoped trace

   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain) return;

   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode())
   {
   case AMT_C_U8_CCAMSG_OPCODE_UPREG:
   case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
   case AMT_C_U8_CCAMSG_OPCODE_SET:
   case AMT_C_U8_CCAMSG_OPCODE_GET:
      {
         // create and send error  message
         amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         m_poGWMain->enPostMessage(&oErrorMessage);
      }
      break;

   case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
      {
         fi_tclVisitorMessage oMsg(poMessage);
         midw_diagnosticsfi_tclMsgSetDefaultSettingsMethodStart oData;
         (tVoid) oMsg.s32GetData(oData);
         // get pointer to the Defset object
         // save defset msg
         m_oDefSetMessage.u16AppID = poMessage->u16GetSourceAppID();
         m_oDefSetMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
         m_oDefSetMessage.u16RegisterID = poMessage->u16GetRegisterID();
         m_oDefSetMessage.u16FunctionID = poMessage->u16GetFunctionID();
         m_oDefSetMessage.u16SubID = poMessage->u16GetSourceSubID();
         dia_ISystemSettings* pInterface = 0;
         tDiaResult retCode = queryInterface<dia_ISystemSettings>(&pInterface);
         if ( (retCode == DIA_SUCCESS) && pInterface )
         {
            retCode = pInterface->processSystemSetting(DIA_C_UID_SYSTEM_SETTING_TYPE_HMI,*this, tCookieType(this));
         }

         if ( retCode != DIA_SUCCESS )
         {
            DIA_TR_ERR("##### FAILED TO START SYSTEM SETTINGS OPERATION (HMI FACTORY SETTINGS) !!! #####");
         }
      }
      break;

   default:
      // unknown opcode: just delete message
      break;
   }

   (tVoid) poMessage->bDelete();
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSetFactorySettingsStatus
//
//  DESCRIPTION: sends a SetFactorySettingsStatus status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendDefaultSettingsResult(tBool bResult)
{
   dia_tclFnctTrace trc("dia_tclService::bSendDefaultSettingsResult"); //lint !e1788 ok --> scoped trace

   // create result object

//   if (menSysSetState == EN_USERSYSSET_RUNNING && m_bRunSysSet == FALSE) {
      midw_diagnosticsfi_tclMsgSetDefaultSettingsMethodResult oResultData;
      oResultData.SettingsSuccessful = bResult;
      fi_tclVisitorMessage oResultMsg(oResultData);
      oResultMsg.vInitServiceData(DIA_USED_APP_ID, // source
            m_oDefSetMessage.u16AppID, // Target
            AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
            0, // StreamCounter
            m_oDefSetMessage.u16RegisterID, // RegisterID
            m_oDefSetMessage.u16CmdCounter, // nCmdCounter,
            DIA_USED_SRV_ID, // nServiceID,
            m_oDefSetMessage.u16FunctionID, // function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, AMT_C_U16_DEFAULT_NULL, m_oDefSetMessage.u16SubID);

      NORMAL_M_ASSERT(NULL != m_poGWMain);

//      (tVoid) unsetSysAdapterListener<dia_ISystemSetBroadcastListener>(this);

      // send it
      if (NULL != m_poGWMain) {
         if (m_poGWMain->enPostMessage(&oResultMsg) != AIL_EN_N_NO_ERROR)
         {
            (tVoid) oResultMsg.bDelete();
         }
//         menSysSetState = EN_USERSYSSET_IDLE;
      }
//   } else if (menSysSetState == EN_USERSYSSET_RUNNING && m_bRunSysSet == TRUE) {
//      m_bRunSysSet = FALSE;
//      (tVoid) setSysAdapterListener<dia_ISystemSetBroadcastListener>(this);
//
//      dia_ISystemSetBroadcast* pHandler = OSAL_NULL;
//      querySysAdapterInterface<dia_ISystemSetBroadcast>(&pHandler);
//      if (pHandler) {
//         pHandler->startSystemSetAll(EN_TYPE_DEFAULT_HMI, EN_ACTION_EXECUTE);
//         menSysSetState = EN_USERSYSSET_RUNNING;
//      }
//   } else {
//      // UserSysSetState == IDLE
//      NORMAL_M_ASSERT_ALWAYS();
//   }

   return true;
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    vSetFactorySettingsStatus
//
//  DESCRIPTION: handle a SetFactorySettingsStatus message
//
//  PARAMETER:   message to analyse
//
//  RETURNVALUE: NONE
///////////////////////////////////////////////////////////////////////////
//
tVoid
dia_tclService::vSetFactorySettingsStatus(amt_tclServiceData* poMessage)
{
   dia_tclFnctTrace trc("dia_tclService::vSetFactorySettingsStatus"); //lint !e1788 ok --> scoped trace
   // Msg must come from a client
   OSAL_vAssert(poMessage->bIsServerMessage() == FALSE);

   if (OSAL_NULL == m_poGWMain)
      return;

   // the opcode tells what we should do
   switch (poMessage->u8GetOpCode()) {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if (!bUpreg(poMessage)) {
            // create and send error  message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            DIA_TR_INF("Upreg successful: send a status message");
            // Upreg successful : send a status message.
            midw_diagnosticsfi_tclMsgSetFactorySettingsStatusStatus oStatusData;
            // send dummy status data
            vSendStatusMsg(poMessage, oStatusData);
         }
         (tVoid) poMessage->bDelete();
         break;

      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if (!bRelUpreg(poMessage)) {
            // Relupreg failed: send an error message
            amt_tclServiceDataError oErrorMessage(*poMessage, AMT_C_U16_ERROR_RELUPREG_FAILURE);
            m_poGWMain->enPostMessage(&oErrorMessage);
            DIA_ASSERT_ALWAYS();
         } else {
            // Relupreg successful : no action
            DIA_TR_INF("Relreg successful: no action");
         }
         (tVoid) poMessage->bDelete();
         break;

      case AMT_C_U8_CCAMSG_OPCODE_SET: {
         fi_tclVisitorMessage oMsg(poMessage);
         midw_diagnosticsfi_tclMsgSetFactorySettingsStatusSet oData;
         (tVoid) oMsg.s32GetData(oData);

         if (oData.DefSetStateGroup == ((tU8) midw_fi_tcl_e8_DefSetGroups::FI_EN_DEFSET_GROUP__MAINTENANCE)) {
            // This is a coding update result.received from the HMI
            DIA_TR_INF("---  Forwarded Coding Result to Active Session");
            //m_poGWMain->vCompRes(poMessage);
            //dia_Application::getInstance()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventCompRes(poMessage));
            DIA_TR_INF("############################################################");
            DIA_TR_INF("#");
            DIA_TR_INF("# dia_tclDiagSession::tclEventCompRes NO LONGER SUPPORTED !!");
            DIA_TR_INF("#");
            DIA_TR_INF("###################################################");
         } else {
            (tVoid) poMessage->bDelete();
         }
      }
         break;

      default:
         (tVoid) poMessage->bDelete();
         // unknown opcode: just delete message
         break;
   }
}

///////////////////////////////////////////////////////////////////////////
//  FUNCTION:    dia_tclService::bSendSetFactorySettingsStatus
//
//  DESCRIPTION: sends a SetFactorySettingsStatus status message
//
//  PARAMETER:   see FI parameters
//
//  RETURNVALUE: true - success
///////////////////////////////////////////////////////////////////////////
//
tBool
dia_tclService::bSendSetFactorySettingsStatus(tU8 u8Group, tU8 u8Mode, tBool bSuccess, tU16 u16FirstDiffParam) const
{
   dia_tclFnctTrace trc("dia_tclService::bSendSetFactorySettingsStatus"); //lint !e1788 ok --> scoped trace

   // create status result object
   midw_diagnosticsfi_tclMsgSetFactorySettingsStatusStatus oStatusData;
   oStatusData.DefSetStateGroup = u8Group;
   oStatusData.DefSetStateMode.enType = static_cast<midw_fi_tcl_e8_DefSetMode::tenType>(u8Mode);
   oStatusData.DefSetStateSuccessful = bSuccess;
   oStatusData.DefSetStateFirstDifferentParameterId = u16FirstDiffParam;
   vUpdateAllClients(oStatusData, MIDW_DIAGNOSTICSFI_C_U16_SETFACTORYSETTINGSSTATUS);

   return true;
}

void
dia_tclService::onSystemSettingProcessed ( dia_UID /*typeID*/, const dia_SystemSettingExtendedData& /*extData*/, tDiaResult resultCode, tCookieType /*cookie*/ )
{
   (void) bSendDefaultSettingsResult((resultCode == DIA_SUCCESS) ? TRUE : FALSE);
}

void
dia_tclService::onSystemSettingProcessed ( dia_UID /*typeID*/, const dia_SystemSettingExtendedData& /*extData*/, const std::vector<dia_UID>& /*modules*/, tDiaResult resultCode, tCookieType /*cookie*/ )
{
   (tVoid) bSendDefaultSettingsResult((resultCode == DIA_SUCCESS) ? TRUE : FALSE);
}

//-----------------------------------------------------------------------------

void
dia_tclService::vOnConfigItemUpdate ( tU32 ConfigItem )
{
   dia_tclFnctTrace trc("dia_tclService::vOnConfigItemUpdate");

   DIA_TR_INF("INFORM REGISTERED LISTENERS ABOUT UPDATE OF ConfigItem %d", ConfigItem);

   if ( tMapConfigItemEntryNotify.size() )
   {
      std::map<tU32,TNotificationConfigItem>::iterator iter = tMapConfigItemEntryNotify.begin();
      for ( ; iter != tMapConfigItemEntryNotify.end(); iter++)
      {
         if ( ConfigItem == iter->first )
         {
            ahl_tNotification keyNotification = {0,0,0,0};
            keyNotification.u16AppID = iter->second.u16AppID;
            keyNotification.u16RegisterID = iter->second.u16RegisterID;
            keyNotification.u16CmdCounter = iter->second.u16CmdCounter;
            vSendDiagConfigItemEntry(&keyNotification, ConfigItem);
         }
      }
   }
   else
   {
      DIA_TR_INF("NO CONFIGITEM LISTENERS REGISTERED !!!");
   }
}

//-----------------------------------------------------------------------------

void
dia_tclService::vOnConfigItemGetComplete( tU32 ConfigItem, tDiaResult retCode, std::vector<tU8>& data, void* cookie)
{
   dia_tclFnctTrace trc("dia_tclService::vOnConfigItemGetComplete");

   ahl_tNotification* pNot = (ahl_tNotification*) cookie;
   if (pNot)
   {
      midw_diagnosticsfi_tclMsgDiagConfigItemStatus oMyData;
      oMyData.ConfigItem = ConfigItem;
      oMyData.ConfigItemValid = FALSE;

      if (retCode == DIA_SUCCESS)
      {
         oMyData.ConfigItemValid = TRUE;
         oMyData.ConfigItemContent.itemList = data;
      }

      fi_tclVisitorMessage oMyStatusMessage(oMyData);
      oMyStatusMessage.vInitServiceData(
                  CCA_C_U16_APP_DIAGDEBUG,
                  pNot->u16AppID,
                  AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                  0,
                  pNot->u16RegisterID ,
                  pNot->u16CmdCounter,
                  CCA_C_U16_SRV_DIAGDEBUG,
                  MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS
                  );

      if (m_poGWMain->enPostMessage(&oMyStatusMessage) != AIL_EN_N_NO_ERROR)
      {
          // Error in sending the message
          oMyStatusMessage.bDelete();
      }

      OSAL_DELETE pNot;
   } //if (pNot)
}

//-----------------------------------------------------------------------------

void
dia_tclService::vOnConfigItemSetComplete( tU32 ConfigItem, tDiaResult retCode, void* cookie)
{
   dia_tclFnctTrace trc("dia_tclService::vOnConfigItemSetComplete");

   ahl_tNotification* pNot = (ahl_tNotification*) cookie;
   if (pNot)
   {
      if (retCode == DIA_SUCCESS)
      {
         vSendDiagConfigItemEntry(pNot, ConfigItem);
      }
      else
      {
         vSendError(pNot->u16AppID, pNot->u16RegisterID, pNot->u16CmdCounter, MIDW_DIAGNOSTICSFI_C_U16_DIAGCONFIGITEM);
      }

      OSAL_DELETE pNot;
   } //if (pNot)
}

