/************************************************************************
* FILE:         dia_tclDiagSession.h
* PROJECT:      ARION Prototyp
* SW-COMPONENT: Diagnostic application
*----------------------------------------------------------------------
*
* DESCRIPTION: Virtual base class for the diagnostic session
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2004 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 16.02.05  | 3SOFT Drenkhahn    | initial version
*
*************************************************************************/

#ifndef __INCLUDED_DIA_DIAGSESSION__
#define __INCLUDED_DIA_DIAGSESSION__

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __DIA_UNIT_TESTING__

#ifndef __INCLUDED_DIA_LOCK__
#include "common/framework/application/dia_Lock.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_DEFINES__
#include <common/framework/engine/dia_EngineDefs.h>
#endif

#endif

#ifndef __INCLUDED_DIA_INTERFACE_TIMER_LISTENER__
#include "common/interfaces/dia_ITimerListener.h"
#endif

#ifndef __INCLUDE_DIA_FUNCTOR__
#include <common/framework/application/dia_Functor.h>
#endif

#ifndef __INCLUDED_DIA_MAIN__
#include "common/depricated/dia_main.h"
#endif

class dia_MessageBuffer;
class diagnostics_tclApp;

class dia_tclDiagSession
    : public dia_ITimerListener
{
public:

    /// events //////////////////////////////////////////////////////////
    // unique event identifiers
    enum tenEventId
    {
       enEvIdUnknown = 0,
       enEvIdTimeout,
       enEvIdReqRx,
       enEvIdResReadyAndQuit,
       enEvIdConfTxOk,
       enEvIdIdle,
       enEvIdIntMsgRx,
       enNumberOfEvents
    };

    // Base class for all events used in the diag session state machines
    class tclEvent
    {
    public:
        //! ctor without argument, needed in the main thread loop to allocate memory for a generic event
        tclEvent ( void ) : mEventID(enEvIdUnknown), mpArg(OSAL_NULL), mCookie(0), mUID(serial++) {}
        //! ctor with event id and pointer to event parameter data
        tclEvent ( int id, void* arg, void* /*cookie*/ = 0 ) : mEventID(id), mpArg(arg), mCookie(0), mUID(serial++) {}
        //! destructor
        virtual ~tclEvent ( void ) { mpArg = 0; }

        //! return identifier of event type
        virtual int getEventID ( void ) const { return mEventID; }
        //! return unique message ID of the event object (like serial number)
        virtual dia_UID getUID ( void ) const { return mUID; }
        //! return unique message ID of the event object (like serial number)
        virtual intptr_t getCookie ( void ) const { return mCookie; }

    protected:
        //! event type identifier
        int mEventID;
        //! cookie passed to the constructor
        void* mpArg;
        //! cookie
        int mCookie;
        //! unique object ID
        const dia_UID mUID;
        //! static counter used to generate unique object serial numbers
        static dia_UID serial;
    };

    // internal event used send by the internal timer if the processing time limit is reached.
    class tclEventTimeout : public tclEvent
    {
    public:
        tclEventTimeout ( void ) : tclEvent(static_cast<int>(enEvIdTimeout), OSAL_NULL) {}
    };

    // event used if a diagnostic request is received from an outside source
    class tclEventReqRx : public tclEvent
    {
    public:
        tclEventReqRx ( dia_MessageBuffer* poMsgBuffer ) : tclEvent(static_cast<int>(enEvIdReqRx),poMsgBuffer) {}

        dia_MessageBuffer* poMsgBuffer ( void ) { return reinterpret_cast<dia_MessageBuffer*>(mpArg); }

    protected:
        //! deprecated default constructor without implementation
        tclEventReqRx ( void );
    };

    // internal event used if the service handler finished the processing of the
    // service and the response is ready to be send.
    class tclEventResReadyAndQuit : public tclEvent
    {
    public:
        tclEventResReadyAndQuit() : tclEvent( static_cast<int>(enEvIdResReadyAndQuit), OSAL_NULL) {}
    };


    // event used if a external (cca) message was received.  Pointer
    // to the base message object of type amt_tclBaseMessage is given as
    // parameter.
    class tclEventIntMsgRx : public tclEvent
    {

    public:
       tclEventIntMsgRx ( void* pArg, void* cookie = 0 ) : tclEvent(static_cast<int>(enEvIdIntMsgRx),pArg,cookie) {}
       virtual ~tclEventIntMsgRx ( void ) { mpArg = 0; }

       virtual tDiaResult execute ( void ) = 0;

    protected:
        //! deprecated default constructor without implementation
        tclEventIntMsgRx ( void );
    };

#ifndef __DIA_UNIT_TESTING__
    // event used if a external (cca) message was received.  Pointer
    // to the base message object of type amt_tclBaseMessage is given as
    // parameter.
    class tclEventIntMsgRxCCA : public tclEventIntMsgRx
    {
    public:
       tclEventIntMsgRxCCA ( amt_tclBaseMessage* pMessage ) : tclEventIntMsgRx(pMessage) {}

       virtual ~tclEventIntMsgRxCCA ( void )
       {
          if ( mpArg ) {
             amt_tclBaseMessage* pMsg = static_cast<amt_tclBaseMessage*>(mpArg);
             OSAL_DELETE pMsg;
             mpArg = 0;
          }
       }

       virtual tDiaResult execute ( void )
       {
          dia_tclFnctTrace trc("tclEventIntMsgRxCCA::execute()");

          diagnostics_tclApp* pApplication = diagnostics_tclApp::getInstance();
          if ( (!pApplication) || (!mpArg) ) return DIA_E_INVALID_POINTER;
          amt_tclBaseMessage* pMessage = static_cast<amt_tclBaseMessage*>(mpArg);
          pApplication->vOnDispatchMessage(pMessage);
          return DIA_SUCCESS;
       }

    protected:
        //! deprecated default constructor without implementation
       tclEventIntMsgRxCCA ( void );
    };

#endif

    // event used if a external (cca) message was received.  Pointer
    // to the base message object of type amt_tclBaseMessage is given as
    // parameter.
    class tclEventIntMsgRxGeneric : public tclEventIntMsgRx
    {
    public:
       tclEventIntMsgRxGeneric ( dia_Functor* pFunctor ) : tclEventIntMsgRx(pFunctor) {}

       virtual ~tclEventIntMsgRxGeneric ( void )
       {
          if ( mpArg ) {
             dia_Functor* pFunctor = static_cast<dia_Functor*>(mpArg);
             OSAL_DELETE pFunctor;
             mpArg = 0;
          }
       }

       virtual tDiaResult execute ( void )
       {
          dia_tclFnctTrace trc("tclEventIntMsgRxGeneric::execute()");

          if ( !mpArg ) return DIA_E_INVALID_POINTER;
          dia_Functor* pFunctor = static_cast<dia_Functor*>(mpArg);
          return (*pFunctor)();
       }

    protected:
        //! deprecated default constructor without implementation
       tclEventIntMsgRxGeneric ( void );
    };


    // event used if a diagnostic response was send to the outside source and the
    // peer signals the successful confirmation of the transmission
    class tclEventConfTxOk : public tclEvent
    {
    public:

        tclEventConfTxOk()
            : tclEvent(static_cast<int>(enEvIdConfTxOk), OSAL_NULL) {}
    };

    // Event used when the active session returns to idle.
    class tclEventIdle : public tclEvent
    {
    public:
        tclEventIdle()
            : tclEvent(static_cast<int>(enEvIdIdle), OSAL_NULL) {}
    };

public:

//#ifndef __DIA_UNIT_TESTING__
    dia_tclDiagSession ( void );

    virtual ~dia_tclDiagSession();

//    virtual void vLookupSrvcHandler() = 0;

    // query service handler
    virtual tBool bIsServiceActive ( void );
//#endif

    // send an event to the own state machine
    virtual void vEmit( tclEvent* pEvent ) const = 0;

    // fill the internal diag message buffer with the diag request message
    // received by the tclEventReqRx event.

    tDiaResult setDiagnosisMessageBuffer( /*const*/ tclEventReqRx& );
    // free the internal diag message buffer.  Used if the service processing
    // finished

    void deleteDiagnosisMessageBuffer ( void );


    /* ********** class attributes *********** */

    // one timer per session, used in the processing and post processing stage
    dia_Timer oTimer;

    //! lock the session synchronization object
//    void vLockSession ( void );
//    //! unlock the session synchronization object
//    void vUnlockSession ( void );

    //! overloaded method from class dia_ITimerListener
    virtual void vOnTimerElapsed ( dia_TimerID id );

protected:
    //! lock object used to synchronize multiple session objects, because only one session should be active
//    static dia_Lock mSessionMutex;
//    //! the following flag indicates if the given session object has locked the session mutex
//    tBool bIsSessionLocked;

    // pointer to the diagnostic message buffer or NULL
//    dia_tclDiagMsgBuffer* poDiagMsgBuffer;
    dia_MessageBuffer* poDiagMsgBuffer;

//#ifndef __DIA_UNIT_TESTING__
//    OSAL_tMQueueHandle hMsgQueue;
//    OSAL_tThreadID     hThreadId;
    dia_UID            mEngineID;
//#endif
};

#endif
