/************************************************************************
* FILE:         dia_main.cpp
* PROJECT:      ARION Prototyp
* SW-COMPONENT: diagnostics
*----------------------------------------------------------------------
*
* DESCRIPTION: Main application class for the diagnostics component
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2004-2010 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                | Modification
* 16.02.05  | 3SOFT Drenkhahn       | first adjustments
* 17.10.04  | CM-DI/ESA2 Wersuhn    | initial version
* 26.10.05  | John W P Elkins       | Addition of new drv_diagPr
* 09.04.08  | Prakash G.D(RBEI\ECM6)| Adapted for CCA Port Handling
*************************************************************************/

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------

#ifndef __INCLUDED_DIA_COMMON_APPLICATION__
#include "common/framework/application/dia_common_application.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_TRACKER__
#include "common/framework/platform/cca/dia_ServiceTracker.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE__
#include "common/depricated/dia_tclService.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include <common/framework/application/dia_AppController.h>
#endif

#ifndef __INCLUDED_DIA_EVENT_THREAD__
#include <common/framework/application/dia_EventThread.h>
#endif

#ifndef __INCLUDED_DIA_INTERFACE_SYSTEM_CCA__
#include "common/depricated/SystemAdapter/dia_ISystemCCA.h"
#endif

// unified driver framework
#ifndef __INCLUDED_DIA_UDD_DRIVER_FACADE__
#include "common/framework/udd/udd_DriverFacade.h"
#endif

#ifndef __INCLUDED_DIA_DIAGSESSION_UDS__
#include "common/depricated/dia_tclDiagSessionUds.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include "common/framework/application/dia_ThreadMonitor.h"
#endif

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGNOSTICS_SM
#include "trcGenProj/Header/dia_main.cpp.trc.h"
#endif

#define DIAGNOSTICS_S_IMPORT_INTERFACE_GENERIC
#define DIAGNOSTICS_S_IMPORT_INTERFACE_MSG
#include "fc_diagnosis_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

diagnostics_tclApp* diagnostics_tclApp::poApp = OSAL_NULL;
diagnostics_tclApp* diagnostics_tclApp::m_poServer = OSAL_NULL;

OSAL_tSemHandle diagnostics_tclApp::m_hNotTableSem = OSAL_C_INVALID_HANDLE;

#ifdef __STARTUP_REFACTORING_ENABLE__
#ifndef __INCLUDED_DIA_APPLICATION_LAUNCHER__
#include <common/framework/application/dia_ApplicationLauncher.h>
#endif
#endif
using namespace dia;

//-----------------------------------------------------------------------------

diagnostics_tclApp*
diagnostics_tclApp::getInstance ( void )
{
   if ( m_poServer == OSAL_NULL )
   {
      // 1st stage of construction
      m_poServer = new diagnostics_tclApp();
   }

   return m_poServer;

}
//-----------------------------------------------------------------------------

diagnostics_tclApp::diagnostics_tclApp ( void )
   : poCcaService(OSAL_NULL),
#ifndef VARIANT_S_FTR_DIAG_REMOVE_SESSION_POINTER
     poDiagSessionUds(OSAL_NULL),
     poDiagSession(OSAL_NULL),
#endif
     m_poNotTable(NULL),
#ifdef __STARTUP_REFACTORING_ENABLE__
     mpApp(0)
#else
     mpApp(0),
     mpFactory(0)
#endif
{
   ScopeTrace oTrace("diagnostics_tclApp::diagnostics_tclApp");

   // Make sure that ThreadMonitor is properly created
   // After this point traces have the proper Thread ID
   // in the output.
   createInstanceOfThreadMonitor();
   // this is a singleton!
   DIA_ASSERT(poApp == OSAL_NULL);

   m_poPortMngr = OSAL_NULL;
}

//-----------------------------------------------------------------------------

diagnostics_tclApp::~diagnostics_tclApp()
{
    m_poPortMngr = OSAL_NULL;
    poCcaService = OSAL_NULL;
#ifndef VARIANT_S_FTR_DIAG_REMOVE_SESSION_POINTER
    poDiagSessionUds = OSAL_NULL;
    poDiagSession = OSAL_NULL;
#endif
    m_poNotTable = OSAL_NULL;
    m_poServer = OSAL_NULL;
    mpApp = OSAL_NULL;
#ifndef __STARTUP_REFACTORING_ENABLE__
    mpFactory = OSAL_NULL;
#endif
//  (void) unsetInterfaceListener<dia::IApplicationStateControlListener>(this);
}

//-----------------------------------------------------------------------------

tBool
diagnostics_tclApp::bOnInit()
{
   ScopeTrace trc(DIA_TRC_SM,"diagnostics_tclApp::bOnInit()");

   (void) dia::registerThread("DIA_CCA");

#ifdef __STARTUP_REFACTORING_ENABLE__
   tBool retCode = FALSE;
   ApplicationLauncher* pAppLauncher = getInstanceOfApplicationLauncher();
   if ( pAppLauncher && (pAppLauncher->launchApplication() != DIA_FAILED) )
   {
      retCode = TRUE;
   }
   (void) setInterfaceListener<dia::IApplicationStateControlListener>(this);
#else
   tBool retCode = TRUE;
   // create the factory object
   DIA_TR_USR1("diagnostics_tclApp::bOnInit - Create Factory Plugin");
   mpFactory = getInstanceOfFactory();
   if ( mpFactory )
   {
      // initialize level 0
      if ( mpFactory->setup(DIA_EN_INITLEVEL_0) != DIA_SUCCESS ) return FALSE;
      DIA_TR_USR1("diagnostics_tclApp: Successfully initialized (DIA_EN_INITLEVEL_0)");

      (void) setInterfaceListener<dia::IApplicationStateControlListener>(this);

      // initialize level 1
      if ( mpFactory->setup(DIA_EN_INITLEVEL_1) != DIA_SUCCESS ) return FALSE;
      DIA_TR_USR1("diagnostics_tclApp: Successfully initialized (DIA_EN_INITLEVEL_1)");

      // initialize level 2
      if ( mpFactory->setup(DIA_EN_INITLEVEL_2) != DIA_SUCCESS ) return FALSE;
      DIA_TR_USR1("diagnostics_tclApp: Successfully initialized (DIA_EN_INITLEVEL_2)");

      DIA_TR_USR1("diagnostics_tclApp::bOnInit - Create Customer Lookup-Lists");
      (void) mpFactory->makeServiceHandlers(DIA_E_TYPE_CUSTOMER);

      // Create the cca service handler.  It analyses the received cca
      // messages and calls the corresponding handler function
      DIA_TR_USR1("diagnostics_tclApp::bOnInit - Create CCA service handler");
      poCcaService = OSAL_NEW dia_tclService(this);
      DIA_TR_USR1("diagnostics_tclApp::bOnInit - Create ClientHandlers / SystemAdapters");
      mpFactory->makeSystemAdapters();

      // initialize level 3
      if ( mpFactory->setup(DIA_EN_INITLEVEL_3) != DIA_SUCCESS ) return FALSE;
      DIA_TR_USR1("diagnostics_tclApp: Successfully initialized (DIA_EN_INITLEVEL_3)");

      // setup internal event thread
      DIA_TR_USR1("diagnostics_tclApp::bOnInit - Starting worker thread");
      // start event waiting thread for extComms and Defset (not required for early customer requests
      dia_EventThread* pEventThread = getInstanceOfdiaEventThread();
      DIA_ASSERT(NULL != pEventThread);

      // initialize level 4
      if ( mpFactory->setup(DIA_EN_INITLEVEL_4) != DIA_SUCCESS ) return FALSE;
      DIA_TR_USR1("diagnostics_tclApp: Successfully initialized (DIA_EN_INITLEVEL_4)");


      /////////////////////////////////////////////////////////////////////////
      //
      // Runlevel 1 is reached
      //
      /////////////////////////////////////////////////////////////////////////

      getInstanceOfAppController()->vOnEvent(eAppCtrlEvent_InternalSetupDone);

#ifdef VARIANT_S_FTR_ENABLE_FEAT_SET_DIAG_VAG
      // Create CCA Port Manager
      m_poPortMngr = new ahl_tclClientPortMngr(this);
#endif
      // create semaphore to protect the NotificationTable
      DIA_TR_USR1("diagnostics_tclApp::bOnInit - Create Notification table and semaphore");
      tS32 s32Result = OSAL_s32SemaphoreCreate(DIAGNOSTICS_NOTTABLE_SEM_NAME, &m_hNotTableSem, 1);
      if(OSAL_ERROR == s32Result)
      {
         m_hNotTableSem = OSAL_C_INVALID_HANDLE;
         DIA_TR_ERR("diagnostics_tclApp::bOnInit - Creating semaphore FAILED!");
      }
      //create NotificationTable where information about all registered clients is stored
      m_poNotTable = OSAL_NEW ahl_tclNotificationTable (DIAGNOSTICS_FUNCARRAY_SIZE);
      DIA_ASSERT(OSAL_NULL != m_poNotTable);
   }
#endif

    poApp = this;

    return retCode;
}   // End of diagnostics_tclApp::bOnInit

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnApplicationClose(void)
{
   dia_tclFnctTrace trc("diagnostics_tclApp::vOnApplicationClose()");
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnCVMEvent (tU32 u32CVMEvent)
{
   dia_tclFnctTrace trc("diagnostics_tclApp::vOnCVMEvent()");
   DIA_TR_USR1("diagnostics_tclApp::vOnCVMEvent - Received CVM Event: %d ", u32CVMEvent);
}

//-----------------------------------------------------------------------------

tBool
diagnostics_tclApp::bGetServiceVersion ( tU16  u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion )
{
   dia_tclFnctTrace trc("diagnostics_tclApp::bGetServiceVersion()");

   tBool bSuccess = FALSE;

   // we offer only one service DIA_USED_SRV_ID
   if (u16ServiceID == DIA_USED_SRV_ID)
   {
      rfu16MajorVersion = MIDW_DIAGNOSTICSFI_C_U16_SERVICE_MAJORVERSION;
      rfu16MinorVersion = MIDW_DIAGNOSTICSFI_C_U16_SERVICE_MINORVERSION;
      rfu16PatchVersion = 0;
      bSuccess = TRUE;
   }

   return bSuccess;
}

//------------------------------------------------------------------------------

ail_tenCommunicationError
diagnostics_tclApp::enPostMessage ( amt_tclServiceData* poMessage, tBool, tU32 priority )
{
   tU16 srvID  = poMessage->u16GetServiceID();
   tU16 appID  = poMessage->u16GetTargetAppID();
   tU16 funcID = poMessage->u16GetFunctionID();
   tU8  opcode = poMessage->u8GetOpCode();

   DIA_TR_SM("CCA msg TX: AppId: 0x%04x SID: 0x%04x FID: 0x%04x OpCode: 0x%02x",appID,srvID,funcID,opcode);

   return ail_tclOneThreadAppInterface::enPostMessage(poMessage, TRUE, priority);
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnAsyncRegisterConf ( tU16 regID, tU16 appID, tU16 srvID, tU16 subID )
{
   DIA_TR_SM("ServiceRegistrationConfirmation RX - Server: 0x%04x, Service: 0x%04x, RegId: %d, SourceSubId: %d", appID, srvID, regID, subID);

   dia_ISystemCCA* pCCA = 0;
   querySysAdapterInterface<dia_ISystemCCA>(&pCCA);
   DIA_TR_USR1("diagnostics_tclApp::vOnAsyncRegisterConf - Other RegId received and forwarded to CCA SystemAdapter");

   if ( !pCCA || (pCCA->onRegistrationID(regID,appID,srvID,subID) != DIA_SUCCESS) )
   {
      DIA_TR_USR1("diagnostics_tclApp::vOnAsyncRegisterConf - Received unknown RegId. Discarded!");
   }

   dia_ServiceTracker* pSrvTracker = getInstanceOfServiceTracker();
   if ( pSrvTracker ) pSrvTracker->onServiceRegistration(srvID,appID,regID);
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnAsyncRegisterConfirmation ( tU16 regID, tU16 appID, tU16 srvID, tU16 subID )
{
   DIA_TR_SM("ServiceRegistrationConfirmation RX - Server: 0x%04x, Service: 0x%04x, RegId: %d, SourceSubId: %d", appID, srvID, regID, subID);

   dia_ISystemCCA* pCCA = 0;
   querySysAdapterInterface<dia_ISystemCCA>(&pCCA);
   DIA_TR_USR1("diagnostics_tclApp::vOnAsyncRegisterConfirmation - Other RegId received and forwarded to CCA SystemAdapter");

   if ( !pCCA || (pCCA->onRegistrationID(regID,appID,srvID,subID) != DIA_SUCCESS) )
   {
      DIA_TR_USR1("diagnostics_tclApp::vOnAsyncRegisterConfirmation - Received unknown RegId. Discarded!");
   }

   dia_ServiceTracker* pSrvTracker = getInstanceOfServiceTracker();
   if ( pSrvTracker ) pSrvTracker->onServiceRegistration(srvID,appID,regID);
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnServiceState ( tU16 srvID, tU16 appID, tU16 regID, tU8 srvState, tU16 subID )
{
   DIA_TR_SM("ServiceStateChange RX - Server: 0x%04x, Service: 0x%04x, RegId: %d, ServiceState: %d, SourceSubId: %d", appID, srvID, regID, srvState, subID);


   DIA_TR_USR1("diagnostics_tclApp::vOnServiceState - Other ServiceStateChange received and forwarded to CCA SystemAdapter");

   dia_ISystemCCA* pCCA = 0;
   querySysAdapterInterface<dia_ISystemCCA>(&pCCA);
   if ( !pCCA || (pCCA->onServiceState (srvID,appID,regID,srvState,subID) != DIA_SUCCESS) )
   {
      DIA_TR_SYS("diagnostics_tclApp::vOnServiceState - Received unknown ServiceStateChange. Discarded!");
   }

   dia_ServiceTracker* pSrvTracker = getInstanceOfServiceTracker();
   if ( pSrvTracker )
   {
       dia_enServiceState newState = (srvState == AMT_C_U8_SVCSTATE_AVAILABLE) ? DIA_EN_SERVICE_STATE_AVAILABLE : DIA_EN_SERVICE_STATE_NOT_AVAILABLE;
       pSrvTracker->onServiceState(srvID,appID,newState);
   }

   // call the corresponding method of the super class to give AIL the chance to do the rest
   ail_tclOneThreadAppInterface::vOnServiceState(srvID,appID,regID,srvState,subID);
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnNewAppState ( tU32 srcAppState, tU32 destAppState )
{
   ScopeTrace oTrace(DIA_TRC_SM,"diagnostics_tclApp::vOnNewAppStateChange()");

   DIA_TR_SM("--- RECEIVED REQUEST TO CHANGE STATE FROM %d to %d !!!",srcAppState,destAppState );

   dia::IApplicationStateControl* pAppStateCtrl = 0;
   querySysAdapterInterface<dia::IApplicationStateControl>(&pAppStateCtrl);
   if ( pAppStateCtrl )
   {
      (void) setInterfaceListener<dia::IApplicationStateControlListener>(this);
      tDiaResult retCode = pAppStateCtrl->changeApplicationState((dia::ApplicationStateID) srcAppState, (dia::ApplicationStateID) destAppState);
      if ( retCode != DIA_SUCCESS )
      {
         DIA_TR_SM("--- UNABLE TO REQUEST STATE CHANGE (ERRCODE = 0x%08x) !!!",retCode);
         (void) unsetInterfaceListener<dia::IApplicationStateControlListener>(this);
      }
   }
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::onApplicationStateChangePending ( dia::ApplicationStateID srcStateID, dia::ApplicationStateID /*destStateID*/ )
{
   ScopeTrace oTrace(DIA_TRC_SM,"diagnostics_tclApp::onApplicationStateChangePending()");

   // the power manager is calling the application a second ( or a third! ) time
   DIA_TR_INF("State switch already initiated ---  Apply for Delay - DIA_C_U32_MAX_STATE_CHANGE_DELAY");

   // in maxium an application can request a delay 3 times ( 3 * 20sec. = 60sec. )
   // confirm agaim the state you already have entered at Framework level
   vAppStateChanged((tU32) srcStateID, DIA_C_U32_MAX_STATE_CHANGE_DELAY);
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::onApplicationStateChanged ( dia::ApplicationStateID stateID )
{
   ScopeTrace oTrace(DIA_TRC_SM,"diagnostics_tclApp::onApplicationStateChanged(stateID)");

   DIA_TR_SM("##### CHANGED STATE TO 0x%02x #####",stateID);
   vAppStateChanged((tU32) stateID, 0);
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnNewMessage ( amt_tclBaseMessage* poMessage )
{
   if(poMessage == OSAL_NULL)
   {
      DIA_TR_ERR("diagnostics_tclApp::vOnNewMessage - Invalid Message Pointer.");
      return;
   }

   amt_tclBaseMessage* poMyMessage = (amt_tclBaseMessage*) poMessage->poClone();

   if ( poMyMessage )
   {
      amt_tclServiceData oServiceData(poMyMessage);

      tU16 srvID  = oServiceData.u16GetServiceID();
      tU16 appID  = oServiceData.u16GetSourceAppID();
      tU16 funcID = oServiceData.u16GetFunctionID();
      tU8  opcode = oServiceData.u8GetOpCode();

      DIA_TR_SM("CCA msg RX: AppId: 0x%04x SID: 0x%04x FID: 0x%04x OpCode: 0x%02x", appID, srvID, funcID, opcode);

#ifdef VARIANT_S_FTR_DIAG_REMOVE_SESSION_POINTER
      getInstanceOfApplication()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxCCA(poMyMessage));
#else
      if ( poDiagSession == OSAL_NULL && poDiagSessionUds == OSAL_NULL)
      {
         DIA_TR_ERR("diagnostics_tclApp::vOnNewMessage - No valid Session Pointer. Incoming CCA msg discarded.");
         DIA_ASSERT_ALWAYS();
         (void) poMyMessage->bDelete();
         OSAL_DELETE poMyMessage;
      }
      else
      {
         if(poDiagSession == OSAL_NULL)
         {
            DIA_TR_USR1("diagnostics_tclApp::vOnNewMessage - Forwarding incoming CCA msg to UDS session for thread change");
            if ( poDiagSessionUds ) poDiagSessionUds->vEmit(OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxCCA(poMyMessage));
         }
         else
         {
            DIA_TR_USR1("diagnostics_tclApp::vOnNewMessage - Forwarding incoming CCA msg to ACTIVE session for thread change");
            poDiagSession->vEmit(OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxCCA(poMyMessage));
         }
      }
#endif
   }
   else
   {
      (void) poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnDispatchMessage ( amt_tclBaseMessage* poMessage )
{
   dia_tclFnctTrace oTrace("diagnostics_tclApp::vOnDispatchMessage(amt_tclBaseMessage*)");

   // check if the received message is a AMT Service data message
   if ( poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA )
   {
      // cast the received message to a AMT Service data message so we can
      // access the standard members
      amt_tclServiceData oServiceData(poMessage);

      tU16 u16ServiceID   = oServiceData.u16GetServiceID();
      tU16 u16SourceAppID = oServiceData.u16GetSourceAppID();
      tU16 u16FunctionID  = oServiceData.u16GetFunctionID();

      DIA_TR_INF("---  Message Received (APPID=0x%04x, SRVID=0x%04x, FUNID=0x%04x)", u16SourceAppID, u16ServiceID, u16FunctionID);

#ifndef VARIANT_S_FTR_REPLACE_DIA_TCLSERVICE
      //STC2HI: Please implement a SystemAdapter for CCA_C_U16_SRV_DIAGDEBUG (see default case below)
	if ( u16ServiceID == CCA_C_U16_SRV_DIAGDEBUG )
         {
            DIA_TR_INF("---  Received message with service ID --- DIA_USED_SRV_ID !!!");

            // give the message to the cca service handler for analysing.
            // vDispatchMessage is a framework function. it will compare the
            // function ID with the message map an call the corresponding
            // function
            if ( poCcaService != OSAL_NULL )
            {
               poCcaService->vDispatchMessage(&oServiceData);
            }
         }
	else
         {
            dia_ISystemCCA* pCCA = 0;
            querySysAdapterInterface<dia_ISystemCCA>(&pCCA);
            if ( !pCCA || (pCCA->onDispatchMessage(&oServiceData) != DIA_SUCCESS) )
             {
                 // we have received a message with an unkown service ID. this is an   // error
                  DIA_TR_BAS("Received message with service ID --- !! UNKNOWN Sid !! ---");
                 // release msg here.  vDispatchMessage() does this in the other cases
                 (void) poMessage->bDelete();
             }
         }
#else
	dia_ISystemCCA* pCCA = 0;
	querySysAdapterInterface<dia_ISystemCCA>(&pCCA);
	if ( !pCCA || (pCCA->onDispatchMessage(&oServiceData) != DIA_SUCCESS) )
	{
		DIA_TR_BAS("Received message with service ID --- !! UNKNOWN Sid !! ---");
		(void) poMessage->bDelete();
      }
#endif
   }
   else
   {
      DIA_TR_INF("---  Received message with service ID --- !! UNKNOWN format !! ---");
      (void) poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

void
diagnostics_tclApp::vOnUnregister ( tU16 u16ServiceID, tU16 u16RegisterID )
{
   if(u16ServiceID == DIA_USED_SRV_ID)
   {
      if ( (m_poNotTable != OSAL_NULL) && (m_hNotTableSem != OSAL_C_INVALID_HANDLE) )
      {
         ahl_bEnterCritical(m_hNotTableSem);
         m_poNotTable->bRemoveAllEntriesWithRegID(u16RegisterID);
         ahl_bReleaseCritical(m_hNotTableSem);
         DIA_TR_USR1("diagnostics_tclApp::vOnUnregister - Following RegId unregistered for diag service: %d", u16RegisterID);
      }
   }
}
