/**
 * @file WifiSetupManager.cpp
 * @author RBEI/ECO32 Karthikeyan Madeswaran
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "WifiSetupManager.h"
#include "WBLMessages.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLServices", WifiSetupManager, Debug);
using boost::static_pointer_cast;

WifiSetupManager::WifiSetupManager()
{
   LOG_INFO("WifiSetupManager::WifiSetupManager Constructor");
   _poAPSetUp = new WifiSetupService(sWblWifisetupAPPortName);
   WBL_ASSERT(nullptr == _poAPSetUp);
   _poSTASetUp = new WifiSetupService(sWblWifisetupSTAPortName);
   WBL_ASSERT(nullptr == _poSTASetUp);
   _wifiSetupList.clear();
}//WifiSetupManager::WifiSetupManager

WifiSetupManager::~WifiSetupManager()
{
   RELEASE_MEM(_poAPSetUp);
   RELEASE_MEM(_poSTASetUp);
   _wifiSetupList.clear();
}//WifiSetupManager::~WifiSetupManager

void WifiSetupManager::onWBLServiceResponse(const ::boost::shared_ptr<WBLBaseMsg> msg)
{
   LOG_INFO("WifiSetupManager::WifiSetupManager onWBLServiceResponse");
   WifiSetupService* poSetup = nullptr;
   switch (msg->getMsgID())
   {
      case WBL_MSG_RESTORE_SETTINGS_REQ:
      {
         ::boost::shared_ptr<RestoreFactorySettingsMsg> resetMsg = static_pointer_cast<RestoreFactorySettingsMsg>(msg);
         if (WIFI_MODE_STA == resetMsg->getWifiMode())
         {
            poSetup = getSTAWifiSetup();
         }
         else if ((WIFI_MODE_AP1 == resetMsg->getWifiMode()) || (WIFI_MODE_AP2 == resetMsg->getWifiMode()))
         {
            poSetup = getAPWifiSetup();
         }
         if (poSetup)
         {
            poSetup->onRestoreFactorySettingsResponse(resetMsg->getErrorCode(), resetMsg->getMsgToken());
         }
      }
      break;
      default:
         LOG_ERROR("WifiSetupManager::onWBLServiceResponse:Unhandled response");
      break;
   }
}

void WifiSetupManager::onActiveWifiSetupsChanged(const ActiveWifiSetupsMsg& msg)
{
   LOG_INFO("WifiSetupManager::onActiveWifiSetupsChanged Entered");
   WifiSetUpList list;
   msg.getWifiSetUps(list);

   for (auto it = list.begin(); it != list.end(); ++it)
   {
      WifiSetUpItem item = *it;
      if (!isSetupExist(item, _wifiSetupList))
      {
         LOG_INFO("onActiveWifiSetupsChanged new Setup detected %s", item.setupObjPath.c_str());
         _wifiSetupList.push_back(item);
         updateWifiSetup(item);
      }
      else if (!isSetupPropertiesChanged(item))
      {
         LOG_INFO("onActiveWifiSetupsChanged change in setup properties %s", item.setupObjPath.c_str());
         updateWifiSetup(item);
      }
      else
      {
         LOG_INFO("onActiveWifiSetupsChanged No change in the setup %s", item.setupObjPath.c_str());
      }
   }
   if(isSetupRemoved(list))
   {
      LOG_INFO("onActiveWifiSetupsChanged Setup Removed Successfully");
   }
}//WifiSetupManager::onActiveWifiSetupsChanged

bool WifiSetupManager::isSetupExist(const WifiSetUpItem& item, const WifiSetUpList list)
{
   LOG_INFO("WifiSetupManager::isSetupExist Entered");
   for (auto it = list.begin(); it != list.end(); ++it)
   {
      WifiSetUpItem cuurItem = *it;
      if (cuurItem.setupObjPath == item.setupObjPath)
      {
         return true;
      }
   }
   return false;
}//WifiSetupManager::isSetupExist

bool WifiSetupManager::isSetupPropertiesChanged(const WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupManager::isSetupPropertiesChanged Entered");
   for (auto it = _wifiSetupList.begin(); it != _wifiSetupList.end(); ++it)
   {
      WifiSetUpItem cuurItem = *it;
      if (cuurItem.property == item.property)
      {
         return true;
      }
   }
   return false;
}//WifiSetupManager::isSetupPropertiesChanged

bool WifiSetupManager::isSetupRemoved(const WifiSetUpList list)
{
   LOG_INFO("WifiSetupManager::isSetupRemoved Entered");
   for (auto it = _wifiSetupList.begin(); it != _wifiSetupList.end(); ++it)
   {
      WifiSetUpItem item= *it;
      if (!isSetupExist(item, list))
      {
         LOG_INFO("WifiSetupManager::isSetupRemoved Setup Removed %s", (*it).setupObjPath.c_str());
         _wifiSetupList.erase(it);
         //update interface removed
         updateInterfaceRemoved(item);
         return true;
      }
   }
   return false;
}//WifiSetupManager::isSetupRemoved

void WifiSetupManager::updateWifiSetup(WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupManager::updateWifiSetup Entered");
   if (item.property.mode == WIFI_MODE_AP1)
   {
      _poAPSetUp->updateSetupProperties(item);
   }
   else if (item.property.mode == WIFI_MODE_STA)
   {
      _poSTASetUp->updateSetupProperties(item);
   }
}//WifiSetupManager::updateWifiSetup

void WifiSetupManager::updateInterfaceAdded(WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupManager::updateInterfaceAdded Entered");
   PropertiesList propertiesList;
   ::std::string interfacename;
   ::std::string objpath;
   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   ObjectManagerService* objectManager = serviceFactory->getObjectManagerServiceIf();
   if (item.property.mode == WIFI_MODE_AP1)
   {
      _poAPSetUp->getAllProperties(propertiesList);
      _poAPSetUp->getInterfaceName(interfacename);
      _poAPSetUp->getDBusObjectPath(objpath);
   }
   else if (item.property.mode == WIFI_MODE_STA)
   {
      _poSTASetUp->getAllProperties(propertiesList);
      _poSTASetUp->getInterfaceName(interfacename);
      _poSTASetUp->getDBusObjectPath(objpath);
   }
   objectManager->updateInterfaceAddedSignal(objpath,interfacename,propertiesList);
}//WifiSetupManager::updateInterfaceAdded

void WifiSetupManager::updateInterfaceRemoved(WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupManager::updateInterfaceRemoved Entered");
   ::std::vector< ::std::string> interface;
   ::std::string interfacename;
   ::std::string objpath;
   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   ObjectManagerService* objectManager = serviceFactory->getObjectManagerServiceIf();
   if (item.property.mode == WIFI_MODE_AP1)
   {
      _poAPSetUp->getDBusObjectPath(objpath);
      _poAPSetUp->getInterfaceName(interfacename);
   }
   else if (item.property.mode == WIFI_MODE_STA)
   {
      _poSTASetUp->getDBusObjectPath(objpath);
      _poSTASetUp->getInterfaceName(interfacename);
   }
   interface.push_back(interfacename);
   objectManager->updateInterfaceRemovedSignal(objpath,interface);
}//WifiSetupManager::updateInterfaceRemoved

WifiSetupService* WifiSetupManager::getAPWifiSetup()
{
   return _poAPSetUp;
}//WifiSetupManager::getAPWifiSetup

WifiSetupService* WifiSetupManager::getSTAWifiSetup()
{
   return _poSTASetUp;
}//WifiSetupManager::getSTAWifiSetup

//Todo: Ensure Setup Availability
bool WifiSetupManager::isAPSetupAvaialable()
{
   bool bRet = false;
   WifiSetUpItem item;
   if (isSetupExist(_poAPSetUp->getWifiSetupItem(), _wifiSetupList))
   {
      bRet = true;
   }
   return bRet;
}//WifiSetupManager::isAPSetupAvaialable

bool WifiSetupManager::isSTASetupAvaialable()
{
   bool bRet = false;
   if (isSetupExist(_poSTASetUp->getWifiSetupItem(), _wifiSetupList))
   {
      bRet = true;
   }
   return bRet;
}//WifiSetupManager::isSTASetupAvaialable

void WifiSetupManager::getAllProperties(::std::map < ::std::string, ::asf::dbus::DBusVariant >& propertiesMap)
{
WBL_INTENTIONALLY_UNUSED(propertiesMap);
}//WifiSetupManager::getAllProperties

void WifiSetupManager::getDBusObjectPath(::std::string& objPath)
{
   WBL_INTENTIONALLY_UNUSED(objPath);
}//WifiSetupManager::getDBusObjectPath

void WifiSetupManager::getInterfaceName(::std::string& interfaceName)
{
   WBL_INTENTIONALLY_UNUSED(interfaceName);
}//WifiSetupManager::getInterfaceName

} //namespace bosch
} //namespace org
