/**
 * @file LoopbackService.cpp
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_business_logic
 *
 * @brief
 *
 * @{
 */

#include "LoopbackService.h"
#include "WBLPortsDefines.h"
#include "GenlEvent.h"
#include "GenericNetlink.h"
#include "Poller.h"
#include "WBLDefines.h"
#include "WifiRadio.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLServices", cLoopbackService, Debug);

cLoopbackService::cLoopbackService() :
		WBLASFServiceStub(sWBLServicePort), cNlObserver(NL_MESSAGE_TYPE_GENERIC)

{
	int iRet;

	LOG_DEBUG("Creating the loopback stub");
	iRet = cGenNetLink::getInstance()->iRegisterForNlEvents(this);
	if (iRet < 0)
		LOG_ERROR ("Failed to register for the genl events: %s/%d",
				strerror(-iRet), -iRet);
	cGenNetLink::getInstance()->vTriggerUpdate();
}

cLoopbackService::~cLoopbackService()
{
   try
   {
      int iRet;

      iRet = cGenNetLink::getInstance()->iUnRegisterFromNlEvents(this);
      if (iRet < 0)
         LOG_ERROR ("Failed to register for the genl events: %s/%d",
               strerror(-iRet), -iRet);
   }catch(...){}
}

void cLoopbackService::bNetlinkEvent(cNlMessage *msg)
{
	cGenlMessage *gmsg = NULL;
	cRegulatoryUpdate *regUpdate = NULL;
	cMlmeEventConnect *cevent = NULL;
	cMlmeEventDisconnect *devent = NULL;

	if (!msg)
		return;

	switch (msg->getNlMessageId())
	{
	case NL_MESSAGE_TYPE_GENERIC:
	{
		gmsg = static_cast<cGenlMessage *> (msg);
		switch (gmsg->getGenlMessageId())
		{
		case GENL_EVENT_REGULATORY_UPDATE:
		{
			regUpdate = GenlEventToRegulatoryUpdate(static_cast<cGenlRegEvent *> (msg));
			if(regUpdate)
				onRegulatoryUpdate(*regUpdate);
			delete regUpdate;
		}
		break;

		case GENL_EVENT_MLME_CONNECT:
		{
			cevent = GenlEventToMlmeConnectEvent(static_cast<cGenlMlmeConnectEvent *>(msg));
			if (cevent)
				onMlmeConnectUpdate(*cevent);
			delete cevent;
		}
		break;

		case GENL_EVENT_MLME_DISCONNECT:
		{
			devent = GenlEventToMlmeDisConnectEvent(static_cast<cGenlMlmeDisConnectEvent *>(msg));
			if (devent)
				onMlmeDisconnectUpdate(*devent);
			delete devent;
		}
		break;

		default:
			break;
		}
	}
	break;
	default:
		break;
	}
}

void cLoopbackService::onRegulatoryUpdate(const cRegulatoryUpdate &update)
{
	bool signal = false;

	/*
	 * Usually ASF updates only whenever there is a change in the
	 * property which is how ideally it should be. But I need everything
	 * to be dumped at the other end */
	if (getCurrentRegSettings() == update)
		signal = true;

	setCurrentRegSettings(update);

	if (signal)
		sendCurrentRegSettingsNotification();
}

void cLoopbackService::onMlmeConnectUpdate(const cMlmeEventConnect &event)
{
	bool signal = false;

	if (getMediaConnected() == event)
		signal = true;

	setMediaConnected(event);

	if (signal)
		sendMediaConnectedNotification();
}

void cLoopbackService::onMlmeDisconnectUpdate(const cMlmeEventDisconnect &event)
{
	bool signal = false;

	if (getMediaDisconnected() == event)
		signal = true;

	setMediaDisconnected(event);

	if (signal)
		sendMediaDisconnectedNotification();
}

void cLoopbackService::onSetpropertyRequest (const ::boost::shared_ptr< SetpropertyRequest > &setproperty)
{
    std::string message;

    message = setproperty->getMessage();
    LOG_INFO ("Country to be set: %s", message.c_str());
    /* Just fake it */
    sendSetpropertyResponse(true);
}

void cLoopbackService::onStarteventloopRequest (const ::boost::shared_ptr< StarteventloopRequest > &starteventloop)
{
   	(void) starteventloop;

	LOG_INFO("On Start Event Loop");
	sendStarteventloopResponse();
	cPoller::getInstance()->vRunEventLoop();
}

	}
}

/** @} */
