/**
 * @file LastIntendedModeService.cpp
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_business_logic
 *
 * @brief Implements the stub for LastIntendedMode
 * service of WBL.
 *
 * @{
 */

#include <string>

#include "LastIntendedModeService.h"
#include "WBLPortsDefines.h"
#include "DbusHelper.h"
#include "WBLASFComponent.h"
#include "WBLTypes.h"
#include "WBLTypeProperties.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLServices", LastIntendedModeService, Debug);

LastIntendedModeService::LastIntendedModeService(::std::string sPortName) :
            LastIntendedModeStub(sPortName)

{
   LOG_INFO("LastIntendedModeService:: %s", __func__);
   readLIMOnStartup();

} //LastIntendedModeService::LastIntendedModeService

LastIntendedModeService::~LastIntendedModeService()
{
   try
   {
      LOG_INFO("LastIntendedModeService:: %s", __func__);
      updateInterfaceRemoved();
   }catch(...){}
} //LastIntendedModeService::~LastIntendedModeService

void LastIntendedModeService::readLIMOnStartup()
{
   LOG_INFO("LastIntendedModeService:: %s", __func__);
   DBManagerFactory* dbmgrFactory = DBManagerFactory::getInstance();
   IDBManager* dbmgrIf = (dbmgrFactory) ? (dbmgrFactory->getDBManagerIf()) : (nullptr);
   if (dbmgrIf)
   {
      LIMList list;
      if(dbmgrIf->readLIMFromDB(list))
      {
         populateLIMItems(list);
         populateLIMUpdate(list);
      }
   }

} //void LastIntendedModeService::readLIMOnStartup

void LastIntendedModeService::onSetupAdded(const WifiSetUpItem& setupItem)
{
   LOG_INFO("LastIntendedModeService::%s for mode %s", __func__,WIFI_MODE_TO_STR(setupItem.property.mode));
   LIMList limList;
 
   auto limIter = _limItems.find(setupItem.setupObjPath);
   if (limIter != _limItems.end())
   {
      LOG_DEBUG("onSetupAdded:Entry found");
      limIter->second.property.powerState = WIFI_STATE_DEVICE_AVAILABLE;
   }
   else
   {
      LOG_DEBUG("onSetupAdded:Entry add");
      addLIMItem(setupItem);
   }
   if (getLIMItems(limList))
   {
      populateLIMUpdate(limList);
   }

   if(limIter != _limItems.end())
   {
      checkPoweredSetting(limIter->second,setupItem.property.isPowered);
   }

} //void LastIntendedModeService::onSetupAdded

void LastIntendedModeService::checkPoweredSetting(LIMItem item,
      const bool isPowered)
{
   LOG_INFO("LastIntendedModeService::%s", __func__);
   bool isChanged = false;

   if ((!item.property.poweredSetting) && (!isPowered))
   {
      item.property.powerState = WIFI_STATE_POWERED_OFF;
      isChanged = true;
   }
   else if ((item.property.poweredSetting) && (isPowered))
   {
      item.property.powerState = WIFI_STATE_POWERED_ON;
      isChanged = true;
   }
   if (isChanged)
   {
      LIMList limList;
      auto limIter = _limItems.find(item.objPath);
      if (limIter != _limItems.end())
      {
         limIter->second = item;
      }
      if (getLIMItems(limList))
      {
         populateLIMUpdate(limList);
      }
   }

} //void LastIntendedModeService::checkPoweredSetting

void LastIntendedModeService::onSetupRemoved(const ::std::string& objPath)
{
   LOG_INFO("LastIntendedModeService::%s", __func__);
   removeLIMItem(objPath);

} // void LastIntendedModeService::onSetupRemoved

void LastIntendedModeService::onSetupUpdated(const WifiSetUpItem& item)
{
   LOG_INFO("LastIntendedModeService::%s", __func__);

   bool isPropertyChanged = false;
   auto limIter = _limItems.find(item.setupObjPath);
   if (limIter != _limItems.end())
   {
      if ((item.property.mode == WIFI_MODE_AP1) || (item.property.mode == WIFI_MODE_AP2))
      {
         if ((*limIter).second.property.limAPConfig.frequency != item.property.frequency)
         {
            LOG_ERROR("updateLIMItem:Frequency change");
            (*limIter).second.property.limAPConfig.frequency = item.property.frequency;
            isPropertyChanged = true;
         }
         if ((*limIter).second.property.limAPConfig.apType != item.property.apSetup.type)
         {
            LOG_ERROR("updateLIMItem:APType change");
            (*limIter).second.property.limAPConfig.apType = item.property.apSetup.type;
            isPropertyChanged = true;
         }
      }
      if ((WIFI_STATE_UNKNOWN != item.property.powerState)
            && ((*limIter).second.property.powerState != item.property.powerState))
      {
         LOG_DEBUG("updateLIMItem:PowerState change");
         (*limIter).second.property.powerState = item.property.powerState;
         isPropertyChanged = true;

      }

      if(item.property.isPowered != (*limIter).second.property.poweredSetting)
      {
         if ((WIFI_STATE_POWERED_OFF == item.property.powerState)
               && (WIFI_POWER_OFF_USER_REQUEST == item.property.powerOffType))
         {
            LOG_DEBUG("onSetupUpdated:poweredSetting OFF");
            (*limIter).second.property.poweredSetting = false;
         }
         else if (WIFI_STATE_POWERED_ON == item.property.powerState)
         {
            LOG_DEBUG("onSetupUpdated:poweredSetting ON");
            (*limIter).second.property.poweredSetting = true;
         }
         addLIMItemToDB(limIter->second);
      }
   }
   if (isPropertyChanged)
   {
      addLIMItemToDB(limIter->second);
      LIMList limList;
      APConfig apConfig;
      for (auto limIter = _limItems.begin(); limIter != _limItems.end(); ++limIter)
      {
         limList.push_back(limIter->second);
      }
      populateLIMUpdate(limList);
   }

} // void LastIntendedModeService::onSetupUpdated

bool LastIntendedModeService::getLIMItems(LIMList& limList)const
{
   bool isSuccess = true;
   if(_limItems.empty())
   {
      isSuccess = false;
   }
   for(auto limIter = _limItems.begin(); limIter !=_limItems.end() ; ++limIter)
   {
      limList.push_back(limIter->second);
   }
   LOG_DEBUG("LastIntendedModeService::getLIMItems %s",BOOL_TO_STR(isSuccess));
   return isSuccess;
} //LastIntendedModeService::getLIMItems

void LastIntendedModeService::populateLIMItems(const LIMList& limList)
{
   LOG_INFO("LastIntendedModeService:: %s", __func__);

   if (limList.empty())
   {
      LOG_INFO("populateLIMItems:empty list");
      return;
   }
   for (auto limIter = limList.begin(); limIter != limList.end(); limIter++)
   {
      addLIMItem(*limIter);
   }
} //void LastIntendedModeService::populateLIMItems

void LastIntendedModeService::populateLIMUpdate(const LIMList& limList)
{
   LOG_INFO("LastIntendedModeService ::%s", __func__);
   ::std::vector < LastIntendedModeStruct > limUpdate;
   LastIntendedModeStruct limStruct;

   if (limList.empty())
   {
      LOG_ERROR("populateLIMUpdate:empty list");
      return;
   }
   for (auto limIter = limList.begin(); limIter != limList.end(); limIter++)
   {
	  printLIMItem(*limIter);
      if (!limIter->objPath.empty())
      {
         limStruct.setElem1(limIter->objPath);
         svMap limMap;
         populateLIMMap(limMap, *limIter);
         if (!limMap.empty())
         {
            limStruct.setElem2(limMap);
            limUpdate.push_back(limStruct);
         }
      }
   }
   if (!limUpdate.empty())
   {
      LOG_DEBUG("populateLIMUpdate:Setting values");
      /*We explicitly call set here because ASF framework doesn't correctly
        compare if the update holds the same value as the previous one if
        the data is of a complex variant type , this needs to be revisited */
      setLastIntendedMode (limUpdate);
   }

} //void LastIntendedMode::populateLIMUpdate

void LastIntendedModeService::addLIMItem(const WifiSetUpItem& item)
{
   LOG_INFO("LastIntendedModeService:: %s", __func__);

   _limItems[item.setupObjPath].objPath = item.setupObjPath;
   _limItems[item.setupObjPath].property.mode = item.property.mode;
   _limItems[item.setupObjPath].property.powerState = item.property.powerState;

   if(WIFI_STATE_POWERED_ON == item.property.powerState)
   {
      _limItems[item.setupObjPath].property.poweredSetting = true;
   }

   if ((WIFI_MODE_AP1 == item.property.mode) || (WIFI_MODE_AP2 == item.property.mode))
   {
      _limItems[item.setupObjPath].property.limAPConfig.frequency = item.property.frequency;
      _limItems[item.setupObjPath].property.limAPConfig.apType = item.property.apSetup.type;
   }
   addLIMItemToDB(_limItems[item.setupObjPath]);

} // void LastIntendedModeService::addLIMItem

void LastIntendedModeService::addLIMItem(const LIMItem& item)
{
   LOG_INFO("LastIntendedModeService:: %s", __func__);

   _limItems[item.objPath].objPath = item.objPath;
   _limItems[item.objPath].property.mode = item.property.mode;
   _limItems[item.objPath].property.poweredSetting = item.property.poweredSetting;
   
   if ((item.property.mode == WIFI_MODE_AP1) || (item.property.mode == WIFI_MODE_AP2))
   {
      _limItems[item.objPath].property.limAPConfig.frequency = item.property.limAPConfig.frequency;
      _limItems[item.objPath].property.limAPConfig.apType = item.property.limAPConfig.apType;
   }

} //void LastIntendedModeService::addLIMItem

void LastIntendedModeService::removeLIMItem(const ::std::string& objPath)
{
   LOG_INFO("LastIntendedModeService::%s", __func__);

   _limItems.erase(objPath);
   removeLIMItemFromDB(objPath);
   LIMList limList;
   for (auto limIter = _limItems.begin(); limIter != _limItems.end(); ++limIter)
   {
      limList.push_back(limIter->second);
   }
   populateLIMUpdate(limList);

} //void LastIntendedModeService::removeLIMItem

void LastIntendedModeService::addLIMItemToDB(const LIMItem& item)
{
   LOG_INFO ("LastIntendedModeService::%s", __func__);
   WifiSetUpItem apSetupitem;
   IDBManager *dbMgrIf = DBManagerFactory::getInstance()->getDBManagerIf();

   if (dbMgrIf)
   {
      dbMgrIf->writeLIMToDB(item);

      //Update Last Mode AP Configurations
      if (getAPSetupItem(apSetupitem))
      {
         if(apSetupitem.property.apSetup.type != WBL_AP_TYPE_RESTRICTED)
         {
            dbMgrIf->writeAPConfigsToDB(apSetupitem.property.apSetup,false);
         }
      }
   }
} // void LastIntendedModeService::addLIMItemToDB

void LastIntendedModeService::removeLIMItemFromDB(const ::std::string& objPath)
{
   LOG_INFO ("LastIntendedModeService::%s", __func__);

   DBManagerFactory* dbmgrFactory = DBManagerFactory::getInstance();
   IDBManager* dbMgrIf = (dbmgrFactory) ? (dbmgrFactory->getDBManagerIf()) : nullptr;
   WifiMode mode = WIFI_MODE_UNKNOWN;

   bool exists = objPath.find("/wlan0/") != std::string::npos;
   //TO DO:Revisit
   if(exists)
      mode = WIFI_MODE_STA;
   else
      mode = WIFI_MODE_AP1;
   if (dbMgrIf)
   {
      dbMgrIf->removeLIMItem(mode);
   }
} //void LastIntendedModeService::removeLIMItemFromDB

void LastIntendedModeService::printLIMItem(const LIMItem &item)
{
   LOG_DEBUG("printLIMItem::ObjectPath:%s", item.objPath.c_str());
   LOG_DEBUG("printLIMItem::Mode:%s", WIFI_MODE_TO_STR(item.property.mode));
   LOG_DEBUG("printLIMItem::PowerState:%s",POWER_STATE_TO_STR(item.property.powerState));
   LOG_DEBUG("printLIMItem::PowerSetting:%s",BOOL_TO_STR(item.property.poweredSetting));

   if ((item.property.mode == WIFI_MODE_AP1)||(item.property.mode == WIFI_MODE_AP2))
   {
      LOG_DEBUG("printLIMItem::frequency:%s", FREQ_TO_STR(item.property.limAPConfig.frequency));
      LOG_DEBUG("printLIMItem::APType: %s", AP_TYPE_TO_STR(item.property.limAPConfig.apType));
   }

} //LastIntendedModeService::printLIMItem

void LastIntendedModeService::populateLIMMap(svMap& limMap, const LIMItem& item)
{
   LOG_INFO("LastIntendedModeService::%s ", __func__);

   const char* mode = (wblWifiMode.getWifiMode2String(item.property.mode)).c_str();
   const char* powerState = (wblPowerState.getPowerState2String(item.property.powerState)).c_str();

   insert_map_element(limMap, "WifiMode", DBUS_TYPE_STRING, &mode);
   insert_map_element(limMap, "PowerState", DBUS_TYPE_STRING, &powerState);

   if ((item.property.mode == WIFI_MODE_AP1)||(item.property.mode == WIFI_MODE_AP2))
   {
      const char* apType = (wblAPType.getAPType2String(item.property.limAPConfig.apType)).c_str();
      const char* frequency = (wblFrequency.getFrequency2String( item.property.limAPConfig.frequency)).c_str();
      ::asf::dbus::DBusVariant apconfigVariant;
      DBusMessageIter *arrWriteIter = apconfigVariant.getWriteIterator();
      DBusMessageIter dictWriteIter;

      if (dbus_message_iter_open_container(arrWriteIter, DBUS_TYPE_ARRAY,"{sv}", &dictWriteIter))
      {
         dict_append_entry(&dictWriteIter, "Frequency", DBUS_TYPE_STRING, &frequency);
         dict_append_entry(&dictWriteIter, "Type", DBUS_TYPE_STRING, &apType);
         (void)dbus_message_iter_close_container(arrWriteIter, &dictWriteIter);
      }
      limMap.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("APConfig", apconfigVariant));
   }

} //void LastIntendedModeService::populateLIMMap

void LastIntendedModeService::getAllProperties(::std::map< ::std::string, ::asf::dbus::DBusVariant >& propertiesMap)
{
   LOG_INFO ("%s", __func__);
   _stubDelegate->getAllProperties(propertiesMap);
}

void LastIntendedModeService::getDBusObjectPath(::std::string& objPath)
{
   LOG_INFO ("%s", __func__);
   objPath = _stubDelegate->getDBusObjectPath();
}

void LastIntendedModeService::getInterfaceName(::std::string& interfaceName)
{
   LOG_INFO ("%s", __func__);
   interfaceName = _stubDelegate->getInterfaceName();
}

void LastIntendedModeService::updateInterfaceAdded()
{
   LOG_INFO ("%s", __func__);

   PropertiesList propertiesList;
   ::std::string interfacename;
   ::std::string objpath;

   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   ObjectManagerService* objectManager =
           (serviceFactory) ? (serviceFactory->getObjectManagerServiceIf()) : (nullptr);
   if(objectManager)
   {
      getAllProperties(propertiesList);
      getInterfaceName(interfacename);
      getDBusObjectPath(objpath);
      objectManager->updateInterfaceAddedSignal(objpath,interfacename,propertiesList);
   }
}

void LastIntendedModeService::updateInterfaceRemoved()
{
   LOG_INFO ("%s", __func__);

   ::std::vector < ::std::string > interface;
   ::std::string interfacename;
   ::std::string objpath;

   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   ObjectManagerService* objectManager =
         (serviceFactory) ? (serviceFactory->getObjectManagerServiceIf()) : (nullptr);

   if (objectManager)
   {
      getDBusObjectPath(objpath);
      getInterfaceName(interfacename);
      interface.push_back(interfacename);
      objectManager->updateInterfaceRemovedSignal(objpath, interface);
   }
}
bool LastIntendedModeService::getAPSetupItem(WifiSetUpItem &item)
{
   LOG_INFO("%s", __func__);
   WifiSetUpList wifiSetupList;
   bool bRet = false;
   if (ConflictManagerFactory::getInstance()->getConflictManagerRequestIf()->getWiFiSetUps(wifiSetupList))
   for (auto iter = wifiSetupList.begin(); iter != wifiSetupList.end(); iter++)
   {
      if (WIFI_MODE_AP1 == iter->property.mode || WIFI_MODE_AP2 == iter->property.mode)
      {
         item = *iter;
         bRet = true;
         break;
      }
   }
   return bRet;
}
void LastIntendedModeService::updateAPConfigurations(const APConfig &apConfig)
{
   LOG_INFO ("%s", __func__);
   bool isRestricted = false;
   WifiSetUpItem item;
   if (getAPSetupItem(item))
   {
      if(item.property.apSetup.type != WBL_AP_TYPE_RESTRICTED)
      {
         IDBManager *dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
         if(NULL != dbmgrIf)
         {
            isRestricted = (apConfig.type == WBL_AP_TYPE_RESTRICTED);
            dbmgrIf->writeAPConfigsToDB(item.property.apSetup,isRestricted);
         }
      }
   }
}

} //namespace bosch
} //namespace org

/** @} */
