/**
 * @file ConflictManagementService.cpp
 * @author RBEI/ECO32 Karthikeyan Madeswaran
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#include "ConflictManagementService.h"
#include "WBLASFComponent.h"
#include "WBLServiceInterfaces.h"
#include "WBLTypeProperties.h"
#include "DbusIfTypes.h"
#include "DbusParser.h"
#include "DbusHelper.h"
#include "ConflictManagementTypes.h"
#include "WBLDefines.h"
#include "WBLHelper.h"

namespace org
{
namespace bosch
{

static DbusParser < WBLServiceInterfaces > wblDbusParser;

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLServices", ConflictManagementService, Debug);

ConflictManagementService::ConflictManagementService(::std::string portName) :
            ConflictmanagementStub(portName)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   wblDbusParser.setTraces(true);
}//ConflictManagementService::ConflictManagementService

ConflictManagementService::~ConflictManagementService()
{
   try
   {
      updateInterfaceRemoved();
   }catch(...){}
}//ConflictManagementService::~ConflictManagementService

void ConflictManagementService::onPrepareSetupRequest(const ::boost::shared_ptr<PrepareSetupRequest>& request)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   const ::boost::shared_ptr<PrepareSetupMsg> preparesetup(new PrepareSetupMsg);
   WifiMode configMode;
   APConfig apConfig;
   WBLActionType actionType = WIFI_ACTION_ACTIVATE;
   ::std::map<::std::string, ::asf::dbus::DBusVariant> configurationMap = request->getConfiguration();
   WBLErrorCode errorCode;

   if (!parseActionType(configurationMap, actionType))
   {
      LOG_ERROR("ConflictManagementService:onPrepareSetupRequest Invalid ActionType");
      errorCode = WBL_ERR_INVALID_ARGS;
      onPrepareSetupResponse(errorCode, static_cast<uint>(request->getAct()));
      return;
   }

   if (parseWifiMode(configurationMap, configMode))
   {
      if((WIFI_MODE_AP1 == configMode) || (WIFI_MODE_AP2 == configMode))
      {
         if (!parseAPConfig(configurationMap, apConfig))
         {
            LOG_DEBUG("ConflictManagementService:onPrepareSetupRequest Invalid Arguments");
            errorCode = WBL_ERR_INVALID_ARGS;
            onPrepareSetupResponse(errorCode, static_cast<uint>(request->getAct()));
            return;
         }
         else
         {
            ILastIntendedModeService *limServiceIf =
                  WBLServiceFactory::getInstance()->getLastIntendedModeServiceIf();
            if (NULL != limServiceIf)
            {
               limServiceIf->updateAPConfigurations(apConfig);
            }
         }
      }
   }
   else
   {
      LOG_DEBUG("ConflictManagementService:onPrepareSetupRequest Wifi Mode Parsing Failed");
      return;
   }
   preparesetup->setWifiMode(configMode);
   preparesetup->setAPConfigItemParams(apConfig);
   preparesetup->setWifiAction(actionType);
   preparesetup->setMsgID(WBL_MSG_PREPARE_SETUP_REQ);
   preparesetup->setMsgToken(static_cast<uint>(request->getAct()));

   ICmdConflictManager* conflictManagerIf = ConflictManagerFactory::getInstance()->getConflictManagerRequestIf();
   if (NULL != conflictManagerIf)
   {
      conflictManagerIf->onProcessRequest(preparesetup);
   }
} //void ConflictManagementService::onPrepareSetupRequest

bool ConflictManagementService::parseWifiMode(::std::map < ::std::string, ::asf::dbus::DBusVariant >& configurationMap,
      WifiMode &wifiMode)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::map < ::std::string, ::asf::dbus::DBusVariant >::const_iterator configIter;
   ::asf::dbus::DBusVariant valVariant;
   ::std::string mode;
   bool isArgsValid = true;

   for (auto configIter = configurationMap.begin(); configIter != configurationMap.end(); ++configIter)
   {
      valVariant = configIter->second;
      DBusMessageIter* valIter = valVariant.getReadIterator();
      if(configIter->first.compare("Mode") == 0)
      {
          mode = ::std::string((char*)getDbusIterValue(valIter));
          LOG_DEBUG("ConflictManagementService:parseWifiMode %s",mode.c_str());
         break;
      }
   }
   wifiMode = wblWifiMode.getWifiMode2Enum(mode);
   if (WIFI_MODE_UNKNOWN == wifiMode)
   {
      isArgsValid = false;
   }
   return isArgsValid;
}//ConflictManagementService::parseWifiMode

bool ConflictManagementService::parseAPConfig(::std::map < ::std::string, ::asf::dbus::DBusVariant >& data,APConfig& property)
{
   LOG_INFO("ConflictManagementService:parseAPConfig Entered");

   ::std::vector < DbusVariantProperty > ifProperties;
   ::std::vector < int > matchingInterfaces;
   matchingInterfaces.push_back((int) IF_PREPARESETUP);
   bool isArgsValid = true;

   wblDbusParser.parseProperties(ifProperties, matchingInterfaces, data, DBUS_BUS_SYSTEM, "", "", "PrepareSetup"); //TODO - missing params
   for (size_t i = 0; (isArgsValid) && (i < ifProperties.size()); ++i)
   {
      PrepareSetupConfigurationName propertyName =
            static_cast < PrepareSetupConfigurationName >(ifProperties[i].propEnum);
      switch (propertyName)
      {
         case PREPARESETUP_TYPE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            const ::std::string& apType = ifProperties[i].propData.getString();
            LOG_DEBUG("parsePrepareSetupRequest: Type = %s", apType.c_str());
            property.type = wblAPType.getAPType2Enum(apType);
            isArgsValid  = (WBL_AP_TYPE_UNKNOWN != property.type);
            property.isTypeValid = isArgsValid;
            break;
         }
         case PREPARESETUP_SSID_VALUE:
         {
            LOG_DEBUG("parsePrepareSetupRequest: PREPARESETUP_SSID_VALUE");
            WBL_ASSERT(VARIANT_BYTE_ARRAY != ifProperties[i].propData.getType());
            const ByteArray& ssid = ifProperties[i].propData.getByteArray();
            ::std::string strSSID = convertByteArray2String(ssid);
            LOG_DEBUG("parsePrepareSetupRequest SSID is %s", strSSID.c_str());
            property.ssid.SSID = ssid;
            isArgsValid = !(ssid.empty());
            property.isSSIDValid = isArgsValid;
            break;
         }
         case PREPARESETUP_SSID_UTF8:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.ssid.isUTF8 = ifProperties[i].propData.getBool();
            isArgsValid = true;
            LOG_DEBUG("parsePrepareSetupRequest: SSID isUTF8 = %d", property.ssid.isUTF8);
            break;
         }
         case PREPARESETUP_PASSPHRASE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.passphrase = ifProperties[i].propData.getString();
            LOG_DEBUG("parsePrepareSetupRequest: Passphrase = %s", property.passphrase.c_str());
            isArgsValid = !(property.passphrase.empty());
            property.isPassphraseValid = isArgsValid;
            break;
         }
         case PREPARESETUP_SECURITY:
         {
            /* WBL should restrict the security preference from clients
             * Property Not handled*/
            /*WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            const ::std::string& security = ifProperties[i].propData.getString();
            LOG_DEBUG("parsePrepareSetupRequest: Security = %d", security.c_str());
            property.security = wblSecurity.getSecurity2Enum(security);
            isArgsValid = (WBL_SECURITY_UNKNOWN != property.security);
            property.isSecurityValid = isArgsValid ;*/
            break;
         }
         case PREPARESETUP_VISIBILITY:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.isVisible = ifProperties[i].propData.getBool();
            isArgsValid = true;
            property.isVisibilityValid = isArgsValid;
            LOG_DEBUG("parsePrepareSetupRequest: isVisible = %d", property.isVisible);
            break;
         }
         case PREPARESETUP_FREQUENCY:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            const ::std::string& frequency = ifProperties[i].propData.getString();
            LOG_DEBUG("parsePrepareSetupRequest: frequency = %s", frequency.c_str());
            property.frequency = wblFrequency.getFrequency2Enum(frequency);
            isArgsValid = (WBL_FREQ_BAND_UNKNOWN != property.frequency);
            property.isFrequencyValid = isArgsValid;
            break;
         }
         case PREPARESETUP_CHANNEL:
         {
            WBL_ASSERT(VARIANT_UINT16_ARRAY != ifProperties[i].propData.getType());
            property.chanList = ifProperties[i].propData.getUInt16Array();
            property.isChanListValid = true;
            isArgsValid = true;
            break;
         }
         case PREPARESETUP_STATIONSTOBERESERVED:
         {
            WBL_ASSERT(VARIANT_UINT16 != ifProperties[i].propData.getType());
            property.stationsToReserve = ifProperties[i].propData.getUInt16();
            property.isStationsToReserveValid = true;
            LOG_DEBUG("parsePrepareSetupRequest: stationsToReserve = %d", property.stationsToReserve);
            break;
         }
         case PREPARESETUP_SEPARATEAP:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.isSeparateAP = ifProperties[i].propData.getBool();
            isArgsValid = true;
            property.isSeparateAPValid = isArgsValid ;
            LOG_DEBUG("parsePrepareSetupRequest: isSeparateAP = %d", property.isSeparateAP);
            break;
         }
         case PREPARESETUP_MODE:
         case PREPARESETUP_APCONFIG:
         case PREPARESETUP_P2PCONFIG:
         case PREPARESETUP_GOINTENT:
         {
            //Property Not handled
            break;
         }
         default:
         {
            LOG_ERROR("parsePrepareSetupRequest: Invalid access point property %d", propertyName);
            break;
         }
      }
   }
   return isArgsValid;
}//ConflictManagementService::parseAPConfig

bool ConflictManagementService::parseActionType(::std::map<::std::string, ::asf::dbus::DBusVariant>& configurationMap,
      WBLActionType &actionType) const
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   bool isValid = true;
   uint32 actionValue = 0;
   auto configIter = configurationMap.find("Action");
   if (configurationMap.end() != configIter)
   {
      ::asf::dbus::DBusVariant actionVariant = configIter->second;
      DBusMessageIter* actionVariantIter = actionVariant.getReadIterator();
      dbus_message_iter_get_basic(actionVariantIter, &actionValue);
      LOG_DEBUG("parseActionType:ActionValue:%d",actionValue);
      actionType = static_cast<WBLActionType>(actionValue);
      if ((WIFI_ACTION_ACTIVATE != actionType)
      && (WIFI_ACTION_UPDATE != actionType))
      {
         LOG_DEBUG("parseActionType:Setting invalid");
         isValid = false;
      }
   }
   else
   {
      LOG_DEBUG("parseActionType:Setting defaults");
      actionType = WIFI_ACTION_ACTIVATE;
   }
   LOG_INFO("ConflictManagementService:parseActionType:%s,%s",BOOL_TO_STR(isValid),ACTIONTYPE_TO_STR(actionType));
   return isValid;
}//ConflictManagementService::parseActionType

void ConflictManagementService::onPrepareSetupResponse(WBLErrorCode errcode, uint32 actToken)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::string errname = "";
   ::std::string errmsg = "";

   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendPrepareSetupResponse(actToken);
   }
   else
   {
      errname = wblErrorCode.getErrorCode2String(errcode);
      LOG_INFO("ConflictManagementService::onPrepareSetupResponse %s",errname.c_str());
      sendPrepareSetupError(errname, errmsg, actToken);
   }
}//ConflictManagementService::onPrepareSetupResponse

void ConflictManagementService::onActiveWifiSetupsChanged(const ActiveWifiSetupsMsg& msg)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   WifiSetUpList wifisetuplist;
   msg.getWifiSetUps(wifisetuplist);

   ::std::vector < ActiveWiFiSetupsStruct > activeWifiSetups;

   for (auto it = wifisetuplist.begin(); it != wifisetuplist.end(); ++it)
   {
      WifiSetUpItem item = *it;
      printWifiSetUpInfo(item);
      bindWiFiSetup(activeWifiSetups, item);
   }
   notifyActiveWiFiSetups(activeWifiSetups);
}//ConflictManagementService::onActiveWifiSetupsChanged

void ConflictManagementService::printWifiSetUpInfo(const WifiSetUpItem& item) const
{
   LOG_INFO("ConflictManagementService::printWifiSetUpInfo");
   LOG_DEBUG("printWifiSetUpInfo: Object path:%s", item.setupObjPath.c_str());
   LOG_DEBUG("wifiMode:%s interface:%s powerState: \"%s\" "
            "frequency:%s currChannel:%d HealthIndicator:%d",
            WIFI_MODE_TO_STR(item.property.mode),
            item.property.interface.c_str(),
            POWER_STATE_TO_STR(item.property.powerState),
            FREQ_TO_STR(item.property.frequency),
            item.property.currChannel,
            item.property.connectionStatus.healthIndicatorValue);

   if ((WIFI_MODE_AP1 == item.property.mode) || (WIFI_MODE_AP2 == item.property.mode))
   {
      ::std::string ssid = convertByteArray2String(item.property.apSetup.SSID);
      LOG_DEBUG(
            "SSID: \"%s\" "
            "passphrase:\"%s\" "
            "macAddress:\"%s\" "
            "type:\"%s\" "
            "security:\"%s\" "
            "isVisible:\"%d\" ",
            ssid.c_str(), item.property.apSetup.passphrase.c_str(),
            item.property.apSetup.macAddress.c_str(),
            AP_TYPE_TO_STR(item.property.apSetup.type),
            SECURITY_TO_STR(item.property.apSetup.security),
            item.property.apSetup.isVisible);
   }
}//ConflictManagementService::printWifiSetUpInfo

void ConflictManagementService::insertSupportedFrequencies(const APSetup &apSetup,
      ::std::map<::std::string, ::asf::dbus::DBusVariant> &setupsMap)
{
    LOG_INFO("ConflictManagementService::%s", __func__);
    ::asf::dbus::DBusVariant techVariant;
    DBusMessageIter *iter1,iter2,iter3,iter4,iter5,iter6;
    DBusMessageIter iterChan1,iterChan2,iterChan3;
    ::std::string frequency, techSupport, technology,channel;
    uint16 freqBand;

    if (apSetup.supportedFrequencies.empty())
        return;

    iter1 = techVariant.getWriteIterator();
    (void) dbus_message_iter_open_container(iter1, DBUS_TYPE_ARRAY, "a{sv}",&iter2);
    for (auto it = apSetup.supportedFrequencies.begin();it != apSetup.supportedFrequencies.end(); ++it)
    {
      (void) dbus_message_iter_open_container(&iter2, DBUS_TYPE_ARRAY, "{sv}",&iter3);

        //Insert Frequency
        freqBand = it->first;
        frequency = wblFrequency.getFrequency2String(it->first);
        dict_append_entry(&iter3, "Frequency", DBUS_TYPE_STRING,&frequency);

        //Insert Technologies
        const char* strTech = "SPITechnologies";
        (void) dbus_message_iter_open_container(&iter3,DBUS_TYPE_DICT_ENTRY, NULL, &iter4);
            (void) dbus_message_iter_append_basic(&iter4, DBUS_TYPE_STRING,&strTech);
            SPITechnologies tech = (SPITechnologies) it->second;
            (void) dbus_message_iter_open_container(&iter4,DBUS_TYPE_VARIANT, "a{sv}", &iter5);
                (void) dbus_message_iter_open_container(&iter5,DBUS_TYPE_ARRAY, "{sv}", &iter6);
                    technology = "CarPlayWireless";
                    //Update CPW technology
                    techSupport = (tech.isCPWSupport) ? "ALLOWED" : "NOT_ALLOWED";
                    LOG_INFO("ConflictManagementService::%s CPW technology:%s",__func__, techSupport.c_str());
                    dict_append_entry(&iter6, technology.c_str(),DBUS_TYPE_STRING, &techSupport);
                    //Update AAW technology
                    technology = "AndroidAutoWireless";
                    techSupport =(tech.isAAWSupport) ?"ALLOWED" : "NOT_ALLOWED";
                    LOG_INFO("ConflictManagementService::%s AAW technology:%s",__func__, techSupport.c_str());
                    dict_append_entry(&iter6, technology.c_str(),DBUS_TYPE_STRING, &techSupport);
                (void) dbus_message_iter_close_container(&iter5, &iter6);
            (void) dbus_message_iter_close_container(&iter4, &iter5);
        (void) dbus_message_iter_close_container(&iter3, &iter4);

        //Insert Channels
        const char* strChannels = "Channels";
        (void) dbus_message_iter_open_container(&iter3, DBUS_TYPE_DICT_ENTRY, NULL, &iterChan1);
          (void) dbus_message_iter_append_basic(&iterChan1, DBUS_TYPE_STRING, &strChannels);
          (void) dbus_message_iter_open_container(&iterChan1, DBUS_TYPE_VARIANT, "a{sv}", &iterChan2);
              (void) dbus_message_iter_open_container(&iterChan2, DBUS_TYPE_ARRAY, "{sv}", &iterChan3);
              for (auto itr=apSetup.chanList.begin(); itr != apSetup.chanList.end(); itr++ )
              {
                  if (itr->second >= freqBand)
                  {
                      if((freqBand == WBL_FREQ_BAND_2_4_GHZ && itr->second <= WBL_FREQ_BAND_5_GHZ) ||
                             (freqBand == WBL_FREQ_BAND_5_GHZ))
                      {
                          channel = std::to_string(itr->first);
                          uint16 freq = itr->second;
                          dict_append_entry(&iterChan3, channel.c_str(),DBUS_TYPE_UINT16, &freq);
                      }
                  }
              }
              (void) dbus_message_iter_close_container(&iterChan2, &iterChan3);
          (void) dbus_message_iter_close_container(&iterChan1, &iterChan2);
        (void) dbus_message_iter_close_container(&iter3, &iterChan1);
      (void) dbus_message_iter_close_container(&iter2, &iter3);
    }
    (void) dbus_message_iter_close_container(iter1, &iter2);
    setupsMap.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("SupportedFrequencies", techVariant));
}//ConflictManagementService::insertSupportedFrequencies

void ConflictManagementService::addRestrictionsListItem(DBusMessageIter *iterRestrict,::std::string tech,::std::string state)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   DBusMessageIter iterEntry,iterTechElement,iterEntryElement;
   const char* strTech = "SPITechnology";
 
   (void) dbus_message_iter_open_container(iterRestrict, DBUS_TYPE_DICT_ENTRY, NULL, &iterTechElement);
   (void) dbus_message_iter_append_basic(&iterTechElement, DBUS_TYPE_STRING, &strTech);
   (void) dbus_message_iter_open_container(&iterTechElement, DBUS_TYPE_VARIANT, "a{sv}", &iterEntry);
   (void) dbus_message_iter_open_container(&iterEntry, DBUS_TYPE_ARRAY, "{sv}", &iterEntryElement);
   dict_append_entry(&iterEntryElement,"Technology",DBUS_TYPE_STRING, &tech);
   (void) dbus_message_iter_close_container(&iterEntry, &iterEntryElement);
   (void) dbus_message_iter_close_container(&iterTechElement, &iterEntry);
   (void) dbus_message_iter_close_container(iterRestrict, &iterTechElement);
   dict_append_entry(iterRestrict, "State", DBUS_TYPE_STRING, &state);
}//ConflictManagementService::addRestrictionsListItem

void ConflictManagementService::insertTechnologyRestrictions(::std::map<::std::string, ::asf::dbus::DBusVariant> &setupsMap)
{
   LOG_INFO("ConflictManagementService::insertTechnologyRestrictions");
   ::asf::dbus::DBusVariant restrictVariant;
   ::std::string strTech,strState;
   DBusMessageIter *iterRestrict,iterEntry,iterEntryElement;
   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   TechnologyRestrictions techRestrictions = serviceFactory->getWifiSetupManager()->getAPWifiSetup()->getTechnologyRestrictions();

   if(techRestrictions.empty())
     return;

   iterRestrict = restrictVariant.getWriteIterator();
   (void) dbus_message_iter_open_container(iterRestrict, DBUS_TYPE_ARRAY, "a{sv}", &iterEntry);
   for(auto it = techRestrictions.begin(); it != techRestrictions.end(); it++ )
   {
      (void) dbus_message_iter_open_container(&iterEntry, DBUS_TYPE_ARRAY,"{sv}", &iterEntryElement);
      strTech = wblTechRestriction.getWBLTechnology2String(it->first);
      strState = wblRestrictionState.getWBLRestriction2String(it->second);
      addRestrictionsListItem(&iterEntryElement,strTech,strState);
      (void) dbus_message_iter_close_container(&iterEntry, &iterEntryElement);
   }
   (void) dbus_message_iter_close_container(iterRestrict, &iterEntry);
   setupsMap.insert(std::pair < ::std::string, ::asf::dbus::DBusVariant >("Restrictions", restrictVariant));

}//ConflictManagementService::updateRestrictions

void ConflictManagementService::addIpv4ConfigurationItem(DBusMessageIter *iterConfig,IPv4Configuration &ipconfig)
{
   LOG_INFO("ConflictManagementService::addIpv4ConfigurationItem");
   ::std::string strAddress,strNetMask;
   DBusMessageIter iterEntry,iterEntryElement;
   (void) dbus_message_iter_open_container(iterConfig, DBUS_TYPE_ARRAY, "a{sv}", &iterEntry);
   (void) dbus_message_iter_open_container(&iterEntry, DBUS_TYPE_ARRAY,"{sv}", &iterEntryElement);
   strAddress = ipconfig.router;
   strNetMask = ipconfig.netMask;
   dict_append_entry(&iterEntryElement,"Address",DBUS_TYPE_STRING, &strAddress);
   dict_append_entry(&iterEntryElement,"Netmask",DBUS_TYPE_STRING, &strNetMask);
   (void) dbus_message_iter_close_container(&iterEntry, &iterEntryElement);
   (void) dbus_message_iter_close_container(iterConfig, &iterEntry);
}

void ConflictManagementService::insertIPv4Configuration(const APSetup &apSetup,
      ::std::map<::std::string, ::asf::dbus::DBusVariant> &setupsMap)
{
   LOG_INFO("ConflictManagementService::insertIPv4Configuration");
   ::asf::dbus::DBusVariant ipConfigVariant;
   DBusMessageIter *iterIPConfig;

   if(apSetup.ipv4ConfigurationList.empty())
      return;

   iterIPConfig = ipConfigVariant.getWriteIterator();
   for (auto iter = apSetup.ipv4ConfigurationList.begin(); iter != apSetup.ipv4ConfigurationList.end(); iter++)
   {
      IPv4Configuration ipconfig= (IPv4Configuration) *iter;
      addIpv4ConfigurationItem(iterIPConfig,ipconfig);
   }
   setupsMap.insert(std::pair < ::std::string, ::asf::dbus::DBusVariant >("IPv4Configuration", ipConfigVariant));
}

void ConflictManagementService::insertConnectionStatus(const WBLConnectionStatus &connectionStatus,
      ::std::map<::std::string, ::asf::dbus::DBusVariant> &setupsMap)
{
   LOG_INFO("ConflictManagementService::insertConnectionStatus");
   ::asf::dbus::DBusVariant connStatus;
   DBusMessageIter iterEntry,iterEntryElement;
   DBusMessageIter *iterConnStaus = connStatus.getWriteIterator();
   dbus_bool_t bConnectedVal = connectionStatus.isConnected;
   dbus_uint32_t uHealthIndicatorVal = connectionStatus.healthIndicatorValue;
   (void) dbus_message_iter_open_container(iterConnStaus, DBUS_TYPE_VARIANT, "a{sv}", &iterEntry);
   (void) dbus_message_iter_open_container(&iterEntry, DBUS_TYPE_ARRAY,"{sv}", &iterEntryElement);
   dict_append_entry(&iterEntryElement,"Connected",DBUS_TYPE_BOOLEAN, &bConnectedVal);
   dict_append_entry(&iterEntryElement,"HealthIndicator",DBUS_TYPE_UINT32, &uHealthIndicatorVal);
   (void) dbus_message_iter_close_container(&iterEntry, &iterEntryElement);
   (void) dbus_message_iter_close_container(iterConnStaus, &iterEntry);
   setupsMap.insert(std::pair < ::std::string, ::asf::dbus::DBusVariant >("ConnectionStatus", connStatus));
}

void ConflictManagementService::bindWiFiSetup(::std::vector < ActiveWiFiSetupsStruct >& activeWifiSetups,const WifiSetUpItem& item)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   if (item.setupObjPath.empty())
   {
      LOG_ERROR("parseWiFiSetup: Setup not parsed since object path is empty");
      return;
   }
   dbus_uint32_t iVal;
   ::asf::dbus::DBusVariant techVariant,restrictVariant,ipConfigVariant;
   ::std::map < ::std::string, ::asf::dbus::DBusVariant > setupsMap;

   ActiveWiFiSetupsStruct setupsStruct;
   setupsStruct.clear();
   setupsStruct.setElem1(item.setupObjPath);

   std::string strMode = wblWifiMode.getWifiMode2String(item.property.mode);
   insert_map_element(setupsMap, "Mode", DBUS_TYPE_STRING, &strMode);

   std::string strInterface = item.property.interface.c_str();
   insert_map_element(setupsMap, "Interface", DBUS_TYPE_STRING, &strInterface);

   std::string powerState = wblPowerState.getPowerState2String(item.property.powerState);
   insert_map_element(setupsMap, "PowerState", DBUS_TYPE_STRING, &powerState);

   std::string strFreqBand = wblFrequency.getFrequency2String(item.property.frequency);
   insert_map_element(setupsMap, "Frequency", DBUS_TYPE_STRING, &strFreqBand);

   iVal = item.property.currChannel;
   insert_map_element(setupsMap, "CurrentOperatingChannel", DBUS_TYPE_UINT32, &iVal);

   if (WIFI_MODE_AP1 == item.property.mode || WIFI_MODE_AP2 == item.property.mode)
   {
      std::string macAddress = item.property.apSetup.macAddress;
      insert_map_element(setupsMap, "MacAddress", DBUS_TYPE_STRING, &macAddress);
      insertAPConfigElements(item.property.apSetup,setupsMap);
      insertSupportedFrequencies(item.property.apSetup,setupsMap);
      insertTechnologyRestrictions(setupsMap);
      insertIPv4Configuration(item.property.apSetup,setupsMap);
   }
   else
   {
      //insertConnectionStatus(item.property.connectionStatus,setupsMap);
   }
   insertConnectionStatus(item.property.connectionStatus,setupsMap);
   setupsStruct.setElem2(setupsMap);
   activeWifiSetups.push_back(setupsStruct);
}//ConflictManagementService::bindWiFiSetup

void ConflictManagementService::insertAPConfigElements(const APSetup &apSetup,
      ::std::map < ::std::string, ::asf::dbus::DBusVariant > &setupsMap)
{
   LOG_INFO("ConflictManagementService::insertAPConfigElements");
   ::asf::dbus::DBusVariant apConfigVariant;
   DBusMessageIter iterApElement, iterStaElement, dict, iterSta;
   DBusMessageIter *iterAPConfigVariant;
   uint index = 0;
   dbus_bool_t visibility;
   std::string security, passphrase, type;

   iterAPConfigVariant = apConfigVariant.getWriteIterator();
   //open iterApElement
   (void) dbus_message_iter_open_container(iterAPConfigVariant, DBUS_TYPE_ARRAY,
   DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
   DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
   DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &iterApElement);

   uint8* ssid = (uint8*) getMemAlloc(apSetup.SSID.size());

   for (std::vector < uint8 >::const_iterator it = apSetup.SSID.begin(); it != apSetup.SSID.end(); ++it, index++)
      ssid[index] = *it;
   const char* strSSID = "SSID";
   append_dict_entry_fixed_array(&iterApElement, strSSID, &ssid, static_cast<uint>(apSetup.SSID.size()));

   security = wblSecurity.getSecurity2String(apSetup.security);
   dict_append_entry(&iterApElement, "Security", DBUS_TYPE_STRING, &security);

   passphrase = apSetup.passphrase;
   dict_append_entry(&iterApElement, "Passphrase", DBUS_TYPE_STRING, &passphrase);

   visibility = apSetup.isVisible;
   dict_append_entry(&iterApElement, "Visible", DBUS_TYPE_BOOLEAN, &visibility);

   LOG_DEBUG("ConflictManagementService::insertAPConfigElements No. Associated Stations %d ",apSetup.stationList.size());

   if (apSetup.stationList.size())
   {
      (void) dbus_message_iter_open_container(&iterApElement, DBUS_TYPE_DICT_ENTRY, NULL, &iterStaElement);

      const char* strAsSta = "AssociatedStations";
      (void) dbus_message_iter_append_basic(&iterStaElement, DBUS_TYPE_STRING, &strAsSta);

      AssociatedStationList::const_iterator itrSta = apSetup.stationList.begin();
      (void) dbus_message_iter_open_container(&iterStaElement, DBUS_TYPE_VARIANT, "a{sv}", &dict);
      (void) dbus_message_iter_open_container(&dict, DBUS_TYPE_ARRAY, "{sv}", &iterSta);
      for (; itrSta != apSetup.stationList.end(); itrSta++)
      {
         AssociatedStationItem staItem = *itrSta;
         insertAssociatedStation(staItem, &iterSta);
      }
      (void) dbus_message_iter_close_container(&dict, &iterSta);
      (void) dbus_message_iter_close_container(&iterStaElement, &dict);
      (void) dbus_message_iter_close_container(&iterApElement, &iterStaElement);
   }
   type = wblAPType.getAPType2String(apSetup.type);
   dict_append_entry(&iterApElement, "Type", DBUS_TYPE_STRING, &type);

   //close iterApElement
   (void) dbus_message_iter_close_container(iterAPConfigVariant, &iterApElement);

   setupsMap.insert(std::pair < ::std::string, ::asf::dbus::DBusVariant >("APConfig", apConfigVariant));
}//ConflictManagementService::insertAPConfigElements

void ConflictManagementService::insertAssociatedStation(const AssociatedStationItem &staItem,
      DBusMessageIter* iterSta)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   DBusMessageIter iterEntry, iterEntryElement, iterStaDet;
   std::string macAddress, interface, ipAddress, apPath, hostname;

   macAddress = staItem.property.macAddress;
   interface = staItem.property.interface;
   ipAddress = staItem.property.ipAddress;
   hostname = staItem.property.hostName;
   apPath = staItem.property.apPath;
   (void) dbus_message_iter_open_container(iterSta, DBUS_TYPE_DICT_ENTRY, NULL, &iterStaDet);
   const char* strPath = staItem.objPath.c_str();
   (void) dbus_message_iter_append_basic(&iterStaDet, DBUS_TYPE_STRING, &strPath);
   (void) dbus_message_iter_open_container(&iterStaDet, DBUS_TYPE_VARIANT, "a{sv}", &iterEntry);
   (void) dbus_message_iter_open_container(&iterEntry, DBUS_TYPE_ARRAY, "{sv}", &iterEntryElement);
   dict_append_entry(&iterEntryElement, "HostName", DBUS_TYPE_STRING, &hostname);
   dict_append_entry(&iterEntryElement, "MacAddress", DBUS_TYPE_STRING, &macAddress);
   dict_append_entry(&iterEntryElement, "Interface", DBUS_TYPE_STRING, &interface);
   dict_append_entry(&iterEntryElement, "IPAddress", DBUS_TYPE_STRING, &ipAddress);
   dict_append_entry(&iterEntryElement, "AccessPoint", DBUS_TYPE_STRING, &apPath);
   (void) dbus_message_iter_close_container(&iterEntry, &iterEntryElement);
   (void) dbus_message_iter_close_container(&iterStaDet, &iterEntry);
   (void) dbus_message_iter_close_container(iterSta, &iterStaDet);

}//ConflictManagementService::insertAssociatedStation

void ConflictManagementService::notifyActiveWiFiSetups(const ::std::vector< ActiveWiFiSetupsStruct >& activeWiFiSetups)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   
   //! Store the new wifi setups list in Stub & trigger update to clients
   //! @Note: ConflictManagementStub::setActiveWiFiSetups() function is not used to set property 
   //! since validation of variant types is not correctly handled in the ASF generated code.
   ::std::vector< ActiveWiFiSetupsStruct >& activeWiFiSetupsList = getActiveWiFiSetupsMutable();
   activeWiFiSetupsList = activeWiFiSetups;
   sendActiveWiFiSetupsUpdate();
}//ConflictManagementService::notifyActiveWiFiSetups

void ConflictManagementService::onDeActivateSetupRequest(const ::boost::shared_ptr < DeActivateSetupRequest >& request)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   ::std::string wifiMode = request->getWiFiMode();
   LOG_DEBUG("onDeActivateSetupRequest: WifiMode = %s", wifiMode.c_str());

   const ::boost::shared_ptr < DeActivateSetupMsg > deactivatesetup(new DeActivateSetupMsg);
   deactivatesetup->setWifiMode(wblWifiMode.getWifiMode2Enum(wifiMode));
   deactivatesetup->setMsgID(WBL_MSG_DEACTIVATE_SETUP_REQ);
   deactivatesetup->setRequestType(WIFI_POWER_OFF_USER_REQUEST);
   deactivatesetup->setMsgToken(static_cast<uint>(request->getAct()));

   ConflictManagerFactory* factory = ConflictManagerFactory::getInstance();
   ICmdConflictManager* conflictManagerIf = (NULL != factory) ? (factory->getConflictManagerRequestIf()) : (NULL);
   if (NULL != conflictManagerIf)
   {
      conflictManagerIf->onProcessRequest(deactivatesetup);
   }
}//ConflictManagementService::onDeActivateSetupRequest

void ConflictManagementService::onDeactivateSetupResponse(WBLErrorCode errcode, uint32 actToken)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::string errname = "";
   ::std::string errmsg = "";

   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendDeActivateSetupResponse(actToken);
   }
   else
   {
      errname = wblErrorCode.getErrorCode2String(errcode);
      sendDeActivateSetupError(errname, errmsg, actToken);
   }
}//ConflictManagementService::onDeactivateSetupResponse

void ConflictManagementService::onWifiPriorityList(const PriorityList& list)
{
   LOG_INFO("ConflictManagementService::onWifiPriorityList");
   ::std::vector< ::std::string > priorityList;
   std::string priorityItem;
   for (auto itr = list.begin(); itr != list.end(); itr++)
   {
      priorityItem = wblWifiMode.getWifiMode2String(*itr);
      priorityList.push_back(priorityItem);
   }
   setWiFiPriorityList(priorityList);
}//ConflictManagementService::onWifiPriorityList

void ConflictManagementService::onWiFiPriorityListSet(const ::boost::shared_ptr < WiFiPriorityListSet >& payload)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::vector< ::std::string > list = payload->getWiFiPriorityList();
   PriorityList prioList;
   int modeValue = WIFI_MODE_UNKNOWN;
   for (auto itr= list.begin(); itr != list.end(); itr++)
   {
      modeValue = wblWifiMode.getWifiMode2Enum(*itr);
      prioList.push_back((WifiMode)modeValue);
   }
   IWBLPriorityManager* _wblPriorityManagerIf = NULL;
   WBLPriorityManagerFactory *_pmFactory = WBLPriorityManagerFactory::getInstance();
   if (NULL != _pmFactory)
   {
      _wblPriorityManagerIf = _pmFactory->getWBLPriorityManagerIf();
   }
   _wblPriorityManagerIf->onPMChangeinModes(prioList);
}//ConflictManagementService::onWiFiPriorityListSet

void ConflictManagementService::onConfigurationTimerSet(const ::boost::shared_ptr < ConfigurationTimerSet >& payload)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   uint timer = payload->getConfigurationTimer();
   IWBLPriorityManager* _wblPriorityManagerIf = NULL;
   WBLPriorityManagerFactory *_pmFactory = WBLPriorityManagerFactory::getInstance();
   if (NULL != _pmFactory)
   {
      _wblPriorityManagerIf = _pmFactory->getWBLPriorityManagerIf();
   }
   _wblPriorityManagerIf->onPMChangeinTimer(timer);
}//ConflictManagementService::onConfigurationTimerSet

void ConflictManagementService::onChangeinConfigurationTimer(const uint16 &timer)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   setConfigurationTimer(timer);
}//ConflictManagementService::onChangeinConfigurationTimer

void ConflictManagementService::onWiFiStationModeUsageRequest(
      const ::boost::shared_ptr < WiFiStationModeUsageRequest >& request)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   WBL_INTENTIONALLY_UNUSED(request);
}//ConflictManagementService::onWiFiStationModeUsageRequest

void ConflictManagementService::onWiFiStationModeUsageResponse(WBLErrorCode errcode, uint32 actToken)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::string errname = "";
   ::std::string errmsg = "";
   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendWiFiStationModeUsageResponse(actToken);
   }
   else
   {
      errname = wblErrorCode.getErrorCode2String(errcode);
      sendWiFiStationModeUsageError(errname, errmsg,actToken);
   }
}//ConflictManagementService::onWiFiStationModeUsageResponse

void ConflictManagementService::onResolveConflictRequest(const ::boost::shared_ptr < ResolveConflictRequest >& request)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   const ::boost::shared_ptr < ResolveConflictMsg > resolveConflict(new ResolveConflictMsg);

   resolveConflict->setConflictId(request->getConflictid());
   resolveConflict->setAction(request->getAction());
   LOG_INFO("ConflictManagementService::onResolveConflictRequest ConflictId:%d",resolveConflict->getConflictId());
   resolveConflict->setMsgToken(static_cast<uint>(request->getAct()));
   resolveConflict->setMsgID(WBL_MSG_RESOLVE_CONFLICT_REQ);
   ConflictManagerFactory* factory = ConflictManagerFactory::getInstance();
   ICmdConflictManager* conflictManagerIf = (NULL != factory) ? (factory->getConflictManagerRequestIf()) : (NULL);
   if (NULL != conflictManagerIf)
   {
      conflictManagerIf->onProcessRequest(resolveConflict);
   }
}//ConflictManagementService::onResolveConflictRequest

void ConflictManagementService::onResolveConflictResponse(WBLErrorCode errcode, uint32 actToken)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   std::string errname = "";
   std::string errmsg = "";
   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendResolveConflictResponse(actToken);
   }
   else
   {
      errname = wblErrorCode.getErrorCode2String(errcode);
      sendResolveConflictError(errname, errmsg, actToken);
   }
}//ConflictManagementService::onResolveConflictResponse

void ConflictManagementService::onConflictDetected(const ConflictDetectedMsg& msg)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::map < std::string, ::asf::dbus::DBusVariant > conflictMap;
   ConflictDetectedItem conflictItem;
   uint32 conflictId;
   msg.getConflictDetectedItem(conflictItem);
   conflictId = conflictItem.conflictID;
   insert_map_element(conflictMap, "ConflictID", DBUS_TYPE_UINT32, &conflictId);
   insertConflictConfigs(conflictMap,conflictItem);
   insertConflictData(conflictMap,conflictItem);
   sendConflictDetectedSignal(conflictMap);
   //conflictMap.clear();
}//ConflictManagementService::onConflictDetected

void ConflictManagementService::insertConflictConfigs(
         ::std::map<std::string, ::asf::dbus::DBusVariant> &conflictMap, const ConflictDetectedItem &conflictItem)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   DBusMessageIter *iterWifiSetup,iterWifiSetupElement;
   ::asf::dbus::DBusVariant reqSetupVariant,curSetupVariant,apConfigVariant;
   std::string strMode,strFreqBand;

   if(conflictItem.reqConfig.mode != WIFI_MODE_UNKNOWN)
   {
      iterWifiSetup = reqSetupVariant.getWriteIterator();
      //open iterWifiSetup
      (void) dbus_message_iter_open_container(iterWifiSetup, DBUS_TYPE_ARRAY,
             DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
             DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
             DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &iterWifiSetupElement);

      strMode = wblWifiMode.getWifiMode2String(conflictItem.reqConfig.mode);
      dict_append_entry(&iterWifiSetupElement, "Mode", DBUS_TYPE_STRING, &strMode);
      if(conflictItem.reqConfig.mode != WIFI_MODE_STA)
      insertRequestedAPConfigData(conflictItem.reqConfig.apConfig,&iterWifiSetupElement);
      //close iterWifiSetup
      (void) dbus_message_iter_close_container(iterWifiSetup, &iterWifiSetupElement);
      conflictMap.insert(std::pair < std::string, ::asf::dbus::DBusVariant >("RequestedFunction", reqSetupVariant));
   }

   if(conflictItem.currConfig.mode != WIFI_MODE_UNKNOWN)
   {
      iterWifiSetup = curSetupVariant.getWriteIterator();
      //open iterWifiSetup
      (void)dbus_message_iter_open_container(iterWifiSetup, DBUS_TYPE_ARRAY,
            DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
            DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
            DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &iterWifiSetupElement);

      strMode = wblWifiMode.getWifiMode2String(conflictItem.currConfig.mode);
      dict_append_entry(&iterWifiSetupElement, "Mode", DBUS_TYPE_STRING, &strMode);
      strFreqBand = wblFrequency.getFrequency2String(conflictItem.currConfig.frequency);
      dict_append_entry(&iterWifiSetupElement, "Frequency", DBUS_TYPE_STRING, &strFreqBand);
      //close iterWifiSetup
      (void)dbus_message_iter_close_container(iterWifiSetup, &iterWifiSetupElement);
      conflictMap.insert(std::pair < std::string, ::asf::dbus::DBusVariant >("CurrentFunctions", curSetupVariant));
   }
}//ConflictManagementService::insertConflictConfigs

void ConflictManagementService::insertRequestedAPConfigData(const APConfig &apConfig,DBusMessageIter* iterAPConfig)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   DBusMessageIter iterAp,iterApElement,iterConfig;
   uint index = 0;
   dbus_bool_t visibility,isSeperate;
   std::string security, passphrase, type, frequency;
   uint32 stationsToBeReserved;

   (void) dbus_message_iter_open_container(iterAPConfig, DBUS_TYPE_DICT_ENTRY, NULL, &iterAp);

   const char* strApConfig = "APConfig";
   (void) dbus_message_iter_append_basic(&iterAp, DBUS_TYPE_STRING, &strApConfig);

   (void) dbus_message_iter_open_container(&iterAp, DBUS_TYPE_VARIANT, "a{sv}", &iterApElement);
   (void) dbus_message_iter_open_container(&iterApElement, DBUS_TYPE_ARRAY, "{sv}", &iterConfig);

   if(apConfig.isTypeValid)
   {
      type = wblAPType.getAPType2String(apConfig.type);
      dict_append_entry(&iterConfig, "Type", DBUS_TYPE_STRING, &type);
   }
   if(apConfig.isSSIDValid)
   {
      uint8* ssid = (uint8*) getMemAlloc(apConfig.ssid.SSID.size());
      for (std::vector < uint8 >::const_iterator it = apConfig.ssid.SSID.begin(); it != apConfig.ssid.SSID.end(); ++it, index++)
         ssid[index] = *it;
      const char* strSSID = "SSID";
      append_dict_entry_fixed_array(&iterConfig, strSSID, &ssid, static_cast<uint>(apConfig.ssid.SSID.size()));
   }
   if(apConfig.isPassphraseValid)
   {
      passphrase = apConfig.passphrase;
      dict_append_entry(&iterConfig, "Passphrase", DBUS_TYPE_STRING, &passphrase);
   }
   if(apConfig.isSecurityValid)
   {
      security = wblSecurity.getSecurity2String(apConfig.security);
      dict_append_entry(&iterConfig, "Security", DBUS_TYPE_STRING, &security);
   }
   if(apConfig.isVisibilityValid)
   {
      visibility = apConfig.isVisible;
      dict_append_entry(&iterConfig, "Visible", DBUS_TYPE_BOOLEAN, &visibility);
   }
   if(apConfig.isFrequencyValid)
   {
      frequency = wblFrequency.getFrequency2String(apConfig.frequency);
      dict_append_entry(&iterConfig, "Frequency", DBUS_TYPE_STRING, &frequency);
   }
   if(apConfig.isStationsToReserveValid)
   {
      stationsToBeReserved = apConfig.stationsToReserve;
      dict_append_entry(&iterConfig, "StationsToBeReserved", DBUS_TYPE_UINT32, &stationsToBeReserved);
   }
   if(apConfig.isSeparateAPValid)
   {
      isSeperate = apConfig.isSeparateAP;
      dict_append_entry(&iterConfig, "SeperateAP",DBUS_TYPE_BOOLEAN, &isSeperate);
   }
   //close iterConfigElement
   (void) dbus_message_iter_close_container(&iterApElement, &iterConfig);
   (void) dbus_message_iter_close_container(&iterAp, &iterApElement);
   (void) dbus_message_iter_close_container(iterAPConfig, &iterAp);
}//ConflictManagementService::insertConflictConfigs

void ConflictManagementService::insertConflictData(::std::map<std::string, ::asf::dbus::DBusVariant> &conflictMap,
         const ConflictDetectedItem &conflictItem)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   DBusMessageIter *iterConflicter,iterConflictElement;
   ::asf::dbus::DBusVariant conflictVariant;
   std::string strIssue,strDetail;
   uint16 stationsToFree;

   iterConflicter = conflictVariant.getWriteIterator();
   //open iterConflicter
   (void) dbus_message_iter_open_container(iterConflicter, DBUS_TYPE_ARRAY,
          DBUS_DICT_ENTRY_BEGIN_CHAR_AS_STRING
          DBUS_TYPE_STRING_AS_STRING DBUS_TYPE_VARIANT_AS_STRING
          DBUS_DICT_ENTRY_END_CHAR_AS_STRING, &iterConflictElement);
   strIssue = wblConflictType.getConflictType2String(conflictItem.conflictType);
   dict_append_entry(&iterConflictElement, "Issue", DBUS_TYPE_STRING, &strIssue);
   strDetail = wblConflictType.conflictIssueDesList.find(conflictItem.conflictType)->second;
   dict_append_entry(&iterConflictElement, "IssueDescription", DBUS_TYPE_STRING, &strDetail);
   stationsToFree = conflictItem.numStaToFree;
   dict_append_entry(&iterConflictElement, "StationsToBeFreed", DBUS_TYPE_UINT16, &stationsToFree);
   //close iterConflicter
   (void) dbus_message_iter_close_container(iterConflicter, &iterConflictElement);
   conflictMap.insert(std::pair < std::string, ::asf::dbus::DBusVariant >("Conflict", conflictVariant));
}//ConflictManagementService::insertConflictDetail

void ConflictManagementService::onConflictResolved(const ConflictResolvedMsg& msg)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::map < std::string, ::asf::dbus::DBusVariant > conflictResolvedMap;
   ConflictResolvedItem item;
   uint32 conflictId;
   msg.getConflictResolvedItem(item);
   conflictId = item.conflictID;
   insert_map_element(conflictResolvedMap, "ConflictID", DBUS_TYPE_UINT32, &conflictId);
   std::string strConflict = wblConflictType.getConflictType2String(item.conflictType);
   insert_map_element(conflictResolvedMap, "Conflict", DBUS_TYPE_STRING, &strConflict);
   std::string strResolution = wblResolutionType.getResolution2String(item.resType);
   insert_map_element(conflictResolvedMap, "Resolution", DBUS_TYPE_STRING, &strResolution);
   sendConflictResolvedSignal(conflictResolvedMap);
}//ConflictManagementService::onConflictResolved

void ConflictManagementService::onWBLServiceResponse(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("ConflictManagementService::%s", __func__);

   WBLBaseMsgID msgID = msg->getMsgID();
   WBLErrorCode errCode = msg->getErrorCode();
   uint32 actToken = msg->getMsgToken();

   if(!actToken)
      return;

   switch (msgID)
   {
      case WBL_MSG_PREPARE_SETUP_REQ:
         onPrepareSetupResponse(errCode, actToken);
      break;
      case WBL_MSG_DEACTIVATE_SETUP_REQ:
         onDeactivateSetupResponse(errCode, actToken);
      break;
      case WBL_MSG_RESOLVE_CONFLICT_REQ:
         onResolveConflictResponse(errCode, actToken);
      break;
      default:
         LOG_ERROR("Unhandled WBL Response");
      break;
   }
}//ConflictManagementService::onWBLServiceResponse

void ConflictManagementService::clearEmptyProperties(::std::map< ::std::string, ::asf::dbus::DBusVariant >& propertiesList)
{
   LOG_INFO("ConflictManagementService::clearEmptyProperties Entered");
   if(getActiveWiFiSetups().empty())
      propertiesList.erase(propertiesList.find("ActiveWiFiSetups"));
   if(getWiFiPriorityList().empty())
      propertiesList.erase(propertiesList.find("WiFiPriorityList"));
}//ConflictManagementService::clearEmptyProperties

void ConflictManagementService::getAllProperties(::std::map< ::std::string, ::asf::dbus::DBusVariant >& propertiesMap)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::map< ::std::string, ::asf::dbus::DBusVariant > propList;
   _stubDelegate->getAllProperties(propList);
   propertiesMap.insert(propList.begin(),propList.end());
   clearEmptyProperties(propertiesMap);
}//ConflictManagementService::getAllProperties

void ConflictManagementService::getDBusObjectPath(::std::string& objPath)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   objPath = _stubDelegate->getDBusObjectPath();
}//ConflictManagementService::getDBusObjectPath

void ConflictManagementService::getInterfaceName(::std::string& interfaceName)
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   interfaceName = _stubDelegate->getInterfaceName();
}//ConflictManagementService::getInterfaceName

void ConflictManagementService::updateInterfaceAdded()
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   PropertiesList propertiesList;
   ::std::string interfacename;
   ::std::string objpath;
   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   ObjectManagerService* objectManager = serviceFactory->getObjectManagerServiceIf();
   getDBusObjectPath(objpath);
   getInterfaceName(interfacename);
   getAllProperties(propertiesList);
   objectManager->updateInterfaceAddedSignal(objpath,interfacename,propertiesList);
}//ConflictManagementService::updateInterfaceAdded()

void ConflictManagementService::updateInterfaceRemoved()
{
   LOG_INFO("ConflictManagementService::%s", __func__);
   ::std::vector< ::std::string> interface;
   ::std::string interfacename;
   ::std::string objpath;
   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   ObjectManagerService* objectManager = serviceFactory->getObjectManagerServiceIf();
   getDBusObjectPath(objpath);
   getInterfaceName(interfacename);
   interface.push_back(interfacename);
   objectManager->updateInterfaceRemovedSignal(objpath,interface);
}//ConflictManagementService::updateInterfaceRemoved()

} //namespace bosch
} //namespace org

/** @} */
