/**
 * @file WpaSupplicantIfaceProxyManager.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include "asf/core/Proxy.h"
#include "asf/core/Logger.h"
#include <dbus/dbus.h>
#include "WBLPortsDefines.h"
#include "WpaSupplicantIfaceProxyManager.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", cWpaSupplicantIfaceProxyManager, Info);

cWpaSupplicantIfaceProxyManager::cWpaSupplicantIfaceProxyManager()
{
}

cWpaSupplicantIfaceProxyManager::~cWpaSupplicantIfaceProxyManager()
{
}

int cWpaSupplicantIfaceProxyManager::createProxy(const ::std::string &portName, const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const eProxyType eType)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cWpaSupplicantIfaceProxy> oProxy;

	if (busName.empty() || objPath.empty()
			|| portName.empty() || PROXY_TYPE_INVAL == eType)
		return -EINVAL;

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (m_oProxyManager.isProxyAvailable (oProxy, oProxyData))
		return -EEXIST;

	LOG_INFO ("Creating Proxy to the dbus remote object [PortName: %s]"
			"[busName: %s] [objPath: %s]", portName.c_str(), busName.c_str(),
			objPath.c_str());

	if (PROXY_TYPE_STATIC == eType)
		oProxy = cWpaSupplicantIfaceProxy::createProxy(portName, *this);
	else if (PROXY_TYPE_WIRED_AT_RUNTIME == eType)
		oProxy = cWpaSupplicantIfaceProxy::createProxy(portName, busName, objPath, busType, *this);

	m_oProxyManager.addProxyInstance(oProxyData, oProxy);
	return 0;
}

int cWpaSupplicantIfaceProxyManager::destroyProxy(const eProxyDestroy eType, const ::std::string &busName,
		const ::DBusBusType &busType, const ::std::string objPath)
{
	unsigned int iIndex = 0;
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cWpaSupplicantIfaceProxy> oProxy;

	if (PROXY_DESTROY_INVAL == eType || (PROXY_DESTROY_ONE == eType &&
			objPath.empty()) || (PROXY_DESTROY_ALL == eType && !objPath.empty()))
		return -EINVAL;

	LOG_INFO ("Destroying the proxy to : %s", (eType == PROXY_DESTROY_ONE) ?
			objPath.c_str() : "All Objects");

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (PROXY_DESTROY_ONE == eType) {
		if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
			return -ENOENT;
		oProxy->sendDeregisterAll();
		m_oProxyManager.removeProxyInstance(oProxyData);
	}
	else if (PROXY_DESTROY_ALL == eType) {
		for( ; iIndex < m_oProxyManager.getNumberOfProxyInstances(); ++iIndex) {
			oProxy = m_oProxyManager[iIndex].proxy;
			oProxy->sendDeregisterAll();
		}
		m_oProxyManager.resetAllProxiesAndClear();
	}

	return 0;
}

int cWpaSupplicantIfaceProxyManager::setCallbackIf(const ::std::string& busName,
		const ::std::string& objPath, const ::DBusBusType busType, cIWpaSupplicantIfaceNotifCallbackIf *cb)
{
	ProxyMetadata oProxyData;

	if (busName.empty() || objPath.empty())
		return -EINVAL;

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	LOG_INFO ("%s the callback %p to the dbus remote object [busName: %s]"
				"[objPath: %s]", cb ? "Adding" : "Removing", cb, busName.c_str(), objPath.c_str());

	if (!cb) {
		m_pProxyCbManager.removeCallbackIf(oProxyData);
	} else {
		if (m_pProxyCbManager.bIsCallbackAvailable(oProxyData, cb))
			return -EEXIST;
		m_pProxyCbManager.addCallbackIf(oProxyData, cb);
	}

	return 0;
}

void cWpaSupplicantIfaceProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cWpaSupplicantIfaceProxy > oProxy;
	std::vector<cIWpaSupplicantIfaceNotifCallbackIf*> cblist;
	std::vector<cIWpaSupplicantIfaceNotifCallbackIf*>::iterator it;

	oProxy = ::boost::static_pointer_cast< cWpaSupplicantIfaceProxy > (proxy);

	LOG_INFO ("Service Available for the Proxy: [PortName: %s], "
			"[busName: %s], [ObjPath: %s], [IfName: %s]", oProxy->getPortName().c_str(),
			oProxy->getDBusBusName().c_str(), oProxy->getDBusObjectPath().c_str(),
			oProxy->getInterfaceName().c_str());

	oProxyData.busName = oProxy->getDBusBusName();
	oProxyData.busType = oProxy->getBusType();
	oProxyData.objPath = oProxy->getDBusObjectPath();

	cblist = m_pProxyCbManager.getAllCallbackIf(oProxyData);
	for (it = cblist.begin(); it != cblist.end(); ++it) {
		LOG_INFO ("wpa_supplicant interface: Registering the callbacks for: %p", *it);
		oProxy->sendBSSAddedRegister(**it);
		oProxy->sendBSSRemovedRegister(**it);
		(*it)->onWpaSupplicantIfaceServiceAvailable(oProxyData.busName, oProxyData.objPath,
			oProxyData.busType, stateChange.getPreviousState(), stateChange.getCurrentState());
	}
}

void cWpaSupplicantIfaceProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cWpaSupplicantIfaceProxy > oProxy;
	std::vector<cIWpaSupplicantIfaceNotifCallbackIf*> cblist;
	std::vector<cIWpaSupplicantIfaceNotifCallbackIf*>::iterator it;

	oProxy = ::boost::static_pointer_cast< cWpaSupplicantIfaceProxy > (proxy);

	LOG_INFO ("Service Unavailable for the Proxy: [PortName: %s], "
			"[busName: %s], [ObjPath: %s], [IfName: %s]", oProxy->getPortName().c_str(),
			oProxy->getDBusBusName().c_str(), oProxy->getDBusObjectPath().c_str(),
			oProxy->getInterfaceName().c_str());

	oProxy->sendDeregisterAll();

	oProxyData.busName = oProxy->getDBusBusName();
	oProxyData.busType = oProxy->getBusType();
	oProxyData.objPath = oProxy->getDBusObjectPath();

	cblist = m_pProxyCbManager.getAllCallbackIf(oProxyData);
	for (it = cblist.begin(); it != cblist.end(); ++it) {
		LOG_INFO ("Triggering Unavailable notification for: %p", *it);
		(*it)->onWpaSupplicantIfaceServiceUnavailable(oProxyData.busName, oProxyData.objPath,
			oProxyData.busType, stateChange.getPreviousState(), stateChange.getCurrentState());
	}
}

act_t cWpaSupplicantIfaceProxyManager::sendDisconnectRequest(const ::std::string &busName, const ::std::string &objPath,
			const ::DBusBusType busType, cWpaDisconnectCallbackIF &cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cWpaSupplicantIfaceProxy > oProxy;

	if (busName.empty() || objPath.empty())
		return DEFAULT_ACT;

	LOG_INFO ("[busName: %s] [objPath: %s] Disconnect",
			busName.c_str(), objPath.c_str());

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return DEFAULT_ACT;

	return oProxy->sendDisconnectRequest(cb);
}

	}
}

/** @} */
