/**
 * @file WpaSupplicantBssProxyManager.h
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include "asf/core/Proxy.h"
#include "asf/core/Logger.h"
#include <dbus/dbus.h>
#include "WBLPortsDefines.h"
#include "WpaSupplicantBssProxyManager.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", cWpaSupplicantBssProxyManager, Info);

cWpaSupplicantBssProxyManager::cWpaSupplicantBssProxyManager()
{
}

cWpaSupplicantBssProxyManager::~cWpaSupplicantBssProxyManager()
{
}

int cWpaSupplicantBssProxyManager::createProxy(const ::std::string &portName, const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const eProxyType eType)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cBSSProxy> oProxy;

	if (busName.empty() || objPath.empty()
			|| portName.empty() || PROXY_TYPE_INVAL == eType)
		return -EINVAL;

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (m_oProxyManager.isProxyAvailable (oProxy, oProxyData))
		return -EEXIST;

	LOG_INFO ("Creating Proxy to the dbus remote object [PortName: %s]"
			"[busName: %s] [objPath: %s]", portName.c_str(), busName.c_str(),
			objPath.c_str());

	if (PROXY_TYPE_STATIC == eType)
		oProxy = cBSSProxy::createProxy(portName, *this);
	else if (PROXY_TYPE_WIRED_AT_RUNTIME == eType)
		oProxy = cBSSProxy::createProxy(portName, busName, objPath, busType, *this);

	m_oProxyManager.addProxyInstance(oProxyData, oProxy);
	return 0;
}

int cWpaSupplicantBssProxyManager::destroyProxy(const eProxyDestroy eType, const ::std::string &busName,
		const ::DBusBusType &busType, const ::std::string objPath)
{
	unsigned int iIndex = 0;
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cBSSProxy> oProxy;

	if (PROXY_DESTROY_INVAL == eType || (PROXY_DESTROY_ONE == eType &&
			objPath.empty()) || (PROXY_DESTROY_ALL == eType && !objPath.empty()))
		return -EINVAL;

	LOG_INFO ("Destroying the proxy to : %s", (eType == PROXY_DESTROY_ONE) ?
			objPath.c_str() : "All Objects");

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (PROXY_DESTROY_ONE == eType) {
		if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
			return -ENOENT;
		oProxy->sendDeregisterAll();
		m_oProxyManager.removeProxyInstance(oProxyData);
	}
	else if (PROXY_DESTROY_ALL == eType) {
		for( ; iIndex < m_oProxyManager.getNumberOfProxyInstances(); ++iIndex) {
			oProxy = m_oProxyManager[iIndex].proxy;
			oProxy->sendDeregisterAll();
		}
		m_oProxyManager.resetAllProxiesAndClear();
	}

	return 0;
}

int cWpaSupplicantBssProxyManager::setCallbackIf(const ::std::string& busName,
		const ::std::string& objPath, const ::DBusBusType busType, cIWpaSupplicantBssNotifCallbackIf *cb)
{
	ProxyMetadata oProxyData;

	if (busName.empty() || objPath.empty())
		return -EINVAL;

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	LOG_INFO ("%s the callback %p to the dbus remote object [busName: %s]"
				"[objPath: %s]", cb ? "Adding" : "Removing", cb, busName.c_str(), objPath.c_str());

	if (!cb) {
		m_pProxyCbManager.removeCallbackIf(oProxyData);
	} else {
		if (m_pProxyCbManager.bIsCallbackAvailable(oProxyData, cb))
			return -EEXIST;
		m_pProxyCbManager.addCallbackIf(oProxyData, cb);
	}

	return 0;
}

void cWpaSupplicantBssProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cBSSProxy > oProxy;
	std::vector<cIWpaSupplicantBssNotifCallbackIf*> cblist;
	std::vector<cIWpaSupplicantBssNotifCallbackIf*>::iterator it;

	oProxy = ::boost::static_pointer_cast< cBSSProxy > (proxy);

	LOG_INFO ("Service Available for the Proxy: [PortName: %s], "
			"[busName: %s], [ObjPath: %s], [IfName: %s]", oProxy->getPortName().c_str(),
			oProxy->getDBusBusName().c_str(), oProxy->getDBusObjectPath().c_str(),
			oProxy->getInterfaceName().c_str());

	oProxyData.busName = oProxy->getDBusBusName();
	oProxyData.busType = oProxy->getBusType();
	oProxyData.objPath = oProxy->getDBusObjectPath();

	cblist = m_pProxyCbManager.getAllCallbackIf(oProxyData);
	for (it = cblist.begin(); it != cblist.end(); ++it) {
		(*it)->onWpaSupplicantBssServiceAvailable(oProxyData.busName, oProxyData.objPath,
			oProxyData.busType, stateChange.getPreviousState(), stateChange.getCurrentState());
	}
}

void cWpaSupplicantBssProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cBSSProxy > oProxy;
	std::vector<cIWpaSupplicantBssNotifCallbackIf*> cblist;
	std::vector<cIWpaSupplicantBssNotifCallbackIf*>::iterator it;

	oProxy = ::boost::static_pointer_cast< cBSSProxy > (proxy);

	LOG_INFO ("Service Unavailable for the Proxy: [PortName: %s], "
			"[busName: %s], [ObjPath: %s], [IfName: %s]", oProxy->getPortName().c_str(),
			oProxy->getDBusBusName().c_str(), oProxy->getDBusObjectPath().c_str(),
			oProxy->getInterfaceName().c_str());

	oProxy->sendDeregisterAll();

	oProxyData.busName = oProxy->getDBusBusName();
	oProxyData.busType = oProxy->getBusType();
	oProxyData.objPath = oProxy->getDBusObjectPath();

	cblist = m_pProxyCbManager.getAllCallbackIf(oProxyData);
	for (it = cblist.begin(); it != cblist.end(); ++it) {
		LOG_INFO ("Triggering Unavailable notification for: %p", *it);
		(*it)->onWpaSupplicantBssServiceUnAvailable(oProxyData.busName, oProxyData.objPath,
			oProxyData.busType, stateChange.getPreviousState(), stateChange.getCurrentState());
	}
}

	}
}

/** @} */
